;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  gps_eph_panel, plotinfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      plotinfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from plotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(plotinfo, ACTION=, GROUP=)
;*
;* MODIFICATION HISTORY:       
;*      written july 2002, Reiner Friedel
;******************************************************************************
FUNCTION  gps_eph_panel, PlotInfo, GROUP=group, ACTION=ACTION

COMMON gps_eph_panel_data, widgetData
COMMON gps_eph

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(gps_eph_control)
control = *plotinfo.USR_PTR1
;control now contains the user defiend control structure for this module.

;--- create the base-window ---------------------------------------------------
title = PlotInfo.panelKind + ' - ' + papco_module_version('gps_eph')
panel_base=papco_WIDGET_BASE(TITLE=title, ROW=1, GROUP_LEADER=group, /MODAL)
panel_left=widget_base(panel_base,COLUMN=1)
panel_right=widget_base(panel_base,COLUMN=1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;--- satellite choice -- sat_id is set in lanl_gps_set_commons ----------------
sats_base=CW_BGROUP(panel_left, sat_id, /FRAME, ROW=1, /EXCLUSIVE, $
                    LABEL_TOP = 'Select GPS satellite ID',  $
                    UVALUE='mpb_sats' ,$
                    SET_VALUE=control.sc_id)

;--- buttons to chose limits behaviour - plot line or average -----------------
opti_base=WIDGET_BASE(panel_left, ROW=1)

average_names=['seg.','av.']
average_base=CW_BGROUP(opti_base, average_names, /FRAME, ROW=1, /EXCLUSIVE, $
                       UVALUE='spb_average', $
                       SET_VALUE=control.seg_av)

reload_names=['Relaod', 'Adopt']
reload_base=CW_BGROUP(opti_base, reload_names, /FRAME, ROW=1, /EXCLUSIVE, $
                      UVALUE='spb_reload', $
                      SET_VALUE=control.reload)

;--- button to chose "select" variable ----------------------------------------
filter_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
filter_names = 'select ' + select_names
filter_base=CW_BGROUP(filter_outer, filter_names, ROW=1, /EXCLUSIVE, $
                  LABEL_TOP='Select filter variable', $
                  UVALUE='spb_filter', SET_VALUE=control.filter)

;--- Add sliders to filter data - L, MLT, MLAT, etc ---------------------------
;    Multiple map bases
select_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)

;--- make select widgets for all variables in "filter_names" ------------------
;    use papco_panel compound widget for this!

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L', $
            control.L_SELECT_RANGE, $
            control.L_SELECT_FROM_VAL, $
            control.L_SELECT_TO_VAL, 4.0, 20.0, 0.1, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L_STAR', $
            control.L_STAR_SELECT_RANGE, $
            control.L_STAR_SELECT_FROM_VAL, $
            control.L_STAR_SELECT_TO_VAL, 4.0, 20.0, 0.1, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLT', $
            control.MLT_SELECT_RANGE, $
            control.MLT_SELECT_FROM_VAL, $
            control.MLT_SELECT_TO_VAL, 0.0, 24.0, 0.5, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLAT', $
            control.MLAT_SELECT_RANGE, $
            control.MLAT_SELECT_FROM_VAL, $
            control.MLAT_SELECT_TO_VAL, -60, 60.0, 2, '(f5.1)')
additionalWData=create_struct(widgets, additionalWData)

;-- select orginal or model coords - ------------------------------------------
names = ['Original Ephemeris', 'Mag Model Coords']
orig_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
orig_base = CW_BGROUP(orig_outer, names , ROW=1, /EXCLUSIVE, $
                   LABEL_LEFT='Coord Source:', $
                   UVALUE='spb_orig', $
                   SET_VALUE=control.orig)

;--- choose an ephemeris variable to plot -------------------------------------
eph_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
spb_eph=CW_BGROUP(eph_outer, eph_names, COLUMN=7,/EXCLUSIVE, $
                  LABEL_TOP='Select ephemeris variable', $
                  UVALUE='spb_eph', SET_VALUE=control.product)
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
;  widgets=papco_panel_Addcolorbar_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(panel_left, control)
additionalWData=create_struct(widgets, additionalWData)

;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_right, plotinfo, 0,  /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
;  widgets=papco_panel_AddZScale_Widgets(panel_right, plotinfo,/VERTICAL)
;  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_right, Action, plotinfo, $
                                     /CONFIG, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the gps_eph example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'gps_eph_panel.help', $
                 eph_outer:eph_outer}

; add control struct, papco_panel structs
widgetData=papco_PANEL_CreateWidgetData(control, thisWidgetsData)
widgetData=papco_PANEL_CreateWidgetData(widgetData, additionalWData)

;--- realize the window -------------------------------------------------------
gps_eph_panel_refresh, /ALL
papco_Panel_Realize, panel_base, widgetData
papco_xmanager, 'gps_eph_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified }

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)
   
END


;******************************************************************************
;* PROCEDURE:     
;*      PRO gps_eph_panel_refresh 
;* 
;* DESCRIPTION:  
;*      Takes refreesh actions on panel widgets controlled by keywords.
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      gps_eph_panel_refresh, /keywords
;******************************************************************************
PRO gps_eph_panel_refresh, ALL=ALL, EPH_LIMITS = EPH_LIMITS, COORDS = COORDS
  
COMMON gps_eph_panel_data, widgetData
  
IF keyword_set(ALL) THEN BEGIN 
    EPH_LIMITS = 1 & COORDS = 1
ENDIF 

; --- Filter Variable multiple bases ------------------------------------------
IF keyword_set(EPH_LIMITS) THEN BEGIN
    wid_ids = [widgetData.l_select_base, widgetData.l_star_select_base, $
               widgetData.mlt_select_base, widgetData.mlat_select_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid) 
    map(widgetData.filter) = 1 
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
ENDIF 

;--- Turn on/off orig/model bases --------------------------------------------
IF keyword_set(COORDS) THEN BEGIN
    IF widgetData.orig EQ 0 THEN BEGIN 
        widget_control, widgetData.eph_outer,  SENSITIVE = 1
        widget_control, widgetData.mag_model_outer_base,  SENSITIVE = 0
    ENDIF ELSE BEGIN
        widget_control, widgetData.eph_outer,  SENSITIVE = 0
        widget_control, widgetData.mag_model_outer_base,  SENSITIVE = 1
    ENDELSE 
END

END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO gps_eph_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO gps_eph_panel_event, event
  
COMMON gps_eph_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

;test uvalue for structure - select variable sliders return structure uvals.
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval
    uval = uval.uvalue
ENDIF

CASE uval OF

    'mpb_sats'    : widgetData.sc_id = event.value
    'spb_average' : widgetData.seg_av = event.value
    'spb_reload'  : widgetData.reload = event.value
    'spb_eph' : widgetData.product = event.value
    'spb_filter' : BEGIN 
        widgetData.filter = event.value
        gps_eph_panel_refresh, /EPH_LIMITS
    END 
    'spb_orig': BEGIN
        widgetData.orig = event.value
        gps_eph_panel_refresh, /COORDS
    END    
    ELSE:  BEGIN
        widgetData=PAPCO_panel_Event(event, widgetData)
    END 
   
ENDCASE
    
END
