;******************************************************************************
;*  NAME:        r_gps_eph
;*
;*  DESCRIPTION: Routine to read gps_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       satid   the gps satellite id
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        November 1999
;*
;*  AUTOR:       R. Friedel
;*
;******************************************************************************
pro r_gps_eph, plotinfo, VERBOSE=VERBOSE, INTMOD=INTMOD, EXTMOD=EXTMOD

COMMON gps_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verb=VERBOSE else verb=0     

; clear data arrays for reuse
input_header=0  &  input_data=0   
  
; loop through from start to end time concatonating data files into one array. 
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
  
WHILE mjdt_end.mjd GE time.mjd do BEGIN 

    ;read one day of data
    r_gps_eph_onefile, plotinfo, time, VERBOSE=VERB, $
                                       INTMOD=INTMOD, EXTMOD=EXTMOD
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    ;if get_err_no ne 0 then begin      
    ;    c_url=papco_getenv('GPS_EPH_FTP_ARCHIVE')
    ;    c_file=fln+'.idl'  &  c_download_dir=path
    ;    papco_get_remote_data, c_url, c_file, c_download_dir
    ;    ;check if file "arrived" and then read it
    ;    if get_err_no eq 0 then begin
    ;       r_gps_eph_onefile, path+fln, VERBOSE=VERBOSE
    ;    endif else begin
    ;        get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
    ;    ENDELSE  
    ;ENDIF    

    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
    endif else begin            ;otherwise concatonate arrays
        ;structures can have different tags. Adopt and fold into tags of
        ;initial structure!
        ndat = n_elements(input_data)
        new_input_data = replicate(input_data_buf(0), ndat)
        FOR i = 0, ndat-1 DO BEGIN
            dummy = new_input_data(i)
            struct_assign, input_data(i), dummy
            new_input_data(i) = dummy
        ENDFOR 
        input_data_buf=[input_data_buf,new_input_data]
      endelse        
      input_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    time.mjd=time.mjd+1                 ;next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then break  
endwhile         
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    message, get_err_msg, /cont
endelse      

end

;******************************************************************************
;*  NAME:        r_gps_eph_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_gps_eph_onefile, plotinfo, time, VERBOSE=VERBOSE, NO_ORIG = NO_ORIG, $
                       INTMOD=INTMOD, EXTMOD=EXTMOD
  
COMMON gps_eph_data, input_header, input_data
COMMON lanl_gps_data, lanl_gps_header, lanl_gps_data
COMMON get_error, get_err_no, get_err_msg
COMMON gps_eph_1, gps_eph_sat_id, gps_eph_names, gps_eph_select_names, $
                  gps_eph_control
COMMON lanl_gps, sat_id, energy_names_1, energy_names_2, energy_names_3, $
    energy_names_41, energy_names_54, $
    fit_energy_names_1, fit_energy_names_2, singles_names_1, energy_names, $
    eph_names, style_names, select_names, energy_intcal, gain, energy_ch, $
    flux_cal_names, coherent_energy, lanl_gps_control

;keep a coomon block of the first gps_eph structure read in a multiple
;day read. The make sure the next days read have all the same model
;structure tags, make new ones if needed!

get_err_no=0  &  get_err_msg='gps_eph onefile read succesfully'

; check keywords and set
if keyword_set(VERBOSE) then VERBOSE=1 else VERBOSE=0

; check env. variable pointing to data directory
if not (papco_check_data_env('GPS_EPH_DATA', PATH=path)) then return

control = *plotinfo.USR_PTR1
intmod = control.INTERNAL_MODEL
extmod = control.EXTERNAL_MODEL
sat_name = gps_eph_sat_id(control.SC_ID)

;assemble the data file from the time range requested
convert_t90_to_date, time, yr, doy, hr, min, sec, mon, cmon, dom, /mjdt   
datestr=string(yr, mon, dom, format='(i4.4,i2.2,i2.2)')
flname = path+sat_name+papco_get_pathsep()+datestr+'_'+sat_name+'.eph'
gps_eph_struct_flname = path+sat_name+papco_get_pathsep()+'gps_eph_struct.idl'

; clear data arrays for reuse
input_header=0  &  input_data=0   
   
new_struct_name='MODEL_'+varprt(intmod)+'_'+varprt(extmod)
new_struct=create_struct(name='UNILIB_3', 'L',0.0,'LSTAR', 0.0, $
                         'MLT',0.0,'MLAT',0.0, $
                         'B_GSM', fltarr(4), 'B_EQU', 0.0)

;read existing file if it exists
filethere=findfile(flname, count = c)

IF c EQ 0 THEN BEGIN
    get_err_no=2 & get_err_msg='File '+flname+' not found'
    message, 'File not found', /cont
    print, '  ', flname
    ;then read original data, extract ephemeris
    r_gps_eph_onefile_original, plotinfo, time, flname

stop


    new_plotinfo = papco_getplotinfostruct()
    new_plotinfo.USR_PTR1 = ptr_new(lanl_gps_control)
      new_control = *new_plotinfo.USR_PTR1
      new_control.sc_id = control.SC_ID
    new_plotinfo.USR_PTR1 = ptr_new(new_control)
    r_lanl_gps, new_plotinfo, sat_name, VERBOSE=VERBOSE

    IF get_err_no EQ 0 THEN BEGIN 
        ;limit data down to one day
        tai_1=utc2tai({mjd:time.mjd, time:0.0})
        tai_2=utc2tai({mjd:time.mjd, time:86400.0*1000})
        index=where((lanl_gps_data.time ge tai_1) AND $
                    (lanl_gps_data.time le tai_2),c)
        IF c NE 0 THEN lanl_gps_data = lanl_gps_data(index) ELSE BEGIN
            get_err_msg =  'no data for timerange'
            get_err_no = 1
            message, get_err_msg, /CONT
            return
        ENDELSE 
        ndat=n_elements(lanl_gps_data)

        ;make new structures for ephemeris data only
        dummy = {TIME:0.0d}
        FOR i = 0, n_elements(gps_eph_names)-2 DO $
          dummy=create_struct(dummy, gps_eph_names(i), 0.0)
        dummy = create_struct(name='GPS_EPHEMERISa',dummy)
        input_data=replicate(dummy, ndat)                   
        FOR i = 0, n_elements(gps_eph_names)-1 DO $
          input_data.(i) = lanl_gps_data.(i)

        ;save the eph data to file
        message, 'Writing original', /cont
        print, '  ', flname
        save, file = flname, input_header, input_data, /COMPRESS

        ;if no gps_eph_struct exists, write the original one
        result = findfile(gps_eph_struct_flname, count = c)
        IF c EQ 0 THEN BEGIN
            gps_eph_struct = input_data(0)
            save, file = gps_eph_struct_flname, gps_eph_struct
        ENDIF
          
    ENDIF ELSE BEGIN
        return
    ENDELSE     
ENDIF ELSE BEGIN
    restore, flname
    message, 'idl saveset read succesfully', /cont
    print, '  ', flname
ENDELSE

;read current gps data structure from file
restore, file = path+gps_eph_sat_id(control.SC_ID)+papco_get_pathsep()+ $
                'gps_eph_struct.idl'
need_tags = tag_names(gps_eph_struct)
tags=tag_names(input_data(0))

;now, see if we have d any missing tags!
n = 0 & todo_tags = ''
FOR i = 0, n_elements(need_tags)-1 DO BEGIN 
    idx = where(need_tags(i) EQ tags, c)
    IF c EQ 0 THEN BEGIN
        IF (n EQ 0) THEN todo_tags = need_tags(i) $
        ELSE todo_tags = [todo_tags, need_tags(i)]
        n = n+1
    ENDIF 
ENDFOR 

IF control.orig EQ 1 THEN begin
    ;now see if we have to add the currently selected mag field too.
    idx = where(new_struct_name EQ tags, c)
    IF c EQ 0 THEN BEGIN
        idx = where(new_struct_name EQ todo_tags, c)
        IF c EQ 0 THEN BEGIN 
            IF (n EQ 0) THEN todo_tags = new_struct_name $
            ELSE todo_tags = [todo_tags, new_struct_name]
            n = n+1
        ENDIF 
        m_new_struct = 1
    ENDIF ELSE BEGIN
        ;if L* is selected, check if for current choice of model it needs to
        ;be done.
        idx2 = where(new_struct_name EQ tags, c)
        idx = where(input_data.(idx2(0)).LSTAR NE -99, c2)
        IF c2 EQ 0 THEN BEGIN 
            IF (n EQ 0) THEN todo_tags = new_struct_name $
            ELSE todo_tags = [todo_tags, new_struct_name]
            n = n+1
        ENDIF 
        m_new_struct = 0
    ENDELSE 
ENDIF

IF n EQ 0 THEN BEGIN
    message, 'No Mag Coords to do', /cont
    return                      ;no mag field coords to do
ENDIF 

FOR i = 0, n-1 DO BEGIN ;loop through all mag coords todo
    
    ;get intmod,extmod from model name tags
    intmod = fix(strmid(todo_tags(i), 6, 1))
    extmod = fix(strmid(todo_tags(i), 8, 1))

    IF m_new_struct THEN BEGIN 
        ;add new structure tag to array
        dummy2=create_struct(input_data(0),todo_tags(i),new_struct)
        dummy3=replicate(dummy2,n_elements(input_data)) 
        ;fill new array with copies of existing array
        struct_assign, input_data, dummy3
        input_data=temporary(dummy3)    
    ENDIF

    ;get the index of the new mag coords tag
    tags=tag_names(input_data(0)) 
    index=where(tags eq todo_tags(i))
    new_idx=index(0)  
    ndat =  n_elements(input_data)

    ;get mag coords (from UNILIB) 
    tai = input_data.time
    pos = dblarr(3, ndat)
    pos(0, *) = input_data.RAD*6371.20d
    pos(1, *) = 90.0d - input_data.LAT
    pos(2, *) = input_data.LONG
    cor_type='SPH' & cor_sys='GED'

    IF control.l_calc EQ 1 THEN BEGIN
        lstar = 1 
    message,' finding L, L*, MLT, MLAT, B_GSM, B_EQU for '+todo_tags(i), /cont
    ENDIF ELSE BEGIN
        lstar = 0
    message,' finding L, MLT, MLAT, B_GSM, B_EQU for '+todo_tags(i), /cont
    ENDELSE 

    get_mag_unilib, tai, pos, cor_type, cor_sys, $
                    bvec, bequ, Lval, mlt, ilat, error, $
                    INTMOD=INTMOD, EXTMOD = EXTMOD, $
                    LSTAR = lstar, VERBOSE = VERBOSE
      
    input_data.(new_idx).L = Lval
    input_data.(new_idx).Lstar = lstar
    input_data.(new_idx).mlt = mlt 
    input_data.(new_idx).mlat = ilat
    input_data.(new_idx).b_gsm = bvec
    input_data.(new_idx).b_equ = bequ

ENDFOR

;save the new eph data to file
message, 'Writing new', /cont
print, '  ', flname
save, file = flname, input_header, input_data, /COMPRESS

;save the new gps eph structure to file
message, 'Writing new structure contents to file', /cont
gps_eph_struct = input_data(0)
save, file = gps_eph_struct_flname, gps_eph_struct

;if model chosen was MODEL_0_5 then also output to ascii file
;only if L* is done!
IF (new_struct_name EQ 'MODEL_0_5') AND (control.l_calc EQ 1) THEN BEGIN
    message, 'Writing ascii version', /cont
    m_gps_eph_ascii, sat_id,  FILE = flname
ENDIF 

END
  


;******************************************************************************
; gets and makes straight eph files from original source. For GPS,
; that is the original gps data fiels from Tom Cayton.
;******************************************************************************
PRO r_gps_eph_onefile_original, plotinfo, time, flname, VERBOSE = VERBOSE

COMMON gps_eph_data, input_header, input_data
COMMON lanl_gps_data, lanl_gps_header, lanl_gps_data
COMMON get_error, get_err_no, get_err_msg
COMMON gps_eph_1, gps_eph_sat_id, gps_eph_names, gps_eph_select_names, $
                  gps_eph_control
COMMON lanl_gps, sat_id, energy_names_1, energy_names_2, energy_names_3, $
    energy_names_41, energy_names_54, $
    fit_energy_names_1, fit_energy_names_2, singles_names_1, energy_names, $
    eph_names, style_names, select_names, energy_intcal, gain, energy_ch, $
    flux_cal_names, coherent_energy, lanl_gps_control

control = *plotinfo.USR_PTR1
sat_name = gps_eph_sat_id(control.SC_ID)

new_plotinfo = papco_getplotinfostruct()
new_plotinfo.USR_PTR1 = ptr_new(lanl_gps_control)
new_control = *new_plotinfo.USR_PTR1
new_control.sc_id = control.SC_ID
new_plotinfo.USR_PTR1 = ptr_new(new_control)

r_lanl_gps, new_plotinfo, sat_name, VERBOSE=VERBOSE

IF get_err_no EQ 0 THEN BEGIN 
    ;limit data down to one day
    tai_1=utc2tai({mjd:time.mjd, time:0.0})
    tai_2=utc2tai({mjd:time.mjd, time:86400.0*1000})
    index=where((lanl_gps_data.time ge tai_1) AND $
                (lanl_gps_data.time le tai_2),c)
    IF c NE 0 THEN lanl_gps_data = lanl_gps_data(index) ELSE BEGIN
        get_err_msg =  'no data for timerange'
        get_err_no = 1
        message, get_err_msg, /CONT
        return
    ENDELSE 
    ndat=n_elements(lanl_gps_data)

    ;make new structures for ephemeris data only
    dummy = {TIME:0.0d}
    FOR i = 0, n_elements(gps_eph_names)-2 DO $
      dummy=create_struct(dummy, gps_eph_names(i), 0.0)
    dummy = create_struct(name='GPS_EPHEMERIS',dummy)
    input_data=replicate(dummy, ndat)           
    FOR i = 0, n_elements(gps_eph_names)-1 DO $
      input_data.(i) = lanl_gps_data.(i)

    ;save the eph data to file
    message, 'Writing original', /cont
    print, '  ', flname
    save, file = flname, input_header, input_data, /COMPRESS
ENDIF

END 

;------------------------------------------------------------------------------
;program to output eph files to ascii
PRO m_gps_eph_ascii, sat_id,  FILE = FILE, BATCH = BATCH
  
; check env. variable pointing to data directory
if not (papco_check_data_env('GPS_EPH_DATA', PATH=path)) then return

gps_eph_data=path+sat_id+'/'
  
IF keyword_set(FILE) THEN files = [file] ELSE $
  files=findfile(gps_eph_data+'*'+sat_id+'.eph',count=c)
  
FOR i=0, n_elements(files)-1 DO BEGIN

    papco_sepdir, files(i), p, f, e
    out_file = p+f+'_eph.txt'
    
    result = findfile(out_file,  count = c2)
    IF c2 EQ 1 THEN CONTINUE 

    restore, files(i)
    print, out_file

    ;write output file time as decimal day, same as input

    openw, u, out_file, /get_lun        
    FOR j=0, n_elements(input_data)-1 DO BEGIN
        utc = tai2utc(input_data(j).time)
        doy = utc2doy(utc)
        dday = doy + utc.TIME / 86400000.0d
        printf,u,strcenter(varprt(dday),10)+' '+$
               strcenter(varprt(input_data(j).LAT),12)+' '+ $
               strcenter(varprt(input_data(j).LONG),10)+' '+ $
               strcenter(varprt(input_data(j).RAD),10)+' '+ $
               strcenter(varprt(input_data(j).RHO),10)+' '+ $
               strcenter(varprt(input_data(j).Z),10)+' '+ $
               strcenter(varprt(input_data(j).MLONG),10)+' '+ $
               strcenter(varprt(input_data(j).LT89),10)+' '+ $
               strcenter(varprt(input_data(j).BBO),10)+' '+ $
               strcenter(varprt(input_data(j).LOCT),10)+' '+ $
               strcenter(varprt(input_data(j).BSAT),10)+' '+ $
               strcenter(varprt(input_data(j).BEQU),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.L),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.LSTAR),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.MLT),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.MLAT),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(0)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(1)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(2)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(3)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_EQU),10)
    ENDFOR  
    close,u  &  free_lun,u

ENDFOR 

END   

