;******************************************************************************
;* PROCEDURE:     
;*      p_gps_eph_line, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_gps_eph_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_gps_eph_line, panel, plotinfo, OUTPUT=OUTPUT, $
                    OVERPLOT=OVERPLOT, SUBTABLE=SUBTABLE, $
                    PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                    _EXTRA=extra_par

; The following common blocks are needed:
COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_gps_eph. 
COMMON gps_eph_data, input_header, input_data
COMMON gps_eph                              ;see gps_eph_commons for contents
COMMON gps_eph_tick1, time, yray, tick_plotinfo, tick_label, utitle2
COMMON papco_color_names
  
IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.time ge tai_1) AND (input_data.time le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1 & time = data.time-tai_1

control = *plotinfo.USR_PTR1
sat_name = sat_id(control.SC_ID)
utitle = 'gps eph '+sat_name

;select data to plot
Re=6371.0
;set bad data flag
nodata = -99 & n_ch = 0 & tick_label = ''
idx = where(data.rad eq 0, cc) 
IF cc EQ n_elements(data) THEN GOTO, noplot

eph_units = ['deg','deg','Re','Re','Re','deg','L','','Hr','nT','nT','deg']
tags = TAG_NAMES(input_data(0))

model_num = varprt(control.internal_model)+'_'+$
                     varprt(control.external_model)
mag_model = 'MODEL_'+model_num

CASE control.orig OF 
    0: BEGIN                    ;original epehemeris data
        model_num = 'orig'
        CASE 1 OF
            (control.product LE 10): BEGIN
                yray = data.(control.product+1)
            END 
            (plotinfo.ioptions(1) EQ 11): BEGIN  ;mlat, derived coordinate
                pos = strpos(tags, 'RHO') & idx = where(pos NE -1, c)
                rho = data.(idx(0)) 
                pos = strpos(tags, 'Z') & idx = where(pos NE -1, c)
                z = data.(idx(0)) 
                yray = atan(z/rho)*!RADEG
            END     
      
        ENDCASE 

        ;work out bad data and set -99 for L calc
        IF eph_names(control.product) EQ 'LT89' THEN $         
            FOR i = 0, n_elements(yray)-2 DO $
                IF yray(i+1) EQ yray(i) THEN yray(i+1) = -99

        utitle = utitle+'!C'+eph_names(control.product)
        uytitle = eph_units(control.product)
        tick_label = eph_names(control.product) +'('+uytitle+')'
        ;set data tag names for filtering
        l_tag = 'LT89' & mlt_tag = 'LOCT'
    END         
    1: BEGIN                    ;Model ephemeris / mag field data
        nodata = -1e31
        model_num = varprt(control.internal_model)+'_'+$
                             varprt(control.external_model)
        IF control.MAG_DYN THEN dyn_str = '_DYNA' ELSE dyn_str = '_STAT'
        mag_model = 'MODEL_'+model_num+dyn_str
        tags = TAG_NAMES(input_data(0))
        mag_idx = where(tags EQ mag_model, c)
        IF c EQ 0 THEN BEGIN 
            message, 'Magnetic model data for '+mag_model+' not found', /cont
            GOTO, noplot
        ENDIF 

        mag_coord_names = papco_make_choice_names(control.MAG_COORDS_INFO)
        mag_coord_units = papco_make_choice_names(control.MAG_COORDS_UNITS)

        CASE 1 OF
            (control.mag_coords LE 1): BEGIN ;L coords are per 9 PA's
                yray = data.(mag_idx(0)).(control.mag_coords)(0, *)
                yray = transpose(yray)
            END  
            (control.mag_coords GE 2) AND (control.mag_coords LE 6): BEGIN
                yray = data.(mag_idx(0)).(control.mag_coords)
            END 
            (control.mag_coords EQ 7): BEGIN ;Btot, derived
                yray = sqrt(data.(mag_idx(0)).b1^2 + $
                            data.(mag_idx(0)).b2^2 + $
                            data.(mag_idx(0)).b3^2)
            END 
            (control.mag_coords EQ 8): BEGIN ;Btot/Bmin
                btot = sqrt(data.(mag_idx(0)).b1^2 + $
                            data.(mag_idx(0)).b2^2 + $
                            data.(mag_idx(0)).b3^2)
                bmin =  data.(mag_idx(0)).bmin
                idx = where(bmin NE 0, c)
                yray = fltarr(n_elements(data)) & yray(*) = nodata
                IF c NE 0 THEN yray(idx) = btot(idx)/bmin(idx)
            END 
        ENDCASE 
        uytitle = mag_coord_units(control.mag_coords)
        utitle=utitle+'!C' + $
               mag_model+' '+mag_coord_names(control.mag_coords)
        idx = where(yray EQ nodata, cc)
        IF cc NE 0 THEN yray(idx) = nodata
        tick_label = mag_coord_names(control.mag_coords) + $
                     '('+uytitle+':'+model_num+')'
    END 
ENDCASE

idx=where(yray EQ !VALUES.F_INFINITY,c) 
IF c NE 0 THEN yray(idx) = nodata

;now limit data according to variables selected
tag_idx = where(TAG_NAMES(data) EQ mag_model)
tag_idx = tag_idx(0)

;do L-range if that is selected
IF control.L_SELECT_RANGE THEN BEGIN
    sel = data.(tag_idx).L
    idx = where((sel LE control.L_SELECT_FROM_VAL) OR $
                (sel GE control.L_SELECT_TO_VAL), c)
    IF c NE 0 THEN yray(idx) = nodata
    utitle = utitle+'!C'+string(control.L_SELECT_FROM_VAL, $
                             control.L_SELECT_TO_VAL, $
                             format=control.L_SELECT_RANGE_FRMT)+$
                             ' ('+ model_num+')!C'
END   

;do L_STAR-range if that is selected
IF control.L_STAR_SELECT_RANGE THEN BEGIN
    sel = data.(tag_idx).LSTAR
    idx = where((sel LE control.L_STAR_SELECT_FROM_VAL) OR $
                (sel GE control.L_STAR_SELECT_TO_VAL), c)
    IF c NE 0 THEN yray(idx) = nodata
    utitle = utitle+'!C'+string(control.L_STAR_SELECT_FROM_VAL, $
                             control.L_STAR_SELECT_TO_VAL, $
                             format=control.L_STAR_SELECT_RANGE_FRMT)+$
                             ' ('+ model_num+')'
END  

;do MLT-range if that is selected
IF control.MLT_SELECT_RANGE THEN BEGIN
    sel = data.(tag_idx).MLT
    if control.MLT_SELECT_FROM_VAL lt control.MLT_SELECT_TO_VAL THEN $
      idx = where((sel LE control.MLT_SELECT_FROM_VAL) OR $
                  (sel GT control.MLT_SELECT_TO_VAL), c) $
    ELSE $
      idx = where((sel GT control.MLT_SELECT_TO_VAL) AND $
                  (sel LT control.MLT_SELECT_FROM_VAL), c)
    IF c NE 0 THEN yray(idx) = nodata
    utitle = utitle+'!C'+string(control.MLT_SELECT_FROM_VAL, $
                             control.MLT_SELECT_TO_VAL, $
                             format=control.MLT_SELECT_RANGE_FRMT)+$
                             ' ('+ model_num+')'
END 
  

;do MLAT-range if that is selected
IF control.MLAT_SELECT_RANGE THEN BEGIN
    sel = data.(tag_idx).MLAT
    idx = where((sel LE control.MLAT_SELECT_FROM_VAL) OR $
                (sel GT control.MLAT_SELECT_TO_VAL), c)
    IF c NE 0 THEN yray(idx) = nodata
    utitle = utitle+'!C'+string(control.MLAT_SELECT_FROM_VAL, $
                             control.MLAT_SELECT_TO_VAL, $
                             format="(f5.1,'<'+tag+'<',f5.1)")+$
                             ' ('+ model_num+')'
END   

IF output EQ 2 THEN return

panelset,panel	         		;sets the panel position viewport

IF (yscl(panel(0),0) eq 1) THEN BEGIN
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
ENDIF ELSE BEGIN
    papco_ylim, time, yray, yst, yen, LOG = yscl(panel(0),3), BAD_DATA = nodata
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
ENDELSE 

;use autromatic "nice" y-lables
papco_y_label, yst, yen, log=yscl(panel(0),3)
  
; set up extra plot keywords, common ones first
extra_plotPar={yrange:[yst, yen], xrange:[xut1, xut2], $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this routine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar    
  
; now do the actual data plot, data only, no axis
IF n_ch EQ 0 THEN BEGIN         ;ephemeris coords
    
    lanl_gps_gap_plot, time, yray,  AVERAGE=control.seg_av, $
      NODATA = nodata, _extra=extra_plotPar
    ;papco_gap_plot,time, yray, nodata, _extra=extra_plotPar, xstyle=5,ystyle=5

ENDIF ELSE BEGIN                ;mag field data
    
    FOR i = 0, n_ch-1 DO BEGIN
        IF n_ch NE 1 then extra_plotPar.color = colors(i)
        yray = yray2(idx2(i), *)
        lanl_gps_gap_plot, time, yray,  AVERAGE=control.seg_av, $
          NODATA = nodata, _extra=extra_plotPar
        ;papco_gap_plot, time, yray, nodata, xstyle=5, ystyle=5, $
        ;                                    _extra=extra_plotPar
    ENDFOR 
    ; plot channel info inside of panel.
    papco_in_side_label, panel, in_label, NCOL = colors, _extra=extra_plotPar

ENDELSE

; plot y-axis label at left of plot. Use scalable routine!  
IF NOT keyword_set(overplot) THEN left_side_label,panel, uytitle, /rot90       

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot info right of the panel. Multiple pnale labels are handled.
papco_rs_lbl, panel, utitle, /LINE, _extra=extra_Par

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='gps_eph template sample data - single energy channel line'
;    time_label='Seconds since start of day'
    time_label='TAI'
    time = data.time  ; this line only if output in TAI
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, RS_LABEL=RS_LABEL
ENDIF   

return

;jump to here if no data to plot 
noplot: 
  message,'No data to plot', /cont
  plotted_x = !x  &  plotted_y = !y
 
END 
