;******************************************************************************
;*  NAME:        r_gps_eph
;*
;*  DESCRIPTION: Routine to read gps_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       satid   the gps satellite id
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        November 1999
;*
;*  AUTOR:       R. Friedel
;*
;******************************************************************************
pro r_gps_eph, plotinfo, VERBOSE=VERBOSE

COMMON gps_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verb=VERBOSE else verb=0     

; clear data arrays for reuse
input_header=0  &  input_data=0   
  
; loop through from start to end time concatonating data files into one array. 
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
  
WHILE mjdt_end.mjd GE time.mjd do BEGIN 

    ;read one day of data
    r_gps_eph_onefile, plotinfo, time, VERBOSE=VERB
    IF get_err_no NE 0 THEN goto, next_day
  
    ;check for an error in get_error - and then get the file remotely!
    ;comment this section out if you do not want this feature.
    ;if get_err_no ne 0 then begin      
    ;    c_url=papco_getenv('GPS_EPH_FTP_ARCHIVE')
    ;    c_file=fln+'.idl'  &  c_download_dir=path
    ;    papco_get_remote_data, c_url, c_file, c_download_dir
    ;    ;check if file "arrived" and then read it
    ;    if get_err_no eq 0 then begin
    ;       r_gps_eph_onefile, path+fln, VERBOSE=VERBOSE
    ;    endif else begin
    ;        get_err_msg='No data locally, remote get failed!C  ' + get_err_msg
    ;    ENDELSE  
    ;ENDIF    

    ;if a file was succesfully read, concatonate!
    IF no_files_read EQ 0 THEN BEGIN    ;start the array buffer
        input_data_buf=temporary(input_data)
    ENDIF else begin           ;otherwise concatonate arrays
        ;structures can have different tags. Adopt and fold into tags of
        ;initial structure!
        ndat = n_elements(input_data)
        new_input_data = replicate(input_data_buf(0), ndat)
        FOR i = 0, ndat-1 DO BEGIN
            dummy = new_input_data(i)
            struct_assign, input_data(i), dummy
            new_input_data(i) = dummy
        ENDFOR 
        input_data_buf=[input_data_buf,new_input_data]
    ENDELSE         
    input_data=0                      ;clear array for reuse
    no_files_read=no_files_read+1

    next_day:
    get_err_no = 0
    time.mjd=time.mjd+1                 ;next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then break  
ENDWHILE

; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    message, get_err_msg, /cont
ENDELSE  

END 

;******************************************************************************
;*  NAME:        r_gps_eph_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_gps_eph_onefile, plotinfo, time, VERBOSE = VERBOSE,  SAVE = SAVE
  
COMMON gps_eph_data, input_header, input_data
COMMON lanl_gps_data, lanl_gps_header, lanl_gps_data
COMMON get_error, get_err_no, get_err_msg
COMMON gps_eph, gps_eph_sat_id, gps_eph_names, gps_eph_select_names, $
                  gps_eph_control
COMMON lanl_gps, sat_id, energy_names_1, energy_names_2, energy_names_3, $
    energy_names_41, energy_names_54, $
    fit_energy_names_1, fit_energy_names_2, singles_names_1, energy_names, $
    eph_names, style_names, select_names, energy_intcal, gain, energy_ch, $
    flux_cal_names, lanl_gps_control

get_err_no=0  &  get_err_msg='gps_eph onefile read succesfully'

; check keywords and set
if keyword_set(VERBOSE) then VERBOSE=1 else VERBOSE=0

; check env. variable pointing to data directory
if not (papco_check_data_env('GPS_EPH_DATA', PATH=path)) THEN return

control = *plotinfo.USR_PTR1

intmod = control.INTERNAL_MODEL
extmod = control.EXTERNAL_MODEL
sat_name = gps_eph_sat_id(control.SC_ID)

;assemble the data file from the time range requested
convert_t90_to_date, time, yr, doy, hr, min, sec, mon, cmon, dom, /mjdt   
datestr=string(yr, mon, dom, format='(i4.4,i2.2,i2.2)')
fln = datestr+'_'+sat_name+'.eph'
flname = path+sat_name+papco_get_pathsep()+fln
eph_struct_flname = path+sat_name+papco_get_pathsep()+'eph_struct.idl'

IF keyword_set(SAVE) THEN BEGIN
    IF verbose THEN message, 'Saving', /info
    print, '  ', flname
    save, file = flname, input_header, input_data, /COMPRESS
    return
ENDIF

; clear data arrays for reuse
input_header=0  &  input_data=0

;read existing file, or make it
filethere=findfile(flname, count = c)
IF c EQ 0 THEN BEGIN
    get_err_no=2 & get_err_msg='File '+flname+' not found'
    message, 'File not found', /cont
    print, '  ', flname
    ;then read original data, extract ephemeris
    r_gps_eph_onefile_original, plotinfo, time, flname

    IF get_err_no EQ 0 THEN BEGIN     
        ;if no gps_eph_struct exists, write the original one
        result = findfile(eph_struct_flname, count = c)
        IF c EQ 0 THEN BEGIN
            eph_struct = input_data(0)
            save, file = eph_struct_flname, eph_struct
        ENDIF
    ENDIF ELSE BEGIN
        return
    ENDELSE     
ENDIF ELSE BEGIN
    restore, flname
    message, fln, /cont
ENDELSE

;now make/add/check mag coords in input_data.
;need to "hand over" time in TAI,coord system info, actual eph coords. 

;GPS eph in in geographic lat/long
tai = input_data.time
ndat =  n_elements(input_data)
pos = dblarr(3, ndat)

rad = input_data.RAD*6371.0d
lon = input_data.LONG
colat = 90.0d - input_data.LAT

cor_type='CAR' & cor_sys='GEO'
pos(0, *) = rad*sin(colat/!radeg)*cos(lon/!radeg)
pos(1, *) = rad*sin(colat/!radeg)*sin(lon/!radeg)
pos(2, *) = rad*cos(colat/!radeg)

papco_add_mag, control, cor_type, cor_sys, $
               eph_struct_flname, new_struct_name, tai, pos, input_data, error

input_header = {model_name:new_struct_name}

IF error EQ 2 THEN return ;no new structures added to input_data

;save the new eph data to file
message, 'Writing new', /cont
print, '  ', flname
save, file = flname, input_header, input_data, /COMPRESS

;save the new gps eph structure to file
message, 'Writing new structure contents to file', /info 
eph_struct = input_data(0)
save, file = eph_struct_flname, eph_struct


;if model chosen was MODEL_0_5 then also output to ascii file
;only if L* is done!
IF (new_struct_name EQ 'MODEL_0_5') AND (control.l_calc EQ 1) THEN BEGIN
    message, 'Writing ascii version', /cont
    m_gps_eph_ascii, sat_id,  FILE = flname
ENDIF 

END
  

;******************************************************************************
; gets and makes straight eph files from original source. For GPS,
; that is the original gps data files from Tom Cayton.
;******************************************************************************
PRO r_gps_eph_onefile_original, plotinfo, time, flname, VERBOSE = VERBOSE

COMMON mjdt, mjdt_start, mjdt_end
COMMON gps_eph_data, input_header, input_data
COMMON lanl_gps_data, lanl_gps_header, lanl_gps_data
COMMON get_error, get_err_no, get_err_msg
COMMON gps_eph, gps_eph_sat_id, gps_eph_names, gps_eph_select_names, $
                  gps_eph_control
COMMON lanl_gps, sat_id, energy_names_1, energy_names_2, energy_names_3, $
    energy_names_41, energy_names_54, $
    fit_energy_names_1, fit_energy_names_2, singles_names_1, energy_names, $
    eph_names, style_names, select_names, energy_intcal, gain, energy_ch, $
    flux_cal_names, lanl_gps_control

;keep original time range
old_start = mjdt_start & old_end = mjdt_end

;set time for one day
mjdt_start = time & mjdt_end = time & mjdt_end.mjd = mjdt_end.mjd+1

control = *plotinfo.USR_PTR1
sat_name = gps_eph_sat_id(control.SC_ID)

new_plotinfo = papco_getplotinfostruct()
new_control = lanl_gps_control
new_control.sc_id = control.SC_ID
new_plotinfo.USR_PTR1 = ptr_new(new_control)

r_lanl_gps, new_plotinfo, sat_name, VERBOSE=VERBOSE

IF get_err_no EQ 0 THEN BEGIN 
    ;limit data down to one day
    tai_1=utc2tai({mjd:time.mjd, time:0.0})
    tai_2=utc2tai({mjd:time.mjd, time:86400.0*1000})
    index=where((lanl_gps_data.time ge tai_1) AND $
                (lanl_gps_data.time le tai_2),c)
    IF c NE 0 THEN lanl_gps_data = lanl_gps_data(index) ELSE BEGIN
        get_err_msg =  'no data for timerange'
        get_err_no = 1
        message, get_err_msg, /CONT
        GOTO, done
    ENDELSE 
    ndat=n_elements(lanl_gps_data)

    ;make new structures for ephemeris data only
    dummy = {TIME:0.0d}
    FOR i = 0, n_elements(gps_eph_names)-2 DO $
      dummy=create_struct(dummy, gps_eph_names(i), 0.0)
    dummy = create_struct(name='GPS_EPHEMERIS',dummy)
    input_data=replicate(dummy, ndat)           
    FOR i = 0, n_elements(gps_eph_names)-1 DO $
      input_data.(i) = lanl_gps_data.(i)

    ;save the eph data to file
    message, 'Writing original', /cont
    print, '  ', flname
    save, file = flname, input_header, input_data, /COMPRESS
ENDIF

done:
;restore original time range
mjdt_start = old_start & mjdt_end = old_end

END 

;******************************************************************************
;program to output eph files to ascii
PRO m_gps_eph_ascii, sat_id,  FILE = FILE, BATCH = BATCH
  
papco, /set_only

; check env. variable pointing to data directory
if not (papco_check_data_env('GPS_EPH_DATA', PATH=path)) then return

gps_eph_data=path+sat_id+'/'
  
IF keyword_set(FILE) THEN files = [file] ELSE $
  files=findfile(gps_eph_data+'*'+sat_id+'.eph',count=c)
  
FOR i=0, n_elements(files)-1 DO BEGIN

    papco_sepdir, files(i), p, f, e
    out_file = p+f+'_eph.txt'
    
    result = findfile(out_file,  count = c2)
    IF c2 EQ 1 THEN CONTINUE 

    restore, files(i)
    print, out_file

    ;write output file time as decimal day, same as input
    ;Tom wants bad L-values set to +20
    idx = where (input_data.MODEL_0_5.L LE 0, c)
    IF c NE 0 THEN input_data(idx).MODEL_0_5.L = 20
    idx = where (input_data.MODEL_0_5.LSTAR LE 0, c)
    IF c NE 0 THEN input_data(idx).MODEL_0_5.LSTAR = 20

    openw, u, out_file, /get_lun        
    FOR j=0, n_elements(input_data)-1 DO BEGIN
        utc = tai2utc(input_data(j).time)
        doy = utc2doy(utc)
        dday = doy + utc.TIME / 86400000.0d
        printf,u,strcenter(varprt(dday),10)+' '+$
               strcenter(varprt(input_data(j).LAT),12)+' '+ $
               strcenter(varprt(input_data(j).LONG),10)+' '+ $
               strcenter(varprt(input_data(j).RAD),10)+' '+ $
               strcenter(varprt(input_data(j).RHO),10)+' '+ $
               strcenter(varprt(input_data(j).Z),10)+' '+ $
               strcenter(varprt(input_data(j).MLONG),10)+' '+ $
               strcenter(varprt(input_data(j).LT89),10)+' '+ $
               strcenter(varprt(input_data(j).BBO),10)+' '+ $
               strcenter(varprt(input_data(j).LOCT),10)+' '+ $
               strcenter(varprt(input_data(j).BSAT),10)+' '+ $
               strcenter(varprt(input_data(j).BEQU),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.L),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.LSTAR),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.MLT),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.MLAT),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(0)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(1)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(2)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_GSM(3)),10)+' '+ $
               strcenter(varprt(input_data(j).MODEL_0_5.B_EQU),10)
    ENDFOR  
    close,u  &  free_lun,u

ENDFOR 

END   

;******************************************************************************
PRO m_gps_eph_batch, sat_name, date1, date2

COMMON gps_eph
COMMON mjdt, mjdt_start, mjdt_end

papco, /set_only

new_plotinfo = papco_getplotinfostruct()
new_plotinfo.USR_PTR1 = ptr_new(gps_eph_control)
new_control = *new_plotinfo.USR_PTR1

idx = where(sat_name EQ sat_id, c)
IF c  EQ 0 THEN BEGIN
    message, 'GPS satellite '+sat_name+' not found', /cont & return
ENDIF

new_control.sc_id = idx(0)     ; choose satellite
new_control.orig = 1           ; do model coords
new_control.INTERNAL_MODEL = 0 ; IGRF
new_control.EXTERNAL_MODEL = 5 ; OP77
new_control.L_CALC = 1         ; do L*

new_plotinfo.USR_PTR1 = ptr_new(new_control)

r = convert_timeToSeconds(date1, /MJDT, /T90) & mjdt1 = r.value
r = convert_timeToSeconds(date2, /MJDT, /T90) & mjdt2 = r.value

mjdt_start = mjdt1 & mjdt_end = mjdt1

FOR i = mjdt1.mjd, mjdt2.mjd DO BEGIN
    mjdt_start.mjd = i
    mjdt_end.mjd = i+1

    r_gps_eph, new_plotinfo    

END


END 
