;******************************************************************************
;*  NAME:        r_heo_eph
;*
;*  DESCRIPTION: Routine to read heo_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 2003
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
PRO r_heo_eph, plotinfo, VERBOSE=VERBOSE

common heo_eph_data, input_header, input_data
common mjdt, mjdt_start, mjdt_end
common get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

get_err_no=0  &  get_err_msg='heo_eph read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('HEO_EPH_DATA', PATH=path)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. PAPCO standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
  time=mjdt_start  &  no_files_read=0  &  day_count=0.0
   
WHILE mjdt_end.mjd ge time.mjd do begin
     
    ;read one day of data
    r_heo_eph_onefile, plotinfo, time, VERBOSE=VERBOSE
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        ;structures can have different tags. Adopt and fold into tags of
        ;initial structure!
        ndat = n_elements(input_data)
        new_input_data = replicate(input_data_buf(0), ndat)
        FOR i = 0, ndat-1 DO BEGIN
            dummy = new_input_data(i)
            struct_assign, input_data(i), dummy
            new_input_data(i) = dummy
        ENDFOR 
        input_data_buf=[input_data_buf,new_input_data]
      endelse        
      input_data=0                      ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    day_count=day_count+1
    time.mjd=time.mjd+1                 ;next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then break  
endwhile         
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=temporary(input_data_buf)
endif else begin
    get_err_no=1
    get_err_msg='No files found for timerange'
    return
endelse      

END

;******************************************************************************
;*  NAME:        r_heo_eph_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_heo_eph_onefile, plotinfo, time, VERBOSE=VERBOSE, NO_ORIG = NO_ORIG, $
                       INTMOD=INTMOD, EXTMOD=EXTMOD, SAVE = SAVE

common aero_heo_data, aero_heo_header, aero_heo_data
COMMON heo_eph_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON heo_eph       ;see heo_eph_commons for contents
COMMON aero_heo      ;see aero_eph_commons for contents

get_err_no=0  &  get_err_msg='heo_eph onefile read succesfully'

; check keywords and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  
; check env. variable pointing to dat directory
if not (papco_check_data_env('HEO_EPH_DATA', PATH=path)) then return

control = *plotinfo.USR_PTR1
intmod = control.INTERNAL_MODEL
extmod = control.EXTERNAL_MODEL
sat_name = strlowcase(heo_eph_sats(control.SAT_ID))

;assemble the data file from the time range requested
convert_t90_to_date, time, yr, doy, hr, min, sec, mon, cmon, dom, /mjdt   
datestr=string(yr, mon, dom, format='(i4.4,i2.2,i2.2)')
flname = path+sat_name+papco_get_pathsep()+datestr+'_'+sat_name+'.eph'
eph_struct_flname = path+sat_name+papco_get_pathsep()+'eph_struct.idl'

IF keyword_set(SAVE) THEN BEGIN
    message, 'Writing ', /cont
    print, '  ', flname
    save, file = flname, input_header, input_data, /COMPRESS, VERBOSE = VERBOSE
    message, 'Writing current structure contents to file', /cont
    heo_eph_struct = input_data(0)
    save, file = path+sat_name+papco_get_pathsep()+'heo_eph_struct.idl', $
                 heo_eph_struct
    return 
endif

; clear data arrays for reuse
input_header=0  &  input_data=0   
   
new_struct_name='MODEL_'+varprt(intmod)+'_'+varprt(extmod)
new_struct=create_struct(name='UNILIB_3', 'L',0.0,'LSTAR', 0.0, $
                         'MLT',0.0,'MLAT',0.0, $
                         'B_GSM', fltarr(4), 'B_EQU', 0.0)

;read existing file if it exists
filethere=findfile(flname, count = c)

IF c EQ 0 THEN BEGIN
    get_err_no=2 & get_err_msg='File '+flname+' not found'
    message, 'File not found', /cont
    print, '  ', flname
    ;then read original data, extract ephemeris
    new_plotinfo = papco_getplotinfostruct()
    new_plotinfo.USR_PTR1 = ptr_new(aero_heo_control)
      new_control = *new_plotinfo.USR_PTR1
      new_control.sat_id = control.SAT_ID
    new_plotinfo.USR_PTR1 = ptr_new(new_control)
    r_aero_heo, new_plotinfo, VERBOSE=VERBOSE

    IF get_err_no EQ 0 THEN BEGIN 
        ;limit data down to one day
        tai_1=utc2tai({mjd:time.mjd, time:0.0})
        tai_2=utc2tai({mjd:time.mjd, time:86400.0*1000})
        index=where((aero_heo_data.tai ge tai_1) AND $
                    (aero_heo_data.tai le tai_2),c)
        IF c NE 0 THEN aero_heo_data = aero_heo_data(index) ELSE BEGIN
            get_err_msg =  'no data for timerange'
            message, get_err_msg, /CONT
            return
        ENDELSE 
        ndat=n_elements(aero_heo_data)

        ;make new structures for ephemeris data only
        dummy = {TAI:0.0d}
        FOR i = 0, n_elements(heo_eph_products)-1 DO $
          dummy=create_struct(dummy, heo_eph_products(i), 0.0)
        dummy = create_struct(name='HEO_EPHEMERIS',dummy)
        input_data=replicate(dummy, ndat)  
        input_data.tai = aero_heo_data.tai
        tags = tag_names(aero_heo_data(0))                
        FOR i = 0, n_elements(heo_eph_products)-1 DO BEGIN
            idx = where(tags EQ heo_eph_products(i))
            input_data.(i+1) = aero_heo_data.(idx(0))
        ENDFOR 
        r_heo_eph_onefile, plotinfo, time, /SAVE
    ENDIF ELSE BEGIN
        return
    ENDELSE     
ENDIF ELSE BEGIN
    restore, flname
    message, 'idl saveset read succesfully', /cont
    print, '  ', flname
ENDELSE

;read current heo data structure from file
restore, file = path+sat_name+papco_get_pathsep()+ 'heo_eph_struct.idl'
need_tags = tag_names(heo_eph_struct)
tags=tag_names(input_data(0))

;now, see if we have to any missing tags!
n = 0 & todo_tags = ''
FOR i = 0, n_elements(need_tags)-1 DO BEGIN 
    idx = where(need_tags(i) EQ tags, c)
    IF c EQ 0 THEN BEGIN
        IF (n EQ 0) THEN todo_tags = need_tags(i) $
        ELSE todo_tags = [todo_tags, need_tags(i)]
        n = n+1
    ENDIF 
ENDFOR 

IF control.orig EQ 1 THEN begin
    ;now see if we have to add the currently selected mag field too.
    idx = where(new_struct_name EQ tags, c)
    IF c EQ 0 THEN BEGIN
        idx = where(new_struct_name EQ todo_tags, c)
        IF c EQ 0 THEN BEGIN 
            IF (n EQ 0) THEN todo_tags = new_struct_name $
            ELSE todo_tags = [todo_tags, new_struct_name]
            n = n+1
        ENDIF 
    ENDIF 
ENDIF

IF n EQ 0 THEN return ;no mag field coords to do

FOR i = 0, n-1 DO BEGIN ;loop through all mag coords todo
    
    ;get intmod,extmod from model name tags
    intmod = fix(strmid(todo_tags(i), 6, 1))
    extmod = fix(strmid(todo_tags(i), 8, 1))

    ;add new structure tag to array
    dummy2=create_struct(input_data(0),todo_tags(i),new_struct)
    dummy3=replicate(dummy2,n_elements(input_data)) 
    ;fill new array with copies of existing array
    struct_assign, input_data, dummy3
    input_data=temporary(dummy3)    
 
    message,' finding L, L*, MLT, MLAT, B_GSM, B_EQU for '+todo_tags(i), /cont
  
    ;get the index of the new mag coords tag
    tags=tag_names(input_data(0)) 
    index=where(tags eq todo_tags(i))
    new_idx=index(0)  
    ndat =  n_elements(input_data)

    ;get mag coords (from UNILIB) 
    tai = input_data.tai
    pos = dblarr(3, ndat)
    pos(0, *) = input_data.RADDIST*6371.20d
    pos(1, *) = 90.0d - input_data.GEO_LAT
    pos(2, *) = input_data.GEO_LONG
    cor_type='SPH' & cor_sys='GED'
    lstar = 1 ; do L* calc
    lstar = 0 ; quicker for testing!

    get_mag_unilib, tai, pos, cor_type, cor_sys, $
                    bvec, bequ, Lval, mlt, ilat, error, $
                    INTMOD=INTMOD, EXTMOD = EXTMOD, $
                    LSTAR = lstar, VERBOSE = VERBOSE
      
    input_data.(new_idx).L = Lval
    input_data.(new_idx).Lstar = lstar
    input_data.(new_idx).mlt = mlt 
    input_data.(new_idx).mlat = ilat
    input_data.(new_idx).b_gsm = bvec
    input_data.(new_idx).b_equ = bequ

ENDFOR

r_heo_eph_onefile, plotinfo, time, /SAVE
      
END 
