;***************************************************************************
;***************************************************************************
;* 
;* FILE:
;*    $PAPCO_modules/hydra_moments_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_PATH/papco_panel.pro
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ
;*       $PAPCO_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $PAPCO_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for PAPCO to run, read the 
;*    file WIDINIT.PRO
;*
;* COMMON-BLOCKS:
;*    hydra_moments_panel_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {hydra_moments_panel_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  hydra_moments_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, hydra_moments_panel(PAPCO_getPlotInfoStruct())
;*
;* MODIFICATION HISTORY:       
;*     november 1995, written by R. Friedel at Goddard
;***************************************************************************
;***************************************************************************


;***************************************************************************
;* 
;* FUNCTION:     
;*      FUNCTION  hydra_moments_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,	
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=PAPCO_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     november 1995, written by R. Friedel at Goddard
;*
;***************************************************************************
FUNCTION  hydra_moments_panel, aPlotInfo, GROUP=group, ACTION=ACTION
   COMMON hydra_moments_panel_DATA, widgetData

   COMMON hydra_moments_labels, $
    moments_labels

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ----------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	 column=1, $
	 GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets-------------
;    that control the height of that plot
;   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   widgets= { sl_height:0L, panelHeight:1 } 
   additionalWData=create_struct(widgets, additionalWData)
   

; --- Choose plot species---------------------------------------------------
   type_base = WIDGET_BASE(panel_base, /FRAME, /row,/EXCLUSIVE)
   typeNames = ['electrons',$
                'ions' ]
   spb_type = lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
    spb_type(i) = WIDGET_BUTTON(type_base, $
                                VALUE=typeNames(i),$
                                UVALUE='spb_type'+strtrim(string(i),2))

   aPlotInfo.typeVector(0)=aPlotInfo.typeVector(0) < (N_ELEMENTS(typeNames)-1)
   aPlotInfo.typeVector(0)=aPlotInfo.typeVector(0) > 0
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1


;  decode id into label and component
   hydra_moments_panel_map, label, component, aPlotInfo.typeVector(1), $
     /to_label
   additionalWData=create_struct({label:label,$
                                  component:component}, additionalWData)   

; --- Choose plot variable--------------------------------------------------
   type_base=WIDGET_BASE(panel_base, column=5, /FRAME, /EXCLUSIVE)
   
   hydra_moments_panel_map, map, /get_map
   s= uniq(map(*,0),sort(map(*,0)))
   s1= sort(s)
   labels= map(s(s1),0)
   comp_select= map(s(s1),1) ne ''

   additionalWData=create_struct({comp_select:comp_select, $
                                  labels:labels, $
                                  comp_base:0L }, additionalWData)
   typenames = labels
   
   spb_type=lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
     spb_type(i)=WIDGET_BUTTON(type_base, $
                               VALUE=typeNames(i), $
                               UVALUE='label:'+strtrim(i,2))
   
   r= ( where( labels eq label ))(0)
   widget_control,  spb_type(r(0)), set_button=1
   
;  component base
   comp_base= widget_base( panel_base, /row )
   x= widget_label( comp_base, value='vector component: ')
   x= cw_bgroup(comp_base, ['x','y','z'], /exclusive, /row, $
                button_uvalue='component:'+['x','y','z'], uvalue='component' )
   r= ( where( ['x','y','z'] eq component ) )(0)
   if r(0) eq -1 then begin
       widget_control, comp_base, sensitive=0
   endif else begin
       widget_control, x, set_value=r(0)
   endelse
   
   additionalWData.comp_base= comp_base


;decom; --- Choose color-coding option --------------------------------------
;decom   type_base = WIDGET_BASE(panel_base, /FRAME, /row, /EXCLUSIVE)
;decom   typeNames = ['No color-coding','Color-code alignment with B']
;decom   spb_type = lonarr(N_ELEMENTS(typeNames))
;decom   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
;decom    spb_type(i) = WIDGET_BUTTON(type_base, $
;decom                                VALUE=typeNames(i),$
;decom                                UVALUE='spb_colo'+strtrim(string(i),2))
;decom   aPlotInfo.typeVector(2)=aPlotInfo.typeVector(2) < (N_ELEMENTS(typeNames)-1)
;decom   aPlotInfo.typeVector(2)=aPlotInfo.typeVector(2) > 0
;decom   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(2)),
;SET_BUTTON=1

;   filter base
   filter_base= widget_base( panel_base, /frame, /column, /nonexclusive )
   x= widget_button( filter_base, $
                     value='Apply patch filter '+$
                     '(status le 0 or 5th digit of status=3, '+$
                     'or b_status ne 1)', uvalue='patch_filter' )
   widget_control, x, set_button= aPlotInfo.typeVector(2) and 1
   x= widget_button( filter_base, $
                     value='Return Current filter', $
                     uvalue='rcfilter' )
   widget_control, x, set_button= (aPlotInfo.typeVector(2) and 2) / 2
   
;   hydra_findfile widget
   id= cw_hydra_findfile( panel_base, /moments, date='5/4/98', $
                          uvalue='findfile', /events, file=aPlotInfo.filename )
   additionalWdata= create_struct( { findfile_id:id, $
                                     filename:aPlotInfo.filename }, $
                                   additionalWData )
   
;   color highlighting widget (hydra_plot keywords)
   w= hydra_plot_widget( panel_base, /frame, uvalue='hp_kw' )
   v= { $
        codei4:aPlotInfo.typeVector(1), $
        reference:aPlotInfo.options(15), $
        poly_above_ref: ( aPlotInfo.ioptions(15) and 4 ) / 4, $
        poly_below_ref: ( aPlotInfo.ioptions(15) and 2 ) / 2, $
        draw_reference: ( aPlotInfo.ioptions(15) and 1 ) / 1 $
      }
   widget_control, w, set_value=v
   additionalWData= create_struct( { soptions:aPlotInfo.soptions, $
                                     ioptions:aPlotInfo.ioptions, $
                                     options:aPlotInfo.options }, $
                                   additionalWData )

;--- add the widgets for choosing the plot-style -------------------------------
   widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)

   additionalWData=create_struct(widgets, additionalWData)

;--- now, add the widgets for choosing the y-axis-ranges -----------------------
   widgets= hydra_ylim_widget(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

; Reload button
   type_base=WIDGET_BASE(panel_base, /FRAME, /row )
   spb= widget_button( type_base, $
                       value='Reload Data', $
                       uvalue='spb_reload' )

;--- finally, the Actionbuttons (help, cancel, done) are added to the ----------
;    editor window.
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /config)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'hydra_moments_panel.help', $
                    typeVector:aPlotInfo.typeVector, $
                    spb_type:spb_type}
   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_panel_realize, panel_base, widgetData, yoffset=0, xoffset=300

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'hydra_moments_panel', panel_base, /MODAL

; convert panel information into code.
   hydra_moments_panel_map, $
     widgetData.label, $
     widgetData.component, id, /to_id
   widgetData.typeVector(1)=id

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={ typeVector:widgetData.typeVector, $
                soptions:widgetData.soptions, $
                ioptions:widgetData.ioptions, $
                options:widgetData.options }
   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END




;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO hydra_moments_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     november 1995, written by R. Friedel at Goddard
;*
;***************************************************************************
PRO hydra_moments_panel_event, event
COMMON hydra_moments_panel_DATA, widgetData
WIDGET_CONTROL,GET_UVALUE=uval, event.id

COMMON hydra_moments_data, $
  moments_species, $
  moments_header, $
  moments_ele_data, $
  moments_ion_data

IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
    widgetData.typeVector(0)=fix(strfrom(uval,8))
    
ENDIF ELSE IF strleft(uval,6) EQ 'label:' THEN BEGIN
    ilabel= fix(strfrom(uval,6))
    widgetData.label= widgetData.labels(ilabel)
    widget_control, widgetData.comp_base, $
      sensitive= widgetData.comp_select(ilabel)
    
;decom   ENDIF ELSE IF strleft(uval,8) EQ 'spb_colo' THEN BEGIN
;decom      widgetData.typeVector(2)=fix(strfrom(uval,8))
    
ENDIF ELSE IF uval EQ 'spb_configure' THEN begin
    hydra_moments_configure

endif else if uval eq 'findfile' then begin
    widgetData.filename= event.value

ENDIF ELSE IF uval EQ 'patch_filter' THEN BEGIN
    widgetData.typeVector(2)= $
      widgetData.typeVector(2) and ( 65535-1 ) or $
      event.select * 1

endif else if uval eq 'rcfilter' then begin
    widgetData.typeVector(2)= $
      widgetData.typeVector(2) and ( 65535-2 ) or $
      event.select * 2
    print, widgetData.typeVector(2)

ENDIF ELSE IF uval EQ 'spb_reload' THEN BEGIN
    ispecies = widgetData.typeVector(0)
    IF (n_elements(moments_header) GT 0) THEN $
      moments_header(ispecies).reload_flag=1

ENDIF ELSE IF uval EQ 'hp_kw' THEN BEGIN
    v= event.value
    widgetData.typeVector(1)= v.codei4           
    widgetData.options(15)= v.reference
    widgetData.ioptions(15)= $
      v.poly_above_ref*4 + $
      v.poly_below_ref*2 + $
      v.draw_reference*1   
endif else if strleft(uval,9) eq 'component' then begin
    widgetData.component= strfrom(event.value,10)
ENDIF ELSE BEGIN
    widgetData=  hydra_ylim_widget_event( event, widgetData, handled=h )
    if not h then widgetData=PAPCO_panel_Event(event, widgetData)
ENDELSE

END

   




















