;***************************************************************************
;***************************************************************************
;* 
;* FILE:
;*    $PAPCO_PATH/hydra_skew_panel.pro
;* 
;* DESCRIPTION: 
;*
;* FILES YOU MAY WANT TO READ
;*       $PAPCO_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $PAPCO_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for PAPCO to run, read the 
;*    file addpapco_init.PRO in $PAPCO_MODULES
;*
;* COMMON-BLOCKS:
;*    hydra_skew_panel_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {hydra_skew_panel_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  hydra_skew_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, hydra_skew_panel(PAPCO_getPlotInfoStruct())
;*       which calls up the window.
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     December 1995, modified by R.Friedel at Goddard
;*     
;***************************************************************************
;***************************************************************************

;***************************************************************************
;* 
;* FUNCTION:     
;*      FUNCTION  hydra_skew_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,	
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=PAPCO_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;*
;***************************************************************************
FUNCTION  hydra_skew_panel, aPlotInfo, GROUP=group, ACTION=ACTION

   COMMON hydra_skew_panel_DATA, widgetData, drawID

; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}

;--- create the base-window ----------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	 column=1, $
	 GROUP_LEADER=group)

;--- the first part of the window will consist in the widgets ------------------
;    that control the height of that plot
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

   typeVector= aPlotInfo.typeVector

; --- Choose plot type----------------------------------------------------------
 
   spec_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1, /EXCLUSIVE) 
   specNames=['electrons', $
              'ions' ]
   spb_spec=lonarr(N_ELEMENTS(specNames))
   
   FOR i=0, N_ELEMENTS(specNames)-1 DO $
     spb_spec(i)=WIDGET_BUTTON(spec_base, $
                                VALUE=specNames(i), $
                                UVALUE='spb_spec'+strtrim(string(i), 2))
   
   WIDGET_CONTROL, spb_spec(typeVector(0)), SET_BUTTON=1

; hard-code typevector(2) to 4...  that's a skew plot in r_hydra_spec

; --- Choose Skew Plot type --------------------------
   skew_base= widget_base( panel_base, /frame, /row )   

   button_base= widget_base( skew_base, /frame, /column, /exclusive )
   button_names= [ 'Skew', 'Anisotropy', 'Binary Skew' ]

   spb_button= lonarr( n_elements( button_names ) )
   for i=0, n_elements( button_names )-1 do $
     spb_button(i)= widget_button( button_base, value=button_names(i), $
                                   uvalue= 'spb_type'+strtrim(i,2) )
   widget_control, spb_button( typeVector(2) ), set_button=1

   drawID= widget_draw( skew_base, frame=3, xsize=60, ysize=60 )
   
; --- Choose spectrogram resolution -------------------
   type_base=WIDGET_BASE(panel_base, /row) 
   base2= widget_base( type_base, /FRAME, COLUMN=1, /EXCLUSIVE) 
   typenames=[ 'AutoSelect', $
               'Survey Resolution (13.8s)', $
               'Sweep Resolution (1.15s)' $
             ]
   
   spb_type=lonarr(N_ELEMENTS(typeNames))
   
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
     spb_type(i)=WIDGET_BUTTON(base2, $
                               VALUE=typeNames(i), $
                               UVALUE='spb_tres'+strtrim(string(i), 2))
   
   WIDGET_CONTROL, spb_type(typeVector(1)), SET_BUTTON=1

; options
   base2= widget_base( type_base, /FRAME, COLUMN=1, /nonexclusive )
   b= widget_button( base2, value='Shift By S/C Pot' )
   widget_control, b, $
     set_button= aPlotInfo.typeVector(3) and 1, $
     set_uvalue='spb_shift'+ strtrim( aPlotInfo.typeVector(3) and 1,2 )  

; --- Configure button -----------------
   type_base=WIDGET_BASE(panel_base, /FRAME, /row )

   spb= widget_button( type_base, $
                       value='Reload Data', $
                       uvalue='spb_reload' )

; ---add y range scaling widgets
   widgets= papco_panel_addYscale_widgets( panel_base, aPlotInfo )
   additionalWData=create_struct(widgets, additionalWData)   
                       

;  Z Scale widgets
   widgets=PAPCO_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /config)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'hydra_skew_panel.help', $
                    typevector:aPlotInfo.typeVector, $
                    spb_type:spb_type}

   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData
   widget_control, panel_base, yoffset=0

   widget_control, drawID, get_value=w
   wold= !d.window
   wset, w
   case typeVector(2) of
       0: hydra_skew_drawangles, red=[0,30], blue=[150,180]
       1: hydra_skew_drawangles, red=[0,30,150,180], blue=[75,105]
       2: hydra_skew_drawangles, red=[0,30], blue=[150,180]
   endcase
   wset, wold

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'hydra_skew_panel', panel_base, /MODAL

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={typeVector:widgetData.typeVector }

   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO hydra_skew_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;*
;***************************************************************************
PRO hydra_skew_panel_event, event

   COMMON hydra_skew_panel_DATA, widgetData, drawID
   WIDGET_CONTROL,GET_UVALUE=uval, event.id

   common hydra_spec_data, $
     data_context, $            ; selects current data set
     spec_header

; in this editor, we don't need a special event-processing, so we just
; pass the event on to 'PAPCO_panel_Event'. If you need to process
; events, you should check 'event' for your events and call
; 'PAPCO_panel_Event', if event was not generated by your new widgets

   value_idx= strlen( uval ) - 1    
   IF strleft(uval, 8) EQ 'spb_spec' THEN BEGIN
       widgetData.typeVector(0)= fix( strfrom( uval, 8 ) )
   endif else if strleft(uval, 8) eq 'spb_type' then begin
       widgetData.typeVector(2)= fix( strfrom( uval, 8 ) )
       widget_control, drawID, get_value=w
       wold= !d.window
       wset, w
       case widgetData.typeVector(2) of
           0: hydra_skew_drawangles, red=[0,30], blue=[150,180]
           1: hydra_skew_drawangles, red=[0,30,150,180], blue=[75,105]
           2: hydra_skew_drawangles, red=[0,30], blue=[150,180]
       endcase
       wset, wold       
   ENDIF ELSE IF strleft(uval, 8) EQ 'spb_tres' then begin
       widgetData.typeVector(1)= fix( strfrom(uval, 8 ) )       
       data_context = hyd_dc( widgetData.typeVector(1), 4 )
   ENDIF else if strleft(uval,value_idx) eq 'spb_shift' THEN BEGIN 
       widgetdata.typeVector(3)= $
         widgetdata.typeVector(3) and ( not long(1) ) $
         or 1-fix(strfrom(uval, value_idx))
       widget_control, event.id, $
         set_uvalue= 'spb_shift'+ strtrim( widgetData.typeVector(3) and 1,2 )
   endif ELSE if uval eq 'spb_configure' then begin
       hydra_skew_configure
   endif else if uval eq 'spb_reload' then begin
       data_context = hyd_dc( widgetData.typeVector(1), 0 )
       if n_elements(spec_header) gt 0 then $
         spec_header(data_context+[1,2,3]).reload_flag=1       
   endif else if strleft(uval,10) eq 'spb_sample' then begin
       widgetData.typeVector(3)= fix(strfrom(uval, 10))
   endif else begin
       widgetData=PAPCO_panel_Event(event, widgetData)
   ENDELSE

END

   




















