;***************************************************************************
;* file hydra_spec_slice.pro
;*
;* this files contains the routines needed to enable the extra mouse
;* feautures provided for in PAPCO. Mouse zooms is part of main papco.
;* Currently possible:
;*
;*    mouse views slice
;*    mouse views slice (polar)
;*    mouse accesses level zero
;* 
;* Again here the procedure names must adhere to a strict naming convention so 
;* that they can be automatically called when needed.
;*
;* The slice types supported by these routines must be ENABLED by the user, so 
;* that PAPCO will call them. This is done by setting the value of 
;* slice_type in draw_plot_type.pro (contained in papcoadd_plot_type.pro).
;* slice_type is defined in the following way:
;*
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swwitches. Currently
;*                      Three are supported (this may be extended later)
;*                       1 = mouse zooms (default, always possible)
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8, 16, 32, 64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arythmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* The mouse does slice function uses the time returned by PAPCO and the
;* get_data_call specified for the plot_type to read the data and to draw a 
;* slice at the clicked position. This is commonly used in spectra color plots 
;* to draw a spectrum at that time, or a pitch angle distribution etc, or to
;* call up any other plot at that time as specified by the user.
;*
;* The mouse accesses level zero uses the time passed and calls up another
;* widget application to read and work with level zero data. This application
;* must be intirely independant of PAPCO and is only called from PAPCO via
;* a mouse click in the panel required for the time required.
;*
;* If your new data contains several sub-data types, one slice and/or level zero
;* routine has to be provided for each type, since in general they would do
;* different things. See $PAPCO_MODULES/crres_slices.pro for such an example.
;*
;*    PRO DrawSlice_hydra_spec, panelNr, seconds, yValue
;*        -> Draw a slice for hydra_spec
;*
;*    PRO levelzero_hydra_spec, seconds, yvalue
;*        -> This procedure calls the widget for level zero data acces 
;*           for this plottype.
;*    
;* MODIFICATION HISTORY:       
;*    December 1995, written by R.Friedel at Goddard
;*     april 1996, R. Friedel
;*
;***************************************************************************

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO DrawSlice_hydra_spec, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*	Draw a slice for hydra_spec, providing default functionality and
;*      calling a seperate window for the plot.
;*      slice_type in draw_plot_type must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		time of mouse click, a float
;*	yvalue		y-value of mouse click, a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*	
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by PAPCO_DrawSLice (in papco_cursor)
;*
;* MODIFICATION HISTORY:       
;*     april 1996, R. Friedel
;***************************************************************************
PRO DrawSlice_hydra_spec, panelNr, seconds, yValue

   COMMON PAPCO_DRAWSLICE, sliceWidgetData
   COMMON PLOT_COMPOSER, widgetData

; add in here and other common blocks that you need to make your plot data
; available here. The common block included here is a dummy for illustration.
; t90 is the time array in t90 format.
; PAPCO uses the the getdatacall you set in papco_add_plot_type to read the
; data you need for the slice. As long as you then stay in one panel with the 
; mouse, the data is not read again. When you swwitch panels, the new data is 
; read. 

   common hydspctrmstuff,hydspctrm
   common iowa_style,en_both

   WIDGET_CONTROL,/HOURGLASS
   thisPlot=widgetData.plotsDrawn(panelNr)

; make same data plot array as used in p_hydra_spec

   thisPlot=widgetData.plotsDrawn(panelNr)
   type=thisPlot.typeVector  

; set the panel window for the slice to full window

   !p.position=[0.1,0.1,0.9,0.9]

   t90=hydspctrm(*).tim.t90

   case type(0) of
     0:begin			;electrons
         wele=where(hydspctrm.energy lt 0)
         ymat=hydspctrm.lnf(*,wele)
         e_ch=reverse(hydspctrm.energy(wele))*(-1)
         xst=e_ch(0)
         xen=e_ch(n_elements(e_ch)-1)
         ;invert the array to have increasing energy y-scale
         for i=0,hydspctrm.na-1 do ymat(i,*)=reverse(ymat(i,*),2)
         ctit2='HYDRA Electrons'
         cytit='Flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
         cxtit='Energy Channels (keV)'
         this_Par={ylog:1,xlog:1,xminor:0}
       end
     1:begin			;ions
         wion=where(hydspctrm.energy gt 0)
         ymat=hydspctrm.lnf(*,wion)
         e_ch=hydspctrm.energy(wion)
         xst=e_ch(0)
         xen=e_ch(n_elements(e_ch)-1)
         ctit2='HYDRA Ions'
         cytit='Flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
         cxtit='Energy Channels (keV)'
         this_Par={ylog:1,xlog:1,xminor:0}
       end
     2:begin			;both iowa style
         ymat=hydspctrm.lnf
         en_both=abs(hydspctrm.energy)
         e_ch=findgen(n_elements(en_both))
         xst=0
         xen=n_elements(en_both)-1        
         ctit2='HYDRA Spectrum!CIowa style'
         cytit='Flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
         cxtit='Energy Channels (keV)'
         this_Par={xtickformat:'exponate',xminor:0,xticks:11}
       end
     3:begin			;both iowa style, iowa routine
         ymat=hydspctrm.lnf
         en_both=abs(hydspctrm.energy)
         e_ch=findgen(n_elements(en_both))
         xst=0
         xen=n_elements(en_both)-1   
         ctit2='HYDRA Spectrum!CIowa style'
         cytit='Flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
         cxtit='<-- IONS   Energy Channels (keV)   ELECTRONS -->'
         this_Par={xtickformat:'exponate',xminor:0,xticks:11}
       end
   endcase
   ctit='POLAR HYDRA Energy Spectrum'

; select the index for the time chosen out of the time array
   DiffTimes=ABS(t90 - seconds)
   result=MIN(DiffTimes, timeIndex)	; this sets timeIndex to the index with
					; t90(index) is closest to 'seconds'

; select the z-scale for the color plot to be the y-scale range here. For
; color plots the zscale is used in the common yscl block.

   yst=widgetData.plotsDrawn(panelNr).yscl_min * 0.9
   yen=widgetData.plotsDrawn(panelNr).yscl_max * 1.1

; select the single spectrum out of ymat to plot

   yarr=ymat(timeIndex,*)

; set up extra plot keywords, common ones first

  extra_plotPar_common={xrange:[xst, xen], yrange:[yst,yen], $
               noerase:1, ticklen:-0.015, ytitle:cytit, $
               title:ctit2, xstyle:3, ystyle:3, color:1, xtitle:cxtit}

; add keyword structure specific to the subtype of plot chosen here
  
   extra_plotPar_common=create_struct(extra_plotPar_common, this_par)

; do the plot

  erase
  plot,e_ch,yarr,_extra=extra_plotPar_common

; these two info strings are user definable ouput strings to label the slice 
; window.

   convert_t90_to_Date,seconds, year, day_oy, hour, min, sec, $
			      month, cmonth, day_om 

;find corresponding z-value for returned y-value

   DiffChan=ABS(e_ch - yValue)
   result=MIN(DiffChan, ChanIndex)	
   

   info1=string(day_om,cmonth,year,format="('Date: ',i2.2,' ',a3,' ',i2.2,' ')")
   info1=info1+string(hour,min,sec,format="('  Time: ',i2.2,':',i2.2,':',i2.2)")
   info2=string(yarr(ChanIndex), yValue, $
        format="('Z-value: ',f8.4,'   Y-value: ',f9.2)")

; Replace these labels and include your plot! 

; user defined routine for plotting the slice included here. An example is in 
; $PAPCO_MODULES/panels&slices/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; (in t90 format of course), plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie ephemeris data, etc.

   WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
   WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2

END


;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO DrawSlice_hydra_spec_survey, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*	Draw a slice for hydra_spec, providing default functionality and
;*      calling a seperate window for the plot.
;*      slice_type in draw_plot_type must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		time of mouse click, a float
;*	yvalue		y-value of mouse click, a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*	
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by PAPCO_DrawSLice (in papco_cursor)
;*
;* MODIFICATION HISTORY:       
;*     april 1996, R. Friedel
;***************************************************************************
PRO DrawSlice_hydra_spec_survey, panelNr, seconds, yValue

  DrawSlice_hydra_spec,panelNr, seconds, yValue

end

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO levelzero_hydra_spec, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*	This procedure calls the widget for level zero data acces 
;*      for this plottype. 
;*      slice_type in draw_plot_type must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*	seconds		t90 time at which mouse is clicked
;*	yvalue		y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	levelzero_hydra_spec, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     april 1996, R. Friedel
;*
;***************************************************************************
pro levelzero_hydra_spec, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $PAPCO_MODULES/addpapco_swe_plots.pro.
; Provided as paramters to be passed to the new application are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swwitched to (e.g. to get a new
; time off the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('PAPCO_main') 
; to swwitch back to the PAPCO main calling widget. 

; call the levelzero widget.  
;   plot_type_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if PAPCO
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_plot_type, 
; where plot_type is the name of the panel you have set in papcoadd_plot_type.
    print, 'got to the level zero interface  ',seconds, yvalue
END


;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO levelzero_hydra_spec, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*	This procedure calls the widget for level zero data acces 
;*      for this plottype. 
;*      slice_type in draw_plot_type must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*	seconds		t90 time at which mouse is clicked
;*	yvalue		y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	levelzero_hydra_spec, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     april 1996, R. Friedel
;*
;***************************************************************************
pro levelzero_hydra_spec, seconds, yvalue

common auspapco,x_papco,y_papco

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $PAPCO_MODULES/addpapco_swe_plots.pro.
; Provided as paramters to be passed to the new application are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be swwitched to (e.g. to get a new
; time off the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('PAPCO_main') 
; to swwitch back to the PAPCO main calling widget. 


; call the levelzero widget.
   print,'hydra_spec_slice : seconds = ',seconds 

x_papco = double(seconds)
y_papco = yvalue

print,'hydra_spec_levelzero : x_papco = ',x_papco

levelzero_hyd,papco=1


END
