;***************************************************************************
;***************************************************************************
;* 
;* FILE:
;*    $PAPCO_modules/hydra_tpphr_panel.pro
;* 
;*    This file belongs to the PAPCO-project. See $PAPCO_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains a demonstration on how to implement a PAPCO-
;*    panel-editor. It makes use of all the features provided by 
;*    $PAPCO_PATH/papco_panel.pro
;*    It constructs an editor for a very simple PAPCO-plot, that has only
;*    default-functionality.
;*
;* FILES YOU MAY WANT TO READ
;*       $PAPCO_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $PAPCO_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for PAPCO to run, read the 
;*    file WIDINIT.PRO
;*
;* COMMON-BLOCKS:
;*    hydra_tpphr_panel_DATA	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    PAPCO_PANEL_EDITOR-structure (see $PAPCO_PATH/papco_panel.pro for a
;*    description). 
;*       {hydra_tpphr_panel_DATA, 
;*	     ; imagine the PAPCO_PANEL_EDITOR-structure was inserted...
;*	      
;*    For a more complete description see $PAPCO_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  hydra_tpphr_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, hydra_tpphr_panel(PAPCO_getPlotInfoStruct())
;*
;* MODIFICATION HISTORY:       
;*     november 1995, written by R. Friedel at Goddard
;***************************************************************************
;***************************************************************************


;***************************************************************************
;* 
;* FUNCTION:     
;*      FUNCTION  hydra_tpphr_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,	
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=PAPCO_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     november 1995, written by R. Friedel at Goddard
;*
;***************************************************************************
FUNCTION  hydra_tpphr_panel, aPlotInfo, GROUP=group, ACTION=ACTION
   COMMON hydra_tpphr_panel_DATA, widgetData
   
   COMMON hydra_tpphr_labels, $
    tpphr_labels
   
; if GROUP is not specified, the group-leader will be the desktop
   IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
   
; set the label for the action-button
   IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'
   
; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
   additionalWData={canceled:1}
   
;--- create the base-window ----------------------------------------------------
   panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
                          column=1, $
                          GROUP_LEADER=group)
   
;--- the first part of the window will consist in the widgets-------------
;    that control the height of that plot
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
   
; --- Choose plot resolution------------------------------------------------
   type_base = WIDGET_BASE(panel_base, /FRAME, COLUMN=1,/EXCLUSIVE)
   typeNames = ['1.15 s (sweep resolution)', $
                '3.45 s (2 sweep)', $
                '5.75 s (3 sweep)', $
                '12.65 s (block resolution)']
   
   spb_type = lonarr(N_ELEMENTS(typeNames))
   
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
    spb_type(i) = WIDGET_BUTTON(type_base, $
                                VALUE=typeNames(i),$
                                UVALUE='spb_type'+strtrim(string(i),2))
   
   aPlotInfo.typeVector(0)=aPlotInfo.typeVector(0) < (N_ELEMENTS(typeNames)-1)
   aPlotInfo.typeVector(0)=aPlotInfo.typeVector(0) > 0
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(0)), SET_BUTTON=1
   
   
; --- Choose plot variable--------------------------------------------------
   type_base=WIDGET_BASE(panel_base, column=4, /FRAME, /EXCLUSIVE)
   
   IF n_elements(tpphr_labels) EQ 0 THEN $
    hydra_define_structure,/tpphr_lab
   
   typenames = tpphr_labels(*,0)
   
   spb_type=lonarr(N_ELEMENTS(typeNames))
   
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
    spb_type(i)=WIDGET_BUTTON(type_base, $
                              VALUE=typeNames(i), $
                              UVALUE='spb_varb'+strtrim(string(i), 2))
   
   aPlotInfo.typeVector(1)=aPlotInfo.typeVector(1) < (N_ELEMENTS(typeNames)-1)
   aPlotInfo.typeVector(1)=aPlotInfo.typeVector(1) > 0
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(1)), SET_BUTTON=1

; --- Choose error_overplot status------------------------------------------
   type_base = WIDGET_BASE(panel_base, /FRAME, COLUMN=1,/EXCLUSIVE)
   typeNames = ['NO ERROR BARS', $
                'WITH ERROR BARS' $
               ]
   spb_type = lonarr(N_ELEMENTS(typeNames))

   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
    spb_type(i) = WIDGET_BUTTON(type_base, $
                                VALUE=typeNames(i),$
                                UVALUE='spb_errp'+strtrim(string(i),2))

   aPlotInfo.typeVector(2)=aPlotInfo.typeVector(2) < (N_ELEMENTS(typeNames)-1)
   aPlotInfo.typeVector(2)=aPlotInfo.typeVector(2) > 0
   WIDGET_CONTROL, spb_type(aPlotInfo.typeVector(2)), SET_BUTTON=1
   
;--- add the widgets for choosing the plot-style -------------------------------
   widgets=PAPCO_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
   
   additionalWData=create_struct(widgets, additionalWData)
   
;--- now, add the widgets for choosing the y-axis-ranges -----------------------
   widgets= hydra_ylim_widget(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
; Reload button
   type_base=WIDGET_BASE(panel_base, /FRAME, /row )
   spb= widget_button( type_base, $
                       value='Reload Data', $
                       uvalue='spb_reload' )
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ----------
;    editor window.
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo, /config)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   if hydra_new_papco(vers=vers) and vers ge 9.0 then begin
       thisWidgetsData={helpfile:'hydra_tpphr_panel.help', $
                        typeVector:aPlotInfo.typeVector, $
                        spb_type:spb_type}
   endif else begin
       thisWidgetsData={helpfile:'hydra_tpphr_panel.help', $
                        typeVector:aPlotInfo.typeVector, $
                        spb_type:spb_type}
   endelse
   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
   
;--- realize the window
   PAPCO_Panel_Realize, panel_base, widgetData
   
; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the 
; data contained in aPlotInfo would be returned.
   xmanager, 'hydra_tpphr_panel', panel_base, /MODAL
   
   
; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={ typeVector:widgetdata.typeVector }
   additionalRData=PAPCO_panel_ReturnData(widgetData)
   
   return, create_struct(ReturnData, additionalRData)
END




;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO hydra_tpphr_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     november 1995, written by R. Friedel at Goddard
;*
;***************************************************************************
PRO hydra_tpphr_panel_event, event
   COMMON hydra_tpphr_panel_DATA, widgetData
   WIDGET_CONTROL,GET_UVALUE=uval, event.id
   
   COMMON hydra_tpphr_data, $
    tpphr_nsweep_index, tpphr_nsweep, $
    tpphr_header, $
    tpphr_data1, $
    tpphr_data2, $
    tpphr_data3, $
    tpphr_data6
   
   IF strleft(uval, 8) EQ 'spb_type' THEN BEGIN
      widgetData.typeVector(0)=fix(strfrom(uval,8))
      
   ENDIF ELSE IF strleft(uval,8) EQ 'spb_varb' THEN BEGIN
      widgetData.typeVector(1)=fix(strfrom(uval,8))

   ENDIF ELSE IF strleft(uval,8) EQ 'spb_errp' THEN BEGIN
      widgetData.typeVector(2)=fix(strfrom(uval,8))

   ENDIF ELSE IF uval EQ 'spb_reload' THEN BEGIN
      ires = widgetData.typeVector(0)
      IF (n_elements(tpphr_header) GT 0) THEN $
       tpphr_header(ires).reload_flag=1
      
   ENDIF ELSE BEGIN
      widgetData=  hydra_ylim_widget_event( event, widgetData, handled=h )
      if not h then widgetData=PAPCO_panel_Event(event, widgetData)
   ENDELSE
   
END

   




















