pro r_hydra_skew_l1, date_str, start_time, end_time, $
                     data_header, data_values, $
                     aligned_header, aligned_data, $
                     opposed_header, opposed_data

common get_error, get_err_no, get_err_msg

;  load data from level-one files.  
;  History:
;    Jun 97.  Code taken from spectra utility. JBF

   myname= 'r_hydra_skew_l1'

; index space definitions to avoid future confusion
;   day space      0 is first record in the day
;   izoom space    0 is the first block returned to spectra
;   mode space     0 is the first block in a new mode (edible data chunk)

   forward_function spectra_l1_correct
   forward_function spectra_l1_pitch

   if not keyword_set( data_header ) then begin
       print, 'Creating data header for debugging purposes! '
       hydra_define_structure, /spec_header
       common hydra_spec_data, dc, stn, sh
       data_header= sh(0)       
   endif                         

   data_header.reader_version= 2.0
   aligned_header.reader_version= 2.0
   opposed_header.reader_version= 2.0

   time_limit= getenv('HYDRA_L1_SPEC_TIME_LIMIT')
   if time_limit eq '' then time_limit = 9999 $ ; minutes
   else time_limit= long( time_limit )
   
   if end_time - start_time gt time_limit*60L then begin
       get_err_msg = 'To avoid rediculously long reads, data set is !c' + $
         'limited to '+strtrim(time_limit,2)+' minutes. The environment!c' + $
         'variable HYDRA_L1_SPEC_TIME_LIMIT specifies this time limit.'
       get_err_no= 1
       return
   endif       
   
   data_values1= { hydra_spec_data, $
                   time:double(0), $ ; seconds since start_time
                   mode:byte(0), $   ; telemetry mode specifies energies
                   spectrum:fltarr(111), $
                   species:byte(0), $ ; 1=electrons 2=ions 3=both
                   sc_pot:float(0), $
                   sc_pot_qual:byte(0), $
                   psi_status:byte(0) $
                 }

retry:
   f= hydra_findfile( date_str, /level1 )
   if get_err_no ne 0 then return
   fcal= hydra_findfile( date_str, /ddcal, version=l1version )

   if (fcal ne '' and f ne '') then begin
       CDF= cdf_open( f )     
       CAL= cdf_open( fcal )
       get_err_no=0             ; tell papco all is well
   endif else begin
       hydra_retrieve_data, $
         'hydra_spec_l1', '19'+date_str, status
       if (status eq 'okay') then goto, retry else return
   endelse

   print, myname+': Reading level1 file '+f+'...'
   
   b_sec= data_header.time_resolution

   print, 'Averaging for '+strtrim(b_sec,2)+' second resolution.'

   izoom_start= long(start_time/b_sec)
   izoom_end= long(end_time/b_sec)+1
   n_blocks= izoom_end-izoom_start+1

   subblocks= fix( 13.8/b_sec +0.5 )
   recstart= izoom_start/subblocks
   reccount= n_blocks/subblocks+1
   nblocks= reccount*subblocks
   
   data_values= replicate( data_values1, nblocks )
   aligned_data= replicate( data_values1, nblocks )
   opposed_data= replicate( data_values1, nblocks )

;  get total number of blocks in the file
   cdf_control, CDF, variable='BLOCK_STATUS', get_var_info=r
   nrec= r.maxrec

;  get starting record and number of records
   cdf_varget, CDF, 'BLOCK_TIME', block_epoch, $
     rec_start=0, rec_count=nrec
   rectime= block_epoch mod 86400000 / 1000
   rtime= where( rectime le izoom_start*b_sec )
   if rtime(0) ne -1 then recstart= rtime(n_elements(rtime)-1) else recstart=0

   if (reccount gt nrec-recstart) then begin
       reccount= nrec-recstart 
       nblocks= reccount*subblocks
   endif

   if reccount le 1 then begin
       get_err_msg='No data found at or after this time'
       print, get_err_msg
       get_err_no=1
       return
   endif

   cdf_control, CDF, variable='BLOCK_STATUS', get_var_info=r
   if (reccount gt r.maxrec-recstart) then begin
       reccount= r.maxrec-recstart 
       nblocks= reccount*subblocks
   endif

   usesangles = 1

; pitch cache files won't be used
   l1pitch=-1

   cdf_varget, CDF, 'BLOCK_STATUS', block_status, $
     rec_start=recstart, rec_count= reccount
   block_status=reform(block_status)

; poke in flags to make spectra process 45min at a time.
   r=indgen((reccount/200)+1)*200
   block_status(r)= block_status(r) or 2
   
; rnew is where mode or bias levels change
   rnew= where(( block_status and 2 ) ne 0 ) + recstart
   if rnew(0) eq -1 then rnew=0 ; else $
   nnew= n_elements(rnew)

;  get the overflow block numbers
   cdf_control, CDF, get_var_info=v, variable='OV_BLK_NUM'
   if (v.maxrec ge 0) then begin
       cdf_varget, CDF,'OV_BLK_NUM',ov_blk_num,rec_start=0,rec_count=v.maxrec+1
   endif else begin
       ov_blk_num=-1
   endelse

;  GET THE TIME-STAT FILE INFO
   cdf_varget, CDF, 'BLOCK_TIME', block_epoch, $
     rec_start=recstart, rec_count= reccount

   block_time0= fltarr(reccount)
   block_time0(*)= ( block_epoch(*) mod (86400*1000L) ) / 1000
   block_time0_= rebin( block_time0, nblocks, /sample )
   r=indgen(reccount)*subblocks
   for isub=1,subblocks-1 do $
     block_time0_(r+isub)= block_time0_(r+isub) + isub*b_sec
   
;  make itime array 
   itimeoff= long(block_time0_(0)/b_sec)
   itime= long(block_time0_/b_sec)-itimeoff
   
   r= where( itime lt nblocks)  ; might pull off too many 
   itime=itime(r)
   icont=indgen(n_elements(itime)) ; continuous indeces

;  now fill the arrays
   block_time0= fltarr(nblocks)
   block_time0(itime)= ((block_time0_))(icont)
   data_values(itime).time= ((block_time0_))(icont)
   aligned_data(itime).time= ((block_time0_))(icont)
   opposed_data(itime).time= (temporary(block_time0_))(icont)
   
   
   cdf_varget, CDF, 'BLOCK_MODE', block_model1, $
     rec_start=recstart, rec_count= reccount
   block_var= rebin(block_model1,nblocks, /sample)
   block_mode0= intarr(nblocks)
   data_values(itime).mode= block_var(icont)
   aligned_data(itime).mode= block_var(icont)
   opposed_data(itime).mode= block_var(icont)
   
   cdf_varget, CDF, 'SC_POT', block_var, $
     rec_start=recstart, rec_count= reccount
   r= where(block_var eq -1e-31)
   if r(0) ne -1 then block_var(r)=1000.1 ; fill
   block_var= rebin( reform( block_var ), nblocks, /sample ) 
   block_potential0= make_array( nblocks, /float, value=1001 )
   block_potential0(itime)= block_var(icont)
   data_values(itime).sc_pot= block_var(icont)
   aligned_data(itime).sc_pot= block_var(icont)
   opposed_data(itime).sc_pot= block_var(icont)

   cdf_varget, CDF, 'POT_QUALITY', pot_quality, $
     rec_start=recstart, rec_count= reccount

   data_values(itime).sc_pot_qual= pot_quality(icont)
   aligned_data(itime).sc_pot_qual= pot_quality(icont)
   opposed_data(itime).sc_pot_qual= pot_quality(icont)

   cdf_varget, CDF, 'SWEEP_POT_MAX', hr_sc_pot, $
     rec_start=recstart, rec_count= reccount

;  How close can the s/c potential be before we decide to throw out
;  the point?  ( sc_pot / pot_fraction ) > observed_energy !
   cdf_varget, CDF, 'POT_FRACTION', pot_fraction

;  CHECK TO SEE IF MFE HIGH RES BFIELD DATA EXISTS IN L1 FILE
   cdf_control, CDF, var='B_HIGH_RAW', get_var_info=r 
   if n_elements( r ) gt 0 then begin
       if r.maxrec gt -1 then begin
           highres=1
       endif else begin
           highres=0
       endelse
   endif 

;  GET ANGLES INFORMATION
   cdf_varget, CDF, 'PHASE_START', phase_start, $
     rec_start=recstart, rec_count=reccount
   phase_start= reform(phase_start)

;  GET OVERFLOW BLOCKS
   if (ov_blk_num)(0) ne -1 then begin
       cdf_varget, CDF, 'OV_INDEX', ov_index   
       cdf_varget, CDF, 'OV_BLK_NUM', rec_count=ov_index, ov_block_num       
   endif else begin
       ov_block_num=-1
   endelse

;  AVERAGE THE DATA
   nen= 111
   mode_length= intarr(51)

   aligned_pr= aligned_header.pitch_range * !dtor
   opposed_pr= opposed_header.pitch_range * !dtor
   
   for imode= 0, nnew-1 do begin

       rnewi= rnew(imode)
       recstartoff= rnewi-recstart ; goes from mode space to izoom space
       
       modei= block_model1(recstartoff)

       cdf_varget1, CDF, 'BLOCK_INDEX', rec_start=rnewi, offset
       offset= offset-1
       
       cdf_varget, CAL, 'LENGTH', length, rec_start= modei-1

       cdf_varget, CAL, 'CAL_ENERGY_INDEX', energy_index, rec_start= modei-1

       rele= where( energy_index(*,0) lt 0 )
       rion= where( energy_index(*,0) gt 0 )

       mode_length( modei ) = length

       cdf_varget, CAL, 'CAL_ENERGY_SEQ_BOX', energy_seq, rec_start= modei-1

       cdf_varget, CDF, 'ELE_BIAS', elebiasi, rec_start= rnewi
       cdf_varget, CDF, 'ION_BIAS', ionbiasi, rec_start= rnewi
       
       counts_field= string(modei,format='(i3.3)')+'_e_'+ $
         string(elebiasi(0),format='(i2.2)')+'_'+ $
         string(elebiasi(1),format='(i2.2)')+'_i_'+ $
         string(ionbiasi(0),format='(i2.2)')+'_'+ $
         string(ionbiasi(1),format='(i2.2)')

       if imode eq nnew-1 then nrecmode=recstart+reccount-rnewi else $
         nrecmode=rnew(imode+1)-rnewi
       print, ' '
       print, 'averaging ('+strtrim(rnewi-recstart,2)+'/'+$
         strtrim(reccount,2)+')...', format='(A,$)'
       cdf_varget, CDF, 'COUNTS_'+counts_field, counts, $
         rec_start=offset, rec_count=nrecmode

       if ( (usesangles and (l1pitch eq -1)) or $
            (  highres and (b_sec lt 13.8) ) ) then begin
           cdf_varget, CDF, 'PHIB_'+counts_field, phib, $
             rec_start=offset, rec_count=nrecmode
           
           cdf_varget, CDF, 'THETAB_'+counts_field, thetab, $
             rec_start=offset, rec_count=nrecmode
           
           correct_bfield, phib, thetab, bmag, bvalid, rnewi, nrecmode, CDF
       endif 
       
       if ( usesangles and (l1pitch ne -1) ) then begin
           cdf_varget, l1pitch, 'PITCH_ANGLE', alphas, $
             rec_start=rnewi, rec_count=nrecmode
       endif

       split= modei eq 10 and subblocks gt 1       
       
       spectra_l1_average, /init, $
         energy_index=energy_index, energy_seq=energy_seq, $
         split=split, intdata=intdata
       
       nen= n_elements( intdata(0).energy_interp )
       r55= indgen(nen) + 55 - (nen/2)
       
       data_header.mode_energy( r55, modei-1 ) = $
         10. ^ intdata(0).energy_interp( * )
       data_header.mode_energy( 0:54, modei-1 ) = $
         -1.0 * data_header.mode_energy( 0:54, modei-1 )
       data_header.mode_energy( 55, modei-1 ) = 0.0 ; very important!

       r= spectra_l1_correct( /init, rawid=CDF, calid=CAL, $
                              mode=modei, $
                              ele_bias=elebiasi, ion_bias=ionbiasi, $
                              /sigma_cc )

       r= spectra_l1_pitch( /init,  $
                            mode=modei, rele=rele, rion=rion, $
                            cdfcal= CAL )
       
       spec_out= fltarr( nen, subblocks )
; average each block...
       for ii= 0, nrecmode-1 do begin                                 
           print, '.', format="(A,$)"
           icounts= counts(*,*,ii)
           datamask= ( (counts(*,*,ii) and -16384) eq 0 )
           
;            sc_pot filter code was here!

           if ( block_status(ii+recstartoff) and 1 ) then begin ; overflow
               icounts=long(icounts)
               me=where(ov_block_num eq ii+rnewi+1) ; ii to day space
               for i=0, n_elements(me)-1 do begin
                   cdf_varget, CDF, 'OV_STEP_NUM', rec_start=me(i), st
                   cdf_varget, CDF, 'OV_DET_NUM', rec_start=me(i), det
                   cdf_varget, CDF, 'OV_COUNTS', rec_start=me(i), ovcounts
                   icounts(st-1,det-1)=ovcounts
               endfor
           endif

           counts1= spectra_l1_correct( icounts, ii+rnewi, sigma_cc= dcounts1 )

           if ( usesangles ) then begin
               if ( l1pitch eq -1 ) then begin ;calculate pitch
                   pitch= spectra_l1_pitch( $
                                            phase_start(ii+recstartoff),$
                                            thetab(*,ii),phib(*,ii), $
                                            bvalid(*,ii) $
                                          )
               endif else begin ; read it from l1 cache file
                   ipitch= alphas(*,*,ii)
                   pitch= (float( ipitch )+0.5)*!pi/256
               endelse
           endif


;         calculate the three spectrograms
;         all 
           spectra_l1_average, counts1, datamask, subblocks, $
             all_out

;         aligned
           datamask1= datamask
           r= where( pitch gt aligned_pr(1) )
           if r(0) ne -1 then datamask1(r)= 0
           spectra_l1_average, counts1, datamask1, subblocks, $
             sigma_counts= dcounts1, $
             aligned_out, sigma_spec_out= sigma_aligned_out
           
;         opposed
           datamask1= datamask
           r= where( pitch lt opposed_pr(0) )
           if r(0) ne -1 then datamask1(r)= 0
           spectra_l1_average, counts1, datamask1, subblocks, $
             sigma_counts= dcounts1, $
             opposed_out, sigma_spec_out= sigma_opposed_out

           itime= long( block_time0( (ii+recstartoff)*subblocks ) $
                        / b_sec ) - itimeoff
           

;         skew
           r= where( aligned_out ne 1e-20 and opposed_out ne 1e-20 ) 
           rn= where( aligned_out eq 1e-20 or opposed_out eq 1e-20 )
           if r(0) ne -1 then begin
               rr= where( all_out(r) ge 0.1 )
               rrn= where( all_out(r) lt 0.1 )
               if rrn(0) ne -1 then spec_out(r(rrn))= 1e-20
               if rr(0) ne -1 then begin
                   r= r(rr)
                   spec_out(r)= $
                     ( aligned_out(r) - opposed_out(r) ) / $
                     sqrt( sigma_aligned_out(r) + sigma_opposed_out(r) )
               endif
           endif

           if rn(0) ne -1 then spec_out(rn)= 1e-20

           data_values( itime+indgen(subblocks) ).spectrum(r55) = spec_out
           aligned_data( itime+indgen(subblocks) ).spectrum(r55) = aligned_out
           opposed_data( itime+indgen(subblocks) ).spectrum(r55) = opposed_out

           if ( pot_quality(recstartoff+ii) and 2 ) eq 2 then begin
               if subblocks eq 1 then begin
                   data_values( itime ).sc_pot= $
                     total( hr_sc_pot( *, recstartoff+ii ) ) / 12 + 2.0
                   aligned_data( itime ).sc_pot= $
                     total( hr_sc_pot( *, recstartoff+ii ) ) / 12 + 2.0
                   opposed_data( itime ).sc_pot= $
                     total( hr_sc_pot( *, recstartoff+ii ) ) / 12 + 2.0
               endif else begin
                   r= indgen( subblocks < 12 ) * ( 12 / subblocks ) 
                   data_values( itime+indgen(subblocks) ).sc_pot= $
                     hr_sc_pot( r, recstartoff+ii ) + 2.0
                   aligned_data( itime+indgen(subblocks) ).sc_pot= $
                     hr_sc_pot( r, recstartoff+ii ) + 2.0
                   opposed_data( itime+indgen(subblocks) ).sc_pot= $
                     hr_sc_pot( r, recstartoff+ii ) + 2.0
               endelse
           endif
           
       endfor
   endfor
   
;  get "stamp" info
   CDF_varget, CAL, 'ELE_RGAIN_VERSION_'+counts_field(0), rgains_vers 
   CDF_varget, CAL, 'ELE_IGAIN_VERSION_'+counts_field(0), igains_vers
   CDF_varget, CDF, 'GLINT_MASK_VERSION', gmver
   CDF_varget, CAL, 'CAL_ENERGY_VERSION', isweep_version, rec_start=modei-1
   data_header.intersweep_version= $
     min( [ min( isweep_version ), data_header.intersweep_version ] )
   data_header.rgains_version= min([ rgains_vers, data_header.rgains_version ])
   data_header.igains_version= min([ igains_vers, data_header.igains_version ])
   data_header.glint_mask_version= $
     min ([ gmver,data_header.glint_mask_version])
   data_header.data_version= min( [ l1version, data_header.data_version] )

   CDF_varget, CDF, 'SC_POT_VERSION', scpot_vers
   data_header.sc_potential_version= $
     min([ scpot_vers, data_header.sc_potential_version ])

;   CDF_varget, cdf, 'CALIBRATION_VERSION', 
   data_header.calibration_version=l1version
   
   cdf_close, CDF
   cdf_close, CAL
   if (l1pitch ne -1) then cdf_close, l1pitch

   return
end

;           if ( pot_quality(recstartoff+ii) and 2 ) eq 2 then begin
;               sc_pot_en= rebin( hr_sc_pot( *, recstartoff+ii ), 384, /sample )
;               r= where( energy_seq( rele, 0 ) gt $
;                         -sc_pot_en(rele)/pot_fraction )
;               if r(0) ne -1 then begin
;                   ien_mask= rele(r)
;                   for idet=0,10,2 do datamask( ien_mask, idet )= 0
;               endif
;               r= where( energy_seq( rele, 1 ) gt -sc_pot_en(rele) )
;               if r(0) ne -1 then begin
;                   ien_mask= rele(r)
;                   for idet=1,11,2 do datamask( ien_mask, idet )= 0
;               endif
;           endif               
