pro r_hydra_spec_survey, date_str, start_time, end_time, $
                         data_header, data_values

; Read Spectrogram Survey
; Reads in one day of survey data.
;
; HISTORY:
;    v1.0  June 30, 97    Jeremy Faden    First rev coded and released.
;    v1.3  July 1, 97     Jeremy Faden    Fixed time tag bug -- whoops.
;    v1.4  July 23, 97    JBF             Time tags changed to double,
;                                         float is too dangerous
;                                         hydra_findfile function used
;    v1.5  Sept 2, 97     jbf             more versions checked, and
;                                            min(curr,this) is returned.

  common get_error, get_err_no, get_err_msg

  my_name= 'r_hydra_spec_survey'
  data_header.reader_version= 1.4
;  1.4 uses hydra_findfile
;  1.41 handles energy modes with less than 110 energies.

  retry:
  f= hydra_findfile( date_str, /survey, version=version )
  if f eq '' then begin
      hydra_retrieve_data, $
        'hydra_spec_survey', '19'+date_str, status
      if (status eq 'okay') then goto, retry else return
  endif

  if version lt 1.20 then begin
      get_err_msg= 'Hydra Survey file versions less than 1.20 no longer!c'+$
        'support.  To update your data files, please visit!c '+$
        'Hydra''s data site. '
      get_err_no=1
      return
  endif

  CDF= cdf_open(f)
  data_header.data_version= version
  get_err_no=0

  cdf_varget, cdf, 'b_sec', b_sec
  data_header.time_resolution=b_sec
  
  cdf_varget, cdf, 'MODE', mode, rec_count=6272
  rvalid= where( mode gt 0 )

  if rvalid(0) eq -1 then begin
      get_err_msg=my_name+': survey file contains no data.'
      print, get_err_msg
      get_err_no=1
      return
  endif

  cdf_varget, cdf, 'TIME', time, rec_count=6272
  rec_start= ( where( time(rvalid) ge start_time ) )(0)
  if rec_start(0) eq -1 then begin
      get_err_msg= my_name+': No data found before start_time '
      print, get_err_msg
      return
  endif

  rec_start= rvalid( rec_start )

  r= where( time lt end_time $
            and time gt start_time, count )

  if count le 1 then begin
      print, my_name+': no data found'
      return
  endif 
  
  rec_count= r(count-1) - r(0) + 1

  data_values1= { hydra_spec_data, $
                  time:double(0), $
                  mode:byte(0), $
                  spectrum:fltarr(111), $
                  species:byte(0), $ ; 1=ele 2=ions 3=both
                  sc_pot:float(0), $
                  sc_pot_qual:byte(0), $
                  psi_status:byte(0) $
                }
  
  data_values= replicate( data_values1, count )

  data_values(*).time= time(r)
  r=r-r(0)

  cdf_varget, cdf, 'MODE_ENERGY', rec_count=50, mode_energy 
  nen= n_elements( mode_energy(*,0) )

  r55= indgen(nen) + 55 - (nen/2)

  modes= where( total( abs( mode_energy ), 1 ) gt 0 )
  for i = 0, n_elements( modes )-1 do begin      
      imode= modes(i)
      data_header.mode_energy(r55,imode)= mode_energy(*,imode)
  endfor

  case data_header.spec_type_id of
      0: zvar_name='SPEC'
      1: zvar_name='SPEC_ALIGNED'
      2: zvar_name='SPEC_PERP'
      3: zvar_name='SPEC_OPPOSED'
      4: zvar_name='SPEC_SKEW'
      else: begin
          get_err_msg= my_name+': internal error.'
          print, get_err_msg
          get_err_no=1
          return
      end
  endcase

  cdf_varget, cdf, zvar_name, rec_start=rec_start, rec_count=rec_count, $
    spec_data
  
  if zvar_name eq 'SPEC_SKEW' then begin
      spec_data_temp= float( spec_data ) / 10
      rnodata= where( temporary(spec_data) eq -32768 )
      if rnodata(0) ne -1 then spec_data_temp(rnodata)= 1e-20
  endif else begin
      spec_data_temp= 10 ^ ( float( temporary(spec_data) ) / 1000 )
  endelse
  spec_data= temporary( spec_data_temp )

  data_values(*).spectrum(r55)= spec_data(*,r)

  cdf_varget, cdf, 'SC_POT', rec_start=rec_start, rec_count=rec_count, $
    sc_pot
  data_values(*).sc_pot= sc_pot(r)

  data_values(*).sc_pot_qual= 1 ; Goddard

  cdf_varget, cdf, 'MODE', rec_start=rec_start, rec_count=rec_count, $
    mode
  data_values(*).mode= mode(r)

  cdf_varget, cdf, 'GLINT_MASK_VERSION', glint_mask_version
  data_header.glint_mask_version= $
    min( [ glint_mask_version, data_header.glint_mask_version ] )
  
  cdf_varget, cdf, 'CALIBRATION_VERSION', calibration_version
  data_header.calibration_version= $
    min( [ calibration_version, data_header.calibration_version ] )

  cdf_varget, cdf, 'RGAINS_VERSION', rgains_version
  data_header.rgains_version= $
    min( [ rgains_version, data_header.rgains_version ] )

  cdf_varget, cdf, 'IGAINS_VERSION', igains_version
  data_header.igains_version= $
    min( [ igains_version, data_header.igains_version ] )

;  cdf_varget, cdf, 'ISWEEP_VERSION', intersweep_version, rec_start=modei-1
  cdf_varget, cdf, 'ISWEEP_VERSION', intersweep_version ; rec_start=modei-1
  data_header.intersweep_version= $
    min( [ intersweep_version, data_header.intersweep_version ] )

  cdf_varget, cdf, 'SC_POTENTIAL_VERSION', sc_potential_version
  data_header.sc_potential_version= $
    min( [ sc_potential_version, data_header.sc_potential_version ] )
  
  return
end


