;+
; NAME: hydra_color_bar
;
; PURPOSE:
; This IDL-procedure plots a colorbar on the right side of the screen
; in a standard way. (i.e. multiple plots will have colorbars that
; look similar.
; Intended to be very general yet easy to use, by allowing the user
; to override the automatic location of the colorbar relative to the
; plot, and to print only the ztitle or colorbar image.  Also, plot
; keywords not recognized are passed onto the axis command, so for
; instance, tickformats can be specified (use ytickformat).
;
;	
; CATEGORY:
;       Plotting (Color)
;
; CALLING SEQUENCE:
;
;       hydra_COLOR_BAR, [ZMIN],[ZMAX],[ztitle], $
;            zrange=zrange, zlog=zlog, ztitle=ztitle $
;            ypos=ypos, xpos=xpos, nobar=nobar, _extra=e
;          
;
; INPUTS:     in accordance with PLOT_CLR
;             ZMIN     float bottom value of color axis (optional)
;             ZMAX     float top value of color axis (optional)
;             ZTITLE   label for colorbar.  (optional)
;             zrange, ztitle have functionality of ZMIN,ZMAX,ZTITLE
;             /zlog    use log axis
;             ypos=[y1,y2]  specify location of colorbar in normal
;                 coordinates relative to current plot window.
;                  i.e. [ 0.5,1 ] is half heieht of current plot.
;             xpos=[y1,y2]  specify location of colorbar like with y

;	
; PROCEDURE:  location of the colorbar is determined from
;                the current plot position, ypos, and xpos specs.
;             colorbar colors are drawn with polyfills.
;             axis is used to draw axis.  
;             ztitle is printed.
;
; MODIFICATION HISTORY:
;     Written for PaPCo, October 7, 1997, Jeremy Faden
;
PRO hydra_COLOR_BAR, ZMIN,ZMAX,ztitle, $
                     zlog=zlog, zrange=zrange, ztitle=ztitle_key, $
                     ypos=ypos, xpos=xpos, nobar=nobar, _extra=e

  common papco_color, papco_color, r_papco, g_papco, b_papco


; PROCESS INPUT PARAMETERS
  nobar= keyword_set(nobar)

  IF N_PARAMS(0) LT 3 THEN begin
      if not nobar then begin
          zmin= zrange(0)
          zmax= zrange(1)
      endif
      if keyword_set( ztitle_key ) then ztitle=ztitle_key else ztitle=''
  endif

  zlog= keyword_set(zlog)

  if n_elements( ypos ) ne 2 then ypos=[ 0, 1. ]
  if n_elements( xpos ) ne 2 then xpos=[ 1+1/11., 1+1/11.+1/30. ]

; find colorbar range
  cindex= where( papco_color(*,2) eq 1 )
  if (cindex(0) eq -1) then begin
      print, 'Color system failure, consult jbf@space-theory.physics.uiowa.edu'
      print, 'Using color set 0'
      cindex=0
  endif
  color_range= PAPCO_get_Colorindices( cindex )
  cbase= color_range(0)
  clen= color_range(1)-color_range(0)


  ctop= color_range(1)
  
; save current viewport and data-screen-relation:
   pa=!p
   xa=!x
   ya=!y
   colrat=clen/255.0
   ticklen=-.2

   scalex=(!p.position(2)-!p.position(0))
   scaley=(!p.position(3)-!p.position(1))
   !p.position(0)= !p.position(0)+ xpos(0)*scalex
   !p.position(2)= !p.position(0)+ xpos(1)*scalex
   SIZE=SQRT(scaley/scalex)*1.5
   xrange=[0.,100.]
   yrange=[ZMIN,ZMAX]
   xstyle=4+1
;
   linlogbit= keyword_set( zlog )

   ystyle = 1+4
   zfac= DOUBLE(zmax-zmin)/clen
   PLOT,[0,100],[zmin,zmin],color=!p.background,/noerase
   
;*******************create color bar (linear color sequence)
   FOR I=1,clen DO BEGIN
       ZINT1=(I-1)*zfac+zmin
       ZINT2=(I)*zfac+zmin
       color= cbase+I-1
       POLYFILL,[0.,0.,100.,100.],[ZINT1,ZINT2,ZINT2,ZINT1],col=color
   ENDFOR
   
;*******************axes and annotation (only here do we have to distinguish
;                   linear and logarithmic scaling)
   IF zlog  THEN BEGIN          ; logarithmic
       axis, yaxis=1, ylog=ylog, yrange=[zmin,zmax], ticklen=-0.6
   ENDIF ELSE BEGIN             ; linear
       !y.ticks=FIX(zfac)/20
       !x.style=4
       !p.color=1
       AXIS,yaxis=1,ystyle=1,charsize=1.0, _extra=e
   ENDELSE
      
;      ; reset plot position for Z axis title
      !p.position(1) = ymin
      !p.position(3) = ymax
      
  
  !p.font=-1
  !p.position(0)=!p.position(2)
  !p.position(2)=!p.position(0)+scalex/8.0
  !y.range=[0.,100.]
  PLOT,[0.,100.],[100., 100.] , /noerase, /nodata,  ystyle = 5,  xstyle = 5
  XYOUTS,75.,50.,ZTITLE,alig=0.5,chars=SIZE,orient=-90.,color=1
  !x=xa
  !y=ya
  !p=pa
  return
end
