;$Id: fitf_plot.pro,v 1.1.1.1 2003/11/04 21:24:01 friedel Exp $
function fitf_plot, state, datatime = datatime
;+
; NAME:
;       fitf_plot
;
; PURPOSE:
;       
;       This function facilitates plotting to the draw window of the
;       hydra_fitf widget.
;
; CATEGORY:
;       
;       hydra_fitf
;
; CALLING SEQUENCE:
;       
;               result=fitf_plot(state, datatime = datatime)
;
; INPUTS:
;       state:  The state vector of the widget from which plotting
;               parameters are to be taken.
;
; KEYWORD PARAMETERS:
;       state.mode(state.select).diag:   A swwitch to turn off informaitonal x-windows
;                     messages about getting data and plotting.
;
;       datatime:     Return the time range plotted.
;
; OUTPUTS:
;       
;       result:       0 for failure, 1 for success.
;
; SIDE EFFECTS:
;       A new plot is put in the slice window draw widget.
;
; EXAMPLE:
;
;        child = widget_info(hydra_fitf.top, /child)
;        widget_control, child, get_uvalue = state
;
;        result=fitf_plot(state, datatime = datatime)
;
;        if (state.papco ne 0) then begin ; move papco cursor
;            striketime = strmid(state.args.path, 4, 2)+'/'+$
;              strmid(state.args.path, 6, 2)+'/'+$
;              strmid(state.args.path, 0, 4)+' '+strmid(datatime(0), 0, 8)
;            hydra_strike, striketime
;        endif                   ; move papco cursor    
;
;        See the documentation in init_fitf for a description of
;        state.  See the documentation of hydra_fitf.pro for a
;        description of where state is generally stored.
;
; Written by:   Eric E. Dors, 1 March 1998.
;
; MODIFICATION HISTORY:
;
;       Fri May 14 08:19:28 1999, Eric E. Dors
;       <edors@universe.lanl.gov>
;
;		Prob was not being written out for Maxwellian fits, fixed.
;
;       Tue Mar 16 21:31:56 1999, Eric E. Dors
;       <edors@universe.lanl.gov>
;
;		Made the energy error propigation conditional on
;		enough data points to compute it.
;
;               Added error message for when fitf_read fails.
;
;       Sun Feb 21 15:34:46 1999, Eric E. Dors
;       <edors@universe.lanl.gov>
;
;		Commented out patch code which wrote PostScript
;		characters instead of vector characters.
;
;       Sun Feb 21 15:13:48 1999, Eric E. Dors
;       <edors@universe.lanl.gov>
;
;               Updated example to be consistent with change in
;               state.args.path.  It now uses four character years
;               instead of two character years.
;
;       Thu Aug 27 16:52:50 1998, Eric E. Dors  <edors@universe.lanl.gov>
;
;               nodiagnostics keyword removed and the state vector is
;               checked for state.mode(state.select).diag to decide
;               weather to plot informational messages or not.
;
;       Fri Apr 10 13:56:00 1998, Eric Dors <eed@cis>
;
;               Fixed error in reporting angular range.  Range was
;               specifield from amin to amin.  Now it correctly
;               specifies amin to amax.
;
;       Mon Mar 16 14:01:11 1998, Eric Dors <eed@cis>
;
;               Previous fix for foregound color didn't work.  White
;               background in widget window couldn't see anything in
;               the !p.color color because, !p.color isn't updated by
;               PAPCO when a different background color is selected.
;               A hack was added to get around this problem, now we
;               check to see if the device is X, if so: use papco's
;               color table entry 1 for the foreground, if not: use
;               black for the foreground.  Another problem to be fixed
;               in the future is to reset the color table so that the
;               full 255 color table is available to this routine when
;               printing, not the small subset table declaired by PAPCO.
;
;       Mon Mar 9 19:32:15 1998, Eric Dors <eed@cis.physics.uiowa.edu>
;
;               Changed the color fg to be !p.color instead of the
;               papco forground index (1).  Now the forground color is
;               appropriately changed when output is swwitched from
;               screen to printer or printer to screen.
;
;-

COMMON papco_color_names
IF ( state.papco EQ 0) THEN BEGIN
    hydra_setcolors, C
    red = c.red
    yellow = c.yellow
    blue = c.blue
    green = c.green
    fg = !p.color
ENDIF ELSE BEGIN
    ;;This is a hack to get printing to work correctly in PAPCO.  It
    ;;is not clear on how to tell papco to reset the color tables for
    ;;a different device other than X11 (or for X11 for that matter).
    ;;Specifically, it would be nice to have a way to get the
    ;;forground and background colors of the current device without
    ;;each routine having its own look-up table for device to fg
    ;;color.  Especially since PAPCO advertises that colors 0 and 1 ar
    ;;the background and forground.  Use of the !p.color and
    ;;!p.background colors kept by IDL is precluded because PAPCO
    ;;doesn't update them for the background and forground selected by
    ;;the user.  Most annoyingly, the default forground and background
    ;;are not updated into the !p variables.  Also there is a problem
    ;;here that because we don't reinitialize the color table for
    ;;printing, a full 255 color color table is not available like it
    ;;should be.
    IF (!d.name EQ 'X') THEN BEGIN
        fg = 1                  ;PAPCO sets the default color for X11
    ENDIF ELSE BEGIN
        c = setcolors()
        red = c.red
        yellow = c.yellow
        blue = c.blue
        green = c.green
        fg = !p.color             ;black should be an appropriate fg color
    ENDELSE
ENDELSE

print,'plotting...'
widget_control, /hourglass

retval = fitf_read(state.args.path,state.args.t0, state.select, $
                   state.mode(state.select).den, $
                   state.mode(state.select).dalph,$
                   state.mode(state.select).asel, $
                   state.mode(state.select).hres, $
                   state.mode(state.select).seg_idx, $
                   energy,f,sigmaf, nbenergy, datatime, $
                   next = state.mode(state.select).next, $
                   prev = state.mode(state.select).prev, $
                   nocorrect = (state.mode(state.select).correct EQ 0), $
                   nopotential = (state.mode(state.select).potential EQ 0), $
                   parent = state.base_id, error = error)
IF (retval EQ 0) THEN BEGIN
    IF (keyword_set(state.mode(state.select).diag)) THEN BEGIN
        ret = widget_message(/error, 'Error: fitf_read failed.', $
                             dialog_parent = state.base_id)
    ENDIF
    noerase = !p.noerase
    !p.noerase = 0
    plot,[1],xstyle=4,ystyle=4,/nod 
    xyouts, /normal, 0.2, 0.5, 'Error: fitf_read failed.', $
      size = 3, color=fg
    !p.noerase = noerase 
    return, 0
ENDIF


notzero1 = where(f ne 0., n_notzero1)
npts1 = n_elements(f)
IF (n_notzero1 EQ 0) THEN BEGIN
    IF (keyword_set(state.mode(state.select).diag)) THEN BEGIN
        ret = widget_message(/error, 'Error: no data to plot', $
                             dialog_parent = state.base_id)
    ENDIF
    noerase = !p.noerase
    !p.noerase = 0
    plot,[1],xstyle=4,ystyle=4,/nod 
    xyouts, /normal, 0.2, 0.5, 'Error: no data to plot', size = 3, color=fg
    !p.noerase = noerase 
    return, 0
ENDIF
IF (n_notzero1 NE npts1 ) THEN BEGIN
    IF (keyword_set(state.mode(state.select).diag)) THEN BEGIN
        ret = widget_message(/info, 'Info: f is zero at '+$
                             strcompress(/rem,(npts1-n_notzero1))+' points.', $
                             dialog_parent = state.base_id)
    ENDIF
    sigmaf = sigmaf(notzero1)
    nbenergy = nbenergy(notzero1)
    energy = energy(notzero1)
    f = f(notzero1)
ENDIF

notzero2 = where(sigmaf ne 0., n_notzero2)
npts2 = n_elements(sigmaf)
IF ((n_notzero2 NE npts2) AND (n_notzero2 GT 0)) THEN BEGIN
    IF ( keyword_set(state.mode(state.select).diag)) THEN BEGIN
        ret = widget_message(/error, 'Error: sigmaf is zero at '+$
                             strcompress(/rem,(npts2-n_notzero2))+' points.', $
                             dialog_parent = state.base_id)
    ENDIF
    sigmaf = sigmaf(notzero2)
    nbenergy = nbenergy(notzero2)
    energy = energy(notzero2)
    f = f(notzero2)
ENDIF

;sigma_tot^2=sigmaf^2+(deltaE_E * df / d e)^2
IF npts2 GT 3 THEN BEGIN
    smoothf=median(f,3) 
    dfdenergy = deriv(energy, smoothf)
    deltaE_E = state.mode(state.select).denlow
    en_err = (deltaE_E*dfdenergy)
    sigmaf = sqrt(sigmaf^2+en_err^2)
ENDIF

;set up plot window
noerase = !p.noerase
xticks = !x.ticks
xminor = !x.minor
!p.noerase = 0
!x.ticks = 0
!x.minor = 0
plot, /nod, energy, f, psym = 2, yrange=[state.mode(state.select).zrange(0),$
                                         state.mode(state.select).zrange(1)], $
  xrange=[state.mode(state.select).erange(0),$
          state.mode(state.select).erange(1)], $
  xlog = state.mode(state.select).xlog, ylog = state.mode(state.select).ylog, $
  xstyle = 1, ystyle = 1, xtitle = 'Particle Energy (eV)', $
  ytitle = 'Distribution Function (cgs)', color = fg, $
  position = state.mode(state.select).normpos, $
  title = '!ATime: '+state.args.path+' '+datatime(0)+'-'+datatime(1)+'!N'
!p.noerase = noerase
!x.ticks = xticks
!x.minor = xminor
;plot the distribution function data
spoploterr, energy, f, sigmaf, color=fg
;plot the one count level
IF (state.mode(state.select).diag NE 0) THEN BEGIN
    en = state.mode(state.select).erange(0)+findgen(101)/100.*$
      (state.mode(state.select).erange(1)-state.mode(state.select).erange(0))
    oplot, en, state.faconec/en^2, color = yellow

    oplot, energy, state.faconec/energy^2/nbenergy, psym = 10, color = green
ENDIF

;reset fitf_xyouts counter for output at the top right of the plot
bangCcnt = 0

;output species status
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'Species: '+state.species(state.select), color=fg

;output s/c potential mode status
sccode = 0
scmesg = ['Off', 'On', 'Not Avail.']
IF (state.mode(state.select).potential EQ 1) THEN BEGIN
    IF (error(2) EQ 0) THEN BEGIN
        sccode = 1
    ENDIF ELSE BEGIN
        sccode = 2
    ENDELSE
ENDIF
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'S/C Potential subtract: '+scmesg(sccode), color=fg

;output corrected counts mode status
ccmesg = ['Off', 'On']
cccode = (state.mode(state.select).correct EQ 1)
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'Count Correction: '+(['Off', 'On'])(cccode), color=fg

;fit sigma status
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'Error method: '+state.cw_ids.errlabelg(state.mode(state.select).errors), $
  color=fg

;output angular binning status
amin = state.mode(state.select).dalph*(state.mode(state.select).asel-1.)
amax = state.mode(state.select).dalph*(state.mode(state.select).asel)
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'Angle bin: '+strcompress(string(amin, format = '(f4.1)'), /rem)+$
  '-'+strcompress(string(amax, format = '(f4.1)'), /rem)+' degrees', color=fg

;output energy bin size
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'Energy bin: '+strcompress(string(state.mode(state.select).den, $
                                    format = '(f4.2)'), /rem), color=fg

;output seg_idx mode status
;;don't forget that user thinks seg_idx uses ordinal numbers,
;;while we store it as an array index
fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
  'Sub-block: '+strcompress(string((state.mode(state.select).seg_idx+1), $
                                   format = '(i2)'), /rem)+$
  ' of '+strcompress(string(state.mode(state.select).hres, $
                            format = '(i2)'), /rem), color=fg

;remove points below outside of fit range
fitrange = where((energy GE state.mode(state.select).fitrange(0)) and $
                 (energy LE state.mode(state.select).fitrange(1)), n_tofit)
IF (n_tofit GT 0) THEN BEGIN
    print, 'Fit range restricted to ['+$
      strcompress(string(state.mode(state.select).fitrange(0)))+', '+$
      strcompress(string(state.mode(state.select).fitrange(1)))+']'

    sigmaf = sigmaf(fitrange)
    energy = energy(fitrange)
    f = f(fitrange)
ENDIF
;If there are less than two points a bunch of data assume to be arrays
;is no longer, the problem is somewhere in ddeis routines.  No fit
;could be made in this case anyway.
IF (n_elements(f) LE 1) THEN return, 0

;fit to model
retval = FitTempMaxw(energy, f, 0., chi2 = chi2, den = den, $
                     weight=(1d0/sigmaf^2), temp = temp, $
                     line = fit1, xline = xfit, sigma = sig, prob = prob, $
                     errors = state.mode(state.select).errors)

IF (retval NE 0) THEN BEGIN
    oplot, xfit, fit1, color = red
    print, 'Maxwellian fit results:'
    print, 'den =', den, ' +- ', sig(0), format = '(2(a,f7.3))'
    print, 'temp = ', temp, ' +- ', sig(1), format = '(2(a,f6.1))'
    print, 'chi2 = ', chi2, format = '(a,f7.3)'
    print, 'prob = ', prob, format = '(a,f6.2)'
    ;;
    fitf_xyouts, 0.65, 0.93, /normal, bangCcnt=bangCcnt, $
      'Maxwellian fit results:', color=red
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('den =', den, ' !M+!3', sig(0), format='(2(a,f7.3))'), color=red
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('temp = ', temp, ' !M+!3', sig(1), format='(2(a,f6.1))'), $
      color=red
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('!MX!3!E2!N = ', chi2, format='(a,f7.3)'), color=red
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('prob = ', prob, format = '(a,f6.2)'), color=red
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('n =', den, '!9 !7', sig(0), format='(2(a,f7.3))'), color=red
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('T = ', temp, '!9 !7', sig(1), format='(2(a,f6.1))'), $
;      color=red
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('!9C!7!E2!N = ', chi2, format='(a,f7.3)'), color=red
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('prob = ', prob, format = '(a,f6.2)'), color=red
    ;;in general the maxwellian temperature and density don't make
    ;;good initial guesses so make sure that the /maxw_guess swwitch is
    ;;set in call to fittempkappa to notify that the guess is from a
    ;;maxwellian 
    iguess = [den, temp, 2.5]   ;make initial guess for fittempkappa
ENDIF ELSE BEGIN
    IF (keyword_set(state.mode(state.select).diag)) THEN BEGIN
        ret = widget_message(/error, 'Error: Maxwellian fit failed.', $
                             dialog_parent = state.base_id)
    ENDIF
ENDELSE

retval = FitTempKappa(energy, f, 0., $
                      weight = 1d0/sigmaf^2, iguess=iguess, $
                      kappa = kappa, den = denk, temp = tempk, sig = sigk, $
                      line = fit2, xline = xfit, chi2 = chi2k, /maxw_guess, $
                      prob = probk, errors = state.mode(state.select).errors)

IF (retval NE 0) THEN BEGIN
    oplot, xfit, fit2, color = blue
    print, 'Kappa fit results:'
    print, 'den =', denk, ' +- ', sigk(0), format = '(2(a,f7.3))'
    print, 'temp = ', tempk, ' +- ', sigk(1), format = '(2(a,f6.1))'
    print, 'kappa = ', kappa, ' +- ', sigk(2), format = '(2(a,f6.1))'
    print, 'chi2 = ', chi2k, format = '(a,f7.3)'
    print, 'prob = ', probk, format = '(a,f6.2)'
    ;;
    fitf_xyouts, 0.65, 0.93, /normal, bangCcnt = bangCcnt, $
      'Kappa fit results:', color=blue
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('den =', denk, ' !M+', sigk(0), format = '(2(a,f7.3))'), $
      color = blue
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('temp = ', tempk, ' !M+', sigk(1), format='(2(a,f6.1))'), $
      color=blue
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('kappa = ', kappa, ' !M+', sigk(2), format='(2(a,f6.1))'), $
      color=blue
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('!MX!3!E2!N = ', chi2k, format = '(a,f7.3)'), $
      color=blue
    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
      string('prob = ', probk, format = '(a,f6.2)'), color=blue
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('n =', denk, '!9 !7', sigk(0), format = '(2(a,f7.3))'), $
;      color = blue
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('T = ', tempk, '!9 !7', sigk(1), format='(2(a,f6.1))'), $
;      color=blue
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('!9k = ', kappa, ' ', sigk(2), format='(2(a,f6.1))'), $
;      color=blue
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('!9C!7!E2!N = ', chi2k, format = '(a,f7.3)'), $
;      color=blue
;    fitf_xyouts, 0.70, 0.93, /normal, bangCcnt = bangCcnt, /compress, $
;      string('!7prob = ', probk, format = '(a,f6.2)'), color=blue
ENDIF ELSE BEGIN
    IF (keyword_set(state.mode(state.select).diag)) THEN BEGIN
        ret = widget_message(/error, 'Error: Kappa fit failed.', $
                             dialog_parent = state.base_id)
    ENDIF
ENDELSE

time_stamp

IF (state.mode(state.select).filename NE '') THEN BEGIN
    unit = state.mode(state.select).fileunit
    printf, unit, state.args.path+' '+datatime(0)+'-'+datatime(1), $
      den,  sig(0), temp, sig(1), chi2, prob, $
      denk,  sigk(0), tempk, sigk(1), kappa, sigk(2), $
      chi2k, probk, format = '(a,14(2x,g))'
;    format = '(a,2x,2(f7.3,2x),2(f6.1,2x),f7.3,2x,f6.2,'+$
;      '2(f7.3,2x),2(f6.1,2x),2(f6.1,2x),f7.3,2x,f6.2)'
ENDIF

return, 1

END
