;$Id: fittempmaxw.pro,v 1.1.1.1 2003/11/04 21:24:02 friedel Exp $
FUNCTION TempFunctMaxwML,  x, a, f, pder
COMMON FitPassMaxw, accel_p

arg1 = x-accel_p
ex1 = exp(-a(1)*arg1)
f = a(0)*ex1

dumb = where(finite(f) eq 0, num_bad)
IF ( num_bad GT 0 ) THEN BEGIN
    print, 'FitPassMaxwML function evaluation went wacky!'
    return, 0
ENDIF

IF n_params() GE 4 THEN BEGIN
    pder = [[ex1], $
            [-a(0)*ex1*arg1]]
ENDIF
return, 1
END
;=========================================================================
FUNCTION TempFunctMaxwSVD,  x, m
COMMON FitPassMaxw, accel_p

arg1 = x-accel_p
f = fltarr(n_elements(x), m)
f(*,0) = replicate(1.0, n_elements(x))
f(*,1) = -arg1

return, f
end
;=========================================================================
FUNCTION FitTempMaxw, xarr, yarr, emax, errors = errors, $
                      chi2 = chi2, line = fit, xline = xfit, density = den, $
                      potential = pot, temperature = temp, iguess = iguess, $
                      weight = weight, sigma = sig, probability = prob
;+
; NAME:
;       FitTempMaxw
;
; PURPOSE:
;       
;       This function fits the given data to a Maxwellian distribution
;       function and estimate the errors in the parameters.
;
; CATEGORY:
;       
;       hydra_fitf
;
; CALLING SEQUENCE:
;       
;       result = FitTempMaxw(energy, f, emax, chi2 = chi2, density = den, $
;                     weight=w, temperature = temp, $
;                     line = fit1, xline = xfit, sigma = sig, $
;                     probability = prob, errors = errormode)
;
; INPUTS:
;       energy: The energy of the data point.
;               
;       f:      The value of the distribution function for the data point.
;
;       e:      The value of the centroid for the Maxwellian fit function.
;
; KEYWORD PARAMETERS:
;       chi2:   The return value for chi squared by the fitting procedure.
;
;       density: The density result found by the fit.
;
;       temperature: The temperature result found by the fit.
;
;       weight: An array of sigmas for the distribution function.  The
;               array should have the same number of elements as f and
;               energy.
;
;       iguess: An initial guess of the temperature and density.  The
;               input is a two dimensional array [igden,igtemp]
;     
;       line:   The returned values of the distribution function for
;               each energy for the fit function.
;
;       xline:  The returned values of the energy used by the fit.
;
;       sigma:    The extimated errors in the parameters.  A two
;               dimensional array.  [sigdensity, sigtemp].
;
;       probability:   The probability that random fluctuations would produce
;               a value of chi-squared larger than chi2.
;
;       errors: The method to use in calculating the errors.  0: Use
;               the hessian returned by spcurvefit to estimate the
;               curvature near the fit values.  1: Search chi-squared
;               space for the appropriate 68% level contour.  If this
;               keyword is not passed, the errors are not calculated
;
; OUTPUTS:
;
;       result: 0 for failure, 1 for sucess
;
; COMMON BLOCKS:
;       accel_p: The value of the centroid of the fit.  This is in a
;                common block so its value can be shared with
;                TempFunctMaxwML with out making it a fit parameter.
;
; PROCEDURE:
;       
;       The non-linear marquardt-levenburg fit of a Maxwellian is
;       perfomed through use of the function spcurvefit.
;
; EXAMPLE:
;       
;       
;       IDL> Result = FitTempMaxw(energy, f, 0., chi2 = chi2, den = den, $
;       IDL>             weight=(1d0/sigmaf^2), temp = temp, $
;       IDL>             line = fit1, xline = xfit, sigma = sig, prob = prob, $
;       IDL>             errors = 0)
;
;       IDL> plot, energy,f,/xlog,/ylog,psym=1 ;plot the data
;       IDL> oplot, xfig,fit1,/xlog,/ylog      ;plot the fit result
;
;
; Written by:   Eric E. Dors, 1 March 1998.
;
; MODIFICATION HISTORY:
;
;-

COMMON FitPassMaxw, accel_p
IF (n_elements(errors) EQ 0) THEN errors = -1

accel_p = double(emax)

nonz = where(yarr ne 0)

xfit = double(xarr(nonz))
yfit = double(yarr(nonz))
w = double(weight(nonz))

ylog = alog(yfit)

n = n_elements(yarr)
m = 2
nu = n-m

temp =  !values.f_nan
den = !values.f_nan
pot = !values.f_nan
sig = dblarr(m)
sig(*) = !values.f_nan
chi2 = !values.f_nan
prob = !values.f_nan

IF nu GE (m+1) THEN BEGIN 
    a = dblarr(m)
;
;   Here is a code fragment to convert the sigmas to log space.  The
;   idea is that the + and - sigmas are averaged together to get a
;   most probable value.
;    sigma = 1./sqrt(w)
;    lweight = 1./(((alog(yfit+sigma)-ylog)+(ylog-alog(yfit-sigma)))/2.)
;    lweight = yarr^2*w
;
;   Here is a code fragment to fit to model function, make first guess
;   using svd, then modify this guess with the marqadrt-levenburg
;   method
;    stop
;    a = spsvdfit( xfit, ylog, m, weight = lweight, yfit = fit, $
;                FUNCT = 'TempFunctMaxwSVD', singular = sing) 
;    print, 'singular=', sing
;    stop
;
; 
;   then if svd fails give a simpler guess
; ***for now just stick wit the simpler guess
;    IF ((size(a))(0) NE 1) THEN BEGIN
    IF (n_elements(iguess) NE 2) THEN BEGIN
        ;;simple log space slope initial guess
;        a(0) = yfit(0)
;        ii = fix(n/2)         ;an index which can be used to form a
;                                ;linear intial guess   
;        a(1) = -(ylog(ii)-ylog(0))/(xfit(ii)-xfit(0))
;        IF (a(1) LE 0) THEN BEGIN
;            a(1) = -(ylog(n-1)-ylog(0))/(xfit(n-1)-xfit(0))
;        ENDIF
        ;;use the mean energy as an initial guess since that is
        ;;related to K_B*T
        ;;
        ;;integrate to get the E_ave
        fave = xfit(0)*1.6e-12*yfit(0) ;crude approx for density integral from
                                ;0 to emin   
        eave = xfit(0)*1.6e-12*fave ;crude approx for energy integral from 0
                                ;to emin   
                                ;integrate from emin to emax (infinity for our purposes)
        FOR i = 0, n-2 DO BEGIN
            incr = (xfit(i+1)-xfit(i))*1.6e-12*(yfit(i+1)+yfit(i))/2.
            fave = fave + incr
            eave = eave + xfit(i)*incr*1.6e-12
        ENDFOR 
        eave = eave/fave/1.6e-12 ;in eV
        a(0) = (9.11e-28/2./!pi/eave)^(1.5)/sqrt(fave)
        a(1) = 1./eave          ;in eV
        tempinit = 1./a(1)
        deninit = a(0)*(2.*!pi/a(1)*(1.6e-12/9.11e-28))^(1.5) ;cgs
        ;;  eV->ergs -------------------^
        print, 'initial guess for maxw (linear) temp=',tempinit, $
          ';  den=',deninit, format = '(a,f5.1,a,f7.3)'
    ENDIF ELSE BEGIN
        a = dblarr(2)
        a(0) = iguess(0)/((9.11e-28/1.6e-12)/2./!pi/iguess(1))^(1.5)
        a(1) = 1./iguess(1)
        print, 'initial guess for maxw (user) temp=',iguess(1), $
          ';  den=',iguess(0), format = '(a,f5.1,a,f7.3)'
    ENDELSE
;    ENDIF

    itmax = 20

    retval = spcurvefit(xfit,yfit,w,a, curvature = curvature, $
                        FUNCTION_NAME='TempFunctMaxwML', chi2=chi2, $
                        iter = iter, itmax = itmax, yfit = fit) 
    IF (retval EQ 0) THEN return, 0
    ;;
    pot = accel_p
    
    temp = 1./a(1)              ;where temp is measured in KeV (k_b*T)
    den = a(0)*(2.*!pi/a(1)*(1.6e-12/9.11e-28))^(1.5) ;cgs

    IF (errors eq 0) THEN BEGIN 
        ;;estimate errors by using curvature matrix (see scudder write
        ;;up on fitting routines for the best explaination of this and
        ;;Press sec.  15.6
        ;;
        ;;find the sigmas associated with the desired _joint_
        ;;confidence level. For the 68% confidence level this is 1.00
        ;;for one degree of freedom, 2.30 for two degrees of freedom,
        ;;and 3.53 for three degrees of freedom.  See page 212 in
        ;;Bevington or sec 15.6 in Press et al.  get the correct
        ;;dchisquare limit for the current nu (degrees of freedom) and
        ;;the desired confidence limit 1sigma=68.3% dchisq_lim =
        ;;confidence(68.3, m)
        variance = dblarr(m)
        sig = dblarr(m)
        cm = curvature*2.       ;the hessian matrix
        variance(0) = sqrt(cm(1,1)*2.30/(cm(0,0)*cm(1,1)-cm(0,1)^2))
        variance(1) = sqrt(cm(0,0)*2.30/(cm(0,0)*cm(1,1)-cm(0,1)^2))
;;; the following commented out code is the general case for m>2
;;        tau = dblarr(m)
;;        xi = dblarr(m, m)
;;        omega = dblarr(m)
;;        denom = dblarr(m)
;;        FOR l = 0, m-1 DO BEGIN
;;            FOR i = 0, m-1 DO BEGIN
;;                ;;tau(i) = -(cm(i, l) - (1.+cm(i, l))*(i EQ l))
;;                ;;rewrite to make insusceptible to problems when cm
;;                ;;adding 1 to cm is below precision.
;;                tau(i) = -(cm(i,l)*(1-(i eq l))-(i eq l)) 
;;                FOR j = 0, m-1 DO BEGIN
;;                    ;;rewrite to make insusceptible to round-off problems 
;;                    ;;xi(i,j) = cm(i,j)-cm(i,j)*(i EQ l)-cm(i,j)*(j EQ l)+$
;;                    ;;   (1.+cm(i, j))*(i EQ j)*(i EQ l)*(j EQ l)
;;                    xi(i,j) = cm(i,j)*(1 - (i EQ l) - (j EQ l) + $
;;                                       (i EQ j)*(i EQ l)*(j EQ l)) + $
;;                      (i EQ j)*(i EQ l)*(j EQ l)
;;                ENDFOR
;;            ENDFOR
;;            omega= invert(xi) ## tau
;;            FOR i = 0, m-1 DO BEGIN
;;                FOR j = 0, m-1 DO BEGIN
;;                    denom(l) = denom(l)+cm(i, j)*omega(i)*omega(j)
;;                ENDFOR
;;            ENDFOR
;;            variance(l) = sqrt(2.30/denom(l))
;;        ENDFOR
;;;
        sig(1) = variance(1)/a(1)^2

        sig(0) = sqrt((2.*!pi/a(1)*(1.6e-12/9.11e-28))^3 * $
                      (variance(0)^2 + (variance(1)*1.5*a(0)/a(1))^2))
    ENDIF ELSE IF (errors eq 1) THEN BEGIN 
        ;;estimate errors by forming delta chi squared surface on a grid
        print, 'Computing Errors...'
        xsz = 21
        xof = (xsz-1)/2
        ysz = 21
        yof = (ysz-1)/2
        ;;param 1 range in amp
        asamp = (findgen(xsz)-xof)/xof*0.50*a(0) + a(0)
        bsamp = (findgen(ysz)-yof)/yof*0.25*a(1) + a(1)
        ;; adjust this fraction --------^ to pick the relative size of the
        ;; chisquared space constructed (1/2 is an approximate guess, but
        ;; one needs to judge this number by balancing the average width
        ;; of the dchisquare surface at the level for the specified joint
        ;; or single parameter confidence level and the resolution of the
        ;; surfaces required in order to see said level)

        ;; calculated in call to spcurvefit:    chi2 = total(w*(yfit-fit)^2)/nu
        dchisquare = fltarr(xsz, ysz)
        FOR j = 0, ysz-1 DO BEGIN
            FOR i = 0, xsz-1 DO BEGIN
                yf = asamp(i)*exp(-bsamp(j)*(xfit-pot))
                dchisquare(i, j) = total(w*(yfit-yf)^2) - chi2*nu
            ENDFOR
        ENDFOR

        ;;here is a command to view the chi-square surface
        ;;contour,dchisquare,asamp,bsamp,levels=[0,.1,1,2.3,3.53,10] 
        ;;
        ;;find the sigmas associated with the desired _joint_ confidence
        ;;level. dchisquare obeys chisquared statistics for the number of
        ;;degrees of freedom specified by the number of parameters jointly
        ;;varied.  For the 68% confidence level this is 1.00 for one
        ;;degree of freedom, 2.30 for two degrees of freedom, and 3.53 for
        ;;three degrees of freedom.  See page 212 in Bevington or sec 15.6
        ;;in Press et al.
        ;;get the correct dchisquare limit for the current nu (degrees of
        ;;freedom) and the desired confidence limit 1sigma=68.3%
        ;;dchisq_lim = confidence(68.3, m)
        delta_reg = where(dchisquare LT 2.30)

        ;;find the coordinates of the delta_reg surface in (cora,corb) space
        cora=delta_reg mod xsz
        corb=delta_reg/xsz
        mina = min(asamp) & maxa = max(asamp)
        minb = min(bsamp) & maxb = max(bsamp)
        dumb = where(((mina EQ asamp(cora)) OR (maxa EQ asamp(cora)) OR $
                      (mina EQ bsamp(corb)) OR (maxb EQ bsamp(corb)) ), nnn)
        IF (nnn NE 0) THEN BEGIN
            print, 'Sampled dchisquare surface may not be big enough '+ $
              'in fittempmaxw.'
            print, 'Reported sigmas are lower limits on the true sigma.'
        ENDIF

        variance = [(max(asamp(cora))-min(asamp(cora)))/2., $
                    (max(bsamp(corb))-min(bsamp(corb)))/2.]
        sig = variance
        sig(1) = variance(1)/a(1)^2

        sig(0) = sqrt((2.*!pi/a(1)*(1.6e-12/9.11e-28))^3 * $
                      (variance(0)^2 + (variance(1)*1.5*a(0)/a(1))^2))
    ENDIF ELSE BEGIN
        sig = replicate(!values.f_nan, m)
    ENDELSE 

    prob = (1.-igamma(nu/2., chi2*nu/2.))*100.

    return, 1                   ; since f= A exp (- m v^2 / (2KT)), 
                                ; should there be  ^
                                ; a geom. factor   |----here, like 3?
                                ;        = A exp (-E/KT)
                                ;   log f= log A - E/KT
                                ; temperature is therefore -1/a(1)
ENDIF

return, 0

END

