;$Id: print_window.pro,v 1.1.1.1 2003/11/04 21:24:02 friedel Exp $
;; print_window v2.2
;; Eric E. Dors <edors@lanl.gov>
;;
;; For overview of use see the documentation header to the procedure
;; print_window at the bottom of this file.
;;
FUNCTION get_config_filename, filenameout, top_widget = top_widget
;+
; NAME:
;       get_config_filename
;
; PURPOSE: 
;
;       This function locates the fully qualified filename of the
;       configuration file required for the operation of
;       print_window.pro.
;
; CATEGORY:
;       General Untility
;
; CALLING SEQUENCE:
;       
;       Result = get_config_filename(filenameout)
;
; KEYWORD PARAMETERS:
;
;       top_widget: the id of the parent widget base if it exists
;
; OUTPUTS:
;
;       result:   result is set to 1 for sucess and to 0 for failure
;                 of this subroutine to find the configuration filename.
;
;       filenameout:  The fully qualified filename of the
;                     configuration file print_window.cfg.
;
; PROCEDURE:
;
;       First an optional environment variable called PRINT_WINDOW_CFG
;       is searched for.  This environment variable is supposed to
;       contain the path to the configuration file "print_window.cfg".
;       If the environment variable is not found then directory in
;       which print_window.pro resides is searched for the system
;       default configuration file also called print_window.cfg.  If
;       the file is found the configuration information is read in,
;       otherwise the programs writes an error message and exits.
;
; EXAMPLE:
;
;               F = get_config_filename(cfgfilename)
;
; Written by:   Eric E. Dors, 9 September 1997.
;
; MODIFICATION HISTORY:
;
;       Fri Jan 23 19:41:41 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Added top_widget parameter for use with widget_message.
;
;-

cfgpath = getenv('PRINT_WINDOW_CFG')
IF (cfgpath EQ '') THEN BEGIN ;environment variable not specified
    idlpatharr = str_sep(!path, ':') ;make !path into an array of strings
    
    ;search for configuration file in directory local to print_window.pro
    found = 0
    i = 0
    REPEAT BEGIN 
        findstr = findfile(idlpatharr(i)+'/print_window.pro') ne ''
        IF (findstr(0) NE '') THEN found = 1
        i = i+1
    endrep UNTIL((i gt n_elements(idlpatharr)-1) OR (found eq 1))
    i = i-1
    IF (found EQ 0) THEN BEGIN
        ans=widget_message('READ_PRINT_CONFIG (PRINT_WINDOW.PRO): '+$
                           'error while searching for print_window.pro', $
                           /ERROR, TITLE = 'Error', $
                           DIALOG_PARENT = top_widget)
        print, '% READ_PRINT_CONFIG (PRINT_WINDOW.PRO): '+$
          'error while searching for print_window.pro'
        return, 0
    ENDIF
    
    cfgfile = findfile(idlpatharr(i)+'/print_window.cfg')
ENDIF ELSE BEGIN
    cfgfile = findfile(cfgpath+'/print_window.cfg')
ENDELSE

IF (cfgfile(0) EQ '') THEN BEGIN   ;config file not found
    ans=widget_message('READ_PRINT_CONFIG (PRINT_WINDOW.PRO): '+$
                       'error while searching for print_window.cfg', $
                       /ERROR, TITLE = 'Error', $
                       DIALOG_PARENT = top_widget)
    print, '% READ_PRINT_CONFIG (PRINT_WINDOW.PRO): '+$
      'error while searching for print_window.cfg'
    return, 0
ENDIF
    
filenameout = cfgfile(0)

return, 1
end
;=====================================================================
FUNCTION get_cfg_line, unit, oneline
;+
; NAME:
;       get_cfg_line
;
; PURPOSE:
;
;       The purpose of this function is to return one variable
;       expression line configuration file.
;
; CATEGORY:
;       General Utility

; CALLING SEQUENCE:
;       
;       retval = get_cfg_line(unit,oneline)
;
; INPUTS:
;       unit:   The file unit number of the opened configuration file.
;
; OUTPUTS:
;
;       oneline:  One variable expression is returned from the
;                 configuration file.  Comment lines are skipped.
;
;       retval:   1 is returned on success, 0 is returned if eof is
;                 found before a variable definition is found.
;
; PROCEDURE:
;       
;       The configuration file is read in line by line until a non
;       commented line is found and then that line of text is returned
;       to the caller in oneline and 1 is returned in retval.  If eof
;       is hit then the value of oneline is not guarenteed to be
;       anything useful, and retval is set to 0.
;
; EXAMPLE:
;       
;               retval = get_cfg_line(unit, oneline)
;
; Written by:   Eric E. Dors, 9 September 1997.
;
; MODIFICATION HISTORY:
;
;-


oneline = ''
;;skip until a non commented line is read in
REPEAT BEGIN 
    IF (eof(unit) EQ 1) THEN BEGIN
        oneline = ''
        return, 0
    ENDIF
    readf, unit, oneline
ENDREP UNTIL (strpos(oneline, ';') NE 0)

return, 1
END
;=============================================================================
FUNCTION read_print_config, varlist, top_widget = top_widget
;+
; NAME:
;       read_print_config
;
; PURPOSE:
;       The purpose of this subroutine is to read in the printing
;       configuration file called print_window.cfg.  And return a list
;       of variables set in the configuration file to the caller.
;
; CATEGORY:
;       General Utility
;
; CALLING SEQUENCE:
;
;       Result = reat_print_config(varlist)
;
; KEYWORD PARAMETERS:
;
;       top_widget: the id of the parent widget base if it exists
;
; OUTPUTS:
;       
;       varlist:  A 2xn array holding variable [name, definition] of
;                 each of the n variables defined in the configuration
;                 file.
;
;       result:   Return 1 on successful configuration file read, and
;                 0 on failure.
;
; PROCEDURE:
;       
;       get_config_filename is used to find the configuration file,
;       then it is opened and read line by line with get_cfg_line.
;       Since the configuration file is perscribed to have comment
;       lines beginning with ';' and variable setting lines which look
;       like 'name=definition' we separate the names from the
;       definitions and return the result as a 2xn dimensional array
;       of [[name1,def1],[name2,def2],...,[namen,defn]].
;
; EXAMPLE:
;
;               result = read_print_config(varlist)
;
; Written by:   Eric E. Dors, 9 September 1997.
;
; MODIFICATION HISTORY:
;
;       Fri Jan 23 19:42:06 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Added top_widget parameter for use with widget_message.
;
;-

;;get filename
IF (get_config_filename(cfgfile, top_widget = top_widget) EQ 0) THEN return, 0

;;open config file
openr, unit, /get_lun, cfgfile

i = 1
;;define dummy variable element to start out loop with
varlist = [['', '']]
;;read each variable definition line
WHILE (get_cfg_line(unit, oneline) EQ 1) DO BEGIN
    
    sep = str_sep(oneline, '=') ;separate name from definition
    IF (n_elements(sep) NE 2) THEN BEGIN
        ans=widget_message('READ_PRINT_CONFIG(PRINT_WINDOW): configuation line ' + $
                           strcompress(string(i),/remove_all)+' unreadable', $
                           /ERROR, TITLE = 'Error', $
                           DIALOG_PARENT = top_widget)
        print, '% READ_PRINT_CONFIG(PRINT_WINDOW): configuation line ' + $
          strcompress(string(i),/remove_all)+' unreadable'    
        return, 0
    ENDIF
    
    varlist = [[varlist], [sep]] ;concatinate new def to list
    i = i+1
ENDWHILE
varlist = varlist(*, 1:i-1)     ;remove beggining dummy element off the list 
return, 1
END
;=============================================================================
FUNCTION expand_all_vars, varlist, expanded, top_widget = top_widget
;+
; NAME:
;       expand_all_vars
;
; PURPOSE:
;
;       This function takes a list of variable definitions and returns
;       the expanded list.  That is its expands references of
;       variables on the right hand side to their definitions
;
; CATEGORY:
;       General Utility
;
; CALLING SEQUENCE:
; 
;       Result = expand_all_vars(varlist, expanded)
;
; INPUTS:
;
;       varlist:  A 2xn dimensional array of variable and definitions
;                 [[name1,def1],[name2,def2],...,[namen,defn]]
;
; KEYWORD PARAMETERS:
;
;       top_widget: the id of the parent widget base if it exists
;
; OUTPUTS:
;
;       result:   0 for failure, 1 for sucess.
;
;       expanded: The two dimsnesional variable list with variable
;                 references in the definitions expanded using other
;                 variables in the list.
;
; PROCEDURE:
;
;       The variable list is looped through up to max_iter times.
;       Each time through it looks at each variable definition one by
;       one.  It then looks for the first occurance of a variable
;       reference in the definition, and then replaces it with the
;       appropriate definition.  Then it moves on to the next variable
;       and keeps looping until all the variables references in the
;       defintions have been replaced, an undefined variable is found,
;       or the max_iter has been hit.
;
; EXAMPLE:
;
;               result = expand_all_vars(varlist,expanded)
;
; Written by:   Eric E. Dors, 9 September 1997.
;
; MODIFICATION HISTORY:
;
;       Fri Jan 23 19:42:12 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Added top_widget parameter for use with widget_message.
;
;-

max_iter = 25                   ; max number of recursive var. substitutions
expanded = varlist              ; copy input variable list to output
n_exp = (size(expanded))(2)     ; how many variables are defined


;;determine if there are any variable substitutions required, all
;;variables on the right hand side of the equations are of the form
;;$(variable)
done = 1
FOR i = 0, n_exp-1 DO BEGIN
    IF (strpos(expanded(1, i), '$(') NE -1) THEN done = 0
ENDFOR

;;loop until all variables substitutions have been made or an
;;undefined variable is encountered or the maximum iteration limit is
;;encountered. 
j = 0                           
WHILE (done EQ 0) DO BEGIN
    ;;loop through each equation in the variable list
    FOR i = 0, n_exp-1 DO BEGIN
        ;;locate the first variable substitution on the right hand side
        first = strpos(expanded(1, i), '$(') 
        if (first NE -1) THEN BEGIN 
            last = strpos(expanded(1, i), ')', first)
            ;extract the full variable $( and ) and all
            variable = strmid(expanded(1, i), first, last-first+1)
            ;extract just the var name 
            varname = strmid(expanded(1, i), first+2, last-first-2)

            ;find the appropriate substitution
            n = 0
            found = 0
            repeat BEGIN
                IF (expanded(0, n) EQ varname) THEN BEGIN
                    found = 1
                    sub = expanded(1, n)
                ENDIF

                IF ((n EQ n_exp-1) AND (found EQ 0)) THEN BEGIN
                    ans=widget_message('EXPAND_ALL_VARS(PRINT_WINDOW): variable not found.',$
                                       /ERROR, TITLE = 'Error', $
                                       DIALOG_PARENT = top_widget)
                    print, '% EXPAND_ALL_VARS(PRINT_WINDOW): variable not found.'
                    return, 0
                ENDIF
                n = n+1         
            endrep UNTIL (found EQ 1)
            n = n-1             ;subtract one from n to remove the effect 
                                ;of the post increment in the above loop
            
            ;;split out the variable definition and make substitutions
            ;;where necessary
            sep = str_sep(expanded(1, i), variable)
            n_sep = n_elements(sep)
            expanded(1, i) = sep(0)
            FOR m = 1, n_sep-1 DO BEGIN
                expanded(1, i) = expanded(1, i)+sub+sep(m)
            ENDFOR
        ENDIF
    ENDFOR

    j = j+1
    IF (j GE max_iter) THEN BEGIN
        ans=widget_message('EXPAND_ALL_VARS(PRINT_WINDOW): exceded maximum iterations.', $
                           /ERROR, TITLE = 'Error', $
                           DIALOG_PARENT = top_widget)
        print, '% EXPAND_ALL_VARS(PRINT_WINDOW): exceded maximum iterations.'
        return, 0
    endif            

    ;;determine if there are any variable substitutions required, all
    ;;variables are of the form $(variable)
    done = 1
    FOR i = 0, n_exp-1 DO BEGIN
        IF (strpos(expanded(1, i), '$(') NE -1) THEN done = 0
    ENDFOR
ENDWHILE

return, 1
END
;=============================================================================
FUNCTION form_print_cmd, print_file, printer_type, $
                         print_color, lpr_swwitches, ptype_btns, $
                         color_btns, print_command, top_widget = top_widget
;+
; NAME:
;       form_print_cmd
;
; PURPOSE:
;
;       The purpose of this function is to create a string which
;       contains the operating system command to print a file of a
;       given type to the correct printer.
;
; CATEGORY:
;       General Utility
;
; CALLING SEQUENCE:
;
;       Result = form_print_cmd(print_file, printer_type, $
;                         print_color, lpr_swwitches, ptype_btns, $
;                         color_btns, print_command)
;
;
; INPUTS:
;
;       print_file:  The name of the file to be printed.
;
;       printer_type: A value from the list of printer types in
;                     ptype_btns describing the type of printer.
;
;       print_color:  A value from the list of printer color support
;                     in color_btns describing the color printing
;                     options selected.
;
;       lpr_swwitches: A string of user printer swwitches to be used in
;                     addition to the swwitches set in print_window.cfg.
;                     
;       ptype_btns:   A list of printer types.
;
;       color_btns:   A list of printer color options.
;
; KEYWORD PARAMETERS:
;
;       top_widget: the id of the parent widget base if it exists
;
; OUTPUTS:
;       
;       print_command: A string which contains the system command
;                      necessary to print the desired file.
;
;       result:        0 for failure of this routine, 1 for success.
;
; PROCEDURE:
;       
;       Replace the reserved variable queue with the print queue
;       appropriate to the current print job.  Replace the reserved
;       variable file with the file name appropriate for this print
;       job.  Expand all other variables.  Return the print_command
;       requested. 
;
; EXAMPLE:
;               ret_val=form_print_cmd(print_file, printer_type, $
;                         print_color, lpr_swwitches, ptype_btns, $
;                         color_btns, print_command)
;
;               if (ret_val eq 1) then begin
;                     spawn,print_command
;               endif
;
; Written by:   Eric E. Dors, 9 September 1997.
;
; MODIFICATION HISTORY:
;
;       Fri Jan 23 19:42:25 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Added top_widget parameter for use with widget_message.
;
;       Fri Sep 12 15:36:07 1997, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Add recognition of new variable to configuration file
;               called 'file' this shows the position of the file name
;               in the pring command. 
;
;-


;read the printer configuration file
IF (read_print_config(varlist, top_widget = top_widget) EQ 0) THEN return, 0

;find out what variable should be used to determine which print queue
;to print to
CASE print_color OF 
    color_btns(0): BEGIN        ;color
        queuename = 'color'
    END
    color_btns(1): BEGIN        ;b&w
        queuename = 'bw'
    END
ENDCASE

CASE printer_type OF
    PTYPE_BTNS(0): BEGIN        ;PS
        queuename = queuename+'ps'
    END 
    PTYPE_BTNS(1): BEGIN        ;EPS
        ans=widget_message('FORM_PRINT_CMD(PRINT_WINDOW): Cannot print encapsulated postcript?', $
                           /ERROR, TITLE = 'Error', $
                           DIALOG_PARENT = top_widget)
        print, '% FORM_PRINT_CMD(PRINT_WINDOW) ERROR: Cannot print encapsulate postscript'
        return, 0
    END 
    PTYPE_BTNS(2): BEGIN        ;PCL
        queuename = queuename+'pcl'
    END 
    PTYPE_BTNS(3): BEGIN        ;HPGL
        queuename = queuename+'hpgl'
    END 
END

;; replace any instance of the special variable queue {$(queue)} with the
;; name of the queue variable required for this type of print job.
n_vars = (size(varlist))(2)
FOR i = 0, n_vars-1 DO BEGIN
    sep = str_sep(varlist(1, i), '$(queue)')
    n_sep = n_elements(sep)
    varlist(1, i) = sep(0)
    FOR j = 1, n_sep-1 DO BEGIN
        varlist(1, i) = varlist(1, i)+'$('+queuename+')'+sep(j)
    ENDFOR
ENDFOR

;; replace any instance of the special variable file {$(file)} with the
;; name of the file begin printed
n_vars = (size(varlist))(2)
FOR i = 0, n_vars-1 DO BEGIN
    sep = str_sep(varlist(1, i), '$(file)')
    n_sep = n_elements(sep)
    varlist(1, i) = sep(0)
    FOR j = 1, n_sep-1 DO BEGIN
        varlist(1, i) = varlist(1, i)+print_file+sep(j)
    ENDFOR
ENDFOR

;;expand all variables in the print configuration file
IF (expand_all_vars(varlist, expanded, top_widget=top_widget) EQ 0) THEN BEGIN
    return, 0
ENDIF

;;search for the variable print_command 
n_exp = (size(expanded))(2)
FOR i = 0, n_exp-1 DO BEGIN
    IF (expanded(0, i) EQ 'print_command') THEN BEGIN
        print_command = expanded(1, i)
    ENDIF
    
ENDFOR

IF (n_elements(print_command) EQ 0) THEN return, 0

;;return the print command to the caller
return, 1
END

;=====================================================================
PRO print_finish, parent_id = top_widget, noconfirm = noconfirm
;+
; NAME:
;       PRINT_FINISH
;
; PURPOSE: 
;       This procedure completes the actions initiated by PRINT_WINDOW.
;
; CATEGORY:
;       General Utility
;
; CALLING SEQUENCE:
;       PRINT_FINISH, handle
;
; KEYWORD PARAMETERS
;       parent_id:   The widget handle to the controling base widget.
;                    This is used so that informational dialog can be
;                    centered on that widget.
;
;       noconfirm:   if 0 then ask before deleting print file,
;                    otherwise delete without confirmation.
;
; COMMON BLOCKS:
;       print_event:    A common block to pass information between the
;                       event handler and the widget creation routine.
;
;       print_finish:   A common block to hold the selections made
;                       during the PRINT_WINDOW dialogue with the user.
;
; SIDE EFFECTS:
;       The current plot device is changed.
;       If it was requested either a file is created, printed or both.
;
; EXAMPLE:
;       result=print_window(xsize=draw_x, ysize=draw_y)
;       ...
;       if (result) then begin
;            data plotting routines go here
;            time_stamp
;            ...
;            print_finish, parent_id=parent_id
;       endif
;       ...
;
; Written by:   Eric E. Dors, 15 November 1995.
;
; MODIFICATION HISTORY:
;
; HISTORY:
;
;       Fri Jan 23 19:16:46 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Removed top_widget input and created the parent_id ;
;               keyword parameter to contain the base id of the
;               calling widget if it exists.  Also added the
;               noconfirm parameter to prevent the poping of the
;               confirmation dialog for deleting of the print file.
;
;-

COMMON print_event, MAIN, C_BTNS, PT_BTNS, O_BTNS, $
  FNAME_I, LPR_I, COLOR_BTNS, PTYPE_BTNS, ORIEN_BTNS, xsize, ysize, $
  color_t, color_g, bw_t, bw_g, retval

COMMON print_finish, colorp, print_type, orientation, output_file, $
  lpr_swwitches, uvalue

IF (uvalue EQ 'PRINT_B') OR (uvalue EQ 'FILE_B') THEN BEGIN
    device, /CLOSE_FILE
    set_plot, 'x'
ENDIF

IF uvalue EQ 'PRINT_B' THEN BEGIN

    IF output_file(0) EQ '' THEN BEGIN
        print_file = 'idl.' + strlowcase(PTYPE_BTNS(print_type))
    ENDIF ELSE BEGIN
        print_file = output_file(0)
    END

    ret_val = form_print_cmd(print_file, ptype_btns(print_type), $
                             color_btns(colorp), lpr_swwitches, $
                             PTYPE_BTNS, COLOR_BTNS, print_command, $
                             top_widget = top_widget)

    IF (ret_val NE 0) THEN spawn, print_command

    ;;delete file if no file name was specified and the plot was sent
    ;;to the printer
    IF output_file(0) EQ '' THEN BEGIN 
        IF (NOT keyword_set(noconfirm)) THEN BEGIN 
            ans=widget_message('Do you wish to delete the tempoary print file?', $
                               /QUESTION, TITLE = 'Query', $
                               DIALOG_PARENT = top_widget)
            IF ans eq 'Yes' THEN spawn, 'rm '+print_file
        ENDIF ELSE BEGIN
            spawn, 'rm '+print_file
        ENDELSE 
    ENDIF

ENDIF

print, 'Done printing.'

END

;=====================================================================
PRO print_handler, event

COMMON print_event, MAIN, C_BTNS, PT_BTNS, O_BTNS, $
  FNAME_I, LPR_I, COLOR_BTNS, PTYPE_BTNS, ORIEN_BTNS, xsize, ysize, $
  color_t, color_g, bw_t, bw_g, retval

COMMON print_finish, colorp, print_type, orientation, output_file, $
  lpr_swwitches, uvalue

widget_control, event.ID, GET_UVALUE = uvalue
CASE 1 OF
    uvalue EQ 'CANCEL_B': BEGIN 
        print, 'Operation cancled.'
        widget_control,  /DESTROY,  event.TOP
    END 

    (uvalue EQ 'FILE_B') OR (uvalue EQ 'PRINT_B'): BEGIN
        widget_control,     C_BTNS,   GET_VALUE = colorp
        widget_control,    PT_BTNS,   GET_VALUE = print_type
        widget_control,     O_BTNS,   GET_VALUE = orientation
        widget_control,    FNAME_I,   GET_VALUE = output_file
        widget_control,      LPR_I,   GET_VALUE = lpr_swwitches
        widget_control, MAIN, /realize, /HOURGLASS
        print, 'Printing please wait...'
        IF PTYPE_BTNS(print_type) EQ 'EPS' THEN BEGIN
            set_plot, 'ps'
            device, /encapsulated
        ENDIF ELSE IF PTYPE_BTNS(print_type) EQ 'PS' THEN BEGIN
            set_plot, PTYPE_BTNS(print_type)
            ;;make sure encapsulated option is turned off if a
            ;;previous ps plot was made with encapsulated the device
            ;;setup "remembers" that setting
            device, encapsulated = 0 
        ENDIF ELSE BEGIN
            set_plot, PTYPE_BTNS(print_type)
        ENDELSE

        IF (output_file(0) NE '') THEN device, FILENAME = output_file(0)
        IF (COLOR_BTNS(colorp) EQ 'Color') THEN BEGIN
            device, /COLOR, bits=8
;            loadct, color_t
;            gamma_ct, color_g
        ENDIF ELSE BEGIN
            device, /COLOR, bits=8
;            loadct, bw_t
;            gamma_ct, bw_g
        ENDELSE

        IF (ORIEN_BTNS(orientation) EQ ORIEN_BTNS(0)) THEN BEGIN 
            device, /PORTRAIT
            device, /inches, xsize=7.5, ysize= 10.0, $
              xoffset = .5, yoffset = .5
        ENDIF ELSE BEGIN
            device, /LANDSCAPE
            device, /inches, xsize=10.0, ysize= 7.5, $
;              xoffset = 0.5, yoffset = 11.0
              xoffset = 0.5, yoffset = 10.5
        ENDELSE 
        
        widget_control,  /DESTROY,  event.TOP
        
        retval = 1              ;return successful completion
    END

    ELSE: BEGIN
    END

ENDCASE 

END
;=====================================================================
FUNCTION print_window, xsize=xsz, ysize=ysz, GROUP=Group, $
                       color_t=c_t, color_g=c_g, bw_t=b_t, bw_g=b_g, $
                       parent_id = dialog_parent, filename = filename
;+
; NAME:
;       PRINT_WINDOW
;
; PURPOSE: 
;       This function creates a dialogue box for printing
;       information,  the generated events can put IDL into a state to
;       make files or send plots to a printer.
;
; CATEGORY:
;       General Utility
;
; CALLING SEQUENCE:
;
;       Result = PRINT_INFO_WINDOW
;
;       Result: 1 unless print operation was canceled in which case
;               0 is returned.
;
; KEYWORD PARAMETERS:
;       xsize:  Optional keyword to allow removing of device dependent
;       parameters for operations such as tv.
;
;       ysize:  Optional keyword to allow removing of device dependent
;       parameters for operations such as tv.
;
;       color_t: Color table to use for color printing. (39 default)
;       
;       color_g: The gamma factor to use for color printing. (1.0 default)
;
;       bw_t: Color table to use for black and white printing. (0 default)
;       
;       bw_g: The gamma factor to use for black and white
;       printing. (1.0 default)
;
;       group:       Obsolete parameter.
;
;       parent_id:   The widget handle to the controling base widget.
;                    This is used so that informational dialog can be
;                    centered on that widget.
;
;       filename:    default filename
;
; COMMON BLOCKS:
;       print_event:    A common block to pass information between the
;       event handler and the widget creation routine.
;
; SIDE EFFECTS:
;       The current plot device is changed.
;       NOTE By calling print_finish, a device,/close and a
;       set_plot, 'x' will be done to restore to the normal state.
;
; EXAMPLE:
;       result=print_window(xsize=draw_x, ysize=draw_y)
;       ...
;       if (result) then begin
;            data plotting routines go here
;            time_stamp
;            ...
;            print_finish, parent_id=parent_id
;       endif
;       ...
;
; Written by:   Eric E. Dors, 15 November 1995.
;
; MODIFICATION HISTORY:
;
;       Fri Sep 11 11:32:40 1998, Eric E. Dors <edors@universe.lanl.gov>
;
;               Updated the documentation on the "group" keyword
;               parameter (obsolete keyword), made an error statement
;               to flag use of "group" keyword, and moved the
;               initialization of the retval variable from
;               print_handler to print_window.
;
;       Wed Feb 4 05:30:28 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               added support for a default filename from the commandline
;
;       Fri Jan 23 18:46:48 1998, Eric E. Dors <dors@gamma.physics.uiowa.edu>
;
;               Simplified the creation of the main base widget.
;
;               Changed print_window to a function instead of a
;               procedure.  Returns 0 if canceled, 1 otherwise.
;
;               Also added a parent_id parameter to contain the base
;               id of the calling widget if it exists.  This
;               facilitates the centering of widget_message events.
;
;-

COMMON print_event, MAIN, C_BTNS, PT_BTNS, O_BTNS, $
  FNAME_I, LPR_I, COLOR_BTNS, PTYPE_BTNS, ORIEN_BTNS, xsize, ysize, $
  color_t, color_g, bw_t, bw_g, retval

retval = 0                      ;initialize retval to denote fail

IF N_ELEMENTS(xsz) EQ 0 THEN xsize = 0      ELSE xsize = float(xsz)
IF N_ELEMENTS(ysz) EQ 0 THEN ysize = 0      ELSE ysize = float(ysz)
IF N_ELEMENTS(c_t) EQ 0 THEN color_t = 39   ELSE color_t = fix(c_t)
IF N_ELEMENTS(c_g) EQ 0 THEN color_g = 1.0  ELSE color_g = float(c_g)
IF N_ELEMENTS(b_t) EQ 0 THEN bw_t = 0       ELSE bw_t = fix(b_t)
IF N_ELEMENTS(b_g) EQ 0 THEN bw_g = 1.0     ELSE bw_g = float(b_g)
IF N_ELEMENTS(dialog_parent) EQ 1 THEN parent_id = dialog_parent

IF keyword_set(group) THEN BEGIN
    ans=widget_message('PRINT_WINDOW: Obsolete keyword "group" was used.', $
                       /ERROR, TITLE = 'Error', $
                       DIALOG_PARENT = parent_id)
    print, '% PRINT_WINDOW: ERROR, obsolete keyword "group" was used.'
ENDIF

junk   = { CW_PDMENU_S, flags:0, name:'' }

IF (float(!version.release) LT 5.0) THEN BEGIN 
    IF (N_ELEMENTS(dialog_parent) NE 1) THEN BEGIN
        MAIN = WIDGET_BASE(COLUMN=1, /BASE_ALIGN_CENTER, $
                           MAP=1, $
                           TITLE='Printing Information', $
                           UVALUE='MAIN13')
    ENDIF ELSE BEGIN
        MAIN = WIDGET_BASE(GROUP_LEADER=Group, $
                           COLUMN=1, /BASE_ALIGN_CENTER, $
                           MAP=1, $
                           TITLE='Printing Information', $
                           UVALUE='MAIN13')
    ENDELSE         
ENDIF ELSE BEGIN
    MAIN = WIDGET_BASE(COLUMN=1, /BASE_ALIGN_CENTER, $
                       MAP=1, $
                       TITLE='Printing Information', $
                       modal = keyword_set(dialog_parent), $
                       GROUP_LEADER=parent_id, UVALUE='MAIN13')
ENDELSE 

TOP = WIDGET_BASE(MAIN, $
                  COLUMN=2, /BASE_ALIGN_RIGHT, $
                  MAP=1, $
                  TITLE='input_area', $
                  UVALUE='TOP_BASE')

leftTOP = WIDGET_BASE(top, $
                  COLUMN=1, /BASE_ALIGN_RIGHT, $
                  MAP=1, $
                  TITLE='left_col', $
                  UVALUE='TOP_LEFT_BASE')

rightTOP = WIDGET_BASE(top, $
                  COLUMN=1, /BASE_ALIGN_RIGHT, $
                  MAP=1, $
                  TITLE='right_col', $
                  UVALUE='TOP_RIGHT_BASE')

COLOR_BTNS = [ 'Color', $
               'Black and White' ]

C_BASE = WIDGET_BASE(leftTOP, /FRAME, $
                     COLUMN=1, /ALIGN_CENTER, $
                     MAP=1, $
                     UVALUE='C_BASE')

C_BTNS = CW_BGROUP( C_BASE, COLOR_BTNS, $
                    ROW=1, $
                    EXCLUSIVE=1, $
                    LABEL_TOP='Color Options', $
                    SET_VALUE = 1, $
                    UVALUE='C_BTNS')

ORIEN_BTNS = [ 'Portrait', $
               'Landscape' ]

O_BASE = WIDGET_BASE(leftTOP, /FRAME, $
                     COLUMN=1, /ALIGN_CENTER, $
                     MAP=1, $
                     UVALUE='O_BASE')

O_BTNS = CW_BGROUP( O_BASE, ORIEN_BTNS, $
                    ROW=1, $
                    EXCLUSIVE=1, $
                    LABEL_TOP='Orientation', $
                    SET_VALUE = 0, $
                    UVALUE='ORIEN_BTNS')

PTYPE_BTNS = [ 'PS', $
               'EPS', $
               'PCL', $
               'HPGL' ]

PT_BASE = WIDGET_BASE(rightTOP, /FRAME, $
                      COLUMN=1, /ALIGN_CENTER, $
                      MAP=1, $
                      UVALUE='PT_BASE')

PT_BTNS = CW_BGROUP( PT_BASE, PTYPE_BTNS, $
                     ROW=1, $
                     EXCLUSIVE=1, $
                     LABEL_TOP='Printer type', $
                     SET_VALUE = 0, $
                     UVALUE='PT_BTNS')

IF (n_elements(filename) EQ 1) THEN BEGIN
    FNAME_DEF = [ filename ]
ENDIF ELSE BEGIN
    FNAME_DEF = ''
ENDELSE
FNAME_I = CW_FIELD( rightTOP, VALUE=FNAME_DEF, $
                    ROW=1, $
                    xsize = 30, $
                    STRING=1, $
                    TITLE='Filename', $
                    UVALUE='FNAME_I')

LPR_DEF = [ '' ]
LPR_I = CW_FIELD( rightTOP, VALUE=LPR_DEF, $
                  ROW=1, $
                  STRING=1, $
                  TITLE='     extra printer swwitches', $
                  UVALUE='LPR_I')

BOTTOM = WIDGET_BASE(MAIN, $
                     COLUMN = 3, $
                     MAP=1, $
                     TITLE='command_area', $
                     UVALUE='BOTTOM')

PRINT_B = WIDGET_BUTTON( BOTTOM, $
                         UVALUE='PRINT_B', $
                         VALUE='PRINT')

FILE_B = WIDGET_BUTTON( BOTTOM, $
                        UVALUE='FILE_B', $
                        VALUE='FILE')

CANCEL_B = WIDGET_BUTTON( BOTTOM, $
                          UVALUE='CANCEL_B', $
                          VALUE='CANCEL')

IF (n_elements(parent_id) EQ 1) THEN BEGIN
    mgeom = widget_info(main, /geom)
    pgeom = widget_info(parent_id, /geom)
    xoff = fix(pgeom.xoffset + (pgeom.scr_xsize-mgeom.scr_xsize)/2.)
    yoff = fix(pgeom.yoffset + (pgeom.scr_ysize-mgeom.scr_ysize)/2.)
    widget_control, main, xoffset = xoff, yoffset = yoff
ENDIF
;; to center widget: (tried this for the default with no parent, but
;; the problem is there is no easy way of centering widget_message
;; or dialog_message calls and this make widget placement inconsistant)
;;
;;    device, get_screen_size = ssize
;;    xoff = fix((ssize(0)-mgeom.scr_xsize)/2.)
;;    yoff = fix((ssize(1)-mgeom.scr_ysize)/2.)

WIDGET_CONTROL, MAIN, /REALIZE

IF float(!version.release) LT 5.0 THEN BEGIN
    XMANAGER, 'print_window', MAIN, EVENT_HANDLER='print_handler', /MODAL
ENDIF ELSE BEGIN
    XMANAGER, 'print_window', MAIN, EVENT_HANDLER='print_handler'
endelse    

return, retval
END


