;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_kp, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_kp for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_kp, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_kp, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable

COMMON hydra_kp_labels, $
  kp_labels

; The following common blocks are needed:

common time, xut1,xut2          ;common time limits in T90
common mjdt, mjdt1, mjdt2
common get_error, get_err_no, get_err_msg

;   common hydra_yscale, yscl
common yscale, yscl             ;man/auto yscaling
common coordinateSystems, plotted_x, plotted_y
                                ;info on coords used by mouse
common hydra_kp_data, $
  kp_species, $
  kp_header, $
  kp_lr_ele_data, $
  kp_lr_ele_attr, $
  kp_lr_ion_data, $
  kp_lr_ion_attr, $
  kp_hr_ele_data, $
  kp_hr_ele_attr

common papco_color_names

panelset,panel			;sets the panel position viewport

oldy=!y
oldx=!x

goodticks, xtickv, xminor
xticks=n_elements(xtickv)-1

; set duplex font if big_gif is device ; duplex roman
if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
dxut= xut2-xut1

IF (kp_species EQ 0) THEN BEGIN
    data = kp_lr_ele_data
    attr = kp_lr_ele_attr
    str_species = 'Hyd Ele LR KP'
    ispecies = -1
    spec_super='!r!ue'
    res_sub = '6'
ENDIF ELSE IF (kp_species EQ 1) THEN BEGIN
    data = kp_lr_ion_data
    attr = kp_lr_ion_attr
    str_species = 'Hyd Ion LR KP'
    ispecies = 1
    spec_super='!r!ui'
    res_sub = '6'
ENDIF ELSE BEGIN
    data = kp_hr_ele_data
    attr = kp_hr_ele_attr
    str_species = 'Hyd Ele HR KP'
    ispecies = -1
    spec_super='!r!ue'
    res_sub = '1'
ENDELSE

IF n_elements(kp_labels) EQ 0 THEN $
  hydra_define_structure,/kp_lab

ctit2=str_species+'!c'+kp_labels(type(1),2)
cytit=kp_labels(type(1),3)

if getenv( 'hydra_presentation_mode' ) eq '1' then begin
    case type(1) of
        0:ctit2='N!s!dKP'+res_sub+'!n'+spec_super
        1:ctit2='kT!s!dKP'+res_sub+'!n'+spec_super
        else:
    endcase
endif

ffill =  -1.e+31
my_name = '% P_HYDRA_KP: '
get_err_no=0
get_err_msg= my_name +'valid data available for plot'

plot_name=' '
case type(1) of
    0:begin
        t90= data(*).time + ( kp_header(kp_species).start_time - xut1 )
        ydat= data(*).density + 0.01
        dt=attr
        this_Par={ ylog:1 }
        species=ispecies
    end
    1:begin
        t90= data(*).time + ( kp_header(kp_species).start_time - xut1 )
        ydat= data(*).mean_energy
        dt=attr
        this_Par={ ylog:1 }
        species=ispecies
    end
    2:begin                     ; mean energy overplot for spectrogram?
        plot_name='meane_overplot'
        t90= data(*).time + ( kp_header(kp_species).start_time - xut1 )
        ydat= data(*).mean_energy
        dt=attr
        this_Par={ ylog:1 }
        species=ispecies
    END
    10: begin
        this_Par={ ylog:1 }
        t90= data(*).time + ( kp_header(0).start_time - xut1 )
        ydat= data(*).density_fit
        dt=attr
        species=-1
    end
    11: begin
        this_Par={ ylog:1 }
        t90= data(*).time + ( kp_header(0).start_time - xut1 )
        ydat= data(*).mean_energy_fit
        dt=attr
        species=-1         
    end
    12: begin
        this_Par={ ylog:1 }
        t90= data(*).time + ( kp_header(0).start_time - xut1 )
        ydat= data(*).kurtosis
        dt=attr
        species=-1
    end
    13: begin
        this_Par={ ylog:1 }
        t90= data(*).time + ( kp_header(0).start_time - xut1 )
        ydat= data(*).kurtosis_black_frac
        dt=attr
        species=-1
    end
    14: begin
        this_Par={ ylog:0 }
        t90= data(*).time + ( kp_header(0).start_time - xut1 )
        ydat= data(*).density_hot
        dt=attr
        species=-1
    end
    15: begin
        this_Par={ ylog:0 }
        t90= data(*).time + ( kp_header(0).start_time - xut1 )
        ydat= data(*).mean_energy_hot
        dt=attr
        species=-1
    end     
    else: 
endcase

if ( type(1) gt 2 and type(1) lt 10 ) then begin ; statistics from HR kp -> 13.8
    if n_elements( kp_hr_ele_data ) eq 0 then begin
        kp_species=2
        kp_header(kp_species).start_time=xut1
        kp_header(kp_species).end_time=xut2
        kp_header(kp_species).reload_flag=1
        r_hydra_kp
        hreleloaded=1
    endif
    
    nn= n_elements( kp_hr_ele_data(*) )
    dens1= kp_hr_ele_data(*).density
    temp1= kp_hr_ele_data(*).mean_energy
    time1= kp_hr_ele_data(*).time + $
      ( kp_header(2).start_time - xut1 )
    
    b= min( time1 )
    m= 13.8/6
    nn= ( long( ( max(time1) - b ) / m ) / 6 + 1 ) * 6
    dens= fltarr( nn ) + 1e31
    temp= fltarr( nn ) + 1e31
    time= fltarr( nn )
    dt= m
    
    dens( ( time1 - b ) / m ) = dens1
    temp( ( time1 - b ) / m ) = temp1
    time= m * findgen( n_elements(time) ) + b
    
    dens= reform( dens, 6, nn/6 )
    temp= reform( temp, 6, nn/6 )
    time= reform( time, 6, nn/6 )
    valid= make_array( /int, size= size(dens) )
    
;    take averages
    valid(*)=1
    r= where( dens gt 1e30 or temp gt 1e30 ) ; fill - should be the same
    if r(0) ne -1 then begin
        dens(r)= 0
        temp(r)= 0
        valid(r)=0
    endif
    Sdens= total( dens, 1 )
    Stemp= total( temp, 1 )
    N= total( valid, 1 )
    Navg= fltarr( n_elements( N ) ) + 1e31 ; N as in density
    Tavg= fltarr( n_elements( N ) ) + 1e31
    r= where( N gt 0 )     
    if r(0) ne -1 then begin
        Navg(r)= Sdens(r)/N(r)
        Tavg(r)= Stemp(r)/N(r)
    endif
    Nvar= fltarr( n_elements( N ) ) 
    Tvar= fltarr( n_elements( N ) ) 
    N(*)= 0
    for i=0,5 do begin         
        r= where( valid( i,* ) eq 1 )
        if r(0) ne -1 then begin
            Nvar(r)= Nvar(r) + ( dens(i,r) - Navg(r) ) ^ 2
            Tvar(r)= Tvar(r) + ( temp(i,r) - Tavg(r) ) ^ 2
            N(r)= N(r) + 1
        endif
    endfor
    r= where( N gt 1 )
    if r(0) ne -1 then begin
        Nvar(r)= sqrt( Nvar(r) ) / (N(r)-1)
        Tvar(r)= sqrt( Tvar(r) ) / (N(r)-1)
    endif
    r0= where( N le 1 )
    if r0(0) ne -1 then begin
        Nvar(r0)= 1e31
        Tvar(r0)= 1e31
    endif
    time= time(0,*)
    
    rnvalid= where( Nvar gt 1e30 )
    
    t90= time
    
    case type(1) of 
        3: begin
            ydat= Nvar / Navg
            this_Par={ ylog:0 }
        end
        4: begin
            ydat= Tvar / Tavg
            this_Par={ ylog:0 }
        end
        5: begin
            ydat= Navg
            this_Par={ ylog:1 }
        end
        6: begin
            ydat= Tavg
            this_Par={ ylog:1 }
        end
        7: begin
            this_Par= { ylog:0 }             
            itmin= intarr( n_elements( temp(0,*) ) )
            itmax= intarr( n_elements( temp(0,*) ) )             
            indgen= indgen(n_elements(itmin))
            for i=1,5 do begin
                r= where( temp(itmin,indgen)*valid(itmin,indgen) gt $
                          temp(i,*) ) ; find maxs
                if r(0) ne -1 then itmax(r)= i
                r= where( temp(itmin,indgen) lt temp(i,*) ) ; find mins
                if r(0) ne -1 then itmin(r)= i
            endfor
            spin_rate= 2*!pi / 6.012627 ; rad/s
            ydat= ( abs(itmax-itmin) * dt * spin_rate ) mod ( 2*!pi ) / !dtor
        end
        else: ydat= Tavg        ; this will be redefine        
        
    endcase
    if rnvalid(0) ne -1 then ydat(rnvalid)=1e31
    
endif

if type(1) ge 8 and type(1) le 9 then begin
    
    Nion= kp_lr_ion_data(*).density
    Tion= kp_lr_ion_data(*).time + $
      ( kp_header(1).start_time - xut1 )
    Nele= kp_lr_ele_data(*).density
    Tele= kp_lr_ele_data(*).time + $
      ( kp_header(0).start_time - xut1 )
    Temp= kp_lr_ele_data(*).mean_energy / 8.62e-5 ; K, mean_energy=kT
    
; ffill filter must be applied here because it was taken out of
; r_hydra_kp.pro for the lr_ele case, 3/15/99, ppq
    gd = where(Nele ne ffill and Temp ne ffill)
    Nele = Nele(gd)
    Tele = Tele(gd)
    Temp = Temp(gd)
;--------------------------------------
    
    Nvar= Nvar                  ; from above
    Tvar= time
    
;    syncronize times
    b= min( [ Tele, Tion ], max=max )
    m= 13.7
    Ni1= fltarr( long( (max-b) / m + 1 ) ) + 1e31
    Ne1= fltarr( long( (max-b) / m + 1 ) ) + 1e31
    Te1= fltarr( long( (max-b) / m + 1 ) ) + 1e31
    Nv1= fltarr( long( (max-b) / m + 1 ) ) + 1e31
    Time= dblarr( long( (max-b) / m + 1 ) ) + 1e31
    
    Ni1( long( ( Tion - b ) / m ) ) = Nion
    Ne1( long( ( Tele - b ) / m ) ) = Nele
    Te1( long( ( Tele - b ) / m ) ) = Temp
    Nv1( long( ( Tvar - b ) / m ) ) = Nvar
    Time( long( ( Tion - b ) / m ) ) = Tion
    
    r= where( time lt 1e31 )
    
    if r(0) ne -1 then begin
        Nion= Ni1(r)
        Nele= Ne1(r)     
        Tele= Te1(r)
        Time= Time(r)
        Nvar= Nv1(r)
    endif else begin
        Nion= fltarr(5)+ 1e31
        Nele= fltarr(5)+ 1e31
        Tele= fltarr(5)+ 1e31
        Nvar= fltarr(5)+ 1e31
        t90= findgen(5)
    endelse
    
    case type(1) of 
        8: begin
            this_Par={ ylog:0 }             
            ydat= -alog(Nion/Nele)
            t90= Time
        end
        9: begin
            this_Par={ ylog:0 }             
            ydat= -alog(Nion/Nele) * 8.62e-5 * Tele
            t90= time
        end
    endcase
    r= where( abs( Nion-Nele ) lt Nvar, count )
    if r(0) ne -1 then begin
        perc= float(count) / n_elements( ydat )
        message, string(perc,format='(i5.2)')+$
          ' points removed because of N variance', /con
        ydat(r)= 1e31
    endif
endif

; Apply the ffill filter here because it was taken out of
; r_hydra_kp.pro, 3/15/99, ppq
gd = where(ydat ne ffill)
if (gd(0) ne -1) then begin
    t90 = t90(gd)
    ydat = ydat(gd)
endif else begin
    get_err_no=2
    get_err_msg='No valid data in this time range'
    message, get_err_msg, /cont
    return
endelse

ctit=' '                        ; Do not use this variable!

cytit= font+cytit
ctit2= font+ctit2

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF

index=where((t90(1:*) ge 0) and (t90(1:*) le dxut),count)+1
if count ge 2 then begin
    t90=t90(index)
    ydat=ydat(index)
end

; insert breaks where there are data dropouts. (grid the data)
dt= t90(1:*) - t90(0:*)
r= where(dt gt 0)
if r(0) ne -1 then dt=min(dt(r)) else dt=1
nn= max(t90)/dt + 1
ydat_grid= make_array( nn, /float, value=1e31 )
t90_grid= make_array( nn, /float, value=0 )
r= long( t90/dt )
ydat_grid( r ) = ydat
t90_grid( r ) = t90

t90= t90_grid
ydat= ydat_grid

y_kw= hydra_ytick_parm( ydat, fill=1e31, panel=panel, _extra=this_Par )

; set up extra plot keywords, common ones first

extra_plotPar_common={xrange:[0,dxut], $
                      noerase:1, ticklen:-0.03, xticklen:-0.03, $
                      xtickv:xtickv, xminor:xminor, xticks:xticks, $
                      yticklen:0.010, ytitle:cytit, $
                      ythick:1.3, xthick:1.3, max_value:1e30 }

extra_plotPar_common= $
  create_struct( extra_plotPar_common, y_kw )

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

if n_elements(extra_par) EQ 0 then $
  extra_par={color:1}

; add keyword structure set here with the one passed in

extra_plotPar=create_struct( extra_plotPar_common, extra_par ) 

axis_plotPar= create_struct( extra_plotPar_common, {color:1} ) 

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

if panel(0) eq 0 then begin
    xtickformat= 'hydra_timeticks'
endif else begin
    xtickformat= 'noticks'
endelse


; plot the axis in color=1, and only plot if you're not overplotting
if not keyword_set(overplot) then begin
    plot,t90,ydat, /NODATA, $
      xstyle=1, ystyle=1, $
      xtickformat=xtickformat, $
      _extra= axis_plotPar
    if panel(0) eq 0 then x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
endif

; now do the actual data plot
hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5, $
  overplot= keyword_set( overplot )

if not keyword_set(overplot) then begin
    plot,t90,ydat, /NODATA, $
      xstyle=1, ystyle=1, $
      xtickformat=xtickformat, $
      _extra= axis_plotPar
endif

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

; store the coordinate information into common block
plotted_x = !x
plotted_y = !y

!x=oldx
!y=oldy

; don't print the name of the plot if it is over a colorplot.
common plot_composer, widgetData
panelKind= widgetData.plotInfos(*).panelkind
idx= where( widgetData.plotsDrawn(*).panelvector(0) eq panel(0) and $
            widgetData.plotInfos(*).overplot eq 0 and $
            widgetData.plotInfos(*).panelkind ne '' )  
idx= idx(0)
if idx eq -1 then idx=0
colorplot= widgetData.plotsDrawn(idx).zscl_min ne $
  widgetData.plotsDrawn(idx).zscl_max

colorplot=0
if not colorplot then begin
; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
    abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
    lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
    right_side_label,panel,lfs+ctit2, _extra=extra_plotPar
endif

if not overplot then hydra_footprint, kp_header(kp_species)

end






