;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_ppa, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_ppa for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_ppa, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_ppa, panel, type, $
                 OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 _EXTRA=extra_par, subtable=subtable

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Linestyles are also not useful. So the keywords
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; Comments here should include an explanation of the plot options used
; in type, swwitch or channel. Replace type in the call with whatever variables
; you need
   
; One of the common blocks should contain the plot data as returned by the
; data read procedure:
   
   COMMON hydra_ppa_data, $
    ppa_header, $
    ppa_l1_data, $
    ppa_l1_support, $
    ppa_spec_data, $
    ppa_aligned_data, $
    ppa_opposed_data, $
    ppa_perp_data, $
    ppa_skew_data

   COMMON hydra_ppa_labels, $
    ppa_labels
   
; The following common blocks are needed:
   
   common mjdt, mjdt_start, mjdt_end ;common time limits in mjdt
   common time, xut1, xut2
   common yscale, yscl
   common zscale, zscl
   
   common coordinateSystems, plotted_x, plotted_y
                                ;info on coords, used by mouse
                                ;                   functions
   common papco_color_names
   
   panelset,panel               ;sets the panel position viewport
   PAPCO_set_subtable, subtable ;set the color space

   oldy=!y
   oldx=!x
   
;  HYDRA-IOWA uses gif file output, which is converted to pcl.  The
;  following line uses a double stroke font for this case.
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')
   
;  select ticks that fall on even minutes, hours, days, etc...
   goodticks, xtickv, xminor
   xticks= n_elements(xtickv)-1
   
   myname = 'p_hydra_ppa'

; Background flag
   bcor = type(1)               ; 0 means no background removal, 1 means remove it
   IF (not bcor) THEN bcor_str = '!cNo Background Removal' ELSE $
     bcor_str = '!cBackground Removed'
   
; Define the labels
   IF (n_elements(ppa_labels) EQ 0) THEN $
     hydra_define_structure,/ppa_lab
   ctit2 = ppa_labels(type(0),2)
   IF (type(0) EQ 0 or type(0) eq 2) THEN ctit2 = ctit2+bcor_str
   cytit = ppa_labels(type(0),3)
   ctit = ' '
; the variables used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
   
   scpot_time = -1
   case type(0) of
       0:BEGIN
           calc_hydra_ppa_spec,scpot_time,scpot,bcor=bcor,vtcor=1,yticks=yticks, $
             ytickv=ytickv, ylog_init=ylog_init, ytickname=ytickname
           xdat = ppa_spec_data.xdat
           ydat = ppa_spec_data.ydat
           zdat = ppa_spec_data.zdat
           pfill = ppa_spec_data.pfill
           spec = 1
           zmin = .1
           zmax = 1000
           ztit = 'PPA !7R!da!n!x(Counts(!7a!x,E,t)/Viewtime(!7a!x,E,t))'
       end
       1:BEGIN
           dxut= xut2-xut1
           t90 = ppa_l1_data.time + ( ppa_header.start_time - xut1 )
           ydat = ppa_l1_data.background_cps
           pfill = ppa_l1_support.pfill
           spec = 0
       end
       2:BEGIN
           calc_hydra_ppa_spec,scpot_time,scpot,bcor=bcor,vtcor=0,yticks=yticks, $
             ytickv=ytickv, ylog_init=ylog_init, ytickname=ytickname
           xdat = ppa_spec_data.xdat
           ydat = ppa_spec_data.ydat
           zdat = ppa_spec_data.zdat
           pfill = ppa_spec_data.pfill
           spec = 1
           zmin = 10.
           zmax = 100000.
           ztit = 'PPA !7R!da!n!x(Counts(!7a!x,E,t))'
       end
       3:BEGIN
           dxut= xut2-xut1
           t90 = ppa_l1_data.time + ( ppa_header.start_time - xut1 )
           ydat = ppa_l1_data.background_raw
           pfill = ppa_l1_support.pfill
           spec = 0
       end
   endcase
   
;       3:BEGIN
;           IF ppa_header.recalc_flag(type(0)) THEN BEGIN
;               print, myname+': Recalculating '+ctit2
;               ppa_header.recalc_flag(type(0)) = 0
;               calc_hydra_ppa_perp
;               xdat = ppa_perp_data.xdat
;               ydat = ppa_perp_data.ydat
;               zdat = ppa_perp_data.zdat
;           ENDIF ELSE BEGIN 
;               xdat = ppa_perp_data.xdat
;               ydat = ppa_perp_data.ydat
;               zdat = ppa_perp_data.zdat
;           ENDELSE
;       END
;       4:BEGIN
;           IF ppa_header.recalc_flag(type(0)) THEN BEGIN
;               print, myname+': Recalculating '+ctit2
;               ppa_header.recalc_flag(type(0)) = 0
;               calc_hydra_ppa_opposed
;               xdat = ppa_opposed_data.xdat
;               ydat = ppa_opposed_data.ydat
;               zdat = ppa_opposed_data.zdat
;           ENDIF ELSE BEGIN 
;               xdat = ppa_opposed_data.xdat
;               ydat = ppa_opposed_data.ydat
;               zdat = ppa_opposed_data.zdat
;           ENDELSE
;       END
;       5:BEGIN
;           IF ppa_header.recalc_flag(type(0)) THEN BEGIN
;               print, myname+': Recalculating '+ctit2
;               ppa_header.recalc_flag(type(0)) = 0
;               calc_hydra_ppa_skew
;               xdat = ppa_skew_data.xdat
;               ydat = ppa_skew_data.ydat
;               zdat = ppa_skew_data.zdat
;           ENDIF ELSE BEGIN 
;               xdat = ppa_skew_data.xdat
;               ydat = ppa_skew_data.ydat
;               zdat = ppa_skew_data.zdat
;           ENDELSE
;       END
;   ENDCASE
   
   ctit2 = font+ctit2
   cytit = font+cytit
   
   IF (NOT spec) THEN BEGIN     ; Line Plot
       
; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)
       
; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images
       
; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF
       
       index=where((t90(1:*) ge 0) and (t90(1:*) le dxut),count)+1
       if count ge 2 then begin
           t90=t90(index)
           ydat=ydat(index)
       end
       
; insert breaks where there are data dropouts. (grid the data)
       dt= t90(1:*) - t90(0:*)
       r= where(dt gt 0)
       if r(0) ne -1 then dt=min(dt(r)) else dt=1
       nn= max(t90)/dt + 1
       ydat_grid= make_array( nn, /float, value=1e31 )
       t90_grid= make_array( nn, /float, value=0 )
       r= long( t90/dt )
       ydat_grid( r ) = ydat
       t90_grid( r ) = t90
       
       t90= t90_grid
       ydat= ydat_grid
       
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.
       
       hydra_ylim_new, panel(0), type, ydat, yrange=yrange, ylog=ylog
       
; set up extra plot keywords, common ones first
       
       extra_plotPar_common={yrange:yrange, ylog:ylog, xrange:[0,dxut], $
                             noerase:1, ticklen:-0.03, xticklen:-0.03, $
                             xtickv:xtickv, xminor:xminor, xticks:xticks, $
                             yticklen:0.010, ytitle:cytit, $
                             ythick:1.3, xthick:1.3, max_value:1e30 }
       
; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.
       
       if n_elements(extra_par) EQ 0 then $
         extra_par={color:1}
       
; add keyword structure set here with the one passed in
       
       extra_plotPar= create_struct( extra_plotPar_common, extra_par )
       
       axis_plotPar= create_struct( extra_plotPar_common, {color:1} )
       
; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot
       
       if panel(0) eq 0 then begin
           xtickformat= 'hydra_timeticks'
       endif else begin
           xtickformat= 'noticks'
       ENDELSE
       
; plot the axis in color=1, and only plot if if you're not overplotting
       if not keyword_set(overplot) then begin
           plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
             xstyle=1, ystyle=1, xtickformat=xtickformat
           if panel(0) eq 0 then $
             x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
       endif
       
; now do the actual data plot
       hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot )
       
       if not keyword_set(overplot) then begin
           plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
             xstyle=1, ystyle=1, xtickformat=xtickformat
           if panel(0) eq 0 then $
             x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
       endif
       
; store the coordinate information into common block
       plotted_x = !x
       plotted_y = !y
       
       !x=oldx
       !y=oldy
       
; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015
       
       if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
           r= where( (xtickv+xut1) mod 86400 eq 0, count )
           if r(0) ne -1 then begin
               xtickv= xtickv(r)
               xticks= count
           endif else begin
               xtickv= xtickv([0,n_elements(xtickv)-1])
               xticks= 1
           endelse
           
           axis_plotPar.xtickv= xtickv
           axis_plotPar.xticks= xticks
           
           plot, t90, ydat, /NODATA, $
             _extra=axis_plotPar, xtickformat='hydra_juldate', $
             xstyle=1, ystyle=5
           
       endif
       
; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
       abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
       lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
       right_side_label,panel,lfs+ctit2, _extra=extra_plotPar
       
       
   ENDIF ELSE BEGIN             ; spec
       
; Y axis scaling
       ymin=min(ydat) & ymax=max(ydat)
       ylen = ymax-ymin
       ymin = ymin-.001*ylen & ymax = ymax+.001*ylen
       if (yscl(panel(0),0) eq 1) then begin
           ymin=yscl(panel(0),1)
           ymax=yscl(panel(0),2)
           ylog= yscl(panel(0),3)
       endif else BEGIN
           ylog = ylog_init
           yscl(panel(0),1)=ymin
           yscl(panel(0),2)=ymax
           yscl(panel(0),3)=ylog
       endelse
       
; "Z" axis scaling
       if (zscl(panel(0),0) eq 1) then begin
           zmin=zscl(panel(0),1)
           zmax=zscl(panel(0),2)
       endif else begin
           zscl(panel(0),1)=zmin
           zscl(panel(0),2)=zmax
       endelse
       
; set up extra plot keywords, common ones first
       
       extra_plotPar_common={ xrange:[0,xut2-xut1], xstyle:1, $
                              noerase:1, ticklen:-0.03, xticklen:-0.03, $
                              xminor:xminor, xtickv:xtickv, xticks:xticks, $
                              ytitle:cytit, yticklen:-0.010, $
                              ylog:ylog, yrange:[ymin,ymax], ystyle:1 }
       
; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.
       
       if n_elements(extra_par) EQ 0 then $
         extra_par={color:1}
       
; add keyword structure set here with the one passed in (extra)
; add keyword structure specific to the subtype of plot chosen here (this)
       
       extra_plotPar=create_struct(extra_plotPar_common, extra_par )
       
       
       axis_plotPar=create_struct(extra_plotPar_common, {color:1} )
       
; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot
       
       if panel(0) eq 0 then $
         xtickformat='hydra_timeticks' $
       else $
         xtickformat='noticks'
       
; The spectrogram
       if (type(0) eq 0 or type(0) eq 2) then begin
           axis_plotPar=create_struct(axis_plotPar,{yticks:yticks,ytickv:ytickv,ytickname:ytickname})
       endif
       dx= total( xdat(*,1)-xdat(*,0) ) / n_elements(xdat(*,0))
       hydra_plotcolor, zdat, xdat, ydat, /zlog, zrange=[zmin,zmax], $
         _extra=axis_plotPar, dx=dx, $
         xtickformat=xtickformat, nodata=pfill
       if (n_elements(scpot_time) gt 1) then oplot, scpot_time, scpot, line=0, color=1
       
; The colorbar
       hydra_color_bar,zmin,zmax,ztit,/zlog
       
; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
       
       if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
           x_side_label,panel,font+'time (UT)!CDate'
       endif
       
; store the coordinate information into common block
       plotted_x = !x
       plotted_y = !y
       
; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. Since here colorbars are plotted labels are rotated
; by 90 degrees and then plotted.
       
       if not keyword_set(overplot) then $
         right_side_label,panel,ctit2,/rot90
       
   ENDELSE 
   
   hydra_footprint, ppa_header
   
end





