;***************************************************************************
;* PROCEDURE:
;*      p_hydra_spec, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type DATA_TYPE for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*              For this data type has the following meaning:
;*
;*      type(0) 0 for electrons
;*              1 for ions
;*              2 for both in one plot (iowa style).
;*
;*      type(2) 0 data from all pitch angles
;*              1 only field-aligned measurements
;*              2 only perpendicular
;*              3 only field opposed measurements
;*
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_spec, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     written april 1996, Reiner Friedel
;*
;***************************************************************************
pro p_hydra_spec, panel, type, OUTPUT=OUTPUT, $
              OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                  subtable=subtable, _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Linestyles are also not useful. So the keywords
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; Comments here should include an explanation of the plot options used
; in type, swwitch or channel. Replace type in the call with whatever variables
; you need

  common hydra_spec_data, $
    data_context, $             ; selects current data set
    spec_header, $
    spec_data, $
    spec_aligned_data, $
    spec_perp_data, $
    spec_opposed_data, $
    hr_spec_data, $
    hr_spec_aligned_data, $
    hr_spec_perp_data, $
    hr_spec_opposed_data 

; One of the common blocks should contain the plot data as returned by the
; data read procedure.
; The following common blocks are needed:

   common mjdt, mjdt_start, mjdt_end ;common time limits in mjdt
   common time, xut1, xut2
   common yscale, yscl
   common zscale, zscl

   common coordinateSystems, plotted_x, plotted_y
                                ;info on coords, used by mouse
                                ;                   functions
   common papco_color_names

   panelset,panel               ;sets the panel position viewport
   PAPCO_set_subtable, subtable ;set the color space

;  HYDRA-IOWA uses gif file output, which is converted to pcl.  The
;  following line uses a double stroke font for this case.
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

;  select ticks that fall on even minutes, hours, days, etc...
   goodticks, xtickv, xminor
   xticks= n_elements(xtickv)-1

   rfill= 1e-20

;  copy into our local buffer the data set.
   exe_str='data=' + spec_header(data_context).data
   r=execute( exe_str )         

   if n_elements( data ) eq 0 then begin
       plotted_x = !x
       plotted_y = !y
       message, 'Internal Error: No data in memory.', /cont
       return
   endif

   pitch_range= spec_header( data_context ). pitch_range

   t90= data(*).time + ( spec_header(data_context).start_time - xut1 )
   dt= spec_header(data_context).time_resolution
   dxut= xut2-xut1

   rvis= where( t90 ge 0 and t90 le dxut, nvis )
   if nvis lt 2 then begin
       plotted_x = !x
       plotted_y = !y
       print, 'p_hydra_spec: no data in interval'
       spec_matrix= data(*).spectrum(*)
       t90= t90
       return
   endif else begin
       spec_matrix= data(rvis).spectrum(*)
       t90=t90(rvis)
   endelse

   sc_pot= data(rvis).sc_pot
   sc_pot_qual= data(rvis).sc_pot_qual

   mode= data(rvis).mode
   mode_energy= spec_header(data_context).mode_energy

   if max( mode ) gt 0 then begin
       spectra_sort, spec_matrix, mode, mode_energy, avg1, energy
   endif else begin
       n=n_elements(spec_matrix(*,0))/2
       energy_spec= 10^(findgen(n)/(n-1))*4
       energy= [ -1 * reverse(energy_spec), 0., energy_spec ]
       avg1= spec_matrix
       avg1(*)=1e-20
   endelse
   
;  detect psi on
   r= where( data(rvis).psi_status eq 1 )
   psi_on= r(0) ne -1
   
   if data_context lt 5 then res='sv' else res='hr'

   dc= data_context

   if data_context lt 5 then begin
       plot_name= 'papco_spec_sv'
       input_file= 'survey'
   endif else begin
       plot_name= 'papco_spec_l1'
       input_file= 'ddeis_l1_cal'
   endelse
   date= convert_secondstotime( spec_header(dc).start_time, /t90, country=1 )
   if strlen( date ) eq 17 then begin
       date= '19'+ strmid( date,6,2 ) + strmid( date,0,2 ) + strmid( date,3,2 )
   endif else begin
       date= strmid( date,6,4 ) + strmid( date,0,2 ) + strmid( date, 3, 2 )
   endelse

;  information useful for pop-up widgets
   interactive= papco_interactive( draw_id= draw_id )   
   
   plot_version= hyd_prod_vers_v01( product=plot_name, $
                                    dependency_names= $
                                    [ input_file, 'papco_spec_code' ], $
                                    dependency_vers= $
                                    [ spec_header(dc).data_version, $
                                      hydra_code_version(/spec) ] $
                                  )

   if plot_version gt 0.0 then begin
       qf= hyd_prod_qual_v01( product_type=plot_name, $
                              date=date, $
                              product_version=plot_version )
   
       
       if qf gt -1 then begin
           if interactive then begin
               case qf of
                   1: if spec_header(dc).message_flag eq 1 then begin
                       r= widget_message(['Plot is not',$
                                          'the best available.'], $
                                         dialog_parent=draw_id, /info )
                       spec_header(dc).message_flag = 0
                   endif
                   2: if spec_header(dc).message_flag eq 1 then begin
                       r= widget_message(['Plot contains','known flaws.'], $
                                         dialog_parent=draw_id, /info )
                       spec_header(dc).message_flag = 0
                   endif
                   3:
                   else:
               endcase
           endif
           case qf of
               1: qualstr='(NBA)'
               2: qualstr='(F)'
               3: qualstr='(BA)'
               else:
           endcase
       endif 
   endif else begin
       message, 'Unidentified product.', /cont
       dependency_names= $
         [ input_file, 'papco_spec_code' ]
       dependency_vers= $
         [ spec_header(dc).data_version, $
           hydra_code_version(/spec) ]
       print, plot_name, $
         '[', dependency_names + '=' + strtrim( dependency_vers, 2 ), $
         '] --> ', strtrim( plot_version, 2 )
       qf=1
       qualstr='(??)'
   endelse

   if ( psi_on and interactive ) then begin
       w= widget_info( draw_id, /geo )       
       r= hydra_message( [ 'PSI power supply on', 'during this interval' ], $
                         xoffset=w.xoffset-50, /warn, timeout=2. )
   endif

   spec_shift= spec_header(dc).options and 1L

   channelwidth=0.07		;sets % of channel width to get top
				;and bottom bounds
   case type(0) of
     0:begin			;electrons
         rele=where(energy lt 0)
         zmat=rotate(avg1(rele,*),3)
         en_e=energy(rele)
         energy= -reverse(en_e)
         yminarr=reverse(en_e)*(1-channelwidth)*(-1)
         ymaxarr=reverse(en_e)*(1+channelwidth)*(-1)
        ;invert the array to have increasing energy y-scale
         ctit2='HYDRA!cEle'
         zmax = 500.
         r = where( zmat gt zmax , count )
         if ( count gt 0 ) then zmax = 4000.
         zmin = 0.5
       end
     1:begin			;ions
         rspecies=where(energy gt 0)
         zmat=transpose(avg1(rspecies,*))
         en_i=energy(rspecies)
         energy= en_i
         yminarr=reverse(en_i)*(1-channelwidth)
         ymaxarr=reverse(en_i)*(1+channelwidth)
         ctit2='HYDRA!cIon'
         zmax = 100.
         zmin = 0.1
         sc_pot= -1.0 * sc_pot
       end
   endcase

   if getenv( 'hydra_presentation_mode' ) eq '1' then begin
       if spec_shift then cytit='E!dD!n(eV)' else $
         cytit='E!dObs!n(eV)'
   endif else begin
       if spec_shift then cytit='E Debye, eV' else $
         cytit='E obs,eV'
   endelse

   zunits= ( type(3) and 14 ) / 2 ; 1=cnts, 2=dJ/dE, 0=d(energy_flux) 
                                  ; 3=density-normalized 
   case zunits of
       1: begin
           efluxfact =1.
           ztit= 'counts'
       end
       2: BEGIN;rf
           efluxfact = 187970.
           ztit= '(cm!e2!n-s-sr-!4D!xE/E)!e-1!n'
           ztit= '(cm!e2!n-s-sr)!e-1!n';rf
       end
       0: BEGIN;rf
           r= where( zmat ne rfill )
           if r(0) ne -1 then begin
               zmat(r)= zmat(r) * 187970 / $
                 ( make_array( n_elements(zmat(*,0)),$
                               /int,value=1 ) # (energy/1000.) )(r)
           endif
           efluxfact = 1.
           ztit= '(cm!e2!n-s-sr-!4D!xE/E-keV)!e-1!n'
           ztit= '(cm!e2!n-s-sr-keV)!e-1!n';rf
           zmin=1e5 & zmax=1e9
       end
       3: begin

           species= type[0]*2 -1
           
           detector_area= 0.242000D  
           solid_angle= 0.015300D
           delta_time= 0.02395830D      ; true for mode 41
           if species eq -1 then mass= 9.11d-28  else mass= 1.67d-24
           
           energy_mat= replicate(1,n_elements(t90)) # energy 

           facv4= detector_area * solid_angle * delta_time * 0.06 * 0.5 * $
             ( energy_mat * 2 * 1.6d-12 / mass )^2
           r= where( zmat eq rfill )
           f= zmat/facv4
           zmat=f
           if r[0] ne -1 then zmat[r]= rfill
           
           time_t90= double(spec_header(dc).start_time) + t90 
           hd_make_timetags, ttag, time_t90, /width_is_delta12, /t90
           
           case 1 of
               species eq -1 and dc ge 5: id='hrkp_density'               
               species eq  1 and dc ge 5: id='hrkp_density'
               species eq -1 and dc lt 5: id='kp_density_ele'
               species eq  1 and dc lt 5: id='kp_density_ion'
           endcase

           message, 'using '+id+' to normalize the distribution.', /cont
           hyd_access, ttag, id, density

           zmat_max= fltarr(n_elements(t90))

           nen= n_elements( zmat[0,*] )

           normalization= density # replicate(1,nen)
           r= where( zmat ne rfill  and $
                     normalization gt 0. )
           rnv= where( zmat eq rfill  or $
                       normalization le 0. )
           if r[0] ne -1 then begin
               zmat[r]= zmat[r] / normalization[r]
           endif
           if rnv[0] ne -1 then begin
               zmat[rnv]= rfill
           endif
           efluxfact=1.0
           ztit= 'f/density (cgs) '

           if species eq -1 then begin
               zmin=1e-36 & zmax=1e-26
           endif else begin
               zmin=1e-28 & zmax=1e-22
           endelse
       end
   endcase    

   if pitch_range(1) - pitch_range(0) ne 180 then begin
       if pitch_range(0) eq 0 then $
         ctit2= ctit2+ ' !9#!x'
       if pitch_range(1) eq 180 then $
         ctit2= ctit2+ ' -!9#!x'
       if total(pitch_range)/2 eq 90 then $
         ctit2= ctit2+ ' !9x!x'

       ctit2=ctit2+'!c!a!i'+strtrim(fix(pitch_range(0)),2)+'<!7a!x<'+$
         strtrim(fix(pitch_range(1)),2)+'!n'
   endif

   ctit=' '

   ctit2=font+ctit2
   cytit=font+cytit

; axis range vector...

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

  ymin=min(yminarr) & ymax=max(ymaxarr)

; Y axis scaling
  if (yscl(panel(0),0) eq 1) then begin
      ymin=yscl(panel(0),1)
      ymax=yscl(panel(0),2)
      ylog= yscl(panel(0),3)
  endif else begin
      print, 'setting overplot range...'
      ylog=1
      yscl(panel(0),1)=ymin
      yscl(panel(0),2)=ymax
      yscl(panel(0),3)=ylog
  endelse
  
; "Z" axis scaling
  if (zscl(panel(0),0) eq 1) then begin
      zmin=zscl(panel(0),1)/efluxfact
      zmax=zscl(panel(0),2)/efluxfact
  endif else begin
      zscl(panel(0),1)=zmin*efluxfact
      zscl(panel(0),2)=zmax*efluxfact
  endelse

; set up extra plot keywords, common ones first

  extra_plotPar_common={ xrange:[0,xut2-xut1], xstyle:1, $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xminor:xminor, xtickv:xtickv, xticks:xticks, $
                         ytitle:cytit, yticklen:-0.010, ylog:ylog, $
                         yrange:[ymin,ymax], ystyle:1 }

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

  if n_elements(extra_par) EQ 0 then $
    extra_par={color:1}

; add keyword structure set here with the one passed in (extra)
; add keyword structure specific to the subtype of plot chosen here (this)

  extra_plotPar=create_struct(extra_plotPar_common, $
                              extra_par )
  
  
  axis_plotPar=create_struct(extra_plotPar_common, $
                             {color:1} )

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then $
     xtickformat='hydra_timeticks' $
   else $
     xtickformat='noticks'

   if axis_plotPar.ylog ne 0 then $
     ytickformat='hyd_log_ticks' $
   else $
     ytickformat=''
   
;   if panel(0) eq 1 then begin
;       hydra_plot_clr, zmat, t90, yminarr, zrange=[zmin,zmax], /zlog, $
;         nodata=rfill, _extra= axis_plotPar, xtickformat=xtickformat
;   endif else begin
   x1= t90
   x2= t90+dt
   t= reform( [ x1, x2], n_elements(x1), 2 )
   e1= alog10(energy)
   n=n_elements(e1)-1
   de= e1(n) - e1(n-1)
   e2= [e1(1:*),e1(n)+de ]
   e= reform( 10^[ e1, e2], n_elements(e1), 2 )

   goto, skip_data_dump         ; ----------------------------------
; Dumping spectrogram data for other groups...
   convert_t90_to_date,xut1,year1,doy1,h1,m1,s1,mo1,cmo1,dom1
   start_string = string(year1,'(i4.4)') + string(mo1,'(i2.2)') + $
     string(dom1,'(i2.2)') + '  ' + string(h1,'(i2.2)') + ':' + $
     string(m1,'(i2.2)') + ':' + string(s1,'(i2.2)')
   if (type(0) eq 0) then sstring = 'ele' else sstring = 'ion'
   fn = 'mfreeman_'+sstring+'_specdata.txt'
   openw, unit, fn, /get_lun
   printf, unit, 'File: '+fn
   printf, unit, 'Created: '+systime()
   printf, unit, 'Content: Polar/Hydra '+sstring+' spectrogram data'
   printf, unit, 'Start Time: '+start_string
   printf, unit, 't1 unit: Seconds since Start Time'
   printf, unit, 't2 unit: Seconds since Start Time'
   printf, unit, 'energy unit: eV'
   printf, unit, 'Diff. En. Flux unit: (cm2-s-sr-dE/E)^(-1)'
   printf, unit, ''
   printf, unit, $
     '     t1       ', $
     '     t2       ', $
     '  energy      ', $
     'Diff. En. Flux'
   printf, unit, $
     '--------------', $
     '--------------', $
     '--------------', $
     '--------------'
   for ix = 0, n_elements(t(*,0))-1 do BEGIN
       for iy = 0, n_elements(e(*,0))-1 do begin
           printf, unit, t(ix,0), t(ix,1), e(iy,0), zmat(ix,iy)*efluxfact, $
             format='(4(e13.6,1x))'
       endfor
   endfor
   close, unit
   free_lun, unit
   skip_data_dump:              ;--------------------------------
   
   hydra_plotcolor, zmat, t, e, /zlog, zrange=[zmin,zmax], $
     _extra=axis_plotPar, ytickformat=ytickformat, $
     xtickformat=xtickformat, nodata=rfill, dx=dt
;   endelse
   hydra_color_bar,zmin*efluxfact,zmax*efluxfact,ztit,/zlog

; overplot space craft potential
   if ( type(3) and 1 ) eq 0 then begin
       rokay= where( ( sc_pot_qual and 12 ) eq 0 )
       if rokay(0) ne -1 then oplot, t90(rokay), sc_pot(rokay), color=black, $
         min_value=0.1, max_value=1e30

       rwarn= where( ( sc_pot_qual and 12 ) gt 0 )
       if rwarn(0) ne -1 then oplot, t90(rwarn), sc_pot(rwarn), color=black, $
         linestyle=2, min_value=0.1, max_value=1e30 ; dotted
   endif

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     x_side_label,panel,font+'time (UT)!CDate'
  endif

; store the coordinate information into common block
plotted_x = !x & plotted_y = !y

; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. Since here colorbars are plotted labels are rotated
; by 90 degrees and then plotted.

   if not keyword_set(overplot) then $
     right_side_label,panel,ctit2,/rot90

   if (qf ge 0) and (plot_version gt 0.0) then begin
       plot_versionstr= strtrim( string( plot_version, form='(f6.2)' ),2 )
       if data_context gt 4 then res=1 else res=0
       tresstr=(['pspecsv','pspecl1'])(res)
       footprint_str=  tresstr + $
         ':'+ plot_versionstr + qualstr
       hydra_footprint_new, string= footprint_str
   endif else begin
       if data_context le 4 then begin
           hydra_footprint_new, $
             survey=spec_header(data_context).data_version, $
             code= spec_header(data_context).reader_version, notes='(??)'
       endif else begin
           hydra_footprint_new, $
             ddeis_l1_cal=spec_header(data_context).data_version, $
             code= spec_header(data_context).reader_version, notes='(??)'
       endelse
   endelse
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
if KEYWORD_SET(OUTPUT) then BEGIN

    message, 'Writing plot data out to file', /cont
    description='POLAR HYDRA spectra'
    time_label='Seconds since start of day'
    channels=e
    y_label=cytit
    z_label=ztit
    rs_label=ctit2
    nodata=0
    papco_write_data, t, zmat*efluxfact, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF

END 
