;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_sppion, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_sppion for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_sppion, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_sppion, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable



; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

;   common hydra_yscale, yscl
   common yscale, yscl			;man/auto yscaling
   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords used by mouse

   common hydra_sppion_data, sppion_header, sppion_data

   common papco_color_names

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1

; set duplex font if big_gif is device
   if !d.name eq 'Z' then font='!5' else font='' ; duplex roman

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1

   data= sppion_data

 plot_name=' '
 case type(0) of
     0:begin
         ctit2='SPP Ion !cDensity' & cytit='cm!E-3!n'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).density_ion + 0.01
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     end
     1:begin
         ctit2='SPP Ion !cMean Energy' & cytit='eV'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).mean_energy_ion
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     end
     2:begin                    ; mean energy overplot for spectrogram?
         plot_name='ion_meane_overplot'
         ctit2=' ' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).mean_energy_ion
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     3:begin
         ctit2='SPP Ion !cvar_bhat_diag' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).var_bhat_diag
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     4:begin
         ctit2='SPP Ion !clshell' & cytit='R!dE!n'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).lshell
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     5:begin
         ctit2='SPP Ion !cradius' & cytit='R!dE!n'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).radius
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     6:begin
         ctit2='SPP Ion !cCC!dmax!n' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).max_cc(0)
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     7:begin
         ctit2='SPP Ion !cEnergy of CC!dmax!n' & cytit='eV'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).max_cc(1)
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     8:begin
         ctit2='SPP Ion !cE!dmax!n Patched' & cytit='eV'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).max_energy
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     9:begin
         ctit2='SPP Ion !cE!dmin!n Patched' & cytit='eV'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).min_energy
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     10:begin
         ctit2='SPP Ion !cAngle b/t Parl e.v. and B' & cytit='rad'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).angle_b
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     11:begin
         ctit2='SPP Ion !cV_alf' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).v_alf
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     12:begin
         ctit2='SPP Ion !cumag' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_pay(3)
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     13:begin
         ctit2='SPP Ion !cu_parl' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_parl
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     14:begin
         ctit2='SPP Ion !cu_perp_x' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_perp(0)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     15:begin
         ctit2='SPP Ion !cu_perp_y' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_perp(1)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     16:begin
         ctit2='SPP Ion !cu_perp_z' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_perp(2)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     17:begin
         ctit2='SPP Ion !cu_perp_mag' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_perp(3)
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     18:begin
         ctit2='SPP Ion !cux_pay' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_pay(0)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     19:begin
         ctit2='SPP Ion !cuy_pay' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_pay(1)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     20:begin
         ctit2='SPP Ion !cuz_pay' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_pay(2)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     21:begin
         ctit2='SPP Ion !cux_gsm' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_gsm(0)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     22:begin
         ctit2='SPP Ion !cuy_gsm' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_gsm(1)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     23:begin
         ctit2='SPP Ion !cuz_gsm' & cytit='km/s'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).u_gsm(2)
         dt=13.8
         this_Par={ ylog:0 }
         species=1
     END
     24:begin
         ctit2='SPP Ion !ckT!dparl!n' & cytit='eV'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).ktparl
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     25:begin
         ctit2='SPP Ion !ckT!dperp!n' & cytit='eV'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).ktperp
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     26:begin
         ctit2='SPP Ion !cP!dparl!n' & cytit='cgs'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).pres_parl
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     27:begin
         ctit2='SPP Ion !cP!dperp!n' & cytit='cgs'
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).pres_perp
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     28:begin
         ctit2='SPP Ion !crchi2 All' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).rchi2_all
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     29:begin
         ctit2='SPP Ion !crchi2 Patch' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).rchi2_patch
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     30:begin
         ctit2='SPP Ion !cnpatch' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).npatch
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     31:begin
         ctit2='SPP Ion !cnrealizations' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).nrealizations
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
     32:begin
         ctit2='SPP Ion !cAnisotropy' & cytit=' '
         t90= data(*).time_ion + ( sppion_header.start_time - xut1 )
         ydat= data(*).ktparl/data(*).ktperp
         dt=13.8
         this_Par={ ylog:1 }
         species=1
     END
 endcase

; filter out invalid data
 r= where( sppion_data(*).fit_stat_ion ne 1 )
 if r(0) ne -1 then ydat(r)= 1e31

  ctit=' '                      ; Do not use this variable!

  cytit= font+cytit
  ctit2= font+ctit2

; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)

; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images

; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF

  index=where((t90(1:*) ge 0) and (t90(1:*) le dxut),count)+1
  if count ge 2 then begin
      t90=t90(index)
      ydat=ydat(index)
  end

; insert breaks where there are data dropouts. (grid the data)
  dt= t90(1:*) - t90(0:*)
  r= where(dt gt 0)
  if r(0) ne -1 then dt=min(dt(r)) else dt=1
  nn= max(t90)/dt + 1
  ydat_grid= make_array( nn, /float, value=1e31 )
  t90_grid= make_array( nn, /float, value=0 )
  r= long( t90/dt )
  ydat_grid( r ) = ydat
  t90_grid( r ) = t90

  t90= t90_grid
  ydat= ydat_grid


; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.

  if n_elements(yscl) eq 0 then yscl=dblarr(20,4)

  if (yscl(panel(0),0) eq 1) then begin
      print, '% p_hydra_sppion: getting manual scale specifications...'
      yst= yscl(panel(0),1)
      yen= yscl(panel(0),2)
      this_Par.ylog= yscl(panel(0),3)

      IF (yst LE 0 AND this_Par.ylog EQ 1) THEN BEGIN
         print, '% p_hydra_sppion: Negative ymin: swwitching from log to linear...'
         yscl(panel(0),3) =  0
         this_Par.ylog = yscl(panel(0),3)
      ENDIF

  endif else begin              ; use autoscale results
      rv=where(ydat lt 1e31)
      if rv(0) ne -1 then rt=where( t90(rv) ge 0 and t90(rv) le dxut ) $
      else rt=-1
      if (rt(0) ne -1) then begin
          hydra_ylim, ydat(rv(rt)), yst, yen, ylog, _extra=this_Par
          this_Par.ylog=ylog
      endif else begin
          yst=0 & yen=1
      endelse
      yscl(panel(0),1)= yst
      yscl(panel(0),2)= yen
      yscl(panel(0),3)= this_Par.ylog
  endelse

; set up extra plot keywords, common ones first

   extra_plotPar_common={yrange:[yst, yen], xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }

; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
     extra_par={color:1}

; add keyword structure set here with the one passed in

   extra_plotPar= create_struct(extra_plotPar_common, $
                                create_struct( extra_par, this_par ) )

   axis_plotPar= create_struct(extra_plotPar_common, $
                               create_struct( {color:1}, this_par) )

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then $
     xtickformat='hydra_timeticks' $
   else $
     xtickformat='noticks'

; plot the axis in color=1, and only plot if if you're not overplotting
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
          x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03

; hydra events "underplots" shaded regions, highliting specific events.
;       hydra_events
   endif

; now do the actual data plot
   hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y

   !x=oldx
   !y=oldy

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
  abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
  lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
  right_side_label,panel,lfs+ctit2, _extra=extra_plotPar

  hydra_footprint, sppion_header

end





