;***************************************************************************
;*
;* PROCEDURE:
;*      p_hydra_tpphr, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hydra_tpphr for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hydra_tpphr, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hydra_tpphr, panel, type, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
           _EXTRA=extra_par, subtable=subtable

; The following common blocks are needed:

   common time, xut1,xut2 ;common time limits in T90
   common mjdt, mjdt1, mjdt2

;   common hydra_yscale, yscl
   common yscale, yscl			;man/auto yscaling
   common coordinateSystems, plotted_x, plotted_y
                                        ;info on coords used by mouse

   common hydra_tpphr_data, $
    tpphr_nsweep_index, tpphr_nsweep, $
    tpphr_header, $
    tpphr_data1, $
    tpphr_data2, $
    tpphr_data3, $
    tpphr_data6

   common papco_color_names

   COMMON hydra_tpphr_labels, $
    tpphr_labels

   panelset,panel			;sets the panel position viewport

   oldy=!y
   oldx=!x

   goodticks, xtickv, xminor
   xticks=n_elements(xtickv)-1
   
; set duplex font if big_gif is device ; duplex roman
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')
   
; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1
   
   IF (tpphr_nsweep EQ 1) THEN BEGIN 
      data= tpphr_data1
   ENDIF ELSE IF (tpphr_nsweep EQ 2) THEN BEGIN 
      data = tpphr_data2
   ENDIF ELSE IF (tpphr_nsweep EQ 3) THEN BEGIN
      data = tpphr_data3
   ENDIF ELSE BEGIN
      data = tpphr_data6
   ENDELSE 

   rfill = -1.e+31
   errorbars = 0
   dt = data(0).dt

; Define the labels
   IF (n_elements(tpphr_labels) EQ 0) THEN $
    hydra_define_structure,/tpphr_lab

   ctit2 = 'TPPHR Ion '+string(tpphr_nsweep,'(i1)')+' !c'+tpphr_labels(type(1),2)
   cytit = tpphr_labels(type(1),3)
   
; Define the data filter for the 'best' quantities 
; Note: 'rstudy' quantities have their own filter, applied below
   good_best = where( (data.fit_stat_ion EQ 0) OR $
                      (data.fit_stat_ion AND 4) )
   gfit_den = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                    AND data.rstudy_den_stats(6) EQ 0)
   gfit_ux = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                   AND data.rstudy_u_stats(0,6) EQ 0)
   gfit_uy = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                   AND data.rstudy_u_stats(1,6) EQ 0)
   gfit_uz = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                   AND data.rstudy_u_stats(2,6) EQ 0)
   gfit_tparl = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_tparl_stats(6) EQ 0)
   gfit_tperp = where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                      AND data.rstudy_tperp_stats(6) EQ 0)
   gfit_gyrotropy =where((data.fit_stat_ion EQ 0 OR data.fit_stat_ion EQ 4)$
                         AND data.rstudy_gyrotropy_stats(6) EQ 0)
   
   plot_name=' '
   case type(1) of
      0:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).density_ion
         ylog=1 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_den).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_den).density_ion - $
             data(gfit_den).rstudy_den_stats(2) > 0
            erry2 = data(gfit_den).density_ion + $
             data(gfit_den).rstudy_den_stats(3) > 0
         ENDIF
      END 
      1:begin
         t90= data(gfit_den).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_den).rstudy_den_stats(1)
         ylog=1 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_den).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = ( data(gfit_den).rstudy_den_stats(1) - $
                      data(gfit_den).rstudy_den_stats(2) ) > 0
            erry2 = ( data(gfit_den).rstudy_den_stats(1) + $
                      data(gfit_den).rstudy_den_stats(3) ) > 0
         ENDIF
      END
      2:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).mean_energy_ion
         ylog=1 
      end
      3:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).b_avg_angle_cone
         ylog=1 
      END
      4:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).max_energy
         ylog=1 
      END
      5:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).min_energy
         ylog=1 
      END
      6:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).angle_b
         ylog=1 
      END
      7:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).v_alf
         ylog=0 
      END
      8:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_pay(3)
         ylog=1 
      END
      9:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_parl
         ylog=0 
      END
      10:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_perp(0)
         ylog=0 
      END
      11:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_perp(1)
         ylog=0 
      END
      12:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_perp(2)
         ylog=0 
      END
      13:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_perp(3)
         ylog=1 
      END
      14:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_pay(0)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_ux).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_ux).u_pay(0) - data(gfit_ux).rstudy_u_stats(0,2)
            erry2 = data(gfit_ux).u_pay(0) + data(gfit_ux).rstudy_u_stats(0,3)
         ENDIF
      END
      15:begin
         t90= data(gfit_ux).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_ux).rstudy_u_stats(0,1)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_ux).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_ux).rstudy_u_stats(0,1) - $
             data(gfit_ux).rstudy_u_stats(0,2)
            erry2 = data(gfit_ux).rstudy_u_stats(0,1) + $
             data(gfit_ux).rstudy_u_stats(0,3)
         ENDIF
      END
      16:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_pay(1)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uy).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_uy).u_pay(1) - data(gfit_uy).rstudy_u_stats(1,2)
            erry2 = data(gfit_uy).u_pay(1) + data(gfit_uy).rstudy_u_stats(1,3)
         ENDIF
      END
      17:begin
         t90= data(gfit_uy).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_uy).rstudy_u_stats(1,1)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uy).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_uy).rstudy_u_stats(1,1) - $
             data(gfit_uy).rstudy_u_stats(1,2)
            erry2 = data(gfit_uy).rstudy_u_stats(1,1) + $
             data(gfit_uy).rstudy_u_stats(1,3)
         ENDIF
      END
      18:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_pay(2)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uz).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_uz).u_pay(2) - data(gfit_uz).rstudy_u_stats(2,2)
            erry2 = data(gfit_uz).u_pay(2) + data(gfit_uz).rstudy_u_stats(2,3)
         ENDIF
      END
      19:begin
         t90= data(gfit_uz).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_uz).rstudy_u_stats(2,1)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_uz).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_uz).rstudy_u_stats(2,1) - $
             data(gfit_uz).rstudy_u_stats(2,2)
            erry2 = data(gfit_uz).rstudy_u_stats(2,1) + $
             data(gfit_uz).rstudy_u_stats(2,3)
         ENDIF
      END
      20:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_gsm(0)
         ylog=0 
      END
      21:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_gsm(1)
         ylog=0 
      END
      22:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).u_gsm(2)
         ylog=0 
      END
      23:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).ktparl
         ylog=1 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tparl).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_tparl).ktparl - $
             data(gfit_tparl).rstudy_tparl_stats(2)
            erry2 = data(gfit_tparl).ktparl + $
             data(gfit_tparl).rstudy_tparl_stats(3)
         ENDIF
      END
      24:begin
         t90= data(gfit_tparl).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_tparl).rstudy_tparl_stats(1)
         ylog=1 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tparl).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_tparl).rstudy_tparl_stats(1) - $
             data(gfit_tparl).rstudy_tparl_stats(2)
            erry2 = data(gfit_tparl).rstudy_tparl_stats(1) + $
             data(gfit_tparl).rstudy_tparl_stats(3)
         ENDIF
      END
      25:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).ktperp
         ylog=1 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tperp).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_tperp).ktperp - $
             data(gfit_tperp).rstudy_tperp_stats(2)
            erry2 = data(gfit_tperp).ktperp + $
             data(gfit_tperp).rstudy_tperp_stats(3)
         ENDIF
      END
      26:begin
         t90= data(gfit_tperp).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_tperp).rstudy_tperp_stats(1)
         ylog=1 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_tperp).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_tperp).rstudy_tperp_stats(1) - $
             data(gfit_tperp).rstudy_tperp_stats(2)
            erry2 = data(gfit_tperp).rstudy_tperp_stats(1) + $
             data(gfit_tperp).rstudy_tperp_stats(3)
         ENDIF
      END
      27:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).pres_parl
         ylog=1 
      END
      28:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).pres_perp
         ylog=1 
      END
      29:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).gyrotropy
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_gyrotropy).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_gyrotropy).gyrotropy - $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(2)
            erry2 = data(gfit_gyrotropy).gyrotropy + $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(3)
         ENDIF
      END 
      30:begin
         t90= data(gfit_gyrotropy).time_ion + $
          ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(gfit_gyrotropy).rstudy_gyrotropy_stats(1)
         ylog=0 
         IF (type(2) EQ 1) THEN BEGIN
            errorbars = 1
            errx = data(gfit_gyrotropy).time_ion + $
             (tpphr_header(tpphr_nsweep_index).start_time - xut1)+dt/2.
            erry1 = data(gfit_gyrotropy).rstudy_gyrotropy_stats(1) - $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(2)
            erry2 = data(gfit_gyrotropy).rstudy_gyrotropy_stats(1) + $
             data(gfit_gyrotropy).rstudy_gyrotropy_stats(3)
         ENDIF
      END
      31:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).rchi2_patch
         ylog=1 
      END
      32:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).npatch
         ylog=1 
      END
      33:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).nrealizations
         ylog=1 
      END
      34:begin
         t90= data(good_best).time_ion + ( tpphr_header(tpphr_nsweep_index).start_time - xut1 )
         ydat= data(good_best).ktparl/data(good_best).ktperp
         ylog=1 
      END
   ENDCASE 
   
   ctit=' '                     ; Do not use this variable!
   
   cytit= font+cytit
   ctit2= font+ctit2
   
; the variable used above have the following meaning:
; 	cytit : the y-axis label
;	ctit2 : the right-of plot label defining the data
;       ctit  : the label for top panel plot
; the procedure YLIM is optional and sets the ylimit automatically for the data
; in the chosen time-range, leaving a 10% gap at the top and bottom limits.
; this routine is very simple and does not handle spikes or bad data, you may
; need to supply your own procedure for your own data.
; For 3-D color plots you can use the routine AUTORANGE. (both are in
; $PAPCO/PLOT_prog/Common)
   
; select only the time interval to be plotted. This is not needed, but does
; speed up plotting images
   
; Note the kludge to remove the zeroth element, which wasn't being
; filled.  -JBF
   
   index=where((t90 ge 0) and (t90 le dxut),count)
   if count ge 2 then begin
      t90=t90(index)
      ydat=ydat(index)
   end
   
; insert breaks where there are data dropouts. (grid the data)
   dt= t90(1:*) - t90(0:*)
   r= where(dt gt 0)
   if r(0) ne -1 then dt=min(dt(r)) else dt=1
   nn= max(t90)/dt + 1
   ydat_grid= make_array( nn, /float, value=1e31 )
   t90_grid= make_array( nn, /float, value=0 )
   r= long( t90/dt )
   ydat_grid( r ) = ydat
   t90_grid( r ) = t90
   
   t90= t90_grid
   ydat= ydat_grid
   
   
; yscl of common block yscale is an array of dimensions (*,4) and contains info
; on the yscaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin
;         yscl(*,2)  is then ymax
;         yscl(*,3)  is ylog
; when scaling is set to automatic, the automatic y limits are loaded in.
   
   hydra_ylim_new, panel(0), type, ydat, yrange=yrange, ylog=ylog
   
; set up extra plot keywords, common ones first
   
   extra_plotPar_common={yrange:yrange, ylog:ylog, xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:cytit, $
                         ythick:1.3, xthick:1.3, max_value:1e30 }
   
; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.
   
   if n_elements(extra_par) EQ 0 then $
    extra_par={color:1}
   
; add keyword structure set here with the one passed in
   
   extra_plotPar= create_struct( extra_plotPar_common, extra_par )
   
   axis_plotPar= create_struct( extra_plotPar_common, {color:1} )
   
; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot
   
   if panel(0) eq 0 then BEGIN
      xtickformat='hydra_timeticks'
   ENDIF ELSE BEGIN 
      xtickformat='noticks'
   ENDELSE

; plot the axis in color=1, and only plot if if you're not overplotting
   IF NOT keyword_set(overplot) THEN BEGIN 
      plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
       xstyle=1, ystyle=1, xtickformat=xtickformat
      if panel(0) eq 0 then $
       x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   ENDIF 
   
; now do the actual data plot
   hydra_plot, t90, ydat, _extra=extra_plotPar, xstyle=5, ystyle=5, $
    overplot= keyword_set(overplot)
   
   IF (errorbars EQ 1) THEN BEGIN
      FOR ii=0,n_elements(errx)-1 DO BEGIN
         oplot,[errx(ii),errx(ii)],[erry1(ii),erry2(ii)],line=0, $
          color=extra_plotPar.color
         oplot,[errx(ii)-dt/2.,errx(ii)+dt/2.],[erry1(ii),erry1(ii)],line=0, $
          color=extra_plotPar.color
         oplot,[errx(ii)-dt/2.,errx(ii)+dt/2.],[erry2(ii),erry2(ii)],line=0, $
          color=extra_plotPar.color
      ENDFOR
   ENDIF
   
   if not keyword_set(overplot) then begin
       plot,t90,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; store the coordinate information into common block
   plotted_x = !x
   plotted_y = !y
   
   !x=oldx
   !y=oldy
   
; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015
   
   if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
      r= where( (xtickv+xut1) mod 86400 eq 0, count )
      if r(0) ne -1 then begin
         xtickv= xtickv(r)
         xticks= count
      endif else begin
         xtickv= xtickv([0,n_elements(xtickv)-1])
         xticks= 1
      endelse
      
      axis_plotPar.xtickv= xtickv
      axis_plotPar.xticks= xticks
      
      plot, t90, ydat, /NODATA, $
       _extra=axis_plotPar, xtickformat='hydra_juldate', $
       xstyle=1, ystyle=5
      
   endif
   
; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
   abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
   lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
   right_side_label,panel,lfs+ctit2, _extra=extra_plotPar
   
   hydra_footprint, tpphr_header(tpphr_nsweep_index)
   
   return
END 





