pro r_hydra_moments, file=file
   
   common time,xut1,xut2
   common get_error, get_err_no, get_err_msg
   common hydra_moments_data, $
    moments_species, $
    moments_header, $
    moments_ele_data, $
    moments_ion_data
   
   my_name='r_hydra_moments'
   module_name='HYDRA-moments'
   
   IF n_elements( moments_header ) EQ 0 THEN BEGIN 
      mh1={ hydra_moments_dum_header, $
            start_time:long(0), $
            end_time:long(0), $
            data_version:float(0), $
            reload_flag:fix(0) $
          }
      moments_header = replicate(mh1,2)
   ENDIF
   moments_header(moments_species).start_time=xut1
   moments_header(moments_species).end_time=xut2
   moments_header(moments_species).reload_flag=1 
   
   forward_function convert_date_to_t90
   
   get_err_no=1
   get_err_msg= my_name +': unidentified error occurred.'

; read data files for time period
   num_files_read = 0
   time=xut1
   while time-1 lt xut2 do begin
       day_start= time / 86400 * 86400
       start_time= time - day_start
;          end_time is xut2 or midnight, whichever is sooner...
       end_time= min( [ ( (time+1)/86400 +1) * 86400, xut2 ] ) 
       end_time= end_time - day_start 
       convert_t90_to_date,time, year,doy,h,m,s,month,cm,day_om ; Y2K
       if year lt 100 then year=year+1900
       date_str=string(year,month,day_om,format="(i4.4,i2.2,i2.2)")       

       if start_time ne end_time then begin
           r_hydra_moments_oneday, date_str, start_time, end_time, $
             data_values, day_start, xut1, file=file, time_base=base

           if get_err_no eq 0 then BEGIN
              num_files_read = num_files_read + 1
              data_values(*).time= $
                data_values(*).time + ( base - xut1 )
              IF (num_files_read EQ 1) THEN BEGIN
                 data_buf = [data_values]
              ENDIF ELSE BEGIN
                 data_buf = [data_buf,data_values]
              ENDELSE
           ENDIF ELSE BEGIN
              GOTO, jump_out
           ENDELSE

       ENDIF

       time=day_start+86400          ; prepare to read next day
   endwhile
   
   jump_out:

   if (n_elements(data_buf) gt 0) then BEGIN
      IF (moments_species EQ 0) THEN BEGIN
         moments_ele_data = data_buf
      ENDIF ELSE BEGIN
         moments_ion_data = data_buf
      ENDELSE
       get_err_no=0             ; all's well
   endif else begin
       get_err_no=2
       print,get_err_msg
       return
   endelse

   get_err_no=0
   print,'Read '+module_name+': done'

   return
end


pro r_hydra_moments_oneday, date_str, start_time, end_time, $
                            data_values, day_start, xut1, file=file, $
                            time_base= base

   common  get_error, get_err_no, get_err_msg
   common hydra_moments_data, $
    moments_species, $
    moments_header, $
    moments_ele_data, $
    moments_ion_data
   
   myname= 'r_hydra_moments_oneday'
   
   if n_elements( file ) eq 0 then file=''
   if strtrim(file,2) eq '' then begin       
       IF (moments_species EQ 0) THEN BEGIN 
           f = hydra_findfile( date_str, /moments_all)
       ENDIF ELSE BEGIN
           f = hydra_findfile( date_str, /moments_ion)
       ENDELSE
   endif else begin
       f= file
   endelse
   
   mass= ( [ 9.1e-28,1.67e-24 ] )( moments_species ) ; g
   charge= ( [-4.8d-10, 4.8d-10] )( moments_species ) ; esu

   if (f NE '') then begin
      print, myname+': opening '+f+'...'
      cdf = cdf_open(f)
      n= strlen(f)
      moments_header(moments_species).data_version= float( strmid( f, n-8, 4 ))
   endif else BEGIN
      get_err_msg= myname+': Moments file not found for this day'
      print, get_err_msg
      get_err_no = 1
      return
   ENDELSE
   
   cdf_control, cdf, variable='time', get_var_info=r
   IF (r.maxrec LE 0) THEN BEGIN 
      get_err_msg= myname+': Empty moments file ' + f
      print, get_err_msg
      get_err_no = 1
      return
   ENDIF

   hyd_cdf_varget, cdf, 'epoch', rec_start=0, rec_count=r.maxrec, epoch
   r1= where( epoch gt -1., count )
   if r1(0) ne -1 then epoch= total(epoch(r1))/count else epoch=-1 ; robust
   base= double( long( ( epoch / 86400000 ) - 726833 ) * 86400 )

   cdf_epoch, /break, epoch, year, month, day
   date_str= string( year, form='(i4.4)' ) + $
     string( month, form='(i2.2)' ) + $
     string( day, form='(i2.2)' ) 

   hyd_cdf_varget, cdf, 'time', time, rec_count=r.maxrec+1
   rec_start= ( where( time ge start_time ) )(0)
   if rec_start(0) eq -1 then begin
      get_err_msg= myname+': No data found in this time range'
      print, get_err_msg
      get_err_no = 1
      return
   endif
   rec_count= n_elements( where( time(rec_start:*) lt end_time $
                                 and time(rec_start:*) $
                                 gt start_time, $
                                 count ))
   
   if ( count ) eq 0 then begin
      get_err_msg= myname+': No data found in this time range'
      get_err_no= 1
      print, get_err_msg           
      return
   ENDIF

; Define the data structure to return
   data_temp = { hydra_moments_data, $
                 moments_status:long(0), $
                 time:0.0, $
                 time_ssm:0.0, $
                 date:'????????', $
                 mean_energy:0.0, $
                 density:0.0, $
                 u_pay:fltarr(4),$
                 u_gsm:fltarr(4),$
                 u_gse:fltarr(4),$
                 u_perp_pay:fltarr(4),$
                 u_perp_gsm:fltarr(4),$
                 u_parl:0.0,$
                 ktparl:0.0,$
                 ktperp:0.0,$
                 pres_parl:0.0,$
                 pres_perp:0.0,$
                 gyrotropy:0.0, $
                 skew:0.0, $
                 anisotropy:0.0, $
                 b_status:long(0), $
                 b_pay:fltarr(4), $
                 b_pay_sig:fltarr(4), $
                 b_pay_angle_cone:0.0, $
                 b_pay_status:0, $
                 b_gsm:fltarr(4), $
                 b_gse:fltarr(4), $
                 b_proxy:fltarr(4), $
                 v_alf:0.0,$
                 m_tot:0.0,$
                 m_parl:0.0,$
                 m_perp:0.0,$
                 sc_potential:0.0, $
                 q_pay:fltarr(4), $
                 q_gsm:fltarr(4), $
                 q_gse:fltarr(4), $
                 q_perp_pay:fltarr(4), $
                 q_perp_gsm:fltarr(4), $
                 q_parl:0.0, $
                 qparl_over_bmag:0.0, $
                 diagonalization_status:0, $
                 angle_bavg_bproxy:0.0, $
                 angle_bavg_bproxy_sig:0.0, $
                 cosangle_q_bavg:0.0, $
                 cosangle_q_bavg_sig:0.0, $
                 cosangle_u_bavg:0.0, $
                 cosangle_u_bavg_sig:0.0, $
                 qnu_parl:0.0, $ ; Parl current density contribution in microAmps/meter^2
                 eflux_parl:0.0, $ ; ergs/cm^2/s
                 eflux_parl_over_b:0.0 $ ; ergs/cm^2/s/nT
               }
   
   data_values = replicate( data_temp, rec_count )      
   
   hyd_cdf_varget, cdf, 'time',time, $
     rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf, 'sc_potential',sc_potential, $
    rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf, 'n',density, $
    rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf, 't', t, $   ; Scalar temperature, K
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'temp', temp_tensor, $ ; Temperature tensor, K
     rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf, 'v',u, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf,'q',q,$
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'status', moments_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'b_status', b_status, $
    rec_start=rec_start, rec_count=rec_count

; Assume that all of the proper diagonalization is taken
; place when the moments are generated.  The quantities below
; are sensitive to the diagonalization process...
   
   hyd_cdf_varget, cdf,'tparl',tparl,$
    rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf,'tperp',tperp,$
    rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf,'gyrotropy',gyrotropy,$
    rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf,'skew',skew,$
    rec_start=rec_start, rec_count=rec_count
   
   hyd_cdf_varget, cdf,'angle_tparl_bave',angle_bavg_bproxy,$
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf,'mfe_b_avg',b_avg,$
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'b_avg', b_hk, $
     rec_start=rec_start, rec_count=rec_count

   if cdf_varexist(cdf,'b_used') then begin
       hyd_cdf_varget, cdf, 'b_used', b_used, $
         rec_start=rec_start, rec_count=rec_count                  
   endif else begin
       b_used= replicate( 0, rec_count )
   endelse

   r= where( b_used eq -1 )
   if r(0) ne -1 then b_avg(*,r)= 0
   r= where( b_used eq 1 )
   if r(0) ne -1 then b_avg(*,r)= b_hk(*,r)

   hyd_cdf_varget, cdf,'mfe_b_avg_sigma',b_avg_sig,$
    rec_start=rec_start, rec_count=rec_count

   message, 'Using "mfe_b_avg" from the moments file for all parl/perp determinations...', /cont

   hyd_cdf_varget, cdf,'b_proxy',b_proxy,$
    rec_start=rec_start, rec_count=rec_count
   
   cdf_close,cdf   

; Not available in the moments file---------------------
   angle_bavg_bproxy_sig = fltarr(rec_count)
   diagonalization_status = intarr(rec_count) + 1
   b_avg_angle_cone = fltarr(rec_count)
   b_avg_status = intarr(rec_count)
   
   diagonalization_status = reform(diagonalization_status)
   tparl = reform(tparl)
   tperp = reform(tperp)
   gyrotropy = reform(gyrotropy)
   skew = reform(skew)
   angle_bavg_bproxy = reform(angle_bavg_bproxy)
   angle_bavg_bproxy_sig = reform(angle_bavg_bproxy_sig)
   b_avg = reform(b_avg)
   b_avg_sig = reform(b_avg_sig)
   b_avg_angle_cone = reform(b_avg_angle_cone)
   b_avg_status = reform(b_avg_status)
   b_proxy = reform(b_proxy)
   
; ********************** DERIVED PARAMETERS *******************************
   
; Calculate the derived parameters(dp), load the data_values structure
   
   dp_hydra_moments, date_str, time, sc_potential, density, t, u, q, $
     temp_tensor, b_avg, b_avg_sig, b_avg_status, b_avg_angle_cone, $
     diagonalization_status, tparl, tperp, gyrotropy, skew, $
     angle_bavg_bproxy, angle_bavg_bproxy_sig, b_proxy, $
     data_values, charge, mass
   
   data_values.b_status = reform(b_status)
   data_values.moments_status = reform(moments_status)
   
   return
END




