pro r_hydra_spec_survey, date_str, start_time, end_time, $
                         data_header, data_values

; Read Survey Spectrogram 
; Reads in one day of survey data.
;
; HISTORY:
;    v1.0  June 30, 97    Jeremy Faden    First rev coded and released.
;    v1.3  July 1, 97     Jeremy Faden    Fixed time tag bug -- whoops.
;    v1.4  July 23, 97    JBF             Time tags changed to double,
;                                         float is too dangerous
;                                         hydra_findfile function used
;    v1.5  Sept 2, 97     jbf             more versions checked, and
;                                            min(curr,this) is returned.

  common get_error, get_err_no, get_err_msg

  my_name= 'r_hydra_spec_survey'

;  1.4 uses hydra_findfile
;  1.41 handles energy modes with less than 110 energies.

  if !version.os_family eq 'unix' then begin
      dir_sep='/'
      path_sep=':'
  endif else begin
      dir_sep='\'               ;'
      path_sep=';'
  endelse
  
  retry:
  f= hydra_findfile( date_str, /survey, version=version )
  if f eq '' then begin
      if hydra_new_papco() then begin
          f= hydra_net_findfile( date_str, /survey, version=version )        
          if f eq '' then status='not in index' else begin
              s= str_sep( f, dir_sep )
              c_file= s(n_elements(s)-1)
              n= strpos( f, c_file )
              c_url= strmid( f, 0, n )
              c_dir= papco_getenv( 'HYDRA_SURVEY_DATA_PATH' )
              c_dir= str_sep( c_dir, path_sep )
              papco_get_remote_data, c_url, c_file, c_dir(0)
              if get_err_no eq 0 then status= 'okay' else status='not_okay'
          endelse
      endif else begin
          hydra_retrieve_data, $
            'hydra_spec_survey', '19'+date_str, status
      endelse
      if (status eq 'okay') then goto, retry else return
  endif

  if version lt 1.20 then begin
      get_err_msg= 'Hydra Survey file versions less than 1.20 no longer!c'+$
        'support.  To update your data files, please visit!c '+$
        'Hydra''s data site. '
      get_err_no=1
      return
  endif

  if version lt 1.30 and data_header.spec_type_id eq 4 then begin
      get_err_msg= 'Hydra Survey file versions less than 1.30 have !c' + $
        'skew stored in a format that is incompatible with this PaPCo !c' + $
        'modules.  To update your data files, please visit!c '+$
        'Hydra''s data site. '
      get_err_no=1
      return
  endif
      
  CDF= cdf_open(f)
  data_header.data_version= version
  get_err_no=0

  cdf_varget, cdf, 'b_sec', b_sec
  data_header.time_resolution=b_sec
  
  cdf_varget, cdf, 'MODE', mode, rec_count=6272
  rvalid= where( mode gt 0 )

  if rvalid(0) eq -1 then begin
      get_err_msg=my_name+': survey file contains no data.'
      print, get_err_msg
      get_err_no=1
      return
  endif

  cdf_varget, cdf, 'TIME', time, rec_count=6272
  rec_start= ( where( time(rvalid) ge start_time ) )(0)
  if rec_start(0) eq -1 then begin
      get_err_msg= my_name+': No data found before start_time '
      get_err_no= 1
      print, get_err_msg
      return
  endif

  rec_start= rvalid( rec_start )

  r= where( time lt end_time $
            and time gt start_time, count )

  if count le 1 then begin
      print, my_name+': no data found'
      return
  endif 
  
  rec_count= r(count-1) - r(0) + 1

  data_values1= { hydra_spec_data, $
                  time:double(0), $ ; seconds since start_time
                  mode:byte(0), $ ; telemetry mode specifies energies
                  spectrum:fltarr(111), $
                  sigma:fltarr(111), $
                  species:byte(0), $ ; 1=electrons 2=ions 3=both
                  sc_pot:float(0), $
                  sc_pot_qual:byte(0), $
                  psi_status:byte(0), $
                  header_checksum:header_checksum $
                }
 
  
  data_values= replicate( data_values1, count )

  data_values(*).time= time(r)
  r=r-r(0)

  cdf_varget, cdf, 'MODE_ENERGY', rec_count=50, mode_energy 
  nen= n_elements( mode_energy(*,0) )

  r55= indgen(nen) + 55 - (nen/2)

  modes= where( total( abs( mode_energy ), 1 ) gt 0 )
  for i = 0, n_elements( modes )-1 do begin      
      imode= modes(i)
      data_header.mode_energy(r55,imode)= mode_energy(*,imode)
  endfor

  case data_header.spec_type_id of
      0: zvar_name='SPEC'
      1: zvar_name='SPEC_ALIGNED'
      2: zvar_name='SPEC_PERP'
      3: zvar_name='SPEC_OPPOSED'
      4: zvar_name='SPEC_SKEW'
      else: begin
          get_err_msg= my_name+': internal error.'
          print, get_err_msg
          get_err_no=1
          return
      end
  endcase

  cdf_varget, cdf, zvar_name, rec_start=rec_start, rec_count=rec_count, $
    ispec_data

  r0= where( ispec_data eq -32768 )
  
  spec_data= make_array( size= size( ispec_data ), /float, value=1e-20 )
  spec_data= 10 ^ ( (float( temporary(ispec_data) )) / 1000 )
  
  if r0(0) ne -1 then spec_data(r0)= 0.0

  data_values(*).spectrum(r55)= spec_data(*,r)

  cdf_varget, cdf, 'SC_POT', rec_start=rec_start, rec_count=rec_count, $
    sc_pot
  data_values(*).sc_pot= sc_pot(r)

  data_values(*).sc_pot_qual= 1 ; Goddard

  cdf_varget, cdf, 'MODE', rec_start=rec_start, rec_count=rec_count, $
    mode
  data_values(*).mode= mode(r)

  cdf_varget, cdf, 'GLINT_MASK_VERSION', glint_mask_version
  data_header.glint_mask_version= $
    min( [ glint_mask_version, data_header.glint_mask_version ] )
  
  cdf_varget, cdf, 'CALIBRATION_VERSION', calibration_version
  data_header.calibration_version= $
    min( [ calibration_version, data_header.calibration_version ] )

  cdf_varget, cdf, 'RGAINS_VERSION', rgains_version
  data_header.rgains_version= $
    min( [ rgains_version, data_header.rgains_version ] )

  cdf_varget, cdf, 'IGAINS_VERSION', igains_version
  data_header.igains_version= $
    min( [ igains_version, data_header.igains_version ] )

;  cdf_varget, cdf, 'ISWEEP_VERSION', intersweep_version, rec_start=modei-1
  cdf_varget, cdf, 'ISWEEP_VERSION', intersweep_version ; rec_start=modei-1
  data_header.intersweep_version= $
    min( [ intersweep_version, data_header.intersweep_version ] )

  cdf_varget, cdf, 'SC_POTENTIAL_VERSION', sc_potential_version
  data_header.sc_potential_version= $
    min( [ sc_potential_version, data_header.sc_potential_version ] )
  
  cdf_close, cdf

  return
end


