;******************************************************************************
;* file papcoadd_scratch_pad
;*      papcoadd_scratch_pad
;*      papcoadd_scratch_efi
;*      papcoadd_scratch_mvc
;*      papcoadd_scratch_nif
;*      papcoadd_file_ascii
;*      papcoadd_reconnection_events
;*      papcoadd_hd_scratch_pad
;*      papcoadd_hyd_access
;*      papcoadd_event_marks
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a scratch_pad. 
;*
;* THIS IS A WORKING EXAMPLE MODULE WHICH CAN BE INCLUDED INTO papco AS IS
;* (it doesn't plot very exiting data though!)
;* THIS TEMPLATE IS INTENDED AS THE STARTING POINT FOR DEVELOPING OWN
;* MODULES. SIMPLY COPY THE WHOLE scratch_pad DIRECTORY TREE AND RENAME
;* EVERY OCCURRENCE OF "scratch_pad" WITH YOUR MODULE NAME, IN ALL THE FILES
;* AND FILENAMES UNDER THE scratch_pad MODULE DIRECTORY TREE.
;*
;* papco is written such that if the files starting with "papcoadd" exist 
;* in the papco_MODULES directory, then the routines contained therein
;* are utilized to add that plot type into the papco package. 
;* This ensures easy extendibility of papco; also makes papco "configurable" 
;* by adding or leaving out specific plot types.
;*
;* ALL that is needed to include the module is to include the path to the
;* module directory tree in the IDL search path in the papco startup file, 
;* eg assuming all modules are under the directory "papco_BASE":
;*
;* !path = !path +':'+ expand_path('+'+papco_BASE+'/papco_modules/scratch_pad')
;*
;* If the plot type is in itself a sub-menu containing more scratch_pads, then
;* these must be listed at the top of this file, as it is read by papco.
;* the entries for each sub-plot-type must come after the line 
;* "file papcoadd_scratch_pad" and before the "end" line, one entry per line.
;* Each of the routines here must then be duplicated for each sub-plot-type
;* used: see $papco_MODULES/papcoadd_crres_plots for an example.
;*
;* Apart from editing this template the user has to write a panel 
;* editor widget for his scratch_pad. Use the file 
;*    $papco_XX/papco_modules/scratch_pad/papco_interface/scratch_pad_panel.pro
;* as the template.
;*
;* If slice functions are to be supported a slice program needs to be written
;* too. Use the template file
;*    $papco_XX/papco_modules/scratch_pad/papco_interface/scratch_pad_slice.pro.
;*
;* Finally, a data read and data plot routine are required as well. The data
;* read routine will be specific for each scratch_pad. An example for a standard
;* line plot is given in the file 
;*    $papco_XX/papco_modules/scratch_pad/plot_read/p_data_type_line.pro
;* and for a standard color spectrogram in 
;*    $papco_XX/papco_modules/scratch_pad/plot_read/p_data_type_spec.pro. 
;* These templates can also be used as a basis for more complex plots.
;*
;*
;* The Way these add-in routines work:
;* ===================================
;*
;* At papco compile the routine find_modules.pro is used to search the IDL
;* search path, and all the init_module.pro batch files found in the module 
;* sub-directories \papco_interface are written to the papco_add_init.pro
;* batch file and then executed: In this way the user can specify all the 
;* routines that need to be compiled for his module at startup.
;* 
;* At the start-up of papco itself, all module sub-directories 
;* \papco_interface are searched for all files beginning with "papcoadd_". 
;* From the names of these files the scratch_pad name is stripped off and used 
;* to construct all the procedure names contained in this file. The first 
;* routine, "addplottype_scratch_pad", is used to add the button for this 
;* scratch_pad to the papco main window. Since "addplottype_scratch_pad" simply 
;* returns widget ID of the button, the button itself can contain further 
;* sub-menus for plot sub-types (e.g. the button for SWE can contain a menu of
;* buttons for a range of SWE plots, ie moments, spectra, etc).
;*
;* To inform papco of all these sub-plot types (and to avoid having to
;* write a separate file for each sub-plot type) all the types have to
;* be listed in the header of the file papcoadd_scratch_pad.pro. These entries
;* have to finish off with the line ";* end" - see papcoadd_crres_plots.pro
;* for an example. All scratch_pad names are stored in the string array
;* scratch_pads and are placed in a common block:
;*    COMMON ADDINS, scratch_pads
;*
;* If there are no sub-plot types, then the scratch_pad as used in the file name
;* for papcoadd_scratch_pad.pro is used to construct the calls to the procedures
;* in that file. The convention for naming these procedures is
;* "procedure_name_scratch_pad" , so that papco can always construct the right
;* procedure call when needed. 
;*
;* When there are sub-plot types, their names get used to construct the
;* required calls. There are 6 procedures that need to be called for each
;* scratch_pad, so that a set of these 6 procedures has to be written for each
;* sub-plot type. In the same way, you need to write a panel editor for each
;* plot type (sometimes several plot sub-types may share the same editor,
;* e.g. multiple line plots or color spectra, see papcoadd_crres_plots).
;* 
;* In papco, whenever an action for a scratch_pad is taken, the information
;* stored in the structure widgetData is queried. In this structure there
;* is an array of PlotInfo structures, which contain all the relevant 
;* information for a chosen panel. The field plotInfo.panelkind is loaded
;* with the name of the scratch_pad. This is set by the user in this file in
;* the routine addpanel_scratch_pad ( "panelKind='scratch_pad'" ). THIS MUST
;* BE SET CORRECTLY AS ALL THE ROUTINE NAMES USED IN THIS ADD-IN FILE ARE
;* CONSTRUCTED USING THIS INFORMATION!
;*
;* Whenever papco has to undertake some action for a
;* scratch_pad, the field plotInfo.panelkind is compared to the entries in the
;* (COMMON ADDINS) scratch_padS array. If a match is found, the required
;* procedure name is constructed and called If not, an error occurs.
;*
;*    These procedures are contained:
;*
;*    PRO addplottype_scratch_pad,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_scratch_pad, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_scratch_pad, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_scratch_pad
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_scratch_pad, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*           papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_scratch_pad, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure performs the actual data read call and the plot 
;*           call for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_scratch_pad,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this scratch_pad to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      scratch_pads.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_scratch_pad,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addplottype_scratch_pad,plottypes_base,buttonGeometry,pdm_scratch_pad

; VALUE assigns a label for the button
; The scratch_pad portion of UVALUE is used to construct the calls to all
; other procedures in this add-in needed by papco.

; example for simple button

;  pb_scratch_pad = WIDGET_BUTTON(plottypes_base,  $
;                                 VALUE=strcenter('scratch_pad',14), $
;                                 UVALUE='pb_scratch_pad', $
;                                 _extra=buttonGeometry)

;example for simple pull-down for sub-types from the CRRES module:
  MenuDesc = [ $
               { CW_PDMENU_S, 3, strcenter('scratch_pad',10) }, $ 
               { CW_PDMENU_S, 0, ' HYDRA/PWI' }, $
               { CW_PDMENU_S, 0, ' HYDRA/EFI' }, $
               { CW_PDMENU_S, 0, ' HYDRA NIF' }, $
               { CW_PDMENU_S, 0, ' ASCII File'}, $
               { CW_PDMENU_S, 0, ' Reconnection Events'}, $
               { CW_PDMENU_S, 0, ' Event Markers'}, $
               { CW_PDMENU_S, 0, ' hd scratch pad'}, $
               { CW_PDMENU_S, 2, ' hyd_access'} ]
  pdm_scratch_pad = CW_PDMENU(plottypes_base, MenuDesc, $
                              UVALUE='pdm_scratch_pad')
  
;note the convention for naming this widget "pdm_..." for pull-down menu.
;this obviously needs to replace "pb_..." in the procedure call above!

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_scratch_pad, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable with / without orbit.
;*      PAPCO provides "set time by orbit" feature for CRRES and POLAR
;*      so far. Included for compatibility with older CRRES modules
;*      which ONLY use orbit as input to their read/plot routines
;*      (plotting over orbit boundaries is not enabled for CRRES)
;* 
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by orbit
;*                                 1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_scratch_pad, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_scratch_pad, plottable
   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

pro plottable_scratch_efi, plottable
   plottable=1
end

pro plottable_scratch_mvc, plottable
   plottable=1
end

pro plottable_scratch_nif, plottable
   plottable=1
end

pro plottable_file_ascii, plottable
   plottable=1     ; 1 -> plotable without orbit info, by time 
                   ; 0 -> plotable only by orbit number not by time 
END

pro plottable_reconnection_events, plottable
  plottable=1               
END

pro plottable_hd_scratch_pad, plottable
  plottable=1               
END

pro plottable_hyd_access, plottable
  plottable=1               
END

pro plottable_event_marks, plottable
  plottable=1          
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_scratch_pad, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that scratch_pad is capable of
;*      returning data from it's plot routine into a standard PAPCO
;*      array structure. Used for the scratch pad utility module.
;*      
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                                1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_scratch_pad, canget
;*
;* MODIFICATION HISTORY:       
;*      written November 1997, Reiner Friedel
;******************************************************************************
pro canget_scratch_pad, canget
   canget=0      ; 1 -> plot routine can return just data 
                   ; 0 -> feature disabled
END

pro canget_scratch_efi, canget
   canget=0
end

pro canget_scratch_mvc, canget
   canget=0
end

pro canget_scratch_nif, canget
   canget=0
end

pro canget_file_ascii, canget
  canget=  0               ; 1 -> plot routine can return just data to file
                           ; 0 -> feature disabled
END
pro canget_reconnection_events, canget
  canget=  0 
END
pro canget_hd_scratch_pad, canget
  canget=  0 
END
pro canget_hyd_access, canget
  canget=  0 
END
pro canget_event_marks, canget
  canget=  0 
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_scratch_pad, plotInfo, descriptStr
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this scratch_pad. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_scratch_pad, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_scratch_pad, plotInfo, descriptStr
common scratch_pad_control, types
scratch_pad_define_types
descriptStr= types( 1, PlotInfo.typeVector(0) )
END

pro descriptor_scratch_efi, plotInfo, descriptStr
common scratch_efi_control, types
scratch_pad_define_types
descriptStr= types( 1, PlotInfo.typeVector(0) )
end

pro descriptor_scratch_mvc, plotInfo, descriptStr
common scratch_mvc_control, types
scratch_pad_define_types
descriptStr= types( 1, PlotInfo.typeVector(0) )
end

pro descriptor_scratch_nif, plotInfo, descriptStr
  common scratch_nif_control, types
  scratch_pad_define_types
  descriptStr= types( 1, PlotInfo.typeVector(0) )
end

pro descriptor_file_ascii, plotInfo, descriptStr
  descriptStr=descriptStr+'File: ' + $
    strtrim(string(plotInfo.filename), 2) 
END

pro descriptor_event_marks, plotInfo, descriptStr
  descriptStr='Event Marks'
  r= where( plotInfo.options ne 0., count )
  if count eq 1 then begin
      timestr= gen_time_string( plotInfo.options(r(0)) )
      descriptStr= descriptStr+' '+timestr
  endif
end


pro descriptor_reconnection_events, plotInfo, descriptStr
  types=['Alfven Waves', 'Separatrix', 'RDs']
  descriptStr=''
  if plotInfo.typeVector(0) and 1 then $
    descriptStr= descriptStr + types(0)
  if ( plotInfo.typeVector(0) and 2 ) eq 2 then $
    descriptStr= descriptStr + ' ' + types(1)
  if ( plotInfo.typeVector(0) and 4 ) eq 4 then $
    descriptStr= descriptStr + ' ' + types(2)
END

pro descriptor_hd_scratch_pad, plotInfo, descriptStr
  if plotInfo.soptions(3) eq '' then $
    descriptionStr='no label' $
  else $
    descriptStr=plotInfo.soptions(3)
END

pro descriptor_hyd_access, plotInfo, descriptStr
  descriptStr=descriptStr + $
    strtrim(string(plotInfo.filename), 2) 
  hd_table, table, /get_table
  r= where( table(*).id eq plotInfo.filename )
  if r(0) ne -1 then begin
      s= table(r(0)).size
      if s(0) eq 2 then begin
          comp= (['X','Y','Z','Mag','Theta','Phi'])( PlotInfo.typevector(0) )
          case PlotInfo.typeVector(3) of
              0:frame=''
              1:frame='GSM'
              else:frame='??'
          endcase
          descriptStr=descriptStr +  ' ' + comp + ' ' + frame          
      endif
  endif
END
 
;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_scratch_pad
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the scratch_pad.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_scratch_pad
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_scratch_pad,event

  defaults=papco_getPlotInfoStruct() ; returns a basic plotinfo structure
  defaults.typeVector=[0,0,0,0] ; set some tags to desired defaults
  defaults.channel=0            ; channel tag is used for the color table
  defaults.ylinlog=0            ; default linear scale, since log leads to trouble.
  defaults.plotstyle=19
  case event.value of 
      1: panelKind='scratch_pad'
      2: panelKind='scratch_efi'
      3: panelKind='scratch_nif'
      4: panelKind='file_ascii'
      5: panelKind='reconnection_events'
      6: panelKind='event_marks'
      7: panelKind='hd_scratch_pad'
      8: begin
          panelKind='hyd_access'
          defaults.typeVector(0)=3
      end
      else: begin
          message, 'internal error', /cont
          return
      end
  endcase
  panelEditor=panelkind+'_panel'
  papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_scratch_pad, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*      This procedure calls the panel editor when called from 
;*      papco_EditPlot (papco_paneledit.pro). Provided here to allow
;*      several scratch_pads to share a common panel editor. Normally this
;*      procedure does not need to be edited beyond the required name
;*      change for scratch_pad.
;*
;* INPUTS:       
;*      plotNumber      anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*                      the index of the plot in widgetData.plotInfos.
;*            
;* OUTPUTS: 
;*      newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      editor_scratch_pad, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_scratch_pad, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=scratch_pad_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

PRO editor_scratch_efi, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=scratch_efi_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END



PRO editor_scratch_mvc, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=scratch_mvc_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
      GROUP=widgetData.main_base)

END

PRO editor_scratch_nif, plotNumber, newPlotData

  COMMON PLOT_COMPOSER, widgetData

  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=scratch_nif_panel(plotInfo, $
                                ACTION='Update '+plotInfo.panelKind, $
                                GROUP=widgetData.main_base)
  
END

PRO editor_file_ascii, plotNumber, newPlotData
 
  COMMON PLOT_COMPOSER, widgetData
 
  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=file_ascii_panel(plotInfo, $
                              ACTION='Update '+plotInfo.panelKind, $
                              GROUP=widgetData.main_base)
 
END

PRO editor_reconnection_events, plotNumber, newPlotData
  
  COMMON PLOT_COMPOSER, widgetData
  
  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=reconnection_events_panel(plotInfo, $
                                        ACTION='Update '+plotInfo.panelKind, $
                                        GROUP=widgetData.main_base)
  
END

PRO editor_hyd_access, plotNumber, newPlotData
  
  COMMON PLOT_COMPOSER, widgetData
  
  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=hyd_access_panel(plotInfo, $
                               ACTION='Update '+plotInfo.panelKind, $
                               GROUP=widgetData.main_base)
  
END

PRO editor_hd_scratch_pad, plotNumber, newPlotData
  
  COMMON PLOT_COMPOSER, widgetData
  
  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=hd_scratch_pad_panel(plotInfo, $
                                   ACTION='Update '+plotInfo.panelKind, $
                                   GROUP=widgetData.main_base)
  
END

PRO editor_event_marks, plotNumber, newPlotData
  
  COMMON PLOT_COMPOSER, widgetData
  
  plotInfo=widgetData.plotInfos(plotNumber)
  newPlotData=event_marks_panel(plotInfo, $
                                ACTION='Update '+plotInfo.panelKind, $
                                GROUP=widgetData.main_base)
  
END
