;***************************************************************************
;*
;* PROCEDURE:
;*      p_hyd_access, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* DESCRIPTION:
;*	This procedure plots data of type hyd_access for inclusion in PAPCO.
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $PAPCO). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale
;*
;* INPUTS:
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $PAPCO)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the PAPCO structure PAPCO_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are PAPCO_PLOTINFO.swwitch (three element vector) and
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure PAPCO_PLOTINFO
;*              (procedure PAPCO_getPlotInfoStruct in $PAPCO/papco.pro). Try to
;*              avoid this!
;*
;* OUTPUTS:
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.
;*
;* CALLING SEQUENCE:
;*       p_hyd_access, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:
;*     template written December 1995, Reiner Friedel, at Goddard
;*     customized for U of Iowa HYDRA Fall 1996, Jeremy Faden, U of Iowa
;*
;***************************************************************************

pro p_hyd_access, panel, ident, hydra_plot_kw=hp_kw, xrange=xrange, $
                  component=component, phi_center=phi_center, $
                  units=units, $
                  multiplier_string=multiplier_string, $
                  coordinate_frame=coordinate_frame, $
                  OVERPLOT=OVERPLOT, $
                  PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                  _EXTRA=extra_par, subtable=subtable

; The following common blocks are needed:

  if n_elements( xrange ) eq 0 then begin
      common time, xut1, xut2
      xrange=[ xut1, xut2 ]
  endif else begin
      xut1= xrange(0)
      xut2= xrange(1)
  endelse
  
  if ident eq '' then begin
      hyd_err_panel, 'Identifier not specified.', panel
      return
  endif

@hd_include                     ; get hd_fill

  hyd_access, timetags, get_timetags=gen_time(xrange,/t90), $
    ident, data

  if n_elements(timetags) eq 0 then begin      
      hyd_err_panel, 'No timetags returned from hyd_access.', panel
      return
  endif

;  look up identifier in hyd_access table
  hd_table, T, /get_table
  idx= (where(strlowcase(T.id) eq strlowcase(ident)))(0)

  if T[idx].interval ne hd_fill then begin
      hd_insert_fill, timetags, data, interval_sec=T[idx].interval
  endif else begin
      hd_insert_fill, timetags, data
  endelse

;  see if there is a error associated with this identifier.  Error
;  for <ident> is conventionally <ident>_var
  idx_err= ( where( strlowcase(T.id) eq strlowcase(ident)+'_var' ) )(0)
  if idx_err ne -1 then begin
      message, 'getting error for '+ident+' from '+ident+'_var', /cont
      hyd_access, timetags, ident+'_var', data_error
      have_error=1
  endif else begin
      have_error=0
  endelse
  
  
  if ident eq 'inertial_b_mag_scale_length_ratio' or $
    ident eq 'gyroradius_b_mag_scale_length_ratio' then begin
      message, 'Entering kludge in access plotter that draws lines', /cont
      message, 'down to zero.', /cont
      r= where( data eq hd_get_fill() )
      timetags(r).bin_width=0.
      hd_insert_fill, timetags, data
      r= where( data eq hd_get_fill() and timetags.bin_width ne 0 )
      timetags(r).bin_start= timetags(r).bin_start+timetags(r).bin_width
      timetags(r).bin_width=0.
      r= where( data eq hd_get_fill() )
      data(r)=1e-31
  endif
  
  if units ne '' then begin
      runits= hd_convert_units( T(idx).units, units )
      if runits eq hd_fill then begin
          message, 'Invalid units conversion.  Using original units.', /cont
          units= T(idx).units
      endif else begin          
          data_new= make_array( size=size(data), value=hd_fill )
          r= where( data ne hd_fill )                    
          if r(0) ne -1 then data(r)= data(r) * runits
          if have_error then begin
              r= where( data_error ne hd_fill )
              if r(0) ne -1 then data_error(r)= data_error(r) * runits
          endif
      endelse
  endif else begin
      units= T(idx).units
  endelse
  
  if strtrim(multiplier_string,2) ne '' then begin
      exe_str= 'multiplier= '+multiplier_string
      r= execute( exe_str )
      if r ne 1 then begin
          message, 'Syntax error in multiplier.  Using 1.0.', /cont          
          multiplier_string=''
      endif else begin
          data_new= make_array( size=size(data), value=hd_fill )
          r= where( data ne hd_fill )                    
          data(r)= data(r) * multiplier
          if have_error then begin
              r= where( data_error ne hd_fill )
              if r(0) ne -1 then data_error(r)= data_error(r) * multiplier
          endif
          units=  units + ' /(' + multiplier_string +')'
          message, 'applying multiplier to data. multiplier='+$
            strtrim( multiplier,2 ), /cont
      endelse
  endif
  
  if n_elements( timetags ) eq 0 then begin
      hyd_err_panel, 'No data found in this time range.', panel
      return
  end
  
  hyd_access, timetags, 'time_bin_start_t90', time
  time= time - xut1
  hyd_access, timetags, 'time_bin_width_seconds', dt
  message, 'Done reading data.', /cont
  
  label= T(idx).label
  if strmid( label,0,1 ) eq '>' then begin
      title= strtrim( strmid( label,1,120 ) )
  endif else if label ne '' then begin
      title= label
  endif else begin
      title= ident
  endelse

  if strtrim(title,2) eq '' then title=ident 

  time_alias= max(dt)
  
  if n_elements( data(0,*) ) ne 1 then begin
      if coordinate_frame eq 'gsm' then begin
          data= hd_pay_to_gsm( timetags,data )
          if have_error then data_error= hd_pay_to_gsm( timetags,data_error )
          framestr='GSM'
      endif else framestr=''
      case component of
          3: begin 
              data= hd_magnitude( data ) 
              if have_error then data_error= hd_magnitude( data_error )
              compstr= 'Magnitude' 
              pipe= '!9!!!x'
              if strpos(title,'!r') eq -1 then begin
                  title= pipe + title + pipe
              endif else begin                  
                  title= title+'!n!d magnitude'
              endelse
          end
          4: begin
              mag= hd_magnitude( data )
              theta= acos( data(*,2)/mag )              
              r= where( mag eq hd_fill )
              data= temporary( theta ) / !dtor
              if r[0] ne -1 then data[r]= hd_fill
              if have_error then begin
                  message, 'errors not supported for Theta', /cont
                  have_error=0
              endif
              compstr= 'Theta'
              title= title + '!c' + compstr + ' ' + framestr
              units= 'deg'
          end
          5: begin
              data= hd_atan2( data(*,1), data(*,0), $
                              center=phi_center ) / !dtor
              r= where(  data(*,1) eq hd_fill or $
                         data(*,0) eq hd_fill )
              if r[0] ne -1 then data[r]= hd_fill
              if have_error then begin
                  message, 'errors not supported for Phi', /cont
                  have_error=0
              endif
              compstr= 'Phi'
              title= title + '!c' + compstr + ' ' + framestr
              units= 'deg'
          end
          else: begin
              data= data(*,component)
              if have_error then data_error= data_error(*,component)
              compstr= (['X','Y','Z'])(component)
              title= title + '!n!d ' + compstr + ' ' + framestr + '!n'
          endelse
      endcase
  endif

  oldy=!y
  oldx=!x
  
  goodticks, xtickv, xminor
  xticks=n_elements(xtickv)-1
  dxut= xut2-xut1

; set duplex font if big_gif is device ; duplex roman
   if !d.name eq 'Z' then font='!5' else font=getenv('hydra_font')

; the plot calls normally use the arrays t90 for the time axis and yray (ymat
; for color plots) for the y or z axis.
   dxut= xut2-xut1

   r= where( time+dt ge 0 and time-dt le dxut )
   
   if r(0) ne -1 then begin
       y_kw= hydra_ytick_parm( data(r), panel=panel, fill=hd_fill )
   endif else begin
       y_kw= hydra_ytick_parm( data, panel=panel, fill=hd_fill )
   endelse       
   
; set up extra plot keywords, common ones first
   
   extra_plotPar_common={xrange:[0,dxut], $
                         noerase:1, ticklen:-0.03, xticklen:-0.03, $
                         xtickv:xtickv, xminor:xminor, xticks:xticks, $
                         yticklen:0.010, ytitle:units, $
                         ythick:1.3, xthick:1.3, $
                         min_value:-1e30, max_value:1e30 }
   
   extra_plotPar_common= $
     create_struct( extra_plotPar_common, y_kw )
   
; check if the passed in structure extra_par was set. If not, set color to
; default black; which for the color table used by PAPCO is 1.

   if n_elements(extra_par) EQ 0 then $
     extra_par={color:1}

; add keyword structure set here with the one passed in

   extra_plotPar=create_struct( extra_plotPar_common, extra_par ) 

   axis_plotPar= create_struct( extra_plotPar_common, {color:1} ) 

; extend this structure by those keywords specific to the plot being bottom
; panel or not. You might have to add your own ytickformat, or leave it out
; for default plot

   if panel(0) eq 0 then begin
       xtickformat= 'hydra_timeticks'
   endif else begin
       xtickformat= 'noticks'
   endelse
   
; plot the axis in color=1, and only plot if you're not overplotting
   if not keyword_set(overplot) then begin
       plot,[time],[data], /NODATA, $
         xstyle=5, ystyle=5, $
         xtickformat=xtickformat, $
         _extra= axis_plotPar
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!Cdate',XTICKLEN=0.03
   endif
   
; now do the actual data plot
   if n_elements( hp_kw ) then begin
       extra_plotPar1= create_struct( extra_plotPar, hp_kw )
   endif else begin
       extra_plotPar1= extra_plotPar
   endelse

   if have_error then begin
       hydra_plot, [time], [data], dx=dt, $
         _extra=extra_plotPar1, xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot ), $
         error= data_error
   endif else begin
       hydra_plot, [time], [data], dx=dt, $
         _extra=extra_plotPar1, xstyle=5, ystyle=5, $
         overplot= keyword_set( overplot )     
   endelse

   if not keyword_set(overplot) then begin
       plot,[time],[data], /NODATA, $
         xstyle=1, ystyle=1, $
         xtickformat=xtickformat, $
         _extra= axis_plotPar
   endif

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; the routine x_side_label puts x-axis label to the right of the axis.
; Use the keyword XTICKLEN if your x-axis ticklengths are not the
; default 0.015

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
  abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
  lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
  right_side_label,panel,lfs+title, _extra=extra_plotPar

  print, ident

  print



end





