;******************************************************************************
;* file p_scratch_pad_line.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_scratch_pad_line, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_PLOTINFO.swwitch (three element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_scratch_pad_line, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************
pro p_scratch_pad_line, panel, type, $
                        OVERPLOT=OVERPLOT, $
                        PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                        SUBTABLE=SUBTABLE, $
                        _EXTRA=extra_par

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_scratch_pad_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swwitch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:

   common mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
   common yscale, yscl          ;man/auto yscaling
   common coordinateSystems, plotted_x, plotted_y  
                                ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_scratch_pad. 

   common scratch_pad_data, input_header, input_data

   common scratch_pad_control, types

   panelset,panel			;sets the panel position viewport

   common time, xut1, xut2
   dxut= xut2-xut1
   
   common hydra_moments_data, $
    moments_species, $
    moments_header, $
    moments_ele_data, $
    moments_ion_data

   common hydra_kp_data, $
    kp_species, $
    kp_header, $
    kp_lr_ele_data, $
    kp_lr_ele_attr, $
    kp_lr_ion_data, $
    kp_lr_ion_attr, $
    kp_hr_ele_data, $
    kp_hr_ele_attr

   case strlowcase(type) of 
       'omega_pe(n)': begin
           moments_species=0    ; ele
           time= ( moments_header(moments_species).start_time - xut1 ) + $
             moments_ele_data.time
           n_e= moments_ele_data.density
           e= 4.8e-10
           m_e = 9.1e-28
           f= 1./2/!pi*(4*!pi*n_e*e^2/m_e)^0.5 ; Hz
           ydat= f
           title= 'f!dpe!n(n)'
           ytitle='Hz'
       end
       'f_ce(bmag)': begin
           e= 4.8e-10
           m_e = 9.1e-28
           b_mag= moments_ele_data(*).b_pay(3)
           c= 3.0e10
           ydat= e * b_mag / m_e / c / (2*!pi) * 1e-5
           title='f!dce!n(!9!!!xB!9!!!x)'
           ytitle='Hz'
           time= ( moments_header(moments_species).start_time - xut1 ) + $
             moments_ele_data.time
       end
       'p_hrkp(n,t)': begin
           time= ( kp_header(kp_species).start_time - xut1 ) + $
             kp_hr_ele_data.time
           n= kp_hr_ele_data.density
           kT= kp_hr_ele_data.mean_energy
           p_e= n * kT * 1e-12
           ydat= p_e
           title='P(n,t) HRKP'
           ytitle='cgs'
       end
       'avg_p_hrkp(n,t)': begin
           time_B= ( kp_header(kp_species).start_time - xut1 ) + $
             kp_hr_ele_data.time
           n= kp_hr_ele_data.density
           kT= kp_hr_ele_data.mean_energy
           p_e= n * kT * 1e-12
           data_B= p_e
           time_A_1= ( moments_header(moments_species).start_time - xut1 ) + $
             moments_ele_data.time
           r= where( time_A_1 gt 0 and time_A_1 lt dxut )           
           time_A_1= time_A_1(r)
           time_A_2= time_A_1 + 13.8
           message, 'syncronizing times...', /cont
           avg= sp_syncronize_1( time_A_1, time_A_2, time_B, data_B, var=var )
           time= time_A_1
           ydat= avg
           title='avg P(n,t) HRKP'
           ytitle='cgs'
       end
       'delta_p_hrkp(n,t)': begin
           time_B= ( kp_header(kp_species).start_time - xut1 ) + $
             kp_hr_ele_data.time
           n= kp_hr_ele_data.density
           kT= kp_hr_ele_data.mean_energy
           p_e= n * kT * 1e-12
           data_B= p_e
           time_A_1= ( moments_header(moments_species).start_time - xut1 ) + $
             moments_ele_data.time
           r= where( time_A_1 gt 0 and time_A_1 lt dxut )           
           time_A_1= time_A_1(r)
           time_A_2= time_A_1 + 13.8
           message, 'syncronizing times...', /cont
           avg= sp_syncronize_1( time_A_1, time_A_2, time_B, data_B, var=var )
           time= time_A_1
           ydat= var
           title='sigma P(n,t) HRKP'
           ytitle='cgs'
       end
       'dp/p hrkp(n,t)': begin
           time_B= ( kp_header(kp_species).start_time - xut1 ) + $
             kp_hr_ele_data.time
           n= kp_hr_ele_data.density
           kT= kp_hr_ele_data.mean_energy
           p_e= n * kT * 1e-12
           data_B= p_e
           time_A_1= ( moments_header(moments_species).start_time - xut1 ) + $
             moments_ele_data.time
           r= where( time_A_1 gt 0 and time_A_1 lt dxut )           
           time_A_1= time_A_1(r)
           time_A_2= time_A_1 + 13.8
           message, 'syncronizing times...', /cont
           avg= sp_syncronize_1( time_A_1, time_A_2, time_B, data_B, var=var )
           time= time_A_1
           ydat= var / avg
           title='!7d!xP/P(n,t) HRKP'
           ytitle=''
       end
       else: begin
           message, 'Unrecognized type: '+type, /cont
           plotted_x= !x
           plotted_y= !y
           return
       end
   endcase
   
; restrict data to actual time range requested - this makes for faster zooming

   index=where((time ge 0) and (time le (xut2-xut1)),count)
   if count ne 0 then begin
       ydat=ydat(index)
       time=time(index)
   endif

; the variables used above have the following meaning:
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;

   ylog= yscl(panel(0),3)
   yrange_kw= { ylog:yscl(panel(0),3), min_value:-1e29, max_value:1e29 }
   if (yscl(panel(0),0) eq 1) then begin ; manual specs.
       yst= yscl(panel(0),1)
       yen= yscl(panel(0),2)
       yrange_kw= { yrange:[yst,yen], ylog:ylog } 
   endif
   
   extra_plotPar_common= create_struct( yrange_kw, $
                                        { xrange:[0,xut2-xut1], $
                                          ytitle:ytitle } )
   
; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 

  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse     
   
; add keyword structure set here with the one passed in

  extra_plotPar=create_struct( extra_plotPar_common, extra_par)
  axis_plotPar= create_struct( extra_plotPar_common, {color:1} )

; extend this structure by those keywords specific to the plot being bottom
; panel or not. 

   if panel(0) eq 0 then begin
       xtickformat= 'hydra_timeticks'
   endif else begin
       xtickformat= 'noticks'
   ENDELSE

   ; plot the axis in color=1, and only plot if if you're not overplotting
   font= '!3'
   if not keyword_set(overplot) then begin
       plot,time,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif
   
   if n_elements( hydra_plot_kw ) gt 0 then begin
       hp_kw= create_struct( hydra_plot_kw, extra_plotPar )
   endif else begin
       hp_kw= extra_plotPar
   endelse

   hydra_plot, time, ydat, _extra=hp_kw, xstyle=5, ystyle=5, $
     overplot= keyword_set( overplot )

;  replot the axis in color=1, and only plot if if you're not overplotting
   if not keyword_set(overplot) then begin
       plot,time,ydat, _extra=axis_plotPar, /NODATA, $
         xstyle=1, ystyle=1, xtickformat=xtickformat
       if panel(0) eq 0 then $
         x_side_label,panel,font+'time (UT)!CDate',XTICKLEN=0.03
   endif

; set range for overplots
   if yscl(panel(0),0) eq 0 then begin
       if !y.type eq 0 then begin
           yscl(panel(0),1)= !y.crange(0)
           yscl(panel(0),2)= !y.crange(1)
       endif else begin
           yscl(panel(0),1)= 10^!y.crange(0)
           yscl(panel(0),2)= 10^!y.crange(1)
       endelse
   endif

   
; store the coordinate information into common block for slices
   plotted_x = !x
   plotted_y = !y

; plot info right of the panel. If you want overplots to have there own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION
  abunchoflfs='!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c!c'
  lfs= strmid( abunchoflfs, 0, 6*(plots_atthisposition<10) )
  right_side_label,panel,lfs+title, _extra=extra_plotPar

end 
