;******************************************************************************
;* file p_scratch_pad_spec.pro
;*
;* this file is a template for a plot routine which can be used by papco.
;* it includes all the periphery needed, the user simply has to add in
;* the actual plot statement for his/her kind of data.
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_scratch_pad_spec, panel, type, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_PLOTINFO.swwitch (three element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;*      PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_scratch_pad, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;******************************************************************************
pro p_scratch_pad_spec, panel, type, $
                 OVERPLOT=OVERPLOT, $
                 PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                 SUBTABLE=SUBTABLE, $
                 _EXTRA=extra_par

; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_scratch_pad_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags whichare meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type, swwitch. Replace type in the call with whatever variables
; you need, or add more

; Comments here should include an explanation of the plot options used
; in type, swwitch. Replace type in the call with whatever variables
; you need, or add more

; The following common blocks are needed:

   common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
   common yscale, yscl			;man/auto yscaling
   common zscale, zscl			;man/auto zscaling
   common shift_label, down             ;common for x-axis label shifting
   common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_scratch_pad. 

   common scratch_pad_data, input_header, input_data

   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.

   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10

; since there is a common time axis plots look better if they all use the same
; number of xticks. In papco, this number is 6 by default. This is set by papco
; using the IDL graphics system variable !X.TICKS.
; You can overide this for your own plot by setting the varbale
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by papco to default values that gurantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; for the color plot routine, you need to set up arrays containing the upper  
; and lower limits of each channel. Here is an example for CRRES MEB energy 
; channels. There must be as many entries as there are channels in the data
; array! 

 yminarr=[37.,54.,69., 85.,113.,147.,193.,254.,335.,447.] 
 ymaxarr=[54.,69.,85.,113.,147.,193.,254.,335.,447.,602.] 

; for color plots, the y-axis normally refers to a channel. You have to 
; re-scale the data array to the range required before passing it to the
; color plot routine, also the ymin and ymax arrays.

; the plot-calls normally use the arrays time for the time axis and yarr (ymat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. papco convention is to use running secconds
; starting with the no of seconds since the begiing of the start day.
   time=input_data.time
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t

   ymat=input_data.ymat
   ymat=transpose(ymat)
   uztit='flux'
   uytitle='Energy (keV)'
   utitle='Test Spectra!Cdata'

   yst=yminarr(0)
   yen=ymaxarr(n_elements(ymaxarr)-1)

; restrict data to actual time range requested - this makes for faster zooming

;   index=where((time ge xut1) and (time le xut2),count)
;   if count ne 0 then begin
;      ymat=ymat(index,*)
;      time=time(index)
;   endif

; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

  autorange,panel(0),time,ymat,zmax,zmin,1

; the procedure autorange is optional and sets the zlimit automatically for
; the data, attempting to use the full dynamic range of the color table.
; you may need to supply your own procedure for your own data. 
; (in $papco/PLOT_prog/Common)

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling
   
   if (zscl(panel(0),0) eq 1) then begin
      zmin=zscl(panel(0),1)
      zmax=zscl(panel(0),2)
   endif else begin
      zscl(panel(0),1)=zmin
      zscl(panel(0),2)=zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
      yst=yscl(panel(0),1)
      yenx=yscl(panel(0),2)
   endif else begin
      yscl(panel(0),1)=yst
      yscl(panel(0),2)=yen
   endelse

; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override papco defaults.
; See $papco_BASE/papco/papco_draw for the definition of the papco default
; graphic sytem variable settings.

  extra_plotPar_common={yrange:[yst, yen], xrange:[xut1, xut2], $
                        ylog:yscl(panel(0),3) , ystyle:1, ytitle:uytitle}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black. 

  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  

; add keyword structure set here with the one passed in

   extra_plotPar_common=create_struct(extra_plotPar_common, extra_par)

; add keyword structure specific to the subtype of plot chosen here
  
;   extra_plotPar_common=create_struct(extra_plotPar_common, this_par)

; extend this structure by those keywords specific to the plot being bottom
; panel or not. 
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.

   if panel(0) eq 0 then $
     extra_plotPar=create_struct({xtickformat:'MJDT_DATE_4'}, $
                                 extra_plotPar_common) $
   else $
     extra_plotPar=create_struct({xtickformat:'noticks'}, $
                                 extra_plotPar_common)

; plot the axis in color=1, and only plot if if you're not overplotting
; down sets the axis labeling level
   
   down=0
   color1_plotpar=extra_plotPar
   color1_plotpar.color=1
   if not keyword_set(overplot) then $
      plot,time,time, _extra=color1_plotPar, /NODATA

; now do the actual data plot. For the color plots, you need to plot the axis
; and then store the coordinate sytem in the common block. The draw the
; color plot and then redraw the axis once more.

   plot,time,time, _extra=color1_plotPar, /NODATA

; store the coordinate information into common block

   plotted_x = !x
   plotted_y = !y

;do color plot and color bar plot

  linlogbit=zscl(panel(0),3)  ; set linear/log z-scaling
  smbit=0                     ; set if you want to smooth in y-direction

  plot_clr,time,yminarr,ymaxarr,ymat,zmin,zmax,linlogbit,0,smbit
  papco_colorbar,zmin,zmax,0,0,uztit
                    
;replot the axis

   plot,time,time, _extra=color1_plotPar, /NODATA

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; The routine x_side_label puts x-axis label to the right of the axis.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; Or use any other function you define yourself.
; down sets the axis labeling level

  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
     extra_plotPar=create_struct({xtickformat:'MJDT_DATE_3'}, $
                                 extra_plotPar_common) 
     extra_plotPar.color=1
     down=1
     plot,time,time, _extra=extra_plotPar, /NODATA
     x_side_label,panel,'time (UT)!CDate'
  endif

; plot info right of the panel. If you want overplots to have their own label
; you need to add position control for this label using the keyword
; PLOTS_ATTHISPOSITION. Since here papco_colorbars are plotted, labels are rotated
; by 90 degrees and then plotted. This leave lttile room for overplot labels:
; consider putting those inside the panel.

   if not keyword_set(overplot) then $
     right_side_label,panel,utitle,/rot90

end 







