;******************************************************************************
;*  NAME:        r_scratch_pad
;*
;*  DESCRIPTION: Routine to read scratch_pad files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_scratch_pad, needs, panelVector, xut1_in, xut2_in

   common mjdt, mjdt_start, mjdt_end
   common get_error, get_err_no, get_err_msg
   
   common time, xut1, xut2
   if n_elements( xut1_in ) eq 1 then begin
       xut1= xut1_in
       xut2= xut2_in
   endif

   get_err_no=1
   get_err_msg='unidentified error encounterred.'
   
   s= str_sep( needs, ' ' )
   for i=0,n_elements(s)-1 do begin
       case s(i) of
           'HME': hydra_ele_moments=1
           'HHRKP': hydra_hrkp=1
           'HCUR': hydra_currents=1
           'HMVC': hydra_mvc=1
           'HNIF': hydra_nif=1
           'EVGSM': efi_v_gsm=1
           'EV34GSM': efi_v34_scpot=1
           'EXYZ56': efi_xyz56=1
           'EUZPAY': efi_uzpay=1
           else: message, 'unrecognized code: '+s(i), /cont
       endcase
   endfor


;  HYDRA MOMENTS COMMON BLOCK ------------------
   common hydra_moments_data, $
     moments_species, $
     moments_header, $
     moments_ele_data, $
     moments_ion_data
   COMMON moments_loaded, eleloaded, ionloaded
;  ---------------------------------------------

;  provide list of days to load, and xut1 in YYYYMMDD
   convert_t90_to_date,xut1, year,doy,h,m,s,month,cm,day_om 
   if year lt 100 then year= year+1900
   yyyymmdd= strtrim( year*10000 + month*100 + day_om, 2 )

   ndays= (xut2-xut1) / 86400
   for i=1,ndays-1 do begin       
       convert_t90_to_date,xut1+i*86400, year,doy,h,m,s,month,cm,day_om 
       if year lt 100 then year= year+1900
       yyyymmdd= [ yyyymmdd, strtrim( year*10000 + month*100 + day_om, 2 ) ]
   endfor

;  read electron moments data
   if keyword_set( hydra_ele_moments ) then begin
       moments_species=0        ; electrons       
       if n_elements( moments_header ) eq 0 then begin
           yesLoad=1 
       endif else begin
           yesload= $
             (NOT keyword_set(eleloaded) AND moments_species EQ 0)  OR  $
             (NOT keyword_set(ionloaded) AND moments_species EQ 1)  OR  $
             (moments_header(moments_species).start_time gt xut1)   OR  $
             (moments_header(moments_species).end_time lt xut2 )    OR  $
             (moments_header(moments_species).reload_flag )
       endelse
       if yesload then begin
           r_hydra_moments
           if PAPCO_Draw_CheckResult('hydra_moments', $
                                     panelVector, 0, $
                                     [0,0,0,0], $
                                     0 ) then begin           
               moments_header(moments_species).reload_flag=0
               eleloaded=1
           endif
       endif
   endif

   common hydra_kp_data, $
    kp_species, $
    kp_header, $
    kp_lr_ele_data, $
    kp_lr_ele_attr, $
    kp_lr_ion_data, $
    kp_lr_ion_attr, $
    kp_hr_ele_data, $
    kp_hr_ele_attr
   COMMON kp_loaded, lreleloaded, lrionloaded, hreleloaded

   if keyword_set( hydra_hrkp ) then begin
       kp_species= 2
       if n_elements( kp_header ) eq 0 then begin
           mh1={ hydra_kp_dum_header, $
                 start_time:long(0), $
                 end_time:long(0), $
                 data_version:float(0), $
                 reload_flag:fix(0) $
               }
           kp_header = replicate(mh1,3)
       endif
       yesload= ( not keyword_set(hreleloaded) AND kp_species EQ 2) or $
         kp_header(kp_species).start_time gt xut1    or  $
         kp_header(kp_species).end_time lt xut2      or  $
         kp_header(kp_species).reload_flag    
       
       if yesload then begin
           kp_header(kp_species).start_time=xut1
           kp_header(kp_species).end_time=xut2
           kp_header(kp_species).reload_flag=1
           r_hydra_kp
           IF PAPCO_Draw_CheckResult('hydra_kp', $
                                     panelVector, 0, $
                                     [0,0,0,0], $
                                     0) $
             THEN BEGIN
               kp_header(kp_species).reload_flag=0
               IF kp_species EQ 2 THEN hreleloaded =  1
           endif
       endif
       
   endif                        ; hydra_hrkp
   
   if keyword_set( hydra_currents ) then begin
       common hydra_current_data, $
         current_header, $
         current_data
       IF n_elements(current_header) EQ 0 THEN YesLoad1 = 1 ELSE BEGIN 
           YesLoad1= $
             (current_header.start_time gt xut1)     OR  $
             (current_header.end_time lt xut2 )      OR  $
             (current_header.reload_flag ) ; set with panel
       ENDELSE     
       if yesload1 then begin
           r_hydra_current
           IF PAPCO_Draw_CheckResult('hydra_currents', $
                                     panelVector, 0, $
                                     [0,0,0,0], $
                                     0 ) $
             THEN current_header.reload_flag=0
       endif
   endif                        ; hydra_currents
   
   common sp_efi_data, $
     efi_head, efi_data, $
     efi34_head, efi34_data, $
     efi_xyz56_head, efi_xyz56_data, $
     efi_uzpay_head, efi_uzpay_data      
   
   if keyword_set( efi_v_gsm ) then begin
       if n_elements( efi_head ) eq 0 then yesload=1 else begin
           yesload= efi_head.start_time gt xut1   or $
             efi_head.end_time lt xut2
       endelse
       if yesload then begin
           efi_head= { start_time:0L, $
                       end_time:0L, $
                       production_date:'', $
                       file:'' }           
;           read_v_gsm_may29_asc, efi_data
@hd_include
           hyd_access, ttag, 'efi_e_gsm', get_timetags=[150,151], $
             e_gsm, footprint=f
           hyd_access, ttag, 'time_bin_start_ssm', time_ssm
           hyd_access, ttag, 'efi_e_fac', e_fac
           r= where( e_fac eq hd_fill )
           if r(0) ne -1 then e_fac(r)=1e31
           hyd_access, ttag, 'efi_vel_gsm', v_gsm           
           r= where( v_gsm eq hd_fill )
           if r(0) ne -1 then v_gsm(r)=1e31
           efi_data= { time:time_ssm, $
                       production_date:'', $
                       file:strmid(f(1),10,80), $
                       ex_gsm: e_gsm(*,0), $
                       ey_gsm: e_gsm(*,1), $
                       ez_gsm: e_gsm(*,2), $
                       ex_fac: e_fac(*,0), $
                       ey_fac: e_fac(*,1), $
                       ez_fac: e_fac(*,2), $
                       vx: v_gsm(*,0), $
                       vy: v_gsm(*,1), $
                       vz: v_gsm(*,2) }
           efi_head.production_date= efi_data.production_date
           efi_head.file= efi_data.file
           r= convert_timetoseconds( '5/29/96 0:00', /t90 )
           efi_head.start_time= r.value
           efi_head.end_time= r.value+86400
       endif
       get_err_no=0
   endif                        ; efi_v_gsm
       
   if keyword_set( efi_v34_scpot ) then begin
       if n_elements( efi34_head ) eq 0 then yesload=1 else begin
           yesload= efi34_head.start_time gt xut1   or $
             efi34_head.end_time lt xut2
       endelse
       if yesload then begin
           efi34_head= { start_time:0L, $
                         end_time:0L, $
                         production_date:'', $
                         file:'' }
           read_v34_scpot_may29_asc, efi34_data
           efi34_head.production_date= efi34_data.production_date
           efi34_head.file= efi34_data.file
           r= convert_timetoseconds( '5/29/96 0:00', /t90 )
           efi34_head.start_time= r.value
           efi34_head.end_time= r.value+86400
       endif
       get_err_no=0
   endif
   
   if keyword_set( efi_xyz56 ) then begin
       if n_elements( efi_xyz56_head ) eq 0 then yesload=1 else begin
           yesload= efi_xyz56_head.start_time gt xut1   or $
             efi_xyz56_head.end_time lt xut2
       endelse
       if yesload then begin
           efi_xyz56_head= { start_time:0L, $
                             end_time:0L, $ 
                             production_date:'', $
                             file:'' }
           read_xy_z_56_may29_asc, efi_xyz56_data
           efi_xyz56_head.production_date= efi_xyz56_data.production_date
           efi_xyz56_head.file= efi_xyz56_data.file
           r= convert_timetoseconds( '5/29/96 0:00', /t90 )
           efi_xyz56_head.start_time= r.value
           efi_xyz56_head.end_time= r.value+86400
       endif
       get_err_no=0
   endif
   
   if keyword_set( efi_uzpay ) then begin
       if n_elements( efi_uzpay_head ) eq 0 then yesload=1 else begin
           yesload= efi_uzpay_head.start_time gt xut1   or $
             efi_uzpay_head.end_time lt xut2
       endelse
       if yesload then begin
           f= getenv('HYD_EFI_SPECIAL_DATA')
           if f eq '' then f='/opt/hydra/papco/scratch_pad/data/'
           
           for i=0,n_elements(yyyymmdd)-1 do begin
               file= f+yyyymmdd(i)+'_efi_vel56.sdt'
               f1= findfile( file )
               if f1(0) ne '' then begin
                   read_sdt, file, efi_uzpay_head1, efi_uzpay_data1
                   if i eq 0 then begin
                       efi_uzpay_head=efi_uzpay_head1
                       efi_uzpay_data=efi_uzpay_data1
                   endif else begin
                       data_new= [ efi_uzpay_data(*).polar_v_56, $
                                   efi_uzpay_data1(*).polar_v_56 ]
                       time_new= [ efi_uzpay_data(*).time, $
                                   efi_uzpay_data1.time + $
                                   (efi_uzpay_head1.start_time-$
                                    efi_uzpay_head.start_time)*86400 ]
                       efi_uzpay_data= { time:time_new, $
                                         polar_v_56:data_new }
                   endelse
               endif
               get_err_no=0
           endfor
       endif
   endif
   
   if keyword_set( hydra_mvc ) then begin
       common hydra_mvc_common, hydra_mvc_head, $
         hydra_mvc_B_data, hydra_mvc_J_perp_data, hydra_mvc_Press_data,$
         hydra_mvc_Qdata, hydra_mvc_U_perp_ele_data,$
         hydra_mvc_U_perp_ion_data, hydra_mvc_edata, hydra_mvc_u_ele_data,$
         hydra_mvc_u_ion_data, hydra_mvc_J_data, hydra_mvc_vel_km_s, hydra_mvc_traj_re
       if n_elements( hydra_mvc_head ) eq 0 then yesload=1 else begin
           yesload= hydra_mvc_head.start_time gt xut1   or $ 
             hydra_mvc_head.end_time lt xut2
       endelse
       if yesload then begin
           hydra_mvc_head= { start_time:0L, $
                             end_time:0L }
           read_papco_mvc, hydra_mvc_B_data, hydra_mvc_J_perp_data, $
             hydra_mvc_Press_data, hydra_mvc_Qdata, hydra_mvc_U_perp_ele_data,$
             hydra_mvc_U_perp_ion_data, hydra_mvc_edata, hydra_mvc_u_ele_data,$
             hydra_mvc_u_ion_data, hydra_mvc_J_data
           read_papco_mvc2, vel_km_s=hydra_mvc_vel_km_s, traj_re=hydra_mvc_traj_re
           r= convert_timetoseconds( '5/29/96 0:00', /t90 )
           hydra_mvc_head.start_time= r.value
           hydra_mvc_head.end_time= r.value+86400
       endif
       get_err_no=0
   endif

   if keyword_set( hydra_nif ) then begin
       common hydra_nif_common, hydra_nif_head, $
         hydra_nif_B_data, hydra_nif_J_perp_data, hydra_nif_Press_data,$
         hydra_nif_Qdata, hydra_nif_U_perp_ele_data,$
         hydra_nif_U_perp_ion_data, hydra_nif_edata, hydra_nif_u_ele_data,$
         hydra_nif_u_ion_data, hydra_nif_J_data, hydra_nif_vel_km_s, hydra_nif_traj_re
       if n_elements( hydra_nif_head ) eq 0 then yesload=1 else begin
           yesload= hydra_nif_head.start_time gt xut1   or $ 
             hydra_nif_head.end_time lt xut2
       endelse
       if yesload then begin
           hydra_nif_head= { start_time:0L, $
                             end_time:0L }
           read_papco_nif, hydra_nif_B_data, hydra_nif_J_perp_data, $
             hydra_nif_Press_data, hydra_nif_Qdata, hydra_nif_U_perp_ele_data,$
             hydra_nif_U_perp_ion_data, hydra_nif_edata, hydra_nif_u_ele_data,$
             hydra_nif_u_ion_data, hydra_nif_J_data
           read_papco_nif2, vel_km_s=hydra_nif_vel_km_s, traj_re=hydra_nif_traj_re
           r= convert_timetoseconds( '5/29/96 0:00', /t90 )
           hydra_nif_head.start_time= r.value
           hydra_nif_head.end_time= r.value+86400
       endif
       get_err_no=0
   endif

   return
end


