; This program converts differential CPA channels from counts to flux.
; The units of flux are particles/cm^2/sec/sr/keV.
; The programs cpa_read.pro and cpa_int_to_diff.pro should be run before
; runing this program. 
;
; Input
;	satid     = international satellite designator number
;	electrons = differential electron counts/second
;	protons   = differential proton counts/second
;
; Output
;	electrons = differential electron flux (particles/cm^2/sec/sr/keV)
;	protons   = differential proton flux (particles/cm^2/sec/sr/keV)
;
;=======================================================================
; Geoff Reeves
; 95/08/17
;=============

PRO cpa_diff_counts_to_flux, satid, electrons, protons

cpa_electron_fcf = fltarr(12)
cpa_proton_fcf   = fltarr(26)

; -------------------------------------------------------------
; Get the flux conversion factors for the appropriate satellite
; Conversion factors = G*dE where
;	G  = geometric factor
;	dE = High energy - Low energy for the channel (in keV)
; -------------------------------------------------------------

; Nominal electron energies are the same for all satellites
; "LoE" 
cpa_electron_fcf(0) =  3.6e-3 * (45.-30.)
cpa_electron_fcf(1) =  3.6e-3 * (65.-45.)
cpa_electron_fcf(2) =  3.6e-3 * (95.-65.)
cpa_electron_fcf(3) =  3.6e-3 * (140.-95.)
cpa_electron_fcf(4) =  3.6e-3 * (200.-140.)
cpa_electron_fcf(5) =  3.6e-3 * (300.-200.)

; "HiE" 
cpa_electron_fcf(6) =  1.8e-2 * (300.-200.)
cpa_electron_fcf(7) =  1.8e-2 * (400.-300.)
cpa_electron_fcf(8) =  1.8e-2 * (600.-400.)
cpa_electron_fcf(9) =  1.8e-2 * (900.-600.)
cpa_electron_fcf(10) = 1.8e-2 * (1400.-900.)
cpa_electron_fcf(11) = 1.8e-2 * (2000.-1400.)

; Nominal proton energies vary with satellite
case satid of
  '1976-059': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (160.-145.)
	cpa_proton_fcf(1)  = 3.9e-3 * (175.-160.)
	cpa_proton_fcf(2)  = 3.9e-3 * (190.-175.)
	cpa_proton_fcf(3)  = 3.9e-3 * (215.-190.)
	cpa_proton_fcf(4)  = 3.9e-3 * (245.-215.)
	cpa_proton_fcf(5)  = 3.9e-3 * (290.-245.)
	cpa_proton_fcf(6)  = 3.9e-3 * (340.-290.)
	cpa_proton_fcf(7)  = 3.9e-3 * (400.-340.)
	cpa_proton_fcf(8)  = 3.9e-3 * (480.-400.)
	cpa_proton_fcf(9)  = 3.9e-3 * (560.-480.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (500.-400.)
	cpa_proton_fcf(11) = 4.4e-2 * (630.-500.)
	cpa_proton_fcf(12) = 4.4e-2 * (800.-630.)
	cpa_proton_fcf(13) = 4.4e-2 * (1000.-800.)
	cpa_proton_fcf(14) = 4.4e-2 * (1300.-1000.)
	cpa_proton_fcf(15) = 4.4e-2 * (1700.-1300.)
	cpa_proton_fcf(16) = 4.4e-2 * (2700.-1700.)
	cpa_proton_fcf(17) = 4.4e-2 * (4700.-2700.)
	cpa_proton_fcf(18) = 4.4e-2 * (9000.-4700.)
	cpa_proton_fcf(19) = 4.4e-2 * (13000.-9000.)
	cpa_proton_fcf(20) = 4.4e-2 * (26000.-13000.)
	cpa_proton_fcf(21) = 8.0e-2 * (39000.-26000.)
	cpa_proton_fcf(22) = 8.0e-2 * (61000.-39000.)
	cpa_proton_fcf(23) = 8.0e-2 * (92000.-61000.)
	cpa_proton_fcf(24) = 8.0e-2 * (146000.-92000.)
	cpa_proton_fcf(25) = 8.0e-2 * (220000.-146000.)
	end
  '1977-007': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (162.-147.)
	cpa_proton_fcf(1)  = 3.9e-3 * (170.-162.)
	cpa_proton_fcf(2)  = 3.9e-3 * (195.-170.)
	cpa_proton_fcf(3)  = 3.9e-3 * (220.-195.)
	cpa_proton_fcf(4)  = 3.9e-3 * (250.-220.)
	cpa_proton_fcf(5)  = 3.9e-3 * (295.-250.)
	cpa_proton_fcf(6)  = 3.9e-3 * (340.-295.)
	cpa_proton_fcf(7)  = 3.9e-3 * (405.-340.)
	cpa_proton_fcf(8)  = 3.9e-3 * (490.-405.)
	cpa_proton_fcf(9)  = 3.9e-3 * (580.-490.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (540.-450.)
	cpa_proton_fcf(11) = 4.4e-2 * (680.-540.)
	cpa_proton_fcf(12) = 4.4e-2 * (830.-680.)
	cpa_proton_fcf(13) = 4.4e-2 * (1100.-830.)
	cpa_proton_fcf(14) = 4.4e-2 * (1400.-1100.)
	cpa_proton_fcf(15) = 4.4e-2 * (1750.-1400.)
	cpa_proton_fcf(16) = 4.4e-2 * (2800.-1750.)
	cpa_proton_fcf(17) = 4.4e-2 * (4800.-2800.)
	cpa_proton_fcf(18) = 4.4e-2 * (9000.-4800.)
	cpa_proton_fcf(19) = 4.4e-2 * (14000.-9000.)
	cpa_proton_fcf(20) = 4.4e-2 * (25000.-14000.)
	cpa_proton_fcf(21) = 8.0e-2 * (38000.-25000.)
	cpa_proton_fcf(22) = 8.0e-2 * (61000.-38000.)
	cpa_proton_fcf(23) = 8.0e-2 * (92000.-61000.)
	cpa_proton_fcf(24) = 8.0e-2 * (144000.-92000.)
	cpa_proton_fcf(25) = 8.0e-2 * (210000.-144000.)
	end
  '1979-053': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (133.-105.)
	cpa_proton_fcf(1)  = 3.9e-3 * (156.-133.)
	cpa_proton_fcf(2)  = 3.9e-3 * (187.-156.)
	cpa_proton_fcf(3)  = 3.9e-3 * (225.-187.)
	cpa_proton_fcf(4)  = 3.9e-3 * (265.-225.)
	cpa_proton_fcf(5)  = 3.9e-3 * (320.-265.)
	cpa_proton_fcf(6)  = 3.9e-3 * (390.-320.)
	cpa_proton_fcf(7)  = 3.9e-3 * (460.-390.)
	cpa_proton_fcf(8)  = 3.9e-3 * (555.-460.)
	cpa_proton_fcf(9)  = 3.9e-3 * (670.-555.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (480.-390.)
	cpa_proton_fcf(11) = 4.4e-2 * (630.-480.)
	cpa_proton_fcf(12) = 4.4e-2 * (790.-630.)
	cpa_proton_fcf(13) = 4.4e-2 * (1000.-790.)
	cpa_proton_fcf(14) = 4.4e-2 * (1300.-1000.)
	cpa_proton_fcf(15) = 4.4e-2 * (1700.-1300.)
	cpa_proton_fcf(16) = 4.4e-2 * (2700.-1700.)
	cpa_proton_fcf(17) = 4.4e-2 * (4500.-2700.)
	cpa_proton_fcf(18) = 4.4e-2 * (8600.-4500.)
	cpa_proton_fcf(19) = 4.4e-2 * (13000.-8600.)
	cpa_proton_fcf(20) = 4.4e-2 * (26000.-13000.)
	cpa_proton_fcf(21) = 8.0e-2 * (38000.-26000.)
	cpa_proton_fcf(22) = 8.0e-2 * (60000.-38000.)
	cpa_proton_fcf(23) = 8.0e-2 * (91000.-60000.)
	cpa_proton_fcf(24) = 8.0e-2 * (145000.-91000.)
	cpa_proton_fcf(25) = 8.0e-2 * (200000.-145000.)
	end
  '1981-025': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (133.-107.)
	cpa_proton_fcf(1)  = 3.9e-3 * (157.-133.)
	cpa_proton_fcf(2)  = 3.9e-3 * (197.-157.)
	cpa_proton_fcf(3)  = 3.9e-3 * (236.-197.)
	cpa_proton_fcf(4)  = 3.9e-3 * (295.-236.)
	cpa_proton_fcf(5)  = 3.9e-3 * (340.-295.)
	cpa_proton_fcf(6)  = 3.9e-3 * (420.-340.)
	cpa_proton_fcf(7)  = 3.9e-3 * (500.-420.)
	cpa_proton_fcf(8)  = 3.9e-3 * (610.-500.)
	cpa_proton_fcf(9)  = 3.9e-3 * (750.-610.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (470.-400.)
	cpa_proton_fcf(11) = 4.4e-2 * (620.-470.)
	cpa_proton_fcf(12) = 4.4e-2 * (780.-620.)
	cpa_proton_fcf(13) = 4.4e-2 * (1000.-780.)
	cpa_proton_fcf(14) = 4.4e-2 * (1300.-1000.)
	cpa_proton_fcf(15) = 4.4e-2 * (1700.-1300.)
	cpa_proton_fcf(16) = 4.4e-2 * (2600.-1700.)
	cpa_proton_fcf(17) = 4.4e-2 * (4400.-2600.)
	cpa_proton_fcf(18) = 4.4e-2 * (8000.-4400.)
	cpa_proton_fcf(19) = 4.4e-2 * (12000.-8000.)
	cpa_proton_fcf(20) = 4.4e-2 * (24000.-12000.)
	cpa_proton_fcf(21) = 8.0e-2 * (43000.-24000.)
	cpa_proton_fcf(22) = 8.0e-2 * (68000.-43000.)
	cpa_proton_fcf(23) = 8.0e-2 * (110000.-68000.)
	cpa_proton_fcf(24) = 8.0e-2 * (160000.-110000.)
	cpa_proton_fcf(25) = 8.0e-2 * (210000.-160000.)
	end
  '1982-019': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (110.-95.)
	cpa_proton_fcf(1)  = 3.9e-3 * (130.-110.)
	cpa_proton_fcf(2)  = 3.9e-3 * (150.-130.)
	cpa_proton_fcf(3)  = 3.9e-3 * (195.-150.)
	cpa_proton_fcf(4)  = 3.9e-3 * (230.-195.)
	cpa_proton_fcf(5)  = 3.9e-3 * (255.-230.)
	cpa_proton_fcf(6)  = 3.9e-3 * (320.-255.)
	cpa_proton_fcf(7)  = 3.9e-3 * (380.-320.)
	cpa_proton_fcf(8)  = 3.9e-3 * (460.-380.)
	cpa_proton_fcf(9)  = 3.9e-3 * (600.-460.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (520.-370.)
	cpa_proton_fcf(11) = 4.4e-2 * (640.-520.)
	cpa_proton_fcf(12) = 4.4e-2 * (800.-640.)
	cpa_proton_fcf(13) = 4.4e-2 * (1000.-800.)
	cpa_proton_fcf(14) = 4.4e-2 * (1300.-1000.)
	cpa_proton_fcf(15) = 4.4e-2 * (1700.-1300.)
	cpa_proton_fcf(16) = 4.4e-2 * (2700.-1700.)
	cpa_proton_fcf(17) = 4.4e-2 * (4400.-2700.)
	cpa_proton_fcf(18) = 4.4e-2 * (8300.-4400.)
	cpa_proton_fcf(19) = 4.4e-2 * (13000.-8300.)
	cpa_proton_fcf(20) = 4.4e-2 * (24000.-13000.)
	cpa_proton_fcf(21) = 8.0e-2 * (41000.-24000.)
	cpa_proton_fcf(22) = 8.0e-2 * (64000.-41000.)
	cpa_proton_fcf(23) = 8.0e-2 * (100000.-64000.)
	cpa_proton_fcf(24) = 8.0e-2 * (160000.-100000.)
	cpa_proton_fcf(25) = 8.0e-2 * (230000.-160000.)
	end
  '1984-037': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (83.-70.)
	cpa_proton_fcf(1)  = 3.9e-3 * (100.-83.)
	cpa_proton_fcf(2)  = 3.9e-3 * (130.-100.)
	cpa_proton_fcf(3)  = 3.9e-3 * (160.-130.)
	cpa_proton_fcf(4)  = 3.9e-3 * (190.-160.)
	cpa_proton_fcf(5)  = 3.9e-3 * (235.-190.)
	cpa_proton_fcf(6)  = 3.9e-3 * (290.-235.)
	cpa_proton_fcf(7)  = 3.9e-3 * (350.-290.)
	cpa_proton_fcf(8)  = 3.9e-3 * (450.-350.)
	cpa_proton_fcf(9)  = 3.9e-3 * (560.-450.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (490.-380.)
	cpa_proton_fcf(11) = 4.4e-2 * (680.-490.)
	cpa_proton_fcf(12) = 4.4e-2 * (870.-680.)
	cpa_proton_fcf(13) = 4.4e-2 * (1050.-870.)
	cpa_proton_fcf(14) = 4.4e-2 * (1450.-1050.)
	cpa_proton_fcf(15) = 4.4e-2 * (1800.-1450.)
	cpa_proton_fcf(16) = 4.4e-2 * (2900.-1800.)
	cpa_proton_fcf(17) = 4.4e-2 * (4800.-2900.)
	cpa_proton_fcf(18) = 4.4e-2 * (9000.-4800.)
	cpa_proton_fcf(19) = 4.4e-2 * (14000.-9000.)
	cpa_proton_fcf(20) = 4.4e-2 * (24000.-14000.)
	cpa_proton_fcf(21) = 8.0e-2 * (39000.-24000.)
	cpa_proton_fcf(22) = 8.0e-2 * (60000.-39000.)
	cpa_proton_fcf(23) = 8.0e-2 * (90000.-60000.)
	cpa_proton_fcf(24) = 8.0e-2 * (145000.-90000.)
	cpa_proton_fcf(25) = 8.0e-2 * (200000.-145000.)
	end
  '1984-129': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (91.-72.)
	cpa_proton_fcf(1)  = 3.9e-3 * (104.-91.)
	cpa_proton_fcf(2)  = 3.9e-3 * (125.-104.)
	cpa_proton_fcf(3)  = 3.9e-3 * (153.-125.)
	cpa_proton_fcf(4)  = 3.9e-3 * (190.-153.)
	cpa_proton_fcf(5)  = 3.9e-3 * (235.-190.)
	cpa_proton_fcf(6)  = 3.9e-3 * (292.-235.)
	cpa_proton_fcf(7)  = 3.9e-3 * (365.-292.)
	cpa_proton_fcf(8)  = 3.9e-3 * (457.-365.)
	cpa_proton_fcf(9)  = 3.9e-3 * (573.-457.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (480.-360.)
	cpa_proton_fcf(11) = 4.4e-2 * (630.-480.)
	cpa_proton_fcf(12) = 4.4e-2 * (800.-630.)
	cpa_proton_fcf(13) = 4.4e-2 * (1000.-800.)
	cpa_proton_fcf(14) = 4.4e-2 * (1400.-1000.)
	cpa_proton_fcf(15) = 4.4e-2 * (1800.-1400.)
	cpa_proton_fcf(16) = 4.4e-2 * (2800.-1800.)
	cpa_proton_fcf(17) = 4.4e-2 * (4800.-2800.)
	cpa_proton_fcf(18) = 4.4e-2 * (9100.-4800.)
	cpa_proton_fcf(19) = 4.4e-2 * (14500.-9100.)
	cpa_proton_fcf(20) = 4.4e-2 * (25000.-14500.)
	cpa_proton_fcf(21) = 8.0e-2 * (37000.-25000.)
	cpa_proton_fcf(22) = 8.0e-2 * (57000.-37000.)
	cpa_proton_fcf(23) = 8.0e-2 * (88000.-57000.)
	cpa_proton_fcf(24) = 8.0e-2 * (145000.-88000.)
	cpa_proton_fcf(25) = 8.0e-2 * (210000.-145000.)
	end
  '1987-097': begin
	; LoP
	cpa_proton_fcf(0)  = 3.9e-3 * (88.-75.)
	cpa_proton_fcf(1)  = 3.9e-3 * (106.-88.)
	cpa_proton_fcf(2)  = 3.9e-3 * (129.-106.)
	cpa_proton_fcf(3)  = 3.9e-3 * (157.-129.)
	cpa_proton_fcf(4)  = 3.9e-3 * (194.-157.)
	cpa_proton_fcf(5)  = 3.9e-3 * (241.-194.)
	cpa_proton_fcf(6)  = 3.9e-3 * (299.-241.)
	cpa_proton_fcf(7)  = 3.9e-3 * (376.-299.)
	cpa_proton_fcf(8)  = 3.9e-3 * (472.-376.)
	cpa_proton_fcf(9)  = 3.9e-3 * (590.-472.)
	; HiP
	cpa_proton_fcf(10) = 4.4e-2 * (480.-370.)
	cpa_proton_fcf(11) = 4.4e-2 * (670.-480.)
	cpa_proton_fcf(12) = 4.4e-2 * (850.-670.)
	cpa_proton_fcf(13) = 4.4e-2 * (1030.-850.)
	cpa_proton_fcf(14) = 4.4e-2 * (1330.-1030.)
	cpa_proton_fcf(15) = 4.4e-2 * (1720.-1330.)
	cpa_proton_fcf(16) = 4.4e-2 * (2600.-1720.)
	cpa_proton_fcf(17) = 4.4e-2 * (4740.-2600.)
	cpa_proton_fcf(18) = 4.4e-2 * (8960.-4740.)
	cpa_proton_fcf(19) = 4.4e-2 * (13600.-8960.)
	cpa_proton_fcf(20) = 4.4e-2 * (23900.-13600.)
	cpa_proton_fcf(21) = 8.0e-2 * (36400.-23900.)
	cpa_proton_fcf(22) = 8.0e-2 * (58300.-36400.)
	cpa_proton_fcf(23) = 8.0e-2 * (83800.-58300.)
	cpa_proton_fcf(24) = 8.0e-2 * (147000.-83800.)
	cpa_proton_fcf(25) = 8.0e-2 * (192000.-147000.)
	end
endcase

; --------------------------------------------------
; Confert counts/second to particles/cm^2/sec/sr/keV
; by dividing by the flux conversion factors
; --------------------------------------------------

; Electrons
for i = 0,11 do begin
	electrons(i,*) = electrons(i,*)/cpa_electron_fcf(i)
endfor

; Protons
for i = 0,25 do begin
	protons(i,*) = protons(i,*)/cpa_proton_fcf(i)
endfor

end

