;******************************************************************************
;* FUNCTION:   
;*      FUNCTION  lanl_geo_panel, plotinfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	plotinfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from plotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*	plotinfo = lanl_geo_panel(PlotInfo)
;*
;* MODIFICATION HISTORY:       
;*     rewritten March 2002, Reiner Friedel, to work of original files only.
;*     rewritten July 2003, to use control structure.
;******************************************************************************
FUNCTION  lanl_geo_panel, plotinfo, GROUP=group, ACTION=ACTION, $
                              SIZE_ONLY = SIZE_ONLY, SCROLL = scroll, $
                              X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize

COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents
COMMON lanl_geo_panel, WidgetData
  
WIDGET_CONTROL, /HOURGLASS 

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
  
; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to WidgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(lanl_geo_control)
control = *plotinfo.USR_PTR1
;control now contains the user defined control structure for this module.
  
;--- create the base-window ---------------------------------------------------
title = PlotInfo.panelKind + ' - ' + papco_module_version('lanl_geo')
panel_base=papco_WIDGET_BASE(TITLE=title, COLUMN=1, GROUP_LEADER=group, /MODAL)
  
panel_top =    WIDGET_BASE(panel_base, ROW=1, /ALIGN_CENTER)

panel_middle = widget_base(panel_base, ROW = 1, SCROLL = scroll, $
                           X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize)

;panel_middle = WIDGET_BASE(panel_base, ROW=1, /ALIGN_CENTER)
panel_left =   WIDGET_BASE(panel_middle, COLUMN=1, /ALIGN_CENTER)
panel_right =  WIDGET_BASE(panel_middle, COLUMN=1, /ALIGN_CENTER)
panel_bottom = WIDGET_BASE(panel_base, ROW=1, /ALIGN_CENTER)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=PAPCO_panel_AddHeightWidgets(panel_top, plotinfo)
additionalWData=create_struct(widgets, additionalWData)

;--- select satellite to use ------------------------------------------------  
sat_base = WIDGET_BASE(panel_left, ROW = 1,  /ALIGN_CENTER)
mpb_sat=papco_CW_BGROUP(sat_base, sat_names, ROW = 2, /EXCLUSIVE, /FRAME, $
                  LABEL_TOP='Satellite ID', UVALUE='mpb_sat', $
                  BUTTON_IDS = sat_ids, $
                  SET_VALUE=control.sc_id)
  
;--- add plot selector widget -------------------------------------------------
selector_base=WIDGET_BASE(panel_left, row=1, /align_center)

;--- select type of data to use  ----------------------------------------------
names = papco_make_choice_names(control.type_info)
mpb_source=CW_BGROUP(selector_base, names, ROW=2, /EXCLUSIVE, /FRAME, $
                         LABEL_TOP='Data source', UVALUE='mpb_source', $
                         BUTTON_UVALUE = type_uvalues, $
                         SET_VALUE = control.type)

;--- draw line or spec --------------------------------------------------------
names = papco_make_choice_names(control.graph_info)
mpb_graph=papco_CW_BGROUP(selector_base, names, ROW=1, /EXCLUSIVE, /FRAME, $
                          LABEL_TOP='Graph: ', UVALUE='mpb_graph', $
                          BUTTON_IDS = graph_ids, $
                          BUTTON_UVALUE = graph_uvalues, $
                          SET_VALUE=control.graph)

;---- add time res buttons ----------------------------------------------------
names = papco_make_choice_names(control.res_info)
mpb_res=papco_CW_BGROUP(selector_base, names, COLUMN=3, /EXCLUSIVE, /frame, $
                  LABEL_TOP='Resolution : ',BUTTON_IDS = res_ids,  $
                  UVALUE='mpb_res', SET_VALUE=control.res)
  

plot_base =  WIDGET_BASE(panel_left, ROW = 1,  /ALIGN_CENTER)

;--- Plot Data type choice ----------------------------------------------------
pl_type_names = papco_make_choice_names(control.pl_type_info)
mpb_pl_type = papco_CW_BGROUP(plot_base, pl_type_names, COLUMN = 4, $
                      /EXCLUSIVE,/FRAME, $
                      BUTTON_IDS = pl_type_ids, $
                      LABEL_TOP='Plot Type of Data :', $
                      UVALUE='spb_pl_type', $
                      SET_VALUE=control.pl_type)

;-- Choice of Detector--------------------------------------------------------
det_set = bytarr(3)
IF control.det_choice EQ 0 THEN det_set(1) = 1 ELSE FOR i = 0, 2 DO $
 IF (control.det_choice AND 2ll^i) EQ 2ll^i THEN det_set(i)=1 ELSE det_set(i)=0
detector_names = ['1', '2', '3']
mpb_detector = papco_CW_BGROUP(plot_base,detector_names, ROW = 1,/frame, $
                         /NONEXCLUSIVE, LABEL_TOP = 'Use Telescopes:', $
                         UVALUE = 'mpb_detector',BUTTON_IDS = detector_ids, $
                         SET_VALUE = det_set)

;--- button to chose "select" variable ----------------------------------------
filter_outer = widget_base(panel_left, COLUMN = 2, /ALIGN_CENTER)
filter_names = 'select ' + select_names
filter_base=CW_BGROUP(filter_outer, filter_names, ROW = 2, /EXCLUSIVE, /FRAME,$
                  LABEL_TOP='Select filter variable', $
                  UVALUE='spb_filter', SET_VALUE=control.filter)

;--- Add sliders to filter data - L, MLT, MLAT, etc ---------------------------
;    Multiple map bases
select_outer_base=WIDGET_BASE(filter_outer, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)

;--- make select widgets for all variables in "select_names ------------------
;    use papco_panel compound widget for this!
widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L_STAR', $
            control.L_STAR_SELECT_RANGE > 6 < 7.5, $
            control.L_STAR_SELECT_FROM_VAL > 6 < 7.5, $
            control.L_STAR_SELECT_TO_VAL, 6.0, 7.5, 0.05, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L', $
            control.L_SELECT_RANGE > 6 < 7.5, $
            control.L_SELECT_FROM_VAL > 6 < 7.5, $
            control.L_SELECT_TO_VAL, 6.0, 7.5, 0.05, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLT', $
            control.MLT_SELECT_RANGE, $
            control.MLT_SELECT_FROM_VAL, $
            control.MLT_SELECT_TO_VAL, 0.0, 24.0, 0.5, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLAT', $
            control.MLAT_SELECT_RANGE, $
            control.MLAT_SELECT_FROM_VAL, $
            control.MLAT_SELECT_TO_VAL, -15, 15.0, 1, '(f5.1)')
additionalWData=create_struct(widgets, additionalWData)

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(panel_left, control, /MODEL_ONLY)
additionalWData=create_struct(widgets, additionalWData)

;-- map different bases according to what type of data is chosen.  
type_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
type_base=WIDGET_BASE(type_outer_base, /align_center)
  
;--- choices for SOPA/CPA/ESP -------------------------------------------------
 
request_base = WIDGET_BASE(type_base, COLUMN = 2)

;---- get count/ flux option
names=papco_make_choice_names(control.cal_info)
mpb_cal=papco_CW_BGROUP(request_base, names, ROW=1, /EXCLUSIVE, /FRAME, $
                  LABEL_LEFT='Calibration: ', UVALUE='mpb_cal', $
                  SET_VALUE=control.cal, button_ids = cal_ids)  

fld_mu = CW_FIELD(request_base, /ROW,  XSIZE=20, $
                  TITLE = 'Enter mu1, mu2,.. (MeV/G):', $
                  UVALUE= 'fld_mu', VALUE = control.mu)


;--- choices for merged data --------------------------------------------------
merged_base=WIDGET_BASE(type_base, ROW=2)


;--- select for  pitch angle --------------------------------------------------
pitch_base = WIDGET_BASE(panel_left, COLUMN = 3)


pa_base = WIDGET_BASE(pitch_base, ROW = 1, /align_center)
sl_pa =WIDGET_SLIDER(pa_base,XSIZE = 220,/align_center, $; /SUPPRESS_VALUE,
		       /DRAG, TITLE='pitch angle start', $
		       VALUE=control.pa_bin1 <   180 >  0, $
                       UVALUE='sl_pa', MINIMUM=0, MAXIMUM=180, sensitive = 0)


pa_base2 = WIDGET_BASE(pitch_base, ROW = 1)
sl_pa2 =WIDGET_SLIDER(pa_base,XSIZE = 220,/align_center, $; /SUPPRESS_VALUE, 
		      /DRAG, TITLE='pitch angle end', $
		       VALUE=control.pa_bin2 <  180 >  0, $
                       UVALUE='sl_pa2', MINIMUM=0, MAXIMUM=180, sensitive = 0)


fld_pa = CW_FIELD(pitch_base, /ROW,  XSIZE=5, $
                  TITLE = 'PA res. (deg):', $
                  UVALUE= 'fld_pa', VALUE = control.pa_res)


  
;--- remaining common widgets for all data types ------------------------------
  
;--- add channel select buttons -----------------------------------------------
names=strarr(120) 
FOR i=0,119 DO names(i)=strcenter(string(i,format="('empty ',i3.3)"),18)

;--- use ek1, and ek2 as an 120 bit integer, as a
;    logical flag to set channels - a maximum of 120 channels can be set.
set=bytarr(120) 
FOR i=0,59 DO $
    IF (control.ek1 AND 2ll^i) EQ 2ll^i THEN set(i)=1 ELSE set(i)=0
FOR i=60,119 DO $
    IF (control.ek2 AND 2ll^(i-60)) EQ 2ll^(i-60) THEN $
      set(i)=1 ELSE set(i)=0

energy_base =  WIDGET_BASE(panel_left, COLUMN = 2, /align_center)
mpb_energy_ch=CW_BGROUP(energy_base, names, ROW=30, /FRAME, $
                        /NONEXCLUSIVE, /SCROLL, $
                        Y_SCROLL_SIZE=140, X_SCROLL_SIZE=550, $
                        LABEL_TOP='Energy channels', $
                        UVALUE='mpb_energy_ch', $
                        IDS=mpb_energy_ch_ids, SET_VALUE=set)
  
scale_base = WIDGET_BASE(panel_right, ROW = 1)
;--- now, add the widgets for choosing the y-axis-ranges ----------------------
widgets=PAPCO_PANEL_AddYScale_Widgets(scale_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- now, add the widgets for choosing the z-axis-ranges ----------------------
widgets=PAPCO_PANEL_AddZScale_Widgets(scale_base, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- now, add the widgets for choosing own plot descriptions ------------------
widgets=papco_PANEL_AddDescr_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)  
;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_right, plotinfo,0, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)  
;--- add the papco panel widget for choosing the color bar for color plots ----
widgets=papco_panel_Addcolorbar_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)  
  
;--- finally, the Actionbuttons are added to the editor window. ---------------
widgets=PAPCO_panel_AddActionButtons(panel_bottom, Action, plotinfo, $
                                       /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.  
thisWidgetsData={sat_ids:sat_ids, pl_type_ids:pl_type_ids, $
                 fld_mu:fld_mu, $
                 mpb_graph:mpb_graph, graph_ids:graph_ids, $
                 mpb_energy_ch_ids:mpb_energy_ch_ids, $
                 request_base:request_base, $
                 merged_base:merged_base, $
                 helpfile:'lanl_geo_panel.help', $
                 mpb_detector:mpb_detector, $
                 detector_ids:detector_ids, $
                 sl_pa:sl_pa, $ ;, pa_label:pa_label, $
                 sl_pa2:sl_pa2, $ ;, pa_label2:pa_label2, $
                 fld_pa:fld_pa, $
                 mpb_pl_type:mpb_pl_type, $
                 mpb_cal:mpb_cal, cal_ids:cal_ids, $
                 mpb_energy_ch:mpb_energy_ch, $
                 mpb_res:mpb_res, res_ids:res_ids}

; add control struct, papco_panel structs
widgetData=papco_panel_CreateWidgetData(additionalWData, thisWidgetsData)
widgetData=papco_panel_CreateWidgetData(widgetData, control)
  
;--- return size of the window only here --------------------------------------
IF keyword_set(SIZE_ONLY) THEN BEGIN
    size_only = WIDGET_INFO(panel_base, /GEOMETRY)
    ;adjust for non-scroll parts of panel
    size_only.SCR_YSIZE = size_only.SCR_YSIZE + 175
    return, 1
ENDIF 

;--- realize the window -------------------------------------------------------
PAPCO_Panel_Realize, panel_base, WidgetData

WIDGET_CONTROL, panel_base, HOURGLASS=1
lanl_geo_refresh, /ALL
WIDGET_CONTROL, panel_base, HOURGLASS=0
  
papco_xmanager, 'lanl_geo_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

ReturnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified}
  
additionalRData=PAPCO_panel_ReturnData(WidgetData)
  
RETURN, create_struct(ReturnData, additionalRData)

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_geo_refresh, ALL=ALL, SATID=SATID, CHANNEL=CHANNEL
;*
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      if set (e.g. /LID), only the widgets influenced by the information
;*      that the keyword's name describes, are updated.
;*      If the keyword /ALL is set, all of the window is updated.
;*      
;*      ALL     - all
;*      CHANNEL - energy channel
;*
;*
;* CALLING SEQUENCE:
;*      have a look the the event-routine lanl_geo_Panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;*      september 1997, modified by Torsten Heise
;*      september 2003, modified by Arne Aasnes
;******************************************************************************
PRO lanl_geo_refresh, ALL=ALL, CHANNEL=CHANNEL, RES=RES, FILTER = FILTER, $
                      TYPE=TYPE, GRAPH=GRAPH, MU = MU, $
                      DATATYPE = DATATYPE, $
                      DETECTOR = DETECTOR, SATS = SATS

COMMON lanl_geo_panel, WidgetData
COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents

IF KEYWORD_SET(ALL) THEN BEGIN
    CHANNEL=1 & FILTER = 1 &  TYPE = 1 & SATS = 1
ENDIF

; --- Filter Variable multiple bases ------------------------------------------
IF keyword_set(FILTER) THEN BEGIN
    wid_ids = [widgetData.l_select_base, widgetData.l_star_select_base, $
               widgetData.mlt_select_base, widgetData.mlat_select_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid) 
    map(widgetData.filter) = 1 
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
ENDIF 

;--- toggle bases for data types ----------------------------------------------
IF KEYWORD_SET(TYPE) THEN BEGIN
    wid_ids = [widgetData.request_base, widgetData.merged_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid) 
    map(widgetData.type) = 1 
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
    IF widgetData.type THEN BEGIN
        SENS = 0 
        widgetData.pl_type = 0
        WIDGET_CONTROL, widgetData.mpb_pl_type, set_value = 0
        lanl_geo_refresh, /DATATYPE
    ENDIF ELSE sens = 1
    FOR i = 1, n_elements(widgetdata.pl_type_ids)-1 DO $
      WIDGET_CONTROL, widgetdata.pl_type_ids(i), sensitive = sens
ENDIF

;--- Satellite ID's - restrict for merged -------------------------------------
IF keyword_set(SATS) THEN BEGIN
    IF widgetData.type THEN sens = 0 ELSE sens = 1
    FOR i = 0, 7 DO WIDGET_CONTROL, widgetData.sat_ids(i), sensitive = sens
ENDIF 

IF keyword_set(DATATYPE) THEN BEGIN
    wid_ids = [widgetData.sl_pa, widgetData.sl_pa2]
    res_ids = widgetData.res_ids
    cal_ids = widgetdata.cal_ids
    graph_ids = widgetData.graph_ids
    det_ids = widgetData.detector_ids
    CASE WidgetData.pl_type OF 
        0:BEGIN ;Spin Averaged Spectra
           set_cal_sens = [1, 1, 1] & set_graph_sens = [1, 1, 1] 
           set_sens = [0, 0] &  set_res_sens = [1, 1, 1, 1, 1, 1]
            widgetData.ylinlog = 1     
            widget_control, widgetData.mpb_detector, set_value = [1, 1, 1]
            FOR i = 0, 2 DO widget_control, det_ids(i), sensitive = 0
        END 
        1:BEGIN ;All Pitch Angles @ one Energy
            set_cal_sens = [1, 1, 0] & set_graph_sens = [0, 1, 0]
            set_sens = [0, 0] &  set_res_sens = [0, 0, 0, 1, 1, 1]
            widgetData.ylinlog = 0
        END 
        2:BEGIN ;All Energies @ one Pitch Angle
            set_cal_sens = [1, 1, 0] & set_graph_sens = [0, 1, 0]
            set_sens = [1, 0] &  set_res_sens = [0, 0, 0, 1, 1, 1]
            widgetData.ylinlog = 1
        END 
        3:BEGIN ;All Pitch Angles for an Energy Range
            set_cal_sens = [0, 1,  0] & set_graph_sens = [0, 1, 0]
            set_sens = [0, 0] &  set_res_sens = [0, 0, 0, 1, 1, 1]
            widgetData.ylinlog = 0
        END 
        4:BEGIN ;All Energies for a Pitch Angle Range
            set_cal_sens = [1, 1, 0] & set_graph_sens = [0, 1, 0]
            set_sens = [1, 1] &  set_res_sens = [0, 0, 0, 1, 1, 1]
            widgetData.ylinlog = 1
        END 
        5:BEGIN ; anisotropy spectrogram
           set_cal_sens = [1, 1, 0] &  set_graph_sens = [0, 1, 0]
           set_sens = [1, 1] &  set_res_sens = [0, 0, 0, 1, 1, 1]
           widgetData.ylinlog = 1
        end 
        6:BEGIN ;Pitch Angle Range @ an Energy Range (line plot)
            set_cal_sens = [0, 1, 0] & set_graph_sens = [1, 0, 0]
            set_sens = [1, 1] &  set_res_sens = [0, 0, 0, 1, 1, 1]
            widgetData.ylinlog = 1
        END 
        7:BEGIN ;Anisotropy lineplot
           set_cal_sens = [0, 1, 0] & set_graph_sens = [0, 1, 0]
           set_sens = [1, 1] &  set_res_sens = [0, 0, 0, 1, 1, 1]
           widgetData.ylinlog = 0
        END 
    ENDCASE

    FOR i = 0, 2 DO widget_control, cal_ids(i), sensitive = set_cal_sens(i)
    FOR i = 0, 2 DO widget_control, graph_ids(i), sensitive = set_graph_sens(i)
    FOR i = 0,1 DO widget_control, wid_ids(i), sensitive = set_sens(i)
    FOR i = 0, 5 DO widget_control, res_ids(i), sensitive = set_res_sens(i)
    IF widgetData.pl_type GT 0 THEN BEGIN 
       FOR i = 0, 2 DO widget_control, det_ids(i), sensitive = 1
       widget_control, widgetData.mpb_detector, set_value = [0, 1, 0]
       widget_control, widgetData.mpb_res, set_value =  widgetData.res > 3
       IF widgetData.pl_type EQ 3 OR widgetData.pl_type EQ 6 OR widgetData.pl_type EQ 7 THEN widget_control, $
          widgetData.mpb_cal, set_value = 1
       graph_val = where(set_graph_sens EQ 1) 
       widget_control, widgetData.mpb_graph, set_value = graph_val(0)
    endIF 
    papco_panel_refresh, widgetData, /YSCALE
ENDIF

IF keyword_set(DETECTOR) THEN BEGIN 
   IF widgetData.det_choice EQ 0 THEN det_set(1) = 1
   widget_control, widgetData.mpb_detectors, set_value = set
ENDIF 

IF KEYWORD_SET(CHANNEL) THEN BEGIN
    ;first reset all buttons to empty
    button_ids=WidgetData.mpb_energy_ch_ids
    for i=0,119 do begin
      widget_control,button_ids(i), $
        SET_VALUE=strcenter(string(i,format="('empty ',i3.3)"),16)
      widget_control,button_ids(i),sensitive=0
    endfor     

    sat = sat_names(WidgetData.sc_id)
    IF WidgetData.type EQ 1 THEN MERGED = 1 ELSE MERGED = 0
    channels=r_lanl_geo_cal(sat, MERGED = MERGED)
    list = channels.instrument+' '+channels.label &  n=n_elements(list)

    for i=0,(n-1) < 119 do begin
          widget_control,button_ids(i),SET_VALUE=strcenter(list(i),18)
          widget_control,button_ids(i),sensitive=1
    endfor  

    for i=n,119 do begin ;make extra channels insensitive
          widget_control,button_ids(i),$
            SET_VALUE=strcenter(string(i,format="('empty ',i3.3)"),18)
          widget_control,button_ids(i),sensitive=0
    endfor
        
    ;if the channel set switch is zero, select at least 1st channel
    ;otherwise set what is selected
    set=bytarr(120) 
    if WidgetData.ek1+WidgetData.ek2 eq 0 then begin
          set(0)=1
    endif else begin
         for i=0,59 do $
            if (WidgetData.ek1 and 2ll^i) eq 2ll^i then $
              set(i)=1 else set(i)=0
         for i=60,119 do $
            if (WidgetData.ek2 and 2ll^(i-60)) eq 2ll^(i-60) then $
              set(i)=1 else set(i)=0
    endelse
    WIDGET_CONTROL,WidgetData.mpb_energy_ch, SET_VALUE=set
ENDIF  
  
END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_geo_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written october 1995, Reiner Friedel
;*     modified september 1997, Torsten Heise
;******************************************************************************
PRO lanl_geo_panel_event, event

COMMON plot_composer, mainWidgetData
COMMON lanl_geo_panel, WidgetData
COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents
  
WIDGET_CONTROL, GET_UVALUE=uval, event.id

;test uvalue for structure - select variable sliders return structure uvals.
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval
    uval = uval.uvalue
ENDIF
  
CASE uval OF

    'mpb_sat': BEGIN
        WidgetData.sc_id = event.value
        lanl_geo_refresh, /CHANNEL
    END

    'mpb_energy_ch': BEGIN 
        WIDGET_CONTROL,WidgetData.mpb_energy_ch, GET_VALUE=set
        WidgetData.ek1=0  &  WidgetData.ek2=0
        FOR i=0,59 DO IF set(i) THEN WidgetData.ek1=WidgetData.ek1+2ll^i 
        FOR i=60,119 DO IF set(i) THEN WidgetData.ek2=WidgetData.ek2+2ll^(i-60)
    END    
    
    'mpb_graph': BEGIN 
        WidgetData.graph = event.value
        lanl_geo_refresh, /GRAPH
    END 
    
    'mpb_res' : WidgetData.res = event.value
    
    'mpb_cal': BEGIN
        WidgetData.cal = event.value
        lanl_geo_refresh, /MU
    END 
   
    'mpb_source': BEGIN
        WidgetData.type = event.value
        lanl_geo_refresh, /TYPE, /SATS, /CHANNEL
    END 
    'spb_pl_type': BEGIN
        WidgetData.pl_type = event.value
        lanl_geo_refresh, /DATATYPE
    END   
    'mpb_detector':BEGIN
        widget_control, widgetData.mpb_detector, get_value = det_set
        widgetData.det_choice = 0
        FOR i = 0, 2 DO IF det_set(i) THEN $
          widgetData.det_choice = widgetData.det_choice+2ll^i
    END 
       
    'sl_pa': BEGIN
        widgetData.pa_bin1=event.value
    END     

    'sl_pa2':BEGIN
         widgetData.pa_bin2=event.value
    END     


    'spb_filter' : BEGIN 
        widgetData.filter = event.value
        lanl_geo_refresh, /FILTER
    END 

    'pb_cancel' : BEGIN
        mainWidgetData.NEEDTO_READ_DATA=1
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
    END

    ELSE : BEGIN
        WIDGET_CONTROL, widgetData.fld_mu, GET_VALUE = X
        WIDGET_CONTROL, widgetData.fld_pa, GET_VALUE = Y
        widgetData.mu = x
        widgetdata.pa_res = Y
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
    END
    
ENDCASE

END
