;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  lanl_geo_pitch_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;*
;* PLOTINFO VARIABLE ASSIGNMENT:
;*      ioptions(1)     0-local 1-ftp
;*      ioptions(2)     0-count 1-flux
;*      ioptions(3)     0-no local time labels 1-local time labels ??
;*      ioptions(8)     0-PA plot, 1 Spectral plot
;*      ioptions(10)    index of satellite (index of list given in
;*                                  common block sat names)
;*      typeVector(0) \ both together a 128 bit integer for channel 
;*      typeVector(1) / combination (bitwise)
;*      typeVector(2)  an 20 bit integer, as a logical flag to set 
;*                     pitch angle channels.
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=PAPCO_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;* 
;*     written July 2003, Arne Aasnes, to be panel for lanl_geo_pitch
;******************************************************************************
FUNCTION  lanl_geo_pitch_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON LANL_GEO_PITCH_PANEL, WidgetData, lSatNames, rSatNames, lExtNames, rExtNames
COMMON lanl_geo ;see lanl_geo_commons for contents
COMMON time,xut1,xut2
  
WIDGET_CONTROL, /HOURGLASS 

;--- check if the current user is "friedel" by checking PAPCO_PATH
;current = getenv('PAPCO_PATH') & pos=strpos(current, 'friedel') 
;IF pos NE -1 THEN detail = 1 ELSE detail = 0
    
;detail = 0

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
  
; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'
  
; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to WidgetData.
additionalWData={canceled:1}
  
;--- create the base-window ---------------------------------------------------
;    since this is a large widget, make a left and right side
panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, column=1, $
                             GROUP_LEADER=group, /MODAL)
  
panel_top=widget_base(panel_base,row=1, /align_center)
panel_middle = widget_base(panel_base,row=1, /align_center)
panel_left=widget_base(panel_middle,column=1, /align_center)
panel_right=widget_base(panel_middle,column=1, /align_center)
panel_bottom=widget_base(panel_base,row=1, /align_center)
  
;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
widgets=PAPCO_panel_AddHeightWidgets(panel_top, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
  
;--- add plot selector widget -------------------------------------------------
selector_base=WIDGET_BASE(panel_left, row=2, /align_center)
  
;--- data source options ------------------------------------------------------
;names=['locally','via ftp']
;mpb_datasource=papco_CW_BGROUP(selector_base,names,ROW=2,/EXCLUSIVE,/FRAME, $
;                           LABEL_TOP='Data source', UVALUE='mpb_datasource', $
;                           BUTTON_IDS = datasource_ids, $
;                           BUTTON_UVALUE = datasource_uvalues, $
;                           SET_VALUE=aPlotInfo.ioptions(1))  
;widget_control, datasource_ids(1), sensitive = 0
  
;--- select type of data to use  ----------------------------------------------
;names=['SOPA/CPA','MPA/SOPA/CPA/ESP Merged','SOPA/CPA/ESP Pitch']
;mpb_datatype=papco_CW_BGROUP(selector_base, names, ROW=1, /EXCLUSIVE, /FRAME, $
;                         LABEL_TOP='Data types', UVALUE='mpb_datatype', $
;                         BUTTON_IDS = datatype_ids, $
;                         BUTTON_UVALUE = datatype_uvalues, $
;                         SET_VALUE=aPlotInfo.ioptions(3))
;widget_control, datatype_ids(2), sensitive = 0
  
;--- select satellite to use ------------------------------------------------  
names = sat_names
mpb_sat=CW_BGROUP(selector_base, names, ROW = 3, /EXCLUSIVE, /FRAME, $
                       LABEL_TOP='SDatellite ID', UVALUE='mpb_sat', $
                       SET_VALUE=aPlotInfo.ioptions(10))

;IF detail THEN BEGIN 
;;--- sliders to select the MLT-range to be plotted ----------------------------
;  mlt_base=WIDGET_BASE(panel_left, column=1, /align_center, /frame) 
  
;  mlt_rangeNames=['no MLT range', 'select MLT range']
;  mlt_range_base=CW_BGROUP(mlt_base, mlt_rangeNames, ROW=1, /EXCLUSIVE, $
;                           UVALUE='spb_mlt_range', $
;                           SET_VALUE=aPlotInfo.options(7))
  
;  mlt_label_base=WIDGET_BASE(mlt_base, row=1, /align_center)
;  mlt_from_label= WIDGET_LABEL(mlt_label_base, /align_center, $
;  VALUE=strformat(string(aPlotInfo.options(2),format="(f4.1)"),40,/center), $
;                             UVALUE='mlt_from_label')
;  mlt_to_label=WIDGET_LABEL(mlt_label_base, /align_center, $
;  VALUE=strformat(string(aPlotInfo.options(3),format="(f4.1)"),40,/center), $
;                             UVALUE='mlt_to_label')    
  
;  mlt_slider_base=WIDGET_BASE(mlt_base, row=1, /align_center)
;  ; --- from MLT-value --------------------------------------------------------
;  mlt_from=WIDGET_SLIDER(mlt_slider_base, /DRAG, /SUPPRESS_VALUE, xsize=250,$
;                  title=strformat('From MLT',35,/center), $
;         	  VALUE=(aPlotInfo.options(5) * 2) < 49 > 0, $
;                  UVALUE='mlt_from', MINIMUM=0, MAXIMUM=48)	      
  
;  ; --- to MLT-value ----------------------------------------------------------
;  mlt_to=WIDGET_SLIDER(mlt_slider_base, /DRAG, /SUPPRESS_VALUE, xsize=250, $
;                  title=strformat('To MLT',35,/center), $
;		  VALUE=(aPlotInfo.options(6) * 2) < 49 > 0, $
;                  UVALUE='mlt_to', MINIMUM=0, MAXIMUM=48)    


;  ;--- draw line or spec ------------------------------------------------------
;  names=['Line','Color', 'McIlwain']
;  mpb_graph=papco_CW_BGROUP(panel_left, names, ROW=1, /EXCLUSIVE, /frame, $
;                            LABEL_LEFT='Graph: ', UVALUE='mpb_graph', $
;                            BUTTON_IDS = graph_ids, $
;                            BUTTON_UVALUE = graph_uvalues, $
;                            SET_VALUE=aPlotInfo.ioptions(7))
;ENDIF ELSE BEGIN
    mlt_base = 0 & mlt_from = 0 & mlt_to = 0 
    mlt_from_label = '' & mlt_to_label = ''
;ENDELSE 

;-- map different bases according to what type of data is chosen.  
type_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
type_base=WIDGET_BASE(type_outer_base, /align_center)
  
;--- choices for SOPA/CPA data from request program only ----------------------
request_base=WIDGET_BASE(type_base, ROW=2)
request_base1=WIDGET_BASE(request_base, COLUMN = 1)  

;---- get count/ flux option 
names=['counts','flux', 'PSD @ const. mu']
mpb_caltype=CW_BGROUP(request_base1, names, ROW=1, /EXCLUSIVE, /frame, $
                        LABEL_LEFT='Calibration: ', UVALUE='mpb_caltype', $
                        SET_VALUE=aPlotInfo.ioptions(2))  

fld_mu = CW_FIELD(request_base1, /ROW,  XSIZE=40, $
                  TITLE = 'Enter mu1, mu2, ... (keV/nT):', $
                  UVALUE='fld_mu', VALUE = aplotinfo.soptions(0))


 
;--- choices for pitch data ---------------------------------------------------
pitch_base=WIDGET_BASE(type_base, ROW=1)
  
names=strarr(20)
for i=0,19 do begin
    pa1=acos(-1+0.1*i)*180/!PI  &  pa2=acos(-1+0.1*(i+1))*180/!PI
    names(i)=string(fix(pa1),fix(pa2),format="(i3.3,'-',i3.3)")
endfor

;names = strarr(19)
;pa = indgen(20)*10-5
;pa(0) = 0 & pa(19) = 180
;FOR i = 0, 18 DO   names(i) = string(pa(i), pa(i+1), format = "(i3.3,'-',i3.3)")
  
;---- use typevector(2) as an 20 bit integer, as a logical flag to set 
;     pitch angle channels.
  set=bytarr(20)
  for i=0,19 do if (aPlotInfo.typeVector(2) and 2ll^i) eq 2ll^i then $
    set(i)=1 else set(i)=0
  mpb_pitch_ch=CW_BGROUP(panel_left, names, ROW=5, /NONEXCLUSIVE, /frame, $
                      LABEL_TOP='Pitch angle ranges - 0.1 steps in cos(pa)', $
                      UVALUE='mpb_pitch_ch', SET_VALUE=set) 
  
;--- select PA ranges 
;  names=['All','Opposed','Perp.','Aligned','Custom']
;  mpb_pitch_sel=CW_BGROUP(pitch_base, names, ROW=5, /EXCLUSIVE, /frame, $
;                      LABEL_TOP='PA Select', UVALUE='mpb_pitch_sel', $
;                      SET_VALUE=aPlotInfo.ioptions(5))      
  
;--- select action ------------------------------------------------------------
  pitch_action_base=WIDGET_BASE(pitch_base, COLUMN=1)
;  names=['Indiv.','Aver. ']
;  mpb_pitch_act=CW_BGROUP(pitch_action_base, names, ROW=5, $
;                          /EXCLUSIVE, /frame, $
;                          LABEL_TOP='Action', UVALUE='mpb_pitch_act', $
;                          SET_VALUE=aPlotInfo.ioptions(6))      
  
  names=['Pitch ','Spec']
  mpb_pitch_plt=CW_BGROUP(panel_left, names, ROW=5, $
                          /EXCLUSIVE, /frame, UVALUE='mpb_pitch_plt', $
                          SET_VALUE=aPlotInfo.ioptions(8))  
  
;--- select ENERGY  ranges ----------------------------------------------------
;  names=['SOPA E','ESP 1','ESP 2','SOPA P', 'Custom']
;  mpb_energy_sel=CW_BGROUP(pitch_base, names, ROW=5, $
;                          /EXCLUSIVE, /frame, $
;                          LABEL_TOP='EK select', UVALUE='mpb_energy_sel', $
;                          SET_VALUE=aPlotInfo.ioptions(9))
  
;--- remaining common widgets for all data types ------------------------------
  
;--- add channel select buttons -----------------------------------------------
names=strarr(120) 
for i=0,119 do names(i)=strcenter(string(i,format="('empty ',i3.3)"),18)

;--- use typevector(0), and typevector(1) as an 120 bit integer, as a
;    logical flag to set channels - a maximum of 120 channels can be set.
set=bytarr(120) 
for i=0,59 do $
    if (aPlotInfo.typeVector(0) and 2ll^i) eq 2ll^i then set(i)=1 else set(i)=0
for i=60,119 do $
    if (aPlotInfo.typeVector(1) and 2ll^(i-60)) eq 2ll^(i-60) then $
      set(i)=1 else set(i)=0

mpb_energy_ch=CW_BGROUP(panel_left, names, ROW=30, /FRAME, $
                        /NONEXCLUSIVE, /SCROLL, $
                        Y_SCROLL_SIZE=120, X_SCROLL_SIZE=550, $
                        LABEL_TOP='Energy channels', $
                        UVALUE='mpb_energy_ch', $
                        IDS=mpb_energy_ch_ids, SET_VALUE=set)
  
;--- now, add the widgets for choosing the y-axis-ranges ----------------------
  widgets=PAPCO_PANEL_AddYScale_Widgets(panel_left, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;IF detail THEN BEGIN 
;;--- now, add the widgets for choosing the z-axis-ranges ----------------------
;  widgets=PAPCO_PANEL_AddZScale_Widgets(panel_left, aPlotInfo)
;  additionalWData=create_struct(widgets, additionalWData)
;ENDIF
;--- now, add the widgets for choosing own plot descriptions ------------------
  widgets=papco_PANEL_AddDescr_Widgets(panel_right, aPlotInfo, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)  
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_right, aPlotInfo,0, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)  
;--- add the papco panel widget for choosing the color bar for color plots ----
  widgets=papco_panel_Addcolorbar_Widgets(panel_right, aPlotInfo, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)  
  
;--- finally, the Actionbuttons are added to the editor window. ---------------
  widgets=PAPCO_panel_AddActionButtons(panel_right, Action, aPlotInfo, $
                                       /CONFIG, COLUMN=2)
  additionalWData=create_struct(widgets, additionalWData)

  
;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.  
  WidgetData={typeVector:aPlotInfo.typeVector, $
              fld_mu:fld_mu, $
              mpb_energy_ch_ids:mpb_energy_ch_ids, $
;              mpb_energy_sel:mpb_energy_sel, $
              mpb_pitch_ch:mpb_pitch_ch, $
;              mpb_pitch_sel:mpb_pitch_sel, $
              request_base:request_base, $
;              merged_base:merged_base, $
              pitch_base:pitch_base, $
              soptions:aPlotInfo.soptions, $
              ioptions:aPlotInfo.ioptions, $
              options:aPlotInfo.options, $
              mlt_base:mlt_base, $
              mlt_from:mlt_from, $
              mlt_to:mlt_to, $
              mlt_from_label:mlt_from_label, $
              mlt_to_label:mlt_to_label, $              
              helpfile:'lanl_geo_pitch_panel.help', $
              mpb_energy_ch:mpb_energy_ch}
;              mpb_res:mpb_res, $
;              detail:detail
  
  WidgetData=PAPCO_PANEL_CreateWidgetData(WidgetData,additionalWData)
  
;--- realize the window -------------------------------------------------------
  PAPCO_Panel_Realize, panel_base, WidgetData
  WIDGET_CONTROL, panel_base, HOURGLASS=1
  lanl_geo_pitch_refresh, /ALL
  if WidgetData.ioptions(3) then lanl_geo_pitch_refresh, /MERGED
  WIDGET_CONTROL, panel_base, HOURGLASS=0
  
  papco_xmanager, 'lanl_geo_pitch_panel', panel_base

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - e.g. the typeVector-tag.
  
  ReturnData={typeVector:WidgetData.typeVector, $
              soptions:WidgetData.soptions, $
              ioptions:WidgetData.ioptions, $
              options:WidgetData.options}
  
  additionalRData=PAPCO_panel_ReturnData(WidgetData)
  
  RETURN, create_struct(ReturnData, additionalRData)

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_geo_pitch_refresh, ALL=ALL, SATID=SATID, SATlist=SATlist,
;*                            CHANNEL=CHANNEL
;*
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      if set (e.g. /LID), only the widgets influenced by the information
;*      that the keyword's name describes, are updated.
;*      If the keyword /ALL is set, all of the window is updated.
;*      
;*      ALL     - all
;*      LID     - local satellite ID
;*      LLIST   - local satellite list
;*      LEL     - local file extention list
;*      LEID    - local file extention ID
;*      RID     - remote satellite ID
;*      RLIST   - remote satellite list
;*      REL     - remote file extention list
;*      REID  - remote file extention ID
;*      CHANNEL - energy channel
;*      RES     - time resolution buttons
;*
;*
;* CALLING SEQUENCE:
;*      have a look the the event-routine lanl_geo_Panel_event
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;*      september 1997, modified by Torsten Heise
;******************************************************************************
PRO lanl_geo_pitch_refresh, ALL=ALL, CHANNEL=CHANNEL, RES=RES, $
                      PITCH=PITCH, TYPE=TYPE, $
                      PA_SELECT=PA_SELECT, PLOT_SELECT=PLOT_SELECT, $
                      EK_SELECT=EK_SELECT, GRAPH=GRAPH, $
                      MLT=MLT, RANGE_MLT=RANGE_MLT
                   
COMMON LANL_GEO_PITCH_PANEL, WidgetData, lSatNames, rSatNames, lExtNames, rExtNames
COMMON lanl_geo, sat_names
COMMON get_error, get_err_no, get_err_msg
COMMON lanl_geo_satlist, date

IF KEYWORD_SET(ALL) THEN BEGIN
    CHANNEL=1 & TYPE=1  &  PA_SELECT=1 
    MLT=1  &  RANGE_MLT=1
ENDIF

;IF NOT WidgetData.DETAIL THEN BEGIN
    MLT=0  &  RANGE_MLT=0 & GRAPH = 0
;ENDIF 

IF KEYWORD_SET(GRAPH) then begin
END 

IF KEYWORD_SET(TYPE) then begin
    channel=1
    case WidgetData.ioptions(3) of 
      0: begin 
        RES=1 & RLIST=1 & REL=1 & LLIST=1 & LEL=1  & PITCH=0 
      end
      1: begin 
        RES=0 & RLIST=0 & REL=0 & LLIST=0 & LEL=0  & PITCH=0 
      end
      2: begin 
        RES=0 & RLIST=0 & REL=0 & LLIST=0 & LEL=0  & PITCH=1
      end
    endcase   
    widget_control, WidgetData.request_base, MAP=LLIST
;    widget_control, WidgetData.merged_base, MAP=MERGED
    widget_control, WidgetData.pitch_base, MAP=PITCH      
ENDIF 
  
IF KEYWORD_SET(PLOT_SELECT) then begin
    set=bytarr(20)
    WidgetData.typevector(2)=0
    if WidgetData.ioptions(8) eq 0 then begin
      WidgetData.ylinlog=0
      for i=0,19 do begin
        WidgetData.typevector(2)=WidgetData.typevector(2)+2^i
        set(i)=1
      endfor      
;      WIDGET_CONTROL,WidgetData.mpb_pitch_sel, SET_VALUE=0
;      WidgetData.ioptions(5)=0
    endif else BEGIN
      WidgetData.ylinlog=1
      for i=9,10 do begin
        WidgetData.typevector(2)=WidgetData.typevector(2)+2^i
        set(i)=1      
      endfor      
;      WIDGET_CONTROL,WidgetData.mpb_pitch_sel, SET_VALUE=4
;      WidgetData.ioptions(5)=4
    endelse  
    WIDGET_CONTROL,WidgetData.mpb_pitch_ch,SET_VALUE=set 
    papco_panel_Refresh,WidgetData,/yscale
ENDIF 
  
;IF KEYWORD_SET(PA_SELECT) then begin
;    set=bytarr(20)
;    case WidgetData.ioptions(5) of 
;      0: begin
;        WidgetData.typevector(2)=0 
;        for i=0,19 do begin
;          WidgetData.typevector(2)=WidgetData.typevector(2)+2^i
;          set(i)=1
;        endfor
;        WIDGET_CONTROL,WidgetData.mpb_pitch_ch,SET_VALUE=set 
;     end  
;      1: begin
;        WidgetData.typevector(2)=0 
;        for i=0,3 do begin
;          WidgetData.typevector(2)=WidgetData.typevector(2)+2^i
;          set(i)=1
;        endfor
;        WIDGET_CONTROL,WidgetData.mpb_pitch_ch,SET_VALUE=set 
;      end 
;      2: begin
;        WidgetData.typevector(2)=0 
;        for i=8,11 do begin
;          WidgetData.typevector(2)=WidgetData.typevector(2)+2^i
;          set(i)=1
;        endfor
;        WIDGET_CONTROL,WidgetData.mpb_pitch_ch,SET_VALUE=set 
;      end 
;      3: begin
;        WidgetData.typevector(2)=0 
;        for i=16,19 do begin
;          WidgetData.typevector(2)=WidgetData.typevector(2)+2^i
;          set(i)=1
;        endfor
;        WIDGET_CONTROL,WidgetData.mpb_pitch_ch,SET_VALUE=set 
;      end 
;      4:
;    endcase   
;ENDIF 
  
IF KEYWORD_SET(EK_SELECT) then begin
    set=bytarr(120)
    case WidgetData.ioptions(9) of 
      0: begin
        set(0:8)=1
        WidgetData.typevector(0)=0 & WidgetData.typevector(1)=0 
        for i=0,59 do if set(i) then $
          WidgetData.typevector(0)=WidgetData.typevector(0)+2ll^i 
        for i=60,119 do if set(i) then $
          WidgetData.typevector(1)=WidgetData.typevector(1)+2ll^(i-60)
        WIDGET_CONTROL,WidgetData.mpb_energy_ch,SET_VALUE=set 
     end  
      1: begin
        set(10:15)=1
        WidgetData.typevector(0)=0 & WidgetData.typevector(1)=0 
        for i=0,59 do if set(i) then $
          WidgetData.typevector(0)=WidgetData.typevector(0)+2ll^i 
        for i=60,119 do if set(i) then $
          WidgetData.typevector(1)=WidgetData.typevector(1)+2ll^(i-60)
        WIDGET_CONTROL,WidgetData.mpb_energy_ch,SET_VALUE=set 
      end 
      2: begin
        set(10:15)=1
        WidgetData.typevector(0)=0 & WidgetData.typevector(1)=0 
        for i=0,59 do if set(i) then $
          WidgetData.typevector(0)=WidgetData.typevector(0)+2ll^i 
        for i=60,119 do if set(i) then $
          WidgetData.typevector(1)=WidgetData.typevector(1)+2ll^(i-60)
        WIDGET_CONTROL,WidgetData.mpb_energy_ch,SET_VALUE=set 
      end 
      3: begin
        set(16:27)=1
        WidgetData.typevector(0)=0 & WidgetData.typevector(1)=0 
        for i=0,59 do if set(i) then $
          WidgetData.typevector(0)=WidgetData.typevector(0)+2ll^i 
        for i=60,119 do if set(i) then $
          WidgetData.typevector(1)=WidgetData.typevector(1)+2ll^(i-60)
        WIDGET_CONTROL,WidgetData.mpb_energy_ch,SET_VALUE=set 
      end 
      4:
    endcase       
ENDIF 
  
;  IF KEYWORD_SET(MERGED) THEN BEGIN
;    lExtNames='.fluxmerge'
;  ENDIF
  
;  IF KEYWORD_SET(PITCH) THEN BEGIN
;    lSatNames=lanl_geo_satlist(/PITCH) 
;    WIDGET_CONTROL,WidgetData.ls_lSatlist, SET_VALUE=[lSatNames]     
;    WIDGET_CONTROL,WidgetData.ls_rSatlist, SET_VALUE=[''] 
;    lExtNames='.pitch'
;    WIDGET_CONTROL,WidgetData.ls_lExtlist, SET_VALUE=[lExtNames]
;  ENDIF  
  
;  IF KEYWORD_SET(RES) THEN BEGIN

;  ENDIF
  
IF KEYWORD_SET(CHANNEL) THEN BEGIN
    ;first reset all buttons to empty
    button_ids=WidgetData.mpb_energy_ch_ids
    for i=0,119 do begin
      widget_control,button_ids(i), $
        SET_VALUE=strcenter(string(i,format="('empty ',i3.3)"),16)
      widget_control,button_ids(i),sensitive=0
    endfor     

    sat = sat_names(WidgetData.ioptions(10))
    IF WidgetData.ioptions(3) EQ 1 THEN MERGED = 1 ELSE MERGED = 0
    channels=r_lanl_geo_flux_converison(sat, MERGED = MERGED)
    list = channels.instrument+' '+channels.label &  n=n_elements(list)

    for i=0,(n-1) < 119 do begin
          widget_control,button_ids(i),SET_VALUE=strcenter(list(i),18)
          widget_control,button_ids(i),sensitive=1
    endfor  

    for i=n,119 do begin ;make extra channels insensitive
          widget_control,button_ids(i),$
            SET_VALUE=strcenter(string(i,format="('empty ',i3.3)"),18)
          widget_control,button_ids(i),sensitive=0
    endfor
        
    ;if the channel set switch is zero, select at least ist channel
    ;otherwise set what is selected
    set=bytarr(120) 
    if WidgetData.typevector(0)+WidgetData.typevector(1) eq 0 then begin
          set(0)=1
    endif else begin
         for i=0,59 do $
            if (WidgetData.typeVector(0) and 2ll^i) eq 2ll^i then $
              set(i)=1 else set(i)=0
         for i=60,119 do $
            if (WidgetData.typeVector(1) and 2ll^(i-60)) eq 2ll^(i-60) then $
              set(i)=1 else set(i)=0
    endelse
    WIDGET_CONTROL,WidgetData.mpb_energy_ch, SET_VALUE=set
ENDIF  
  
;if keyword_set(MLT) then begin
;      label=string(widgetData.options(5),format="(f4.1)")
;    WIDGET_CONTROL, widgetData.mlt_from_label, $
;      SET_VALUE=strformat(label,35,/center)
;    label=string(widgetData.options(6),format="(f4.1)")
;    WIDGET_CONTROL, widgetData.mlt_to_label, $
;      SET_VALUE=strformat(label,35,/center)
;end    

;  if keyword_set(RANGE_MLT) then begin
;    case widgetData.options(7) of
;      0: begin 
;        WIDGET_CONTROL, widgetData.mlt_from, SENSITIVE=0
;        WIDGET_CONTROL, widgetData.mlt_to, SENSITIVE=0
;        WIDGET_CONTROL, widgetData.mlt_from_label, SENSITIVE=0
;        WIDGET_CONTROL, widgetData.mlt_to_label, SENSITIVE=0
;      end
;      1: begin
;        WIDGET_CONTROL, widgetData.mlt_from, SENSITIVE=1
;        WIDGET_CONTROL, widgetData.mlt_to, SENSITIVE=1
;        WIDGET_CONTROL, widgetData.mlt_from_label, SENSITIVE=1
;        WIDGET_CONTROL, widgetData.mlt_to_label, SENSITIVE=1
;      end 
;    endcase  
;  end  
  
END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_geo_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written october 1995, Reiner Friedel
;*     modified september 1997, Torsten Heise
;******************************************************************************
PRO lanl_geo_pitch_panel_event, event

COMMON LANL_GEO_pitch_PANEL, WidgetData, lSatNames, rSatNames, lExtNames, rExtNames
COMMON PLOT_COMPOSER, mainwidgetData
COMMON lanl_geo, sat_names
  
WIDGET_CONTROL, GET_UVALUE=uval, event.id
  
CASE uval OF

    'mpb_sat': BEGIN
        WidgetData.ioptions(10)=event.value
        lanl_geo_pitch_refresh, /CHANNEL
    END

     'mpb_energy_ch': begin
        WIDGET_CONTROL,WidgetData.mpb_energy_ch, GET_VALUE=set
        WidgetData.typevector(0)=0  &  WidgetData.typevector(1)=0
        for i=0,59 do if set(i) then $
          WidgetData.typevector(0)=WidgetData.typevector(0)+2ll^i 
        for i=60,119 do if set(i) then $
          WidgetData.typevector(1)=WidgetData.typevector(1)+2ll^(i-60)
;        WIDGET_CONTROL,WidgetData.mpb_energy_sel, SET_VALUE=4  
        WidgetData.ioptions(9)=4
    END    
    
;    'mpb_energy_sel': begin
;      WidgetData.ioptions(9)=event.value
;      lanl_geo_pitch_refresh, /EK_SELECT
;    end  
    
;    'mpb_graph': begin
;      WidgetData.ioptions(7)=event.value
;      lanl_geo_pitch_refresh,/GRAPH
;    end
    
;    'mpb_pitch_act': WidgetData.ioptions(6)=event.value
    
    'mpb_pitch_plt': begin
      WidgetData.ioptions(8)=event.value
      lanl_geo_pitch_refresh, /PLOT_SELECT
      lanl_geo_pitch_refresh, /EK_SELECT
    end
    
;    'mpb_pitch_sel': begin
;      WidgetData.ioptions(5)=event.value
;      lanl_geo_pitch_refresh, /PA_SELECT
;    end
    
    'mpb_pitch_ch': begin
      WIDGET_CONTROL,WidgetData.mpb_pitch_ch, GET_VALUE=set
      WidgetData.typevector(2)=0
      index=where(set eq 1,c)
      if c ne 0 then for i=0, c-1 do $
        WidgetData.typevector(2)=WidgetData.typevector(2)+2l^index(i)
;      WIDGET_CONTROL,WidgetData.mpb_pitch_sel, SET_VALUE=4  
;      WidgetData.ioptions(5)=4
    end
    
;    'mpb_ltlab': WidgetData.ioptions(4)=event.value
    
;    'mpb_res' : WidgetData.ioptions(0)=event.value
    
    'mpb_caltype': WidgetData.ioptions(2)=event.value
    
;    'mpb_datasource': begin
;      WidgetData.ioptions(1)=event.value
;      lanl_geo_pitch_refresh, /RES
;    end  
    
;    'spb_mlt_range' : begin
;      widgetData.options(7)=event.value
;      lanl_geo_pitch_refresh, /RANGE_MLT
;    end      
;    'mlt_from'    : begin
;      widgetData.options(5)=event.value / 2.0 
;      lanl_geo_pitch_refresh, /MLT
;    end  
;    'mlt_to'   : begin
;      widgetData.options(6)=event.value / 2.0 
;      lanl_geo_pitch_refresh, /MLT
;    end  
   
    'mpb_datatype': BEGIN
        WidgetData.ioptions(3)=event.value
        case WidgetData.ioptions(3) of
            0: lanl_geo_pitch_refresh, /RES, /CHANNEL
            1: lanl_geo_pitch_refresh, /MERGED, /CHANNEL
            2: lanl_geo_pitch_refresh, /PITCH
        endcase  
        lanl_geo_pitch_refresh, /TYPE
    end

    'pb_cancel' : BEGIN
      mainWidgetData.NEEDTO_READ_DATA=1
      WidgetData=PAPCO_Panel_event(event,WidgetData)        
    END

    ELSE : BEGIN
        WIDGET_CONTROL, widgetData.fld_mu, GET_VALUE = X
        widgetData.soptions(0) = x
        WidgetData=PAPCO_Panel_event(event,WidgetData)        
    END
    
ENDCASE

END
