;******************************************************************************
;* PROCEDURE:     
;*      PRO DrawSlice_lanl_geo, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a slice for lanl_geo
;*      calling a separate window for the plot.
;*      slice_type in draw_lanl_geo must be set to +2 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    set if date needs to be read
;*                  the panel's data is read automatically, use this
;*                  to read in auxiliary data like eph
;*
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      December 1995, written by R.Friedel at Goddard
;******************************************************************************
PRO vert_slice_lanl_geo, panelNr, seconds, yValue, READDATA=READDATA, $
                         NOWIDGET = NOWIDGET, PANEL=PANEL, NUTITLE=NUTITLE
COMMON papco_vert_slice, slice_data
COMMON plot_composer, widgetData
COMMON slice_lanl_geo, time, ymat, utitle, uytitle, ych
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt

COMMON papco_color_names
cols=fixed_color_index

; thisPlot contains the full papco structure of plot information
thisPlot = widgetData.plotsDrawn(panelNr)
thisPlotInfo =widgetData.plotinfos(panelNr)
;get control variables for this panel
control = *thisPlotInfo.USR_PTR1

; select the index for the time chosen out of the time array
DiffTimes=ABS(time - seconds)        ; this sets timeIndex to the index with
result=MIN(DiffTimes, timeIndex)     ; t90(index) is closest to 'seconds'
  
time_str=papco_cursor_time(seconds,panelNr)
time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})

data_tai = tai_1 + time(timeIndex)
data_time_str = tai2utc(data_tai, /ecs)
data_time_str=str_sep(data_time_str,' ')  

; set plotPar structure  
plotpar={ytitle:uytitle, $
         xrange:[thisPlot.zscl_min,thisPlot.zscl_max], $
         yrange:[thisPlot.yscl_min,thisPlot.yscl_max], $
         ylog:1, color:1, yticks:0, xtickformat:"(f8.2)", psym:-4, $
         xlog:1,xticks:0,xminor:0,xtitle:'Energy [keV]'}

slice_utitle=utitle
  
x=10^((alog10(ych(*,0))+alog10(ych(*,1))) /2.0)
plotPar.color=cols(slice_data.overplot)  

;set the number of slices to do
IF slice_data.mult THEN nplot = slice_data.nplot >  1 ELSE nplot = 1

;set up multiple plot sub-windows if slice_data.oplot is NOT set
nx = 1
IF NOT slice_data.oplot THEN BEGIN
    IF nplot GT 1 THEN BEGIN    ;only do if more than one slice is set
        !p.position = 0 & !P.REGION = 0
        nx = fix(sqrt(nplot+1)) & ny = nx
        WHILE nx*ny LT nplot+1 DO nx = nx+1
        !p.multi = [0, nx, ny, 0, 0]
        !X.margin = [12, 1] & !Y.margin = [4, 3]
        skip = 0        
    ENDIF 
ENDIF ELSE BEGIN                ;single plot window
    !p.position=[0.1,0.1,0.9,0.9]
    panelset,[0,1,1]
ENDELSE 

FOR p = 0, nplot-1 DO BEGIN 
    
    ;select data
    y=transpose(ymat((timeIndex+p),*))
    IF p GT 0 THEN linestyle = 2 ;dashed line

    IF p EQ nx-1 THEN skip = 1
    IF nplot GT 1 THEN BEGIN
        plotPar.xtitle = 'Energy [keV] '+varprt(p)
        !p.multi(0) = (nx*ny)-(p+skip)
    ENDIF 
 
    IF slice_data.overplot EQ 0 THEN BEGIN 
        IF p EQ 0 THEN erase 
        plot, x, y, _extra=plotPar, LINESTYLE = linestyle
        IF p EQ 0 THEN FOR j = 0, n_elements(y)-1 DO $
            plots, [ych(j,0),ych(j,1)], [y[j], y[j]], $
              THICK = 2, LINESTYLE = linestyle, _extra=plotPar
        slice_data.bang_x = !x & slice_data.bang_y = !y
    ENDIF ELSE BEGIN  
        !x = slice_data.bang_x & !y = slice_data.bang_y
        oplot, x, y, color = plotPar.color
        IF p EQ 0 THEN FOR j = 0, n_elements(y)-1 DO $
            plots, [ych(j,0),ych(j,1)], [y[j], y[j]], $
              THICK = 2, LINESTYLE = linestyle, color = plotPar.color, psym = plotPar.psym
    ENDELSE 

ENDFOR 
  
slice_utitle=slice_utitle+data_time_str(0)+'!C'+data_time_str(1)
result=str_sep(slice_utitle,'!C')
for i=0,slice_data.rsl_lines do slice_utitle='!C'+slice_utitle
right_side_label,[0,1,1],slice_utitle, _extra=plotPar
slice_data.rsl_lines=slice_data.rsl_lines+n_elements(result)
  
if slice_data.overplot eq 0 then begin
    info1=''
    info2=string(time_str, yValue, format="('time: ',A20,'   Y-val: ',f12.5)")
    WIDGET_CONTROL, slice_data.lbl_slice1, SET_VALUE=info1
    WIDGET_CONTROL, slice_data.lbl_slice2, SET_VALUE=info2
ENDIF 
  
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO levelzero_lanl_geo, seconds, yvalue
;*                                     
;* DESCRIPTION:  
;*      This procedure calls the widget for level zero data access 
;*      for this plottype. 
;*      slice_type in draw_lanl_geo must be set to +4 to enable this feature.
;*
;* INPUTS:       
;*      seconds           T90 time at which mouse is clicked
;*      yvalue            y-value at which mouse is clicked
;*
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      levelzero_lanl_geo, seconds, yvalue
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel at Goddard
;******************************************************************************
pro levelzero_lanl_geo, seconds, yvalue

; The actual procedure is very dependent on what the level zero widget 
; application needs. An example is in $papco_MODULES/addpapco_swe_plots.pro.
; Provided as parameters to be passed to the new application, are the values
; of the mouse pointer when clicked. 
;
; The main widget remains active and can be switched to (e.g. to get a new
; time from the graph) by using a "Parent" button in the new application
; which then executes the line
;            current_appl=xregistered('papco_main') 
; to swittch back to the papco main calling widget. 

; call the levelzero widget.  
;   lanl_geo_levelzero, seconds, yvalue

; the following line is printed on your xterm as a check to see if papco
; correctly called this routine. If there is a problem make sure your naming 
; convention is O.K. : This routine MUST be called levelzero_lanl_geo, 
; where lanl_geo is the name of the panel you have set in papcoadd_lanl_geo.

    print, 'got to the levelzero interface  ',seconds, yvalue

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO WriteSlice_lanl_geo, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw a Write Paneldata slice for lanl_geo
;*      calling a seperate window for the widget
;*      slice_type in draw_ceppad must be set to +32 to enable this feature.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      READDATA    if set the data needs to be re-read from file
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      may 1997, written by R.Friedel
;******************************************************************************
PRO WriteSlice_lanl_geo, panelNr, seconds, yValue, READDATA=READDATA

  common papco_write_paneldata, slice_data
  common plot_composer, widgetdata
   
  common lanl_geo_data, input_header, input_data
  common shift_label, down
  
  down=0
  widget_control,/hourglass

  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  if keyword_set(INTMOD)  then intmod=INTMOD   else intmod=0
  if keyword_set(EXTMOD)  then extmod=EXTMOD   else extmod=5

; use the same typevector, swittch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
  thisPlot=widgetData.plotsDrawn(panelNr)
  type=thisPlot.typeVector
  
; this panel needs the original data from the panel, Call the papco
; routine to read this data if necessary.  
  if READDATA then papco_get_slice_data, panelNr
 
; select the index for the time chosen out of the time array
  DiffTimes=ABS(input_data.time - seconds)
                                       ; this sets timeIndex to the index 
  result=MIN(DiffTimes, tidx)          ; closest to 'seconds'
  
; get ephemeris info  
  rad=input_data(tidx).RADIUS
  lat=input_data(tidx).LATITUDE
  lon=input_data(tidx).LONGITUDE
   
; calculate mag coords (from UNILIB) for current time & eph position  
  mjdt_now=papco_cursor_time(seconds,panelNr,/MJDT)
  tai = utc2tai({mjd:mjdt_now.mjd, time:mjdt_now.t*1000})
  convert_t90_to_date, mjdt_now, year, day_oy, hour, minute, second,$
                       month, cmonth, day_om, /mjdt  
  time_date={iyear:year, imonth:month, iday:day_om, $
             ihour:hour, imin:minute, secs:second, amjd:0.0d}
;  pos={radius:rad,latitude:lat,longitude:lon}
  pos = [rad*6371.0, 90-lat, lon]
  kcompstr = 'SPH' &  kinstr = 'GEO'
  get_mag_unilib, tai, pos, kcompstr, kinstr, bvec, bequ, Lval, maglt, ilat, error, $
          intmod = intmod, extmod = extmod
     
;  get_mag_unilib, tai, pos, kcompstr, kinstr, $ ; inputs
;                    bvec, bequ, Lval, maglt, ilat, error, $         ; outputs 
;                    INTMOD=INTMOD, EXTMOD=EXTMOD,  $
;                    INT_NAME=INT_NAME, EXT_NAME=EXT_NAME, $
;                    LSTAR = LSTAR, KOUTSTR = KOUTSTR, $
;                    PITCH=PITCH, VERBOSE=VERBOSE,  HAVE_MDATE = HAVE_MDATE, $
;                    DRIFTSHELL = DRIFTSHELL

; these strings are user-definable output strings to label the slice window.
; The "write Panelinfo" window has 6 lines of user-definable text.   
  
  info1='LANL GEO ephemeris (geo-> rad, lat, lon, loc time):'
  lct_str=lanl_geo_eph_ticks(0,0,seconds)
  info2=string(rad, lat, lon, lct_str, format="(3(f8.2),a8)")
  
  info3='LANL GEO mag coords (BL-> model B, L, MLT, MLAT):'
  
;  info4=string(fbm, flm, xmlt, xlat, format="(e10.3,3(f8.2))")
  info4 = string(bvec(3), lval, maglt, ilat, format="(4(f8.2))")
  info5='LANL GEO satellite ID:'
  info6=input_header.SAT_ID  
  
; Time and y-value of mouse click are automatically put into labels.
; The calls below put your info strings in the provided user labels (max 6).
;
; NB NB NB:
; However, only info2, info4, info6 are written to file in the form
; info2+info4+info6.
;
; The idea is that info1 will have the labels for the data in info2, info3 the
; labels for the data in info4, and info5 the labels for the data in info6.
; Time and y-value of mouse click are written first, then info2+info4+info6
; followed by a user defined comment from the Write Paneldata window.

  WIDGET_CONTROL, slice_data.lbl_slice1, SET_VALUE=info1
  WIDGET_CONTROL, slice_data.lbl_slice2, SET_VALUE=info2
  WIDGET_CONTROL, slice_data.lbl_slice3, SET_VALUE=info3
  WIDGET_CONTROL, slice_data.lbl_slice4, SET_VALUE=info4
  WIDGET_CONTROL, slice_data.lbl_slice5, SET_VALUE=info5
  WIDGET_CONTROL, slice_data.lbl_slice6, SET_VALUE=info6

END
;*----------------------------------------------------------------------
;* 
;*
;*-----------------------------------------------------------------------

PRO USER_PA_Slice_lanl_geo,panelNr,seconds,yValue, $
    READDATA=READDATA, CLOSE=CLOSE, SOPA = sopa, NOWIDGET = nowidget
;  user_mpa_slice, panelNr, seconds, yValue, 'pitchangle', CLOSE=CLOSE
  COMMON time, xut1, xut2
  COMMON get_error, get_err_no, get_err_msg  
  COMMON USER_GEO_PA_Slice_lanl_geo_pitch, slice_widget
  COMMON PLOT_COMPOSER, widgetData
  COMMON papco_color_names
  
;  As long as you stay in one panel with the 
; mouse, the data is not read again. When you switch panels, the new data is 
; read. 
  
  COMMON lanl_geo
  COMMON lanl_geo_data, geo_pitch_header, geo_pitch_data
  COMMON lanl_geo_slice, utitle, uytitle, uztit
  COMMON lanl_geo_output,xarr, yarr, zmat
  COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  COMMON yscale, yscl           ;man/auto yscaling
  COMMON zscale, zscl           ;man/auto zscaling
  
; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
  plotinfo =widgetData.plotinfos(panelNr)
  plotdrawn=widgetData.plotsDrawn(panelNr)
  control = *plotinfo.USR_PTR1

  
  
  IF papco_Draw_CheckResult(PlotInfo.panelKind, panel, orbitNr, $
                              PlotInfo.typeVector, PlotInfo.channel) $
                              THEN IsRead=1 ELSE IsRead=0

  ind = where(tag_names(geo_pitch_data) EQ 'COUNTS')
  IF ind(0) EQ -1 THEN isread = 0 ELSE isread = 1

  get_data_call = 'r_lanl_geo_pitch'
  IF NOT isread THEN call_procedure, get_data_call, PlotInfo
  
if keyword_set(OVERPLOT) then overplot=OVERPLOT else overplot=0
  IF NOT keyword_set(low_en) THEN low_en = 13
  
  if keyword_set(NOWIDGET) then BEGIN
     slice_widget={resolution:fix(control.pa_res),$
                 low_en:0, $
;                  average:fix(control.mus), $
;                  pitch:fix(control.pa_ctr_3dd), $
;                  label:fix(control.pa_lbl_3dd), $
;                  have_pitch:0, $
                  overplot:overplot, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}    
     goto, plotonly
 ENDIF

; see if slice window needs to be closed --------------------------------------
 if keyword_set(CLOSE) then begin
    if xregistered('USER_PA_Slice_lanl_geo_pitch') then $
      if n_elements(slice_widget) gt 0 then begin
       if slice_widget.base ne 0 then $
          widget_control, slice_widget.base, /destroy
       slice_widget.base=0
    endif
    return
 ENDIF 
 
; if slice window exists, bring it in front. ELSE, create it ------------------
 if not xregistered('USER_PA_Slice_lanl_geo_pitch') then begin
    
    slice_base=widget_base(column=1, group=widgetdata.main_base, $
                           title='PA_Slice_lanl_geo_pitch')
    
;-- setup draw area -----------------------------------------------------------
    draw_base=WIDGET_BASE(Slice_base, COLUMN=1, /FRAME)
    dw_Slice  =WIDGET_DRAW(draw_base, XSIZE=600, YSIZE=500)
    
;-- set up label fields for user data -----------------------------------------
    user_label_base=WIDGET_BASE(slice_base, COLUMN=1, /FRAME)
    lbl_slice1=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
    lbl_slice2=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
   
;-- set up some additional options for this slice -----------------------------
    Opt_base = WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    ef_res = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Pitch angle resolution', $
                     VALUE = control.pa_res, $
                     UVALUE='ef_res')

;    ef_low_en = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
;                     TITLE = 'Lowest energy channel:', $
;                     VALUE = low_en, $
;                     UVALUE='ef_low_en')
;

    
;-- set up buttons for the Actions --------------------------------------------
    Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    pb_help=WIDGET_BUTTON(Butt_Base, Value=strcenter('Help', 12), $
                          UVALUE='pb_help')
    
    pb_print=WIDGET_BUTTON(Butt_Base, Value=strcenter('Print', 12), $
                           UVALUE='pb_print')
    pb_output=WIDGET_BUTTON(Butt_Base, Value=strcenter('Output', 12), $
                           UVALUE='pb_output')    

    pb_done=WIDGET_BUTTON(Butt_Base, Value=strcenter('Close', 12), $
                          UVALUE='pb_done')

    slice_widget={rsl_lines:0, $ 
                 resolution:fix(control.pa_res), $
                 low_en:0, $
                  base:slice_base, $
                  dw_slice:dw_slice, $
		  lbl_slice1:lbl_slice1, $
                  lbl_slice2:lbl_slice2, $
                  pb_output:pb_output, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}
    
    widget_control, slice_base, /realize
    xmanager, 'USER_PA_Slice_lanl_geo_pitch', slice_base, /JUST_REG
    
 ENDIF                          ; end construct slice window
 
 IF keyword_set(NOPLOT) THEN return
 
 WIDGET_CONTROL,/HOURGLASS

;--- now, we need to get the data --------------------------------------------
;    but we will only read the data, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).
 IF slice_widget.lastPanelDrawn NE panelNr THEN begin
    papco_get_slice_data, panelNr
    READDATA=1
 endif else READDATA=0  
  
;--- prepare the drawing widget for plotting ----------------------------------
 if !D.name eq 'X' then begin
    widget_control, slice_widget.dw_slice, get_value=drawindex
    wset, drawindex
 endif  
 
; set the panel window for the slice to full window
 !p.position=[0.1,0.1,0.75,0.95]  
 IF keyword_set(PANEL) THEN panel = panel ELSE panel = [0, 1, 1]
 panelset, panel
 
plotonly:
 
;--- store the current coordinate system --------------------------------------
 old_x = !x  &  old_y = !y  &  old_p = !p
 
; NOW do plot into draw window!!!
; ---------------------------------------------
 
; load the normal colortable
 papco_set_subtable, 0
 
 
; select the index for the time chosen out of the time array
 
;make time of middle of averaged intervals:
 t = (xarr(*, 0)+xarr(*, 1))/2.0
 DiffTimes=ABS(t - seconds)     ; this sets timeIndex to the index with
 result=MIN(DiffTimes, timeIndex) ; time(timeIndex,0) closest to 'seconds'

tai1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
result = min(abs(tai1-geo_pitch_data.tai),addind) &  TimeIndex = TimeIndex+addind

;check for nearest time found. Needs to be within diff*2
 diff = 600

; should return the correct time.
 time_str=papco_cursor_time(seconds,panelNr)
 time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)
 if result gt diff*2 then BEGIN
    message, 'No data found at: '+time_str,/cont
    xl=!p.position(0)  &  xr=!p.position(2)
    yb=!p.position(1)  &  yt=!p.position(3)
    plots, [xl, xr, xr, xl, xl], [yb, yb, yt, yt, yb], COLOR=1, /NORMAL
    goto, no_plot
 ENDIF


; select the index for the energy chosen out of the time array

 en_ind = where(yValue GT yarr(*, 0) AND yValue LT yarr(*, 1))
 

 sc_id = control.sc_id

 chan_arr = bytarr(120)
 FOR i=0,59 DO IF (control.ek1 AND 2ll^i) EQ 2ll^i THEN chan_arr(i)=1 $
              ELSE chan_arr(i)=0
 FOR i=60,119 DO IF (control.ek2 AND 2ll^(i-60)) EQ 2ll^(i-60) THEN chan_arr(i)=1 $
                ELSE chan_arr(i)=0
 sat_info = r_lanl_geo_flux_converison(sat_names(control.sc_id))
 chan_ind = where(chan_arr EQ 1)
 species = sat_info(chan_ind(0)).species
 instrument = sat_info(chan_ind(0)).instrument

 sp_inst_ind = where(sat_info.species EQ species AND sat_info.instrument EQ instrument)
 en_ind = en_ind+min(sp_inst_ind)
; IF sat_info(chan_ind(0)).species EQ 'electrons' THEN species =  0 ELSE species = 1
; species = control.species

;IF NOT keyword_set(pa_res) THEN pa_res = 5

 

;  mpa_energy_info  = r_lanl_mpa_channels(sc_id-8,species)
;  mpa_center_en = reverse((mpa_energy_info(*, 0)+mpa_energy_info(*, 1))/2)

;  IF species EQ 0 THEN BEGIN  ; electrons
;     species = 'electrons'

  det_arr = bytarr(3)
  FOR i = 0, 2 DO IF (control.det_choice AND 2ll^i) EQ 2ll^i THEN det_arr(i) = 1 $ 
         ELSE det_arr(i) = 0
  pol_ind = where(det_arr EQ 1, pol_det)
  IF pol_det EQ 0 THEN BEGIN 
     pol_det = 1
     pol_ind = 1
  ENDIF 
     geo_pa = geo_pitch_data(timeIndex).alpha(pol_ind, *)
     geo_data = geo_pitch_data(timeIndex).counts(pol_ind, *, en_ind)
;  ENDIF ELSE BEGIN  ; ions
;     species = 'ions'
;     mpa_pa = mpa_pitch_data(timeIndex).alpha(*, *)
;     mpa_data = mpa_pitch_data(timeIndex).p_unbinned(en_ind, *, *)
;  ENDELSE 

  nodata = -1e31


  pa_rng = papco_pitch_bin(control.pa_res, pa_idx = pa_idx) 
  pa_bins = n_elements(pa_rng)/2
  pa_data = fltarr(pa_bins)

  geo_data = reform(geo_data,pol_det*32)
  geo_pa = reform(geo_pa, pol_det*32)

  sort_ind = sort(geo_pa)

  sxarr = geo_pa(sort_ind)
  pa_data = geo_data(sort_ind)

;  FOR j = 0, pa_bins-1 DO BEGIN 
;     ind = where(geo_pa(*) GT pa_rng(j, 0) AND geo_pa(*) LE pa_rng(j, 1) $
;                                                 AND geo_pa NE nodata, in_box)
;     IF in_box GT 0 THEN pa_data(j) = total(geo_data(ind))/in_box ELSE pa_data(j) = nodata
;  ENDFOR
  

;  sxarr = pa_rng
  xlog = 0 &  ylog = 1
  
;  lowen = strtrim(string(mpa_energy_info(40-en_ind, 0)), 2) &  lowen = lowen(0)
;  highen = strtrim(string(mpa_energy_info(40-en_ind, 1)), 2) & highen = highen(0)
  
  utitle=sat_names(sc_id)+' '+species+'!C'+sat_info(en_ind).label
;lowen+'-'+                highen+' eV'
  ztitle = ''
  xtitle = 'Pitch Angle (deg)'
  ytitle = 'Counts'
  xticks = 7
  xtickv = [0, 30,60, 90,120,150, 180]
  
  
  zscl(panelNr,3) = plotdrawn.ZLINLOG
  
  
; use the old zscaling if available, else calculate
  IF control.pl_type LT 4 THEN BEGIN 
     yst=plotdrawn.ZSCL_MIN  
     yen=plotdrawn.ZSCL_MAX 
  ENDIF  ELSE BEGIN
;do automatic scaling for full dynamic range
     papco_autorange,pa_data,yst,yen, log=zscl(panelNr,3), exclude=0, nodata=nodata
     
  ENDELSE  
  
  
  
  
  
;yst=plotdrawn.YSCL_MIN  &  yen=plotdrawn.YSCL_MAX
;yst = syarr(0, 0) &  yen = syarr(n_elements(syarr(*, 0))-1, 1)
;  xst = sxarr(0, 0) &  xen = sxarr(n_elements(sxarr(*, 0))-1, 1)
  xst = 0 &  xen = 180
  papco_y_label, yst, yen, log=ylog
  
  extra_plotPar={xrange:[xst,xen], yrange:[yst,yen],  $
               xlog:xlog,ylog:zscl(panelNr,3), $
               color:black,xstyle:1, ystyle:1, $
               xtickformat:'',xtitle:xtitle,xticklen:0.01,$
               xticks:xticks, xminor:0, xtickv:xtickv, yticks:0,$
               yminor:0, ytickv:0,ytickformat:'', ytitle:ytitle}

  if keyword_set(NOWIDGET) then begin
     extra_plotPar.xtickformat='noticks'
     extra_plotPar.xtitle=''
     extra_plotPar=create_struct(extra_plotPar, {ytickformat:'noticks'})
  endif ELSE erase 
  

; nodata specifies the "nodata" flag
; in your data.

userx=[-0.5,0.0,0.5] & usery=[0.0,1.0,0.0] & usersym,userx,usery,/fill
;sxarr = (sxarr(*, 0)+sxarr(*, 1))/2.

PLOT, sxarr, pa_data, ytitle=uztitle, title = title, $
   psym=4,  symsize = 0.75, _extra = extra_plotPar

;papco_plot_colorspec, zmat, sxarr, syarr, nodata=nodata, _extra=extra_plotPar  

;papco_color_bar, _extra=extra_plotPar


no_plot:



if keyword_set(NOWIDGET) then begin
    if NOWIDGET eq 2 then begin
      panelset, PANEL  ;set to normal panel size!
      ;papco_color_bar, _extra=extra_plotPar
      right_side_label,panel,utitle,/rot90
    ENDIF
    ;restore the coordinate-system
    !x = old_x  &  !y = old_y  &  !p = old_p
    return
endif  
 

; plot y-axis label at left of plot. Use scalable routine!  
;left_side_label,panel,uytitle,/rot90

  
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

right_side_label,panel,utitle,/rot90
  
; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the papco philosophy then the time axis is in 
; running seconds since the start of the start day.
   
info1='This string not used yet'
info2=string(time_str, yValue, format="('time: ',A20,'   Y-value: ',f9.2)")
   
; these calls put your info strings in the provided labels of the slice widget
WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2 
  
;--- restore the coordinate-system --------------------------------------------
!x = old_x  &  !y = old_y  &  !p = old_p 
  
;--- keep old panel number- used for detecting panel changed of mouse ---------
slice_widget.lastPanelDrawn = panelNr   

END 
;******************************************************************************
;* PROCEDURE:
;*      PRO USER_PA_Slice_lanl_geo_pitch_Event, Event
;*
;* DESCRIPTION:
;*	This slice displays a PA slice through the data, when the user clicks
;*      in the drawing-widget. This procedure handles events for that window.
;*
;* INPUTS:
;*	Event	 	an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     written November 2003, Arne Aasnes
;******************************************************************************
PRO USER_PA_Slice_lanl_geo_pitch_Event, Event

  COMMON USER_GEO_PA_Slice_lanl_geo_pitch, slice_widget
  COMMON plot_composer, widgetdata
  COMMON slice_calls, slice_calls

  widget_control,event.id,get_uvalue=ev
  
  CASE ev OF 
     'ef_res': BEGIN 
        slice_widget.resolution = Event.value 
;        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
;        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
;        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
;        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue
   end 
;     'ef_low_en':BEGIN 
;        slice_widget.low_en = Event.value
;        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
;        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
;        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
;        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue
;     end 
     'spb_pitch': slice_widget.pitch = event.select
     'spb_label': slice_widget.label = event.select
     'pb_done' :  widget_control, slice_widget.base, /destroy
     'pb_help':   xdisplayfile, papco_helpfile('papco_slice.help'), $
                  title='papco online-help', group=event.top
     'pb_print':  begin         ;loop through slices for all panels if needed
        slice_widget.rsl_lines=0
        for i=0,n_elements(slice_calls)-1 do begin
           close_dev=[i,n_elements(slice_calls)-1]
           papco_printmenu_event,slice_calls(i), CLOSE_DEV=close_dev
        endfor  
     end
     'pb_output': begin
                                ;establish error handler
        if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
        if error_status ne 0 then begin ;call error handler
           papco_error_handler, $
          error_status, proc_call, '', [0,0,0], /NO_ERR_DRAW
           error_status=0 & catch, /CANCEL ;disable error catching
           return
        endif
        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue,/OUTPUT
        catch, /CANCEL          ;disable error catching 
     end  
  else : begin & end
  ENDCASE 
END  
