;*******************************************************************
;*  PROCEDURE
;*      plot_pa,  NUMBERED = numbered, TIME = time,  ENERGY = energy,  $
;*                WRITE_FILE = write_file, SIN_FIT = sin_fit
;*
;*  DESCRIPTION
;*       This procedure plots the pitch angle data from SOPA, which
;*       must be contained in the common block lanl_sopa_pitch_data
;*       (use a_lanl_geo_pitch to read the data)  
;*
;*  KEYWORDS:
;*       NUMBERED: plots the data points as numberes according to 
;*                 azimuthal spin phase 
;*       TIME:     index of time to be plotted
;*       ENERGY:   energy channel to be plotted (10 channels for
;*                 electrons, 12 for protons)  
;*       WRITE_FILE: set this to write image to png file
;*       SIN_FIT:  if set, draws a p(0)*sin(pa)^p(1) fit to the pa
;*                 distribution
;* 
;*  DEPENDS ON:  ~/IDL/lib/mpfit
;*               ~/IDL/lib/mpfitfun
;*               ~/IDL/lib/functions
;*               .r ~/papco_user/papco_user_modules/lanl_geo/plot_read/lanl_geo_pitch_tools
;*       
;*  MODIFICATION HISTORY:
;*       written July 2003, Arne Aasnes at LANL
;*******************************************************************
PRO plot_pa, NUMBERED = numbered, TIME = time,  ENERGY = energy,  $
             WRITE_PNG = write_png,WRITE_PPM = write_PPM, SIN_FIT = sin_fit, symline = symline,  FILE_NAME = file_name, MAX_SCL = MAX_SCL

  COMMON papco_color_names
  COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data
  
  IF NOT keyword_set(ENERGY) THEN energy = 0 
  IF NOT keyword_set(TIME) THEN time = 0

  sat_info = r_lanl_geo_flux_converison(lanl_geo_pitch_data(0).csat_id)
  Instrument = sat_info(energy).instrument
  species = sat_info(energy).species
  IF Instrument EQ 'ESP'  THEN alpha = lanl_geo_pitch_data.alpha_esp $
               ELSE alpha = lanl_geo_pitch_data.alpha
  
  dim = size(alpha, /dimensions)
  counts = lanl_geo_pitch_data.counts
  dim = size(counts, /dimensions)
  counts_st =  reform(counts, dim(0)*dim(1), dim(2), dim(3))
  cleanplot
  !p.position = 0
  !p.background = white
  !y.margin = [7, 4] & !x.margin = [8, 5]
  window, title = 'Sopa Counts vs pitch angle'
  erase 
  

  
  title_str = instrument+' '+species+' counts vs Pitch Angles '+lanl_geo_pitch_data(0).csat_id+'!C'+ $
         'Energy: '+sat_info(energy).label+$
;        ' !C energy channel:'+strcompress(string(energy+1),/remove_all)+ $
   ' date:'+lanl_geo_pitch_data(0).date+ ' time:'+ $
   strmid(strcompress(string(lanl_geo_pitch_data(time).uthours),/remove_all), 0, 4)+' hr'
  ;; trying to calibrate
;  counts(2, *, *, *) = 1.3*counts(2,*,*,*)
;  counts(0, *, *, *) = 1.05*counts(0, *, *, *)
  
  max_y = ceil(1.2*max(counts(*, *, energy, time))) &  max_y = max_y(0)
  IF keyword_set(MAX_SCL) THEN max_y = max_scl
  plot,[0,180],[0, max_y], $
   color = black, ytitle = 'counts', xtitle = 'Pitch angle (deg)', title = title_str, $
    ystyle = 9, xstyle = 9, /nodata
IF keyword_set(symline) THEN oplot, [90, 90], [0, 0.95*max_y], color = black, linestyle = 1
IF keyword_set(numbered) THEN BEGIN 
  FOR i = 0, 31 DO xyouts, alpha(0, i, time),counts(0, i, energy, time), $
     strcompress(string(i+1), /remove_all), color =red, /data, alignment = 0.1, charsize = 0.9
  FOR i = 0, 31 DO xyouts, alpha(1, i, time),counts(1, i, energy, time), $
     strcompress(string(i+1), /remove_all), color =green, /data, alignment = 0.1, charsize = 0.9
  FOR i = 0, 31 DO xyouts, alpha(2, i, time),counts(2, i, energy, time), $
     strcompress(string(i+1), /remove_all), color =blue, /data, alignment = 0.1, charsize = 0.9
ENDIF ELSE BEGIN  
  oplot, alpha(0, *, time),counts(0, *, energy, time), color =red, psym = 1
  oplot, alpha(1, *, time),counts(1, *, energy, time), color =green, psym = 1
  oplot, alpha(2, *, time),counts(2, *, energy, time), color =blue, psym = 1
ENDELSE 

IF keyword_set(sin_fit) THEN BEGIN 
   COMMON  fit_result, result, fit_count, fit_alpha
   alpha_st =  reform(alpha, dim(0)*dim(1), dim(2))
   pa_fit, alpha_st(*, time), counts_st(*, energy, time)

   oplot, fit_alpha, fit_count, color = black

   print, 'Params (p(0)*sin(x*p(1))^p(2)):'+string(result(0))+string(result(1))
ENDIF 

x0 = 0.75
x1 = x0+0.03
y0 = 0.88
ydiff = 0.025
xdiff = 0.01

plots, [x0, x1], [y0, y0], color = red, /normal
plots, [x0, x1], [y0-ydiff, y0-ydiff], color = green, /normal
plots, [x0, x1], [y0-2*ydiff, y0-2*ydiff], color = blue, /normal
xyouts, x1+xdiff, y0, 'Det 1 (30 deg)', /normal, color = black
xyouts, x1+xdiff, y0-ydiff, 'Det 2 (90 deg)', /normal, color = black
xyouts, x1+xdiff, y0-2*ydiff, 'Det 3 (120 deg)', /normal, color = black

spawn, 'echo $user', idnt
xyouts, 0.01, 0.02, 'Generated by:  '+idnt+', '+systime(0), /normal, color = black
IF keyword_set(WRITE_PNG) THEN BEGIN 
   IF keyword_set(FILE_NAME) THEN plotname = file_name ELSE BEGIN 
      path = '~/lanl_geo/sopa/plots/'
      plotname = path+'PA_'+lanl_geo_pitch_data(0).csat_id+'_'+lanl_geo_pitch_data(0).date+'_'+ $
      strmid(strcompress(string(lanl_geo_pitch_data(time).uthours),/remove_all), 0, 5)+ $
      'hr_'+sat_info(energy).label+'.png'
   ENDELSE 
   print, 'Writing '+plotname+' to file'
   
   image = tvrd(true = 1)
   write_png, plotname, image   
ENDIF 
IF keyword_set(write_PPM) THEN BEGIN 
      IF keyword_set(FILE_NAME) THEN plotname = file_name ELSE BEGIN 
      path = '~/lanl_geo/sopa/plots/'
      plotname = path+'PA_'+lanl_geo_pitch_data(0).csat_id+'_'+lanl_geo_pitch_data(0).date+'_'+ $
      strmid(strcompress(string(lanl_geo_pitch_data(time).uthours),/remove_all), 0, 5)+ $
      'hr_'+sat_info(energy).label+'.ppm'
   ENDELSE 
   print, 'Writing '+plotname+' to file'
   
   image = tvrd(true = 1)
   write_ppm, plotname, image 
   
ENDIF 
END 


PRO pa_fit,alpha, counts, cust = cust

  COMMON fit_result, result, fit_count, fit_alpha
  
  start = [ double(max(counts)),1d,0d]
  expr = 'p(0)*sin(x*p(1))^p(2)+p(3)'
  func = 'pa_func'
  IF keyword_set(cust) THEN BEGIN 
     func = 'pa_cust_func' 
     expr = 'p(0)*sin(x*p(1)+p(2))^p(3)+p(4)'
     start = [double(max(counts)*0.9), 2d, double(max(counts)*0.1),0.1d]
  ENDIF 
;func = 'pa_cust_func'
  rerr = sqrt(abs(counts))
  
  result = mpfitfun(func, alpha, counts, rerr, start)
  
  fit_alpha = findgen(181)
  IF keyword_set(cust) THEN fit_count = pa_cust_func(fit_alpha, result) ELSE $
     fit_count = pa_func(fit_alpha, result)
end
;; T1='20020811 00:00:00' & T2='20020812 00:00:00'

 

;; routine to make a ascii file from magnetic field direction
;; the routine readt 
PRO mpa_magdir, SAT_ID = sat_id, T1 = t1, T2 = t2


  COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data

  IF keyword_set(sat_id) AND keyword_set(t1) AND keyword_set(t2) THEN a_lanl_geo, SAT_ID = sat_id,  t1 = t1,  t2 = t2
  IF n_elements(lanl_geo_pitch_data) NE  0 THEN BEGIN 
     filename = '~/lanl_geo/mpa_10min_fielddir_'+lanl_geo_pitch_data(0).date+'_'+sat_id+'.txt'
     
     geo_elements = n_elements(lanl_geo_pitch_data)
     openw, unit, filename, /get_lun
     printf, unit, 'Time (decimal of day)  Theta    Phi'
     FOR i = 0, geo_elements-1 DO printf,unit, lanl_geo_pitch_data(i).uthours, lanl_geo_pitch_data(i).mag_dir(0), lanl_geo_pitch_data(i).mag_dir(1)

     close, unit
  ENDIF 
END

PRO pa_movie, movie_name = movie_name ,ENERGY= energy, _extra = _extra_dat, MPA = mpa

  COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data
  COMMON lanl_mpa_pitch_data, mpa_header, mpa_data
IF NOT keyword_set(movie_name) THEN movie_name = '.output.mpeg'
IF keyword_set(energy) THEN energy = energy ELSE energy = 1

tmp_dir = '~/tmp/movie_tmp/'
spawn, 'mkdir '+tmp_dir

numb = strcompress(indgen(2000), /remove_all)
numb(where(numb LT 10)) = '0'+numb(where(numb LT 10))
numb(where(numb LT 100)) = '0'+numb(where(numb LT 100))
numb(where(numb LT 1000)) = '0'+numb(where(numb LT 1000))

IF not keyword_set(mpa) THEN BEGIN 
   ndat = n_elements(geo_data)
   max_scl = 1.1*max(geo_data.counts( *, *, energy))
   FOR j = 0, ndat-1 DO plot_pa, time = j,file_name = tmp_dir+'tmp.'+numb(j)+'.png', /write_png, energy = energy, $
_extra = _extra_dat, max_scl =max_scl
ENDIF ELSE BEGIN 
   ndat = n_elements(mpa_data)
   max_scl = 1.1*max(mpa_data.e_unbinned(energy, *, *))
   FOR j = 0, ndat-1 DO mpa_plot_pa, time = j,file_name = tmp_dir+'tmp.'+numb(j)+'.png', /write_png, energy = energy, $
_extra = _extra_dat;, max_scl =max_scl

ENDELSE  

make_mpg, prefix = tmp_dir+'tmp.', suffix = 'png', nstart = 0, nend = ndat-1, digits = 4,dims = [150, 120], format = 1, frame_rate = 1, $
 mpeg_file = movie_name
stop

;tmp_inp_info = 'ppmtompeg_info'
;openw, 1, tmp_dir+tmp_inp_info
;printf, 1, 'PATTERN IBBPBBPBBPBBPBB'
;printf, 1, 'OUTPUT '+file_name
;printf, 1, 'INPUT_DIR '+tmp_dir
;printf, 1, 'INPUT '
;printf, 1, 'tmp.*.ppm ['+numb(1)+'-'+numb(ndat-1)+']' 
;printf, 1, 'END_INPUT'
;printf, 1, 'BASE_FILE_FORMAT PPM'
;printf, 1, 'INPUT_CONVERT *'; no conversion necessary
;printf, 1, 'GOP_SIZE 15'
;printf, 1, 'SLICES_PER_FRAME 100'
;printf, 1, 'PIXEL HALF'
;printf, 1, 'RANGE 100'
;printf, 1, 'PSEARCH_ALG LOGARITHMIC'
;printf, 1, 'BSEARCH_ALG SIMPLE'
;printf, 1, 'IQSCALE 100'
;printf, 1, 'PQSCALE 100'
;printf, 1, 'BQSCALE 100'
;printf, 1, 'REFERENCE_FRAME ORIGINAL'
;printf, 1, '

;close, 1

;spawn, 'ppmtompeg '+tmp_dir+tmp_inp_info

;spawn,'rm -r '+tmp_dir

end

PRO plot_dist, mpa_time, species = species, sat_id = sat_id, low_en = low_en, $
               MPA = mpa,GEO = geo, max_val = max_val, min_val = min_val, EN_PA = EN_PA, PA_RES = pa_res

;;---------------------------------------------------------------
;; color plot of plasma density as function of v_par and
;; v_perp. Alternatively, countour plot. 
;;
;; input: mpa_time 
;;        species (0: electrons, 1: protons)
;;
;; Author: AaA, at LANL Nov, 2003 (lousy results so far)
;;---------------------------------------------------------------

  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data
  COMMON lanl_geo_data, geo_pitch_header, geo_pitch_data
  COMMON lanl_mpa,  sat_names, sat_codes,  select_names, select_frmts,dat_names, lanl_mpa_control
;  COMMON lanl_mpa_channels, mpa_yarr, mpa_zmat
;  COMMON lanl_geo_channels, geo_yarr, geo_zmat

  IF NOT keyword_set(species) THEN species = 0
  IF NOT keyword_set(sat_id) THEN BEGIN 
     IF n_elements(mpa_pitch_data) NE 0 THEN sat_id = mpa_pitch_data(0).csat_id $
     ELSE sat_id = sat_names(2) ;'1991-080'
  ENDIF 
  IF NOT keyword_set(low_en) THEN low_en = 13 &  mpa_chan = 40-low_en
  IF NOT keyword_set(pa_res) THEN pa_res = 10
  sc_id = where(sat_id EQ sat_names) &  sc_id = sc_id(0)

; find the geo time that is closest to the given (mpa) time
  geo_diff = min(abs(mpa_pitch_data(mpa_time).tai-geo_pitch_data.tai), geo_time)

;; get the channel information
  geo_sat_info = r_lanl_geo_flux_converison(sat_id)
  mpa_energy_info  = r_lanl_mpa_channels(sc_id,species)
  geo_center_en = (geo_sat_info.ek_low+geo_sat_info.ek_high)/2*1000. ; convert to eV
  mpa_center_en = reverse((mpa_energy_info(*, 0)+mpa_energy_info(*, 1))/2)



;; get the right species data
  IF species EQ 0 THEN geo_chan_ind = where(geo_sat_info.species EQ 'electrons' $
                                        AND geo_sat_info.instrument NE 'ESP', geo_chan) $
    ELSE geo_chan_ind = where(geo_sat_info.species EQ 'ions' $
                                        AND geo_sat_info.instrument NE 'ESP', geo_chan) 

  IF species EQ 0 THEN BEGIN 
     mpa_pa = mpa_pitch_data(mpa_time).e_mag(0:mpa_chan-1, *, *)
     mpa_data = mpa_pitch_data(mpa_time).e_unbinned(0:mpa_chan-1, *, *)
  ENDIF ELSE BEGIN 
     mpa_pa = mpa_pitch_data(mpa_time).p_mag(0:mpa_chan-1, *, *)
     mpa_data = mpa_pitch_data(mpa_time).p_unbinned(0:mpa_chan-1, *, *)
  ENDELSE 

  
  ; taking away the top two energy channels in sopa
  geo_chan_ind = geo_chan_ind(0:geo_chan-3)
  geo_chan = geo_chan-2
  

;; convert geo data to flux (from counts)

  flux_conv = fltarr(31) &  flux_conv(geo_chan_ind) = 1
  sopa_calibration, sat_id, flux_conv
  eflux_conv = flux_conv/geo_sat_info.fcf &  eflux_conv = eflux_conv(geo_chan_ind)
  
  geo_data = geo_pitch_data(geo_time).counts(*, *, geo_chan_ind)* $
             rebin(reform(eflux_conv, 1, 1, geo_chan), 3, 32, geo_chan)

;; convert mpa data to flux (from energy flux)
  IF NOT keyword_set(en_pa) then  mpa_data = mpa_data/rebin(reform(mpa_center_en(0:mpa_chan-1), $
                                                                   mpa_chan, 1, 1), mpa_chan, 6, 24)


  geo_pa = geo_pitch_data(geo_time).alpha


  IF keyword_set(en_PA) THEN BEGIN 
     IF keyword_set(geo) THEN en_chans = mpa_chan+geo_chan ELSE en_chans = mpa_chan
     pa_rng = papco_pitch_bin(pa_res, pa_idx = pa_idx) 
     pa_bins = n_elements(pa_rng)/2
     pa_en_data = fltarr(en_chans,pa_bins)
;     pa_en_counts = fltarr(en_chans,n_elements(pa_rng))
     mpa_data = reform(mpa_data, mpa_chan,6*24)
     mpa_pa = reform(mpa_pa, mpa_chan,6*24)
     FOR i = mpa_chan-1,0, -1 DO BEGIN
        FOR j = 0, pa_bins-1 DO begin
           ind = where(mpa_pa(i, *) GT pa_rng(j, 0) AND mpa_pa(i, *) LE pa_rng(j, 1), in_box)
           IF in_box GT 0 THEN pa_en_data(i, j) = total(mpa_data(i, ind))/in_box
        endFOR 
     ENDFOR 
     IF keyword_set(GEO) THEN BEGIN 
        geo_pa = reform(geo_pa, 3*32)
        FOR j = 0, pa_bins-1 DO BEGIN
           ind = where(geo_pa(*) GT pa_rng(j, 0) AND geo_pa(*) LT pa_rng(j, 1), in_box)
           IF in_box GT 0 THEN FOR i = 0, geo_chan-1 DO pa_en_data(i, j) = $
                                      total(geo_data(ind, geo_chan_ind(i)))/in_box
        endFOR 
     ENDIF 

     pa_en_data = rotate(pa_en_data, 3)
     ysize = 20*en_chans &  xsize = 20*pa_bins

     window,    xsize = xsize, ysize = ysize, title = 'Pitch angle vs Energy'
     tvscl, smooth(congrid(pa_en_data, xsize, ysize),25, /edge_truncate)
     wait, 1
  endIF 


  IF NOT keyword_set(en_pa) THEN BEGIN 

;; convert the data to v_perp and v_para 
   
; geo pitch angles are energy independent, mpa pitch angles already
; have the right dimensions
     geo_V_ppn = sin(geo_pa/!radeg)
     geo_V_pln = cos(geo_pa/!radeg)
     mpa_V_ppn = sin(mpa_pa/!radeg)
     mpa_V_pln = cos(mpa_pa/!radeg)
 


;; get the perp and par coordinates

     geo_V_pp = rebin(reform(geo_V_ppn, 3, 32, 1),  3, 32, geo_chan)* $
                rebin(reform(geo_center_en(geo_chan_ind),1, 1, geo_chan), 3, 32, geo_chan)
     geo_V_pl = rebin(reform(geo_V_pln, 3, 32, 1), 3, 32, geo_chan)* $
                rebin(reform(geo_center_en(geo_chan_ind), 1, 1,geo_chan), 3, 32,geo_chan)
     
     mpa_V_pp = mpa_V_ppn*rebin(reform(mpa_center_en(0:mpa_chan-1), mpa_chan, 1, 1), mpa_chan, 6, 24)
     mpa_V_pl = mpa_V_pln*rebin(reform(mpa_center_en(0:mpa_chan-1), mpa_chan, 1, 1), mpa_chan, 6, 24)
     
;; straighten out all the matrices so they can be plotted

     IF keyword_set(MPA) THEN BEGIN 
        V_pp = reform(mpa_V_pp, (mpa_chan)* 6*24)
        V_pl = reform(mpa_V_pl, (mpa_chan)* 6*24)
        data = reform(mpa_data, (mpa_chan)* 6*24)
     ENDIF ELSE BEGIN 
        V_pp = [reform(mpa_V_pp, (mpa_chan)* 6*24), reform(geo_V_pp, geo_chan*3*32)]
        V_pl = [reform(mpa_V_pl, (mpa_chan)* 6*24), reform(geo_V_pl, geo_chan*3*32)]
        data = [reform(mpa_data, (mpa_chan)* 6*24), reform(geo_data, geo_chan*3*32)]
     ENDELSE 
     IF keyword_set(max_val) THEN BEGIN 
        val_idx = where(abs(V_pp) LT max_val AND abs(V_pl) LT max_val)
        v_pp = v_pp(val_idx)
        v_pl = v_pl(val_idx)
        data = data(val_idx)
     ENDIF
     IF keyword_set(min_val) THEN BEGIN 
        val_idx = where(sqrt(V_pp^2 +V_pl^2) GT min_val)
        v_pp = v_pp(val_idx)
        v_pl = v_pl(val_idx)
        data = data(val_idx)
     ENDIF 
     
     grid_data = fltarr(200, 100) & box_size = 100 ;ev
     grid_count = fltarr(200, 100)
     ndat = n_elements(data)
     
     lims = alog10(findgen(100))
     FOR i = 0, ndat-1 DO BEGIN 
        FOR j = 0, ndat-1 DO BEGIN 
                                ;      x = 
        ENDFOR 
     ENDFOR         
     
     triangulate, V_pl, V_pp, tr, b
     xtemp = trigrid(V_pl, V_pp, data,nx=50,ny=25 ,tr,min_value=0,max_value=25000)
     surface, xtemp,/zlog, color = 254
  endIF ; not en_pa

END 
