;***************************************************************************
;* PROCEDURE:     
;*      FUNCTION lanl_geo_satlist_request
;* 
;* DESCRIPTION:
;*      get list of files available on lanl_geo directory on avatar
;* 
;* INPUTS:       
;*      dates - dates to look for
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      files - list of sats found
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_satlist_request(date)
;*      normally called by lanl_geo_satlist
;*
;* MODIFICATION HISTORY:       
;*      written March 1998, Reiner Friedel
;******************************************************************************
FUNCTION lanl_geo_satlist_request, date, SAT_ID=SAT_ID
  
  local_data='/n/avatar/belly1/home/lanl_geo/'  
  if keyword_set(SAT_ID) then id='_'+SAT_ID+'.text.gz' else id='*'
  sats=['All']
  
  FOR i=0, N_ELEMENTS(date)-1 DO BEGIN
    ;date by default a 4 letter year. These filenames still use 2 letter year.
    if strlen(date(i)) eq 8 then begin
      this_date=strmid(date(i),2,8) 
      year=string(strmid(date(i),0,4),format="(i4.4,'/')")
    endif else begin 
      this_date=date(i)
      year='19'+string(strmid(this_date,0,2),format="(i2.2,'/')")
    endelse

;   look if sats exist in local_data   
    newSats=FINDFILE(local_data+year+this_date+id)
    IF newSats(0) NE '' THEN BEGIN
      FOR m=0, N_ELEMENTS(newSats)-1 DO BEGIN
        dummy=STR_SEP(newSats(m),'/')
        newSats(m)=dummy(N_ELEMENTS(dummy)-1)
      ENDFOR 
      FOR j=0, N_ELEMENTS(newSats)-1 DO BEGIN
        newSats(j)=STRMID(newSats(j),7,8)
        satsSaved=STRPOS(sats, newSats(j))
        index=WHERE(STRPOS(sats, newSats(j)) gt -1)
        IF index(0) EQ -1 THEN sats=[sats, newSats(j)]
      ENDFOR
    ENDIF
    
  ENDFOR
  
  msg=''  &  for i=0,n_elements(sats)-1 do msg=msg+' '+sats(i)
  message, msg, /cont
  RETURN, sats
  
END

;***************************************************************************
;* PROCEDURE:     
;*      FUNCTION lanl_geo_satlist_newList
;* 
;* DESCRIPTION:
;*      get list of all files in the actual directory on the remote host
;* 
;* INPUTS:       
;*      url  - remote hostname (inclusive userid, password,
;                hostname, directory)
;*      arch - path to the Los Alamos archive directory on the local host
;*      file - name of the file where the information shoud be stored
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      returns a status value : 1 one success, 0 one failure
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_satlist_newlist(url, arch, file)
;*      normally called by lanl_geo_satlist
;*
;* MODIFICATION HISTORY:       
;*      september 1997, Torsten Heise
;******************************************************************************
FUNCTION lanl_geo_satlist_newList, url, arch, file
  
  COMMON get_error, get_err_no, get_err_msg
  
  papco_get_remote_file_list, url, arch, file
  
  if get_err_no EQ 0 THEN RETURN, 1 else  RETURN, 0
  
END

;******************************************************************************
;* PROCEDURE:     
;*      FUNCTION lanl_geo_satlist_remote
;* 
;* DESCRIPTION:
;*      for the given date return list of available satellite ID's
;*      found in the file 'c_file' with information of the remote host
;* 
;* INPUTS:       
;*      date - list of dates for that you want to have the sat-list
;*      arch - path to the Los Alamos archive directory on the local host
;*      file - file which contains list of all files on the remote host
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      returns an array of available sat ID's
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_satist_remote(date, arch, file)
;*      normally called by lanl_geo_satlist
;*
;* MODIFICATION HISTORY:       
;*      september 1997, Torsten Heise
;******************************************************************************
FUNCTION lanl_geo_satlist_remote, date, arch, file

  COMMON get_error, get_err_no, get_err_msg
 
  sats='All'
  
  result=FINDFILE(arch+file)
  IF result(0) EQ '' THEN BEGIN
    message,file+'  not found in: '+arch,  /cont
    RETURN, ''
  ENDIF
  
; read all files listed in 'file' in the string array 'files'
  exestr="cat "+arch+file+" |grep "+date
  SPAWN, exestr, files
  
; select the all sats available for 'date' (each sat only ones)
  FOR i=0, N_ELEMENTS(date)-1 DO BEGIN
    index=WHERE(STRPOS(files,date(i)) GE 0)
    IF index(0) NE -1 THEN BEGIN
      newsats=STRMID(files(index),7,8)
      FOR j=0, N_ELEMENTS(newsats)-1 DO BEGIN
        index=STRPOS(sats,newsats(j))
        index=MAX(index)
        IF (index EQ -1) THEN sats=[sats, newsats(j)]
      ENDFOR
    ENDIF
  ENDFOR

  PRINT, '% lanl_geo_satlist_remote: ',sats
  RETURN, sats

END
  
;******************************************************************************
;* PROCEDURE:     
;*      lanl_merged_satlist_local
;* 
;* DESCRIPTION:
;*      for the given date return list of available satellite ID's
;*      found on the local host
;* 
;* INPUTS:       
;*      date - list of dates for that you want to have the sat-list
;*      data - path to the Merged data directory on the local host
;*      arch - path to the Merged data archive directory on the local
;*               host inclusive subdirectorys
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      returns an array of available sat ID's
;* 
;* CALLING SEQUENCE:
;*      result=lanl_merged_satlist_local(date, data, arch)
;*      normally called by lanl_geo_satlist
;*
;* MODIFICATION HISTORY:       
;*      April 1999, Reiner Friedel
;******************************************************************************
FUNCTION lanl_merged_satlist_local, date, data, arch

  sats=['All']
  
  ;date by default a 4 letter year. These filenames still use 2 letter year.
  n_dates=n_elements(date)  &  dates=date
  for i=0, n_dates-1 do dates(i)=strmid(date(i),2,8)
  
  FOR i=0, n_dates-1 DO BEGIN

;   look if sats exist in LA_DATA    
    newSats=FINDFILE(data+dates(i)+'*')
    IF newSats(0) NE '' THEN BEGIN
      FOR m=0, N_ELEMENTS(newSats)-1 DO BEGIN
        dummy=STR_SEP(newSats(m),'/')
        newSats(m)=dummy(N_ELEMENTS(dummy)-1)
      ENDFOR 
      FOR j=0, N_ELEMENTS(newSats)-1 DO BEGIN
        newSats(j)=STRMID(newSats(j),7,8)
        satsSaved=STRPOS(sats, newSats(j))
        index=WHERE(STRPOS(sats, newSats(j)) gt -1)
        IF index(0) EQ -1 THEN sats=[sats, newSats(j)]
      ENDFOR
    ENDIF

;   look if sats exist in LA_DATA_ACHIVE + year
    year=string(strmid(date(0),0,4),format="(i4.4,'/')")
    str=arch+year+dates(i)+'*'
    newSats=FINDFILE(str)
    
    IF newSats(0) NE '' THEN BEGIN
      FOR m=0, N_ELEMENTS(newSats)-1 DO BEGIN
        dummy=STR_SEP(newSats(m),'/')
        newSats(m)=dummy(N_ELEMENTS(dummy)-1)
      ENDFOR 
      FOR j=0, N_ELEMENTS(newSats)-1 DO BEGIN
        newSats(j)=STRMID(newSats(j),7,8)
        satsSaved=STRPOS(sats, newSats(j))
        index=WHERE(STRPOS(sats, newSats(j)) gt -1)
        IF index(0) EQ -1 THEN sats=[sats, newSats(j)]
      ENDFOR
    ENDIF
    
  ENDFOR

  msg=''  &  for i=0,n_elements(sats)-1 do msg=msg+' '+sats(i)
  message, msg, /cont
  
  RETURN, sats

END

  
;******************************************************************************
;* PROCEDURE:     
;*      lanl_pitch_satlist_local
;* 
;* DESCRIPTION:
;*      for the given date return list of available satellite ID's
;*      found on the local host
;* 
;* INPUTS:       
;*      date - list of dates for that you want to have the sat-list
;*      data - path to the pitch data directory on the local host
;*      arch - path to the pitch data archive directory on the local
;*             host inclusive subdirectorys
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      returns an array of available sat ID's for date period
;* 
;* CALLING SEQUENCE:
;*      result=lanl_pitch_satlist_local(date, data, arch)
;*      normally called by lanl_geo_satlist
;*
;* MODIFICATION HISTORY:       
;*      April 2000, Reiner Friedel
;******************************************************************************
FUNCTION lanl_pitch_satlist_local, date, data, arch

  sats=['All']
  
  FOR i=0, N_ELEMENTS(date)-1 DO BEGIN

;   look if sats exist in data directory + year
    year=string(strmid(date(0),0,2)+1900,format="(i4.4,'/')")
    newSats=FINDFILE(data+year+'19'+date(i)+'*')    
    
    IF newSats(0) NE '' THEN BEGIN
      FOR m=0, N_ELEMENTS(newSats)-1 DO BEGIN
        dummy=STR_SEP(newSats(m),'/')
        newSats(m)=dummy(N_ELEMENTS(dummy)-1)
      ENDFOR 
      FOR j=0, N_ELEMENTS(newSats)-1 DO BEGIN
        newSats(j)=STRMID(newSats(j),9,8)
        satsSaved=STRPOS(sats, newSats(j))
        index=WHERE(STRPOS(sats, newSats(j)) gt -1)
        IF index(0) EQ -1 THEN sats=[sats, newSats(j)]
      ENDFOR
    ENDIF

  ENDFOR
  
  msg=''  &  for i=0,n_elements(sats)-1 do msg=msg+' '+sats(i)
  message, msg, /cont
    
  return, sats

END


;******************************************************************************
;* PROCEDURE:     
;*      lanl_geo_satlist_local
;* 
;* DESCRIPTION:
;*      for the given date return list of available satellite ID's
;*      found on the local host
;* 
;* INPUTS:       
;*      date - list of dates for that you want to have the sat-list
;*      data - path to the Los Alamos data directory on the local host
;*      arch - path to the Los Alamos archive directory on the local
;*               host inclusive subdirectorys
;*
;* KEYWORDS:
;*      EXT  returns list of file extensions found instead
;*      
;* OUTPUT: 
;*      returns an array of available sat ID's
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_satlist_local(date, data, arch)
;*      normally called by lanl_geo_satlist
;*
;* MODIFICATION HISTORY:       
;*      september 1997, Torsten Heise
;******************************************************************************
FUNCTION lanl_geo_satlist_local, date, data, arch,  EXT = EXT

  COMMON lanl_geo_panel, WidgetData, lSatNames, rSatNames, lExtNames, rExtNames

  sats=['All'] & exts = ['No extensions found']
  
  ;date by default a 4 letter year. (changed April 2001)
  ;only original ep data files remain two letter year dates.
  n_dates=n_elements(date)  &  dates=date
  
  CASE WidgetData.ioptions(1) OF
      0: data_source = 'local'
      1: data_source = 'ftp'
  ENDCASE

  FOR i=0, N_ELEMENTS(date)-1 DO BEGIN
      
      CASE data_source OF 
          'local': BEGIN        ;look if sats exist in archive directory + year
              year = strmid(dates(i), 0, 4)    
              thisday = strmid(dates(i), 2, 6)
              newSats=FINDFILE(arch+year+'/'+thisday +'*', count = c)
              IF c NE 0 THEN BEGIN
                  found_sats = strarr(c) & found_exts = strarr(c) 
                  FOR j = 0, c-1 DO BEGIN
                      papco_sepdir, newSats(j), dir, file, extension
                      found_sats(j) = strmid(file, 7, 8)
                      IF strmid(extension,strlen(extension)-2, 2) EQ 'gz' THEN $
                         extension = strmid(extension, 0, strlen(extension)-3)
                      found_exts(j) = extension
                  ENDFOR
                  sats = [sats, found_sats]
                  exts = found_exts
              ENDIF         
          END 
          'ftp': BEGIN          ;look if sats exist in local archive directory
              newSats=FINDFILE(data+dates(i)+'*')
              IF newSats(0) NE '' THEN BEGIN
                  FOR m=0, N_ELEMENTS(newSats)-1 DO BEGIN
                      dummy=STR_SEP(newSats(m),'/')
                      newSats(m)=dummy(N_ELEMENTS(dummy)-1)
                  ENDFOR 
                  FOR j=0, N_ELEMENTS(newSats)-1 DO BEGIN
                      newSats(j)=STRMID(newSats(j),9,8)
                      satsSaved=STRPOS(sats, newSats(j))
                      index=WHERE(STRPOS(sats, newSats(j)) gt -1)
                      IF index(0) EQ -1 THEN sats=[sats, newSats(j)]
                  ENDFOR
              ENDIF              
          END
      ENDCASE 
    
  ENDFOR
  
  out_str=''  
  IF keyword_set(EXT) THEN BEGIN
      IF exts(0) NE 'No extensions found' THEN BEGIN
          unique_exts = [found_exts(0)]
          FOR i = 0, n_elements(found_exts)-1 DO BEGIN
              pos = strpos(unique_exts, found_exts(i))
              index = where(pos NE -1, cc)
              IF cc EQ 0 THEN unique_exts = [unique_exts, found_exts(i)]
          ENDFOR
      ENDIF  
      for i=0, n_elements(unique_exts)-1 do out_str=out_str+' '+unique_exts(i)
      message, 'Exts: '+out_str, /cont
      RETURN, unique_exts
  ENDIF ELSE BEGIN
      for i=0, n_elements(sats)-1 do out_str=out_str+' '+sats(i)
      message, 'Sats: '+out_str, /cont
      RETURN, sats
  ENDELSE

END

;******************************************************************************
;* PROCEDURE:     
;*      lanl_geo_dates_needed
;* 
;* DESCRIPTION:
;*      for the given time period (from the PAPCO main widget) return a
;*      list of dates eg [971230, 971231]
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      an array of available dates (format = ddmmyy)
;* 
;* CALLING SEQUENCE:
;*      lanl_geo_dates_needed, date
;*
;* MODIFICATION HISTORY:       
;*      september 1997, Torsten Heise
;******************************************************************************
PRO lanl_geo_dates_needed, date, VERBOSE=VERBOSE

  common plot_composer, widgetdata
  common mjdt, mjdt_start, mjdt_end

  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0
  
; find first date possible
  convert_t90_to_date, mjdt_start, yr, doy, hr, min, sec, mon, cmon, dom, /mjdt
  date= [string(yr, mon, dom, format="(i4.4,i2.2,i2.2)")]

; find next possible dates
  xut=mjdt_start  &  xut.mjd=xut.mjd+1
  while xut.mjd le mjdt_end.mjd do begin
    convert_t90_to_date, xut, yr, day_oy, hr, min, sec, mon, cmon, dom, /mjdt
    nextdate=string(yr, mon, dom, format="(i4.4,i2.2,i2.2)")
    if (xut.mjd eq mjdt_end.mjd)  AND (hr+min+sec eq 0) then $
      goto, done ;last day has no time
    xut.mjd=xut.mjd+1  &  date=[date,nextdate]
  endwhile
  done:
  date_str=''
  for i=0, n_elements(date)-1 do date_str=date_str+date(i)+' '
  if verbose then message, date_str, /cont

end

;******************************************************************************
;* PROCEDURE:     
;*      lanl_geo_check_env_vars
;* 
;* DESCRIPTION:  
;*      check if the necessary environment variables are set correctly
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      MERGDED - if set, return directories for merged data files.
;*      PITCH   - if set, return directories for pitch angle data files.
;*      
;* OUTPUT: 
;*      data - path of the Los Alamos data directory (LA_DATA)
;*      arch - path of the Los Alamos archive directory (LA_DATA_ARCH)
;*      ftp  - hostname and path of the Los Alamos ftp archive (LA_FTP_ARCHIVE)
;* 
;* CALLING SEQUENCE:
;*      lanl_geo_check_env_vars, data, arch, ftp
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;*      January 1997, to allow MJDT format time only
;*      september 1997, rewritten by Torsten Heise
;******************************************************************************
PRO lanl_geo_check_env_vars, data, arch, ftp, MERGED=MERGED, PITCH=PITCH, $
                             ORIGINAL=ORIGINAL

  COMMON get_error, get_err_no, get_err_msg
  
  if keyword_set(ORIGINAL) then begin
    org_path='/n/papco/data/lanl/geo/originals/'
    ORIGINAL=org_path
  endif
  
  if keyword_set(MERGED) then begin
    ;check for env. variable pointing to data binary directory
    if not (papco_check_data_env('LANL_MERGED', path=PATH)) then begin
      strArray=[get_err_msg]
      result=messageBox(strArray, ['OK'], Title='Error discovered', /CENTER)
      return
    endif else data=path
    ;check for env. variable pointing to data archive directory
    if not (papco_check_data_env('LANL_MERGED_ARCHIVE', path=PATH)) then begin
      strArray=[get_err_msg]
      result=messageBox(strArray, ['OK'], Title='Error discovered', /CENTER)
      return
    endif else arch=path
    ;check for env. variable pointing to ftp archive directory
    ftp=''  
    return
  endif
  
  if keyword_set(PITCH) then begin
    ;check for env. variable pointing to data binary directory
    if not (papco_check_data_env('LANL_GEO_PITCH', path=PATH)) then begin
      strArray=[get_err_msg]
      result=messageBox(strArray, ['OK'], Title='Error discovered', /CENTER)
      return
    endif else data=path+'savesets/'
    ;check for env. variable pointing to data archive directory
    arch=''  
    ;check for env. variable pointing to ftp archive directory
    ftp=''  
    return
  endif  
    
    ;check for env. variable pointing to data binary directory
    if not (papco_check_data_env('LANL_EP', path=PATH)) then begin
      strArray=[get_err_msg]
      result=messageBox(strArray, ['OK'], Title='Error discovered', /CENTER)
      return
    endif else data=path
    ;check for env. variable pointing to data archive directory
    if not (papco_check_data_env('LANL_EP_ARCHIVE', path=PATH)) then begin
      strArray=[get_err_msg]
      result=messageBox(strArray, ['OK'], Title='Error discovered', /CENTER)
      return
    endif else arch=path
    ;check for env. variable pointing to ftp archive directory
    if not (papco_check_data_env('LA_FTP_ARCHIVE', path=PATH, /NODIR)) $
      then begin
      strArray=[get_err_msg]
      result=messageBox(strArray, ['OK'], Title='Error discovered', /CENTER)
      return
    endif else ftp=path
      
END

;******************************************************************************
;* PROCEDURE:     
;*      FUNCTION lanl_geo_satlist
;* 
;* DESCRIPTION:  
;*      for the given time period (bye the PAPCO main widget) return a
;*      list of available Los Alamos satellite ID's 
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      FULL    - get list over full time period
;*      LOCAL   - get list of the satellite ID's available on local host
;*      LIST    - get list of the satellite ID's available on remote host
;*      NEWLIST - get new list with satellite ID's available on remote
;*                 host via ftp
;*      REQUEST - get list from local site for rlocal request.
;*      MERGED  - get list for merged data files (MPA/CPA/SOPA/ESP)
;*      
;* OUTPUT: 
;*      A string array of satellite id's eg 1982-097
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_satlist(LOCAL=1)
;*
;* MODIFICATION HISTORY:       
;*      october 1995, written by R. Friedel
;*      January 1997, to allow MJDT format time only
;*      september 1997, rewritten by Torsten Heise
;******************************************************************************
FUNCTION lanl_geo_satlist, FULL=FULL, LOCAL=LOCAL, REMOTE=REMOTE, $
                           NEWLIST=NEWLIST, REQUEST=REQUEST, $
                           MERGED=MERGED, PITCH=PITCH
  
  COMMON lanl_geo_satlist, date
  COMMON PLOT_COMPOSER, widgetData
  COMMON get_error, get_err_no, get_err_msg

; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

  get_err_no=0  &  get_err_msg='% lanl_geo_satlist: o.k.'  
  
  files=''
  lanl_geo_check_env_vars, data, arch, ftp, MERGED=MERGED, PITCH=PITCH
  IF get_err_no NE 0 THEN RETURN, files
  
  lanl_geo_dates_needed, date
  
  year=STRTRIM(STRING('19'),2)+STRMID(date(0),0,2)
  url=ftp
  file='lanl_geo_satlist.text'
    
; get local sat list  
  IF KEYWORD_SET(LOCAL) THEN BEGIN
    files=lanl_geo_satlist_local(date, data, arch)
    RETURN, files
  ENDIF
  
; get new sat list from remote host  
  IF KEYWORD_SET(NEWLIST) THEN BEGIN
    status=lanl_geo_satlist_newList(url, arch, file)
    IF status EQ 0 THEN BEGIN
      PRINT, "% lanl_geo_satlist: coudn't get new List"
      RETURN, ''
    ENDIF
    ; set remote so that the file list is read!
    REMOTE=1
  ENDIF
  
; read satlist from a file  
  IF KEYWORD_SET(REMOTE) THEN BEGIN
    files=lanl_geo_satlist_remote(date, arch, file)
    RETURN, files
  ENDIF
  
; read stalist from local lanl_geo directory on avatar
  IF KEYWORD_SET(REQUEST) THEN BEGIN
    files=lanl_geo_satlist_request(date)
    RETURN, files
  ENDIF  
  
; read local sat list for merged files  
  IF KEYWORD_SET(MERGED) THEN BEGIN
    files=lanl_merged_satlist_local(date, data, arch)
    RETURN, files
  ENDIF    
  
; read local sat list for pitch files  
  IF KEYWORD_SET(PITCH) THEN BEGIN
    files=lanl_pitch_satlist_local(date, data, arch)
    RETURN, files
  ENDIF    
  
END

;******************************************************************************
;* PROCEDURE:     
;*      FUNCTION lanl_geo_satlist_remote
;* 
;* DESCRIPTION:
;*      for the given date return list of available file extensions
;*      found in the file 'file' with information of the remote host
;* 
;* INPUTS:       
;*      date - list of dates for that you want to have the sat-list
;*      arch - path to the Los Alamos archive directory on the local host
;*      file - file which contains list of all files on the remote host
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      returns an array of available file extensions
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_extList_remote(date, arch, file)
;*      normally called by lanl_geo_extList
;*
;* MODIFICATION HISTORY:       
;*      March 1998, Reiner Friedel
;******************************************************************************
FUNCTION lanl_geo_extlist_remote, date, arch, file

  COMMON get_error, get_err_no, get_err_msg
  
  exts=['no ext found']
  
  result=FINDFILE(arch+file)
  IF result(0) EQ '' THEN BEGIN
    print,'% lanl_geo_extlist_list_remote: ',file
    print,'                  not found in: ',arch
    RETURN, ''
  ENDIF
  
; read all files listed in 'file' in the string array 'files'
  exestr="cat "+arch+file+" |grep "+date
  SPAWN, exestr, files
  
; select the all exts available for 'date' (each sat only ones)
  FOR m=0, N_ELEMENTS(files)-1 DO BEGIN
    new=1
    pos=strpos(files(m),'.')
    files(m)=strmid(files(m),pos,strlen(files(m)))
    pos=strpos(files(m),'.gz')
    if pos ne -1 then files(m)=strmid(files(m),0,pos)
    for i=0,n_elements(exts)-1 do if exts(i) eq files(m) then new=0
    if new then begin    ;new type extension
      exts=[exts,files(m)]
    endif  
  ENDFOR   
  
  if n_elements(exts) gt 1 then exts=exts(1:n_elements(exts)-1)
  
  PRINT, '% lanl_geo_extList_remote: ',exts
  RETURN, exts

END

;******************************************************************************
;* PROCEDURE:     
;*      lanl_geo_extlist_local
;* 
;* DESCRIPTION:
;*      for the given date return list of available file extensions
;*      found on the local host
;* 
;* INPUTS:       
;*      date - list of dates for that you want to have the sat-list
;*      data - path to the Los Alamos data directory on the local host
;*      arch - path to the Los Alamos archive directory on the local
;*               host inclusive subdirectorys
;*
;* KEYWORDS:
;*      none
;*      
;* OUTPUT: 
;*      returns an array of available sat extensions
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_extlist_local(date, data, arch)
;*      normally called by lanl_geo_extlist
;*
;* MODIFICATION HISTORY:       
;*      March 1998, Reiner Friedel
;******************************************************************************
FUNCTION lanl_geo_extlist_local, date, data, arch

  return, lanl_geo_satlist_local(date, data, arch,  /EXT)

END

;******************************************************************************
;* PROCEDURE:     
;*      FUNCTION lanl_geo_extlist
;* 
;* DESCRIPTION:  
;*      for the given time period (bye the PAPCO main widget) return a
;*      list of available Los Alamos file types by extention.
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      FULL    - get list over full time period
;*      LOCAL   - get list of file extension available on local host
;*      LIST    - get list of file extension on remote host
;*      NEWLIST - get new list file extension available on remote
;                 host via ftp
;*      
;* OUTPUT: 
;*      A string array of file extensions,  eg .avg.flux.diff
;* 
;* CALLING SEQUENCE:
;*      result=lanl_geo_extlist(LOCAL=1)
;*
;* MODIFICATION HISTORY:       
;*      March 1998, written by R. Friedel
;******************************************************************************
FUNCTION lanl_geo_extlist, FULL=FULL, LOCAL=LOCAL, REMOTE=REMOTE, $
                           NEWLIST=NEWLIST

  COMMON lanl_geo_satlist, date
  COMMON PLOT_COMPOSER, widgetData
  COMMON get_error, get_err_no, get_err_msg
   
; common get_error is used by PAPCO to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise PAPCO will crash at that
; point, too. PAPCO draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information.

  get_err_no=0
  get_err_msg='% lanl_geo_extlist: o.k.'  
  
  exts=''
  lanl_geo_check_env_vars, data, arch, ftp
  IF get_err_no NE 0 THEN RETURN, exts
  
  ;lanl_geo_dates_needed, date
  
  year=STRTRIM(STRING('19'),2)+STRMID(date(0),0,2)
  url=ftp
  file='lanl_geo_satlist.text'
  
; get local sat list 
  IF KEYWORD_SET(LOCAL) THEN BEGIN
    exts=lanl_geo_extlist_local(date, data, arch)
    RETURN, exts
  ENDIF
  
; get new sat list from remote host  
  IF KEYWORD_SET(NEWLIST) THEN BEGIN
    create_subdir, arch, year
    status=lanl_geo_extList_newList(url, arch, file)
    IF status EQ 0 THEN BEGIN
      PRINT, "% lanl_geo_satlist: coudn't get new List"
      RETURN, files
    ENDIF
    REMOTE=1
  ENDIF
  
; read satlist from a file  
  IF KEYWORD_SET(REMOTE) THEN BEGIN
    files=lanl_geo_extList_remote(date, arch, file)
    RETURN, files
  ENDIF
  
END

