;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_geo_line, panel, ioptions, type, smoothing, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;******************************************************************************
pro p_lanl_geo_line, panel, plotinfo, OUTPUT=OUTPUT, $
                     OVERPLOT=OVERPLOT, $
                     PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                     SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
COMMON yscale, yscl		      ;man/auto yscaling
COMMON zscale, zscl		      ;man/auto yscaling
COMMON shift_label, down              ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse
COMMON papco_color_names
color_indx=[black, red, green, blue, magenta, cyan, $
            burgundy, olive, dark_green, teal, royal_blue, violet]

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_lanl_geo. 
COMMON lanl_geo_data, input_header, input_data
COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents

COMMON slice_lanl_geo, time, ymat, utitle, uytitle, ych
COMMON get_error, get_err_no, get_err_msg
   
if keyword_set(OUTPUT) then output=OUTPUT else output=0
  
;set some of the variables from the panel editor
control = *plotinfo.USR_PTR1
sat_id = sat_names(control.sc_id)
nodata = -99

;make space for extra axis if needed!   
panelset, panel              ;sets the panel position viewport

; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
IF c NE 0 THEN data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1  &  time=data.tai-tai_1
ndat=n_elements(data)
  
;select data from array according to channels selected ----------------------- 
set=bytarr(120) 
FOR i = 0,59 DO IF (control.ek1 AND 2ll^i)      EQ 2ll^i      THEN set(i)=1 
FOR i=60,119 DO IF (control.ek2 AND 2ll^(i-60)) EQ 2ll^(i-60) THEN set(i)=1
  
;limit set to max no of data channels --------------------------------------
n_ch = n_elements(tag_names(data))-5
set=set(0:n_ch-1)  &  ch_idx=where(set eq 1,n_ch)
  
IF control.type EQ 1 THEN MERGED = 1 ELSE MERGED = 0

;get channel and cal info
channels = r_lanl_geo_cal(input_header.sat_id, MERGED = MERGED)

verbose = 1

ych=fltarr(n_ch,2)
ych(*, 0) = channels(ch_idx).ek_low
ych(*, 1) = channels(ch_idx).ek_high

ch_lbl = channels(ch_idx).label

IF keyword_set(VERBOSE) THEN BEGIN
    message,'Selected channels:',/cont
    FOR i=0,n_ch-1 DO print, strformat(ch_lbl(i),30,/center)
ENDIF 

CASE control.type OF 
    0:BEGIN                   ;select data for SOPA/CPA plots
        utitle='lanl geo'
        ymat=fltarr(ndat,n_ch)
        for i=0l, ndat-1 do for j=0,n_ch-1 do $
            ymat(i,j)=data(i).(ch_idx(j)+4)
        zst=ych(0,0)  &  zen=ych(n_ch-1,1)
    END
    1:BEGIN                     ;select data for merged 
        utitle='lanl geo merged'
        ymat=fltarr(ndat,n_ch)
        for i=0l, ndat-1 do for j=0,n_ch-1 do $
            ymat(i,j)=data(i).(ch_idx(j)+6)
        zst=ych(0,0)  &  zen=ych(n_ch-1,1)
    END
ENDCASE

;average array to be plotted if needed! data is in ymat, time in time ---------
CASE control.type OF
      0: BEGIN                ;SOPA/CPA
          uytitle = '[cts]s!e-1!n'
          CASE control.res OF
              0: BEGIN & t_av = 10   & av_str = '10 s' & GOTO, no_average & END
              1: BEGIN & t_av = 60    & av_str = '1 min'  & END
              2: BEGIN & t_av = 300   & av_str = '5 min'  & END
              3: BEGIN & t_av = 600   & av_str = '10 min' & END
              4: BEGIN & t_av = 3600  & av_str = '1 hr'   & END
              5: BEGIN & t_av = 86400 & av_str = '1 day'  & END
          ENDCASE
          lanl_geo_average, time, ymat, t_av
      END 
      1: BEGIN                 ;Merged
          uytitle = '[cts]s!e-1!ncm!e-2!nsr!e-1!nkeV!e-1!n'
          idx = where(ymat EQ -999, c)
          IF c NE 0 THEN ymat(idx) = -99
          CASE control.res OF
              0: BEGIN & t_av = 86   & av_str = '86 s' & GOTO, no_average & END
              1: BEGIN & t_av = 86   & av_str = '86 s' & GOTO, no_average & END
              2: BEGIN & t_av = 300   & av_str = '5 min'  & END
              3: BEGIN & t_av = 600   & av_str = '10 min' & END
              4: BEGIN & t_av = 3600  & av_str = '1 hr'   & END
              5: BEGIN & t_av = 86400 & av_str = '1 day'  & END
          ENDCASE
          lanl_geo_average, time, ymat, t_av
      END 
ENDCASE  
no_average:
utitle = utitle+' '+av_str+'!C'+sat_id

;check if any filtering or PSD is needed - get ephemeris data first -----------
IF control.L_SELECT_RANGE OR control.L_STAR_SELECT_RANGE OR $
   control.MLT_SELECT_RANGE OR control.MLAT_SELECT_RANGE OR $
   control.cal EQ 2 THEN BEGIN

    COMMON lanl_geo_eph, vv1, vv2, vv3, lanl_geo_eph_control
    COMMON geo_eph_data, eph_header, eph_data

    new_plotinfo = papco_getplotinfostruct()
    new_control = lanl_geo_eph_control
    new_control.SC_ID = control.SC_ID
    new_control.ORIG = 1 ;mag model coords
    new_control.L_CALC = control.L_CALC
    new_control.EXTERNAL_MODEL = control.EXTERNAL_MODEL
    new_control.INTERNAL_MODEL = control.INTERNAL_MODEL
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    r_geo_eph, new_plotinfo, VERBOSE=VERBOSE
    IF get_err_no NE 0 THEN BEGIN 
        message,'problem getting ephemeris data, cannot filter', /cont
        goto, no_restrict
    ENDIF 
    tag_idx = where(TAG_NAMES(eph_data) EQ eph_header.model_name)
    tag_idx = tag_idx(0)

    filt_t = eph_data.TAI-tai_1

ENDIF

;filter the data using the ephemeris info -------------------------------------
;Handle different time bases - set non-filtered data to nodata flag.

;filter for L
IF control.L_SELECT_RANGE THEN BEGIN
    filt_var = eph_data.(tag_idx).L
    v1 = control.L_SELECT_FROM_VAL
    v2 = control.L_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN ymat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.L_SELECT_RANGE_FRMT)
ENDIF

;filter for L_STAR
IF control.L_STAR_SELECT_RANGE THEN BEGIN
    filt_var = eph_data.(tag_idx).LSTAR
    v1 = control.L_STAR_SELECT_FROM_VAL
    v2 = control.L_STAR_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN ymat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.L_STAR_SELECT_RANGE_FRMT)
ENDIF

;filter for MLT
IF control.MLT_SELECT_RANGE THEN BEGIN

    filt_var = eph_data.(tag_idx).MLT
    v1 = control.MLT_SELECT_FROM_VAL
    v2 = control.MLT_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)

    ;convert mlt to x,y, convert back. Handles midnight crossing.
    x=cos(filt_var*!pi/12)
    y=sin(filt_var*!pi/12)
    n_mlt=atan(y/x)

    idx=where((x lt 0) and y gt 0, c) ;2nd quadrant
    if c ne 0 then n_mlt(idx)=n_mlt(idx)+!pi
    idx=where((x lt 0) and y lt 0, c) ;3nd quadrant
    if c ne 0 then n_mlt(idx)=n_mlt(idx)+!pi
    idx=where((x gt 0) and y lt 0, c) ;4nd quadrant
    if c ne 0 then n_mlt(idx)=2*!pi+n_mlt(idx)
    n_mlt=n_mlt*12/!pi

    ;now select out range needed
    if v1 gt v2 then begin ;crosses over midnight
        idx=where( (n_mlt lt v1) AND (n_mlt gt v2), c)
    endif else begin
        idx=where( (n_mlt lt v1) OR (n_mlt gt v2), c)      
    ENDELSE 

    IF c NE 0 THEN ymat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.MLT_SELECT_RANGE_FRMT)

ENDIF

;filter for MLAT
IF control.MLAT_SELECT_RANGE THEN BEGIN
    filt_var = eph_data.(tag_idx).MLAT
    v1 = control.MLAT_SELECT_FROM_VAL
    v2 = control.MLAT_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN ymat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.MLAT_SELECT_RANGE_FRMT)
ENDIF
no_restrict:

unit_str =''

;apply flux conversion factors if needed
IF control.cal GT 0 THEN BEGIN

    ;do cross calibration (ONERA factors)
    ;electrons=fltarr(10,1) & electrons(*)=1 
    ;sopa_calibration, input_header.sat_id, electrons
    ;elec_cross_cal_fcf = electrons(*)
    ;cal_fcf = 1/ channels.fcf
    ;cal_fcf(0:9) = cal_fcf(0:9) * elec_cross_cal_fcf
    ;message, 'applied geo intercal', /cont
 
    print,  channels(ch_idx).fcf  

    message, 'applied flux conversion factors', /info
    FOR i = 0, n_ch-1 DO ymat(*, i) = ymat(*, i) / channels(ch_idx(i)).fcf

    IF NOT merged THEN BEGIN
        unit_str = 'cm!e-2!nsr!e-1!nkeV!e-1!n'
    ENDIF   

ENDIF    

IF control.cal EQ 2 THEN BEGIN  ;do PSD @ const mu
    ;calculate PSD (from fluxes) at constant mu based on SOPA data...
    ;mu's here in MeV/G, and output is in (c/(MeV cm))^3
    papco_m_mu_arr, control.mu, 1, 10000, 20, mus, mus_str, ylow, yhigh
    n_ch = n_elements(mus) & psd_mat = fltarr(ndat, n_ch)      

    b_mag = eph_data.(tag_idx).B_GSM(3)
    ;interpolate model mag field magnitude data onto data resolution.
    t_mag = eph_data.tai-tai_1
    b_mag_dat = interpol(b_mag, t_mag, time, /spline)
    ndat = n_elements(b_mag_dat)
    ymat_mu=fltarr(ndat,n_ch)

    psd_mat = fltarr(ndat, n_ch)
    FOR i=0,n_ch-1 DO BEGIN 
        mu = mus(i)
        papco_mu_calculation, ych(*,0), ych(*,1), mu, ymat, $
                              b_mag_dat, PSD_at_const_mu, /GEM
        psd_mat(*,i)=PSD_at_const_mu
    ENDFOR 

    idx = where(finite(psd_mat) EQ 0, c)
    IF c NE 0 THEN psd_mat(idx) = nodata 
    
    yminarr = ylow & ymaxarr = yhigh
    ch = mus_str

    ymat=psd_mat
    unit_str = ' s!e3!n km!e-6!n'
    unit_str = ' c!e3!n/ MeV!e3!n cm!e3!n'
    uytitle  = 'Hilmer PSD'
    utitle   = utitle+'!C'+'!4l!3 (MeV/G)'

ENDIF 

uytitle = uytitle+unit_str

IF output GE 1 THEN BEGIN
    message,'plot data constructed',/cont
    IF output EQ 2 THEN return; make data only, no plot
ENDIF 

;set up plot from here onward--------------------------------------------------

if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
endif else BEGIN
    papco_autorange, ymat, yst, yen, LOG = yscl(panel(0),3),  $
        NODATA = -99, exclude=0
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
endelse

; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)
  
if (zscl(panel(0),0) eq 1) then begin
    zst=zscl(panel(0),1)  &   zen=zscl(panel(0),2)
endif else begin
    zscl(panel(0),1)=zst  &   zscl(panel(0),2)=zen
endelse
  
; set up extra plot keywords
extra_plotPar={yrange:[yst,yen], xrange:[xut1,xut2], zrange:[zst,zen],$
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
               xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
  
gap_plotPar=create_struct({color:extra_plotPar.color, $
                           thick:extra_plotPar.thick, $
                           psym:extra_plotPar.psym, $
                           linestyle:extra_plotPar.linestyle, $}
                           xstyle:5, ystyle:5})

; check if we need to do plot or just return arrays to slice.
IF output EQ 3 THEN  GOTO, write_output
  
; use papco rotuine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar
 
; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; do plotting - seperate out line and spec type plots -------------------------
; now do the actual data plot, data only, no axis
  
IF control.graph EQ 0 THEN BEGIN ;multiple line plot  
    FOR i=0,n_ch-1 DO BEGIN 
        ch_label=ch_lbl(i)
        IF n_ch GT 1 THEN BEGIN
            extra_plotPar.color=color_indx(i mod n_elements(color_indx))
            label = ''
        ENDIF ELSE BEGIN
            label = ch_label
        ENDELSE     
        yray=ymat(*,i)
        index=where(yray ne nodata, cc) & IF cc EQ 0 THEN CONTINUE 
        lanl_geo_gap_oplot, time, yray, _extra=extra_plotPar
        c=i/4  &  mult=i mod 4  &  for j=1,c do ch_label='!C'+ch_label
        IF n_ch GT 1 THEN $
          xyouts, !P.position(0)+0.01+0.165*mult, $
                  !P.position(3)-0.015, ch_label, /normal, $
                  charsize=!p.charsize, color=extra_plotPar.color
    ENDFOR    
ENDIF ELSE BEGIN                ;3D color spec plot 
    down=0
    papco_plot_colorspec, zmat, time, yarr, $
                     resx=resx, resy=resy, nodata=nodata, _extra=extra_plotPar
    papco_color_bar, _extra=extra_plotPar    
ENDELSE 
  
; plot y-axis label at left of plot. Use scalable routine!  
IF NOT keyword_set(overplot) THEN left_side_label, panel, uytitle, /rot90

; plot info right of the panel. Multipl labels / panel are handled.
IF control.graph EQ 0 THEN BEGIN ;multiple line plot  
    IF n_elements(label) NE 0 THEN utitle=utitle+'!C'+label
    papco_rs_lbl, panel, utitle, /LINE, _extra=extra_Par
ENDIF ELSE BEGIN 
    papco_rs_lb, panel, utitle, _extra=extra_Par, /rot90
ENDELSE
    
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
write_output:
IF KEYWORD_SET(OUTPUT) THEN BEGIN 
    message,'Writing plot data out to file', /cont
    papco_write_data, time+tai_1, ymat, $
                      DESCRIPTION='lanl_geo '+input_header.sat_id, $
                      TIME_LABEL='TAI', $
                      CHANNELS=ych, Y_LABEL=uytitle, Z_LABEL='', $
                      RS_LABEL=utitle, NODATA=-99

ENDIF   

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

return
  
;jump to here if no data to plot 
noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y  
  
END


;******************************************************************************
;* PROCEDURE:     
;*      lanl_geo_gap_oplot, x, y, _EXTRA=e
;* 
;* DESCRIPTION:  
;*      This procedure plots data without connecting data point if they are
;*      further than 3 hours apart.
;*
;* INPUTS:       
;*      x,y     arrays to be plotted
;*              
;* OUTPUTS: 
;*      none
;* 
;* CALLING SEQUENCE:
;*      omin_gap_oplot, x, y, _EXTRA=e
;*
;* MODIFICATION HISTORY:       
;*     written July 1997, Haje Korth
;******************************************************************************
PRO lanl_geo_gap_oplot, x, y, _EXTRA=extra_plotPar
  
; establish plotting coords for plots  
plot, x, x, /nodata, xstyle=5, ystyle=5, _extra=extra_plotPar
  
; select keywords for plots p[rocedure  
gap_plotPar=create_struct({color:extra_plotPar.color, $
                             thick:extra_plotPar.thick, $
                             psym:extra_plotPar.psym, $
                             linestyle:extra_plotPar.linestyle})  
  
; check for psym=10 and disallow, doesn't work for plots 
if gap_plotPar.psym eq 10 then begin
    gap_plotPar.psym=0
    message,'histogram mode does not work here',/cont
end
      
xray=x  &  yray=y

; work out average spacing along time axis. If actual spacing encountered is
; more than three times the average, don't plot.
ndat=n_elements(xray)
diff_time=abs(xray(1:ndat-2) - xray(0:ndat-1))
av_int=total(diff_time)/(n_elements(diff_time))
   
; work out jumps in y. If a jump is larger than 5 times the average, don't plot
for i=0l,ndat-2 do begin
    int=abs(xray(i+1)-xray(i))
    if int lt av_int*3 then begin
      plots,xray(i),yray(i) 
      plots,xray(i+1),yray(i+1),/continue,noclip=0,_EXTRA=gap_plotPar
    endif
endfor

END 

  
;******************************************************************************
;* PROCEDURE:     
;*      lanl_geo_average
;*
;* MODIFICATION HISTORY:       
;*      written March 2002, Reiner Friedel
;******************************************************************************
PRO lanl_geo_average, time, ymat, t_av

COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt

message, 'averaging to '+varprt(t_av)+ ' second resolution', /cont

tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})   
 
xut1=0  &  xut2=tai_2-tai_1

r = size(ymat, /st) 
IF r.N_DIMENSIONS EQ 1 THEN n_ch = 1 ELSE n_ch = r.dimensions(r.N_DIMENSIONS-1)

;make start and finish times on t_av boundaries.
t1 = (long(xut1) / fix(t_av) ) * t_av + t_av/2
t2 = (long(xut2) / fix(t_av) ) * t_av + t_av/2

;make new time and data array with resolution t_av.
n_dat = (t2-t1) / t_av
newtime = t1+findgen(n_dat)*t_av
newymat = fltarr(n_dat, n_ch) 
newymat(*) = -99.0 ;initialize as bad data values

i = 0l & c = 0l
st_idx = 0l & 
en_idx = 0l
ndat = n_elements(time)

;average data into new time.
WHILE (i LE ndat-1) AND (c LE n_dat-1) DO BEGIN 
    
    ta = newtime(c)-(t_av/2) & tb = newtime(c)+(t_av/2)

    ;get start index for interval.
    WHILE (time(i) LT ta) DO BEGIN
        i = i+1
        IF i GE ndat-1 THEN BREAK
    ENDWHILE 
    i = i < (ndat-1)
    st_idx = i 

    ;now find end index for interval
    WHILE (time(i) LT tb) DO BEGIN
        i = i+1
        IF i GE ndat-1 THEN BREAK
    ENDWHILE 
    i = i < (ndat-1)
    en_idx = i 

    IF st_idx NE en_idx THEN BEGIN
        ;print, 'A ', c, i, ndat, st_idx, en_idx, time(en_idx),  ta, tb
        FOR j = 0, n_ch-1 DO BEGIN
            data = ymat(st_idx:en_idx, j)
            idx = where(data NE 0, valid)
            IF valid NE 0 THEN newymat(c, j) = total(data(idx))/valid
        ENDFOR 
    ENDIF ELSE BEGIN
        ;print, 'B ', c, i, st_idx, en_idx, time(en_idx),  ta, tb
        i = i+1
    ENDELSE 

    c = c+1 < (n_dat-1)

ENDWHILE

time = temporary(newtime)
ymat = temporary(newymat)

END


;******************************************************************************
;* PROCEDURE:     m_lanl_geo_flux_mu
;* 
;* calculate fluxes at a range of constant mu's based on SOPA data...
;******************************************************************************
PRO m_lanl_geo_flux_mu, time, ymat, ych, mus, ymat_mu

common mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
common lanl_geo_data, input_header, input_data
common geo_eph_data, eph_header, eph_data

convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
tai_1=UTC2TAI({YEAR:year, MONTH:mon, DAY:doym, HOUR:hr, MINUTE:min, $
               SECOND:sec, MILLISECOND:0})

r_geo_eph, input_header.sat_id
b_mag = eph_data.MODEL_0_5.B_GSM(3)

;interpolate model mag field magnitude data onto data resolution.
t_mag = eph_data.tai
t_dat = time+tai_1
b_mag_dat = interpol(b_mag, t_mag, t_dat, /spline)

ek = 10^((alog10(ych(*, 0))+alog10(ych(*, 1)))/2)

ndat = n_elements(b_mag_dat)
ymat_mu = fltarr(ndat, n_elements(mus))

FOR i = 0, ndat-1 DO BEGIN
    
    spec = transpose(ymat(i, *))
    
    ;now get the energies corresponding to the mu's requested
    new_ek = mus * b_mag_dat(i)

    ;interpolate to get new_fluxes for those mu's
    new_flux = interpol(spec, ek, new_ek)
    ymat_mu(i, *) = new_flux * new_ek
    
ENDFOR 

END
