;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_geo_pitch, panel, bird, dataType, option, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*            PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*     panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*     plotinfo papco structure PAPCO_PLOTINFO which is set
;*              by the panel editor. 
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_lanl_geo_pitch, panel, currPlotInfo, 
;*                          OVERPLOT=OVERPLOT, $
;*                          PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
;*                          OUTPUT=OUTPUT,SUBTABLE=SUBTABLE, _EXTRA=extra_Par
;*
;* MODIFICATION HISTORY: 
;*        Written July 2003, Arne Aasnes at LANL
;*        Modified, Oct. 2003, AAA at LANL 
;*            added possibility to select the telescopes you want, as
;*            they are not intercalibrated. Normally you'd like to use
;*            the bellyband telescope (2)
;*             
;******************************************************************************
pro p_lanl_geo_pitch, panel, plotInfo, $
                       OVERPLOT=OVERPLOT, $
                       PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                       OUTPUT=OUTPUT,SUBTABLE=SUBTABLE,_EXTRA=extra_Par


; The following common blocks are needed for PAPCO:
  COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  COMMON yscale, yscl           ;man/auto yscaling
  COMMON zscale, zscl           ;man/auto zscaling
  COMMON shift_label, down      ;common for x-axis label shifting
  COMMON coordinateSystems, plotted_x, plotted_y  
                                ;info on coords used by mouse
  
; One of the common blocks should contain the plot data as returned by the
; data read procedure r_lanl_geo_pitch. 
  COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data
  COMMON lanl_geo               ;see lanl_geo_set_commons for contents
  COMMON lanl_geo_slice,  utitle, uytitle, uztitle
;  COMMON lanl_geo_channels, yarr, zmat
  COMMON lanl_geo_output, xarr, yarr, zmat
  COMMON get_error, get_err_no, get_err_msg
  if keyword_set(OUTPUT) then output=OUTPUT else output=0

  IF output LT 2 THEN BEGIN 
  
     papco_Set_SubTable, SUBTABLE ;sets the papco color table index
     panelset,panel             ;sets the panel position viewport

  ENDIF 

  nodata = 1e-31 ; missing value

  control = *plotinfo.USR_PTR1 

; restrict data to actual time range requested - this makes for faster zooming
; geo_pitch data is in TAI. For plotting, always start at zero to get max
; resolution (plot works in float only, large values get rounded! 
  tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
  tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
  index=where((lanl_geo_pitch_data.tai ge tai_1) AND (lanl_geo_pitch_data.tai le tai_2),c)
  if c GT 2 then data=lanl_geo_pitch_data(index) ELSE GOTO, noplot
  xut1=0  &  xut2=tai_2-tai_1  & time=data.tai-tai_1 &  time(0) = 0
  ndat=n_elements(data)




; check the dimensions from the data returned from r_lanl_geo_pitch

  dim = size(data.counts, /dimensions)
  pol_det = dim(0)             ; polar detectors (should be 3)
  azi_det = dim(1)             ; azimuthal sampling (should be 32)


  pl_type = control.pl_type   
; pl_type_info:'0: spin averaged, 1: All PA @ 1 energy, 2: All E @ 1 PA, 
;               3: All PA for E range, 4: All E for PA range, 5: PA @ E range'

  IF control.sc_id GT 8 AND control.ek1 GE 2147483648ll THEN control.ek1 = 1ll
  IF control.ek1 LE 0 THEN control.ek1 = 1ll

  chan_arr = bytarr(120)
  FOR i=0,59 DO IF (control.ek1 AND 2ll^i) EQ 2ll^i THEN chan_arr(i)=1 $
              ELSE chan_arr(i)=0
  FOR i=60,119 DO IF (control.ek2 AND 2ll^(i-60)) EQ 2ll^(i-60) THEN chan_arr(i)=1 $
                ELSE chan_arr(i)=0
  
  sat_info = r_lanl_geo_flux_converison(sat_names(control.sc_id)) ; reads 

  ;chan_arr = chan_arr(0:n_elements(sat_info)-1)
  chan_ind = where(chan_arr EQ 1)

 ;if pl_type is set to all energies,select all energies of the species and instrument 
  IF control.pl_type EQ 2 OR control.pl_type EQ 4 OR control.pl_type EQ 5 THEN BEGIN 
     IF sat_info(chan_ind(0)).species EQ 'electrons' AND sat_info(chan_ind(0)).instrument NE 'ESP'  THEN  $
        chan_ind = where(sat_info.species EQ 'electrons' AND sat_info.instrument NE 'ESP')
     IF sat_info(chan_ind(0)).species EQ 'ions' AND sat_info(chan_ind(0)).instrument NE 'ESP'  THEN  $
        chan_ind = where(sat_info.species EQ 'ions' AND sat_info.instrument NE 'ESP')
     IF sat_info(chan_ind(0)).species EQ 'electrons' AND sat_info(chan_ind(0)).instrument EQ  'ESP'  THEN  $
        chan_ind = where(sat_info.species EQ 'electrons' AND sat_info.instrument EQ  'ESP')
     IF sat_info(chan_ind(0)).species EQ 'ions' AND sat_info(chan_ind(0)).instrument EQ  'ESP'  THEN  $
        chan_ind = where(sat_info.species EQ 'ions' AND sat_info.instrument EQ  'ESP')
     ;remove upper energy channel as it contains only noise
     n_ch = n_elements(chan_ind)
     chan_ind = chan_ind(0:n_ch-2)
  ENDIF ELSE IF control.pl_type EQ 1 THEN chan_ind = chan_ind(0)
  

  all_zmat = data(*).counts(*, *, chan_ind) ; selects only the energies we want

  IF n_elements(chan_ind) EQ 1 THEN all_zmat = reform(all_zmat, pol_det, azi_det, 1, ndat)


  ; select data from the telescopes we want
  det_arr = bytarr(3)
  FOR i = 0, 2 DO IF (control.det_choice AND 2ll^i) EQ 2ll^1 THEN det_arr(i) = 1 ELSE det_arr(i) = 0
  pol_ind = where(det_arr EQ 1, pol_det)
  IF pol_det EQ 0 THEN BEGIN 
     pol_det = 1
     pol_ind = 1
  ENDIF 
  all_zmat = all_zmat(pol_ind, *, *, *)
  
  n_ch = n_elements(chan_ind)
   

  
;**********************************************************************************
; Should add detector intercalibration factors and conversion to flux
; (if chosen)
;**********************************************************************************
;apply flux conversion factors if needed  **new mar 02
  IF (control.cal GT 0) THEN BEGIN 
; AND (sat_info(chan_ind(0)).instrument EQ 'SOPA')  AND (sat_info(chan_ind(0)).species EQ 'electrons')  
                                ;do cross calibration (ONERA factors)
        electrons=fltarr(10,1) & electrons(*)=1 
        sopa_calibration, lanl_geo_pitch_data(0).csat_id, electrons
        elec_cross_cal_fcf = electrons(*)
        cal_fcf = 1/ sat_info.fcf
        cal_fcf(0:9) = cal_fcf(0:9) * elec_cross_cal_fcf
        
        message, 'applied geo intercal', /cont
     

        IF n_ch EQ 1 THEN all_zmat = all_zmat*cal_fcf(chan_ind(0)) ELSE $
        FOR i = 0,  n_ch-1 DO  all_zmat(*, *, i, *) = all_zmat(*, *, i, *)*cal_fcf(chan_ind(i))
        uztit = 'Flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
  ENDIF  ELSE  BEGIN 
     uztit = 'Counts [s!e-1!n]'
  ENDELSE 
 
  
;************************************************************************************
;*     starts binning the data according to plot type
;************************************************************************************

  IF sat_info(chan_ind(0)).instrument EQ 'ESP' THEN alpha = data.alpha_esp ELSE alpha = data.alpha

  alpha_st = reform(alpha(pol_ind, *, *), pol_det*azi_det, ndat) 

  all_zmat_st = reform(all_zmat, pol_det*azi_det,n_ch, ndat)
  
 ; set missing values to NaN to avoid influence on mean 
  mis_ind = where(all_zmat_st lt -1, mis_cnt) &  IF mis_cnt GT 0 THEN all_zmat_st(mis_ind) = !values.f_nan

  IF control.pa_res EQ '' THEN control.pa_res = '10'

; if a range of energies selected, take the mean (only if energy flux)

  IF control.pl_type EQ 3 OR control.pl_type EQ 6 OR control.pl_type EQ 7 AND n_ch GT 1 THEN BEGIN 
     buf_all_zmat = all_zmat_st
     all_zmat_st = fltarr(pol_det*azi_det,1, ndat)
     FOR i = 0, ndat-1 DO BEGIN  
                FOR j = 0, pol_det*azi_det-1 DO all_zmat_st(j,0,i) = mean(buf_all_zmat(j, *, i), /nan)
     ENDFOR  
;     all_zmat_st = rebin(all_zmat_st, pol_det*azi_det,1, ndat)
     
     n_ch = 1
  ENDIF 




; if we want data to be plotted as function of pitch angle
  IF control.pl_type EQ 1 OR control.pl_type EQ 3  THEN BEGIN 
     all_zmat_st = reform(all_zmat_st, pol_det*azi_det, ndat)
;    call function to return an array of pa bins centered around 90 degrees with requested resolution
     pa_rng = papco_pitch_bin(fix(control.pa_res), pa_idx = pa_idx) 
     nbins = n_elements(pa_rng(*, 0))
     zmat = fltarr(nbins, ndat)
     FOR i = 0, ndat-1 DO BEGIN 
        FOR j = 0, nbins-1 DO BEGIN 
           pa_ind = where(alpha_st(*, i) GE pa_rng(j, 0) AND alpha_st(*, i) LT pa_rng(j, 1), pa_cnt)
           IF pa_cnt GT 0 THEN zmat(j, i) = mean(all_zmat_st(pa_ind, i), /nan) ELSE zmat(j, i) = nodata
        endFOR 
     ENDFOR 
  ENDIF 
; if data is to be plotted as function of energy (or for pl_type = 6
; as function of flux) 
  IF control.pl_type EQ 2 OR control.pl_type EQ 4 OR control.pl_type EQ 6 THEN BEGIN 
     
     zmat = fltarr(n_ch, ndat)
     IF control.pl_type EQ 2 THEN pa_arr = [control.pa_bin1-control.pa_res/2l, control.pa_bin1+control.pa_res/2l] ELSE $
     IF control.pl_type EQ 4 OR control.pl_type EQ 6 THEN pa_arr = [control.pa_bin1, control.pa_bin2]
     FOR i = 0, ndat-1 DO BEGIN 
        pa_ind = where(alpha_st(*, i) GE pa_arr(0) AND alpha_st(*, i) LT pa_arr(1), pa_cnt)
        IF pa_cnt GT 0 THEN BEGIN 
           FOR j = 0, n_ch-1 DO zmat(j, i) = mean(all_zmat_st(pa_ind,j, i),/nan)      
        endIF ELSE zmat(*, i) = nodata
     ENDFOR 
  ENDif 
  IF control.pl_type EQ 5 OR control.pl_type EQ 7 THEN BEGIN
     zmat = fltarr(n_ch, ndat)
     FOR i = 0, ndat-1 DO BEGIN 
        FOR j = 0, n_ch-1 DO BEGIN 
        th_zmat = all_zmat_st(*, j, i)
        th_alpha = alpha_st(*, i)
        IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN $
        pa_low = where((th_alpha GT control.pa_bin1-control.pa_res/2. AND th_alpha LT control.pa_bin1+control.pa_res/2.) OR $
                   (th_alpha GT 180-control.pa_bin1-control.pa_res/2. AND th_alpha LT 180-control.pa_bin1+control.pa_res/2.), pa_low_cnt) $
        ELSE pa_low = where(th_alpha GT control.pa_bin1-control.pa_res/2. AND th_alpha LT control.pa_bin1+control.pa_res/2., pa_low_cnt)
        pa_high = where(th_alpha GT control.pa_bin2-control.pa_res/2. AND th_alpha LT control.pa_bin2+control.pa_res/2., pa_high_cnt)
        IF pa_low_cnt GT 0 AND pa_high_cnt GT 0 THEN zmat(j, i) = mean(th_zmat(pa_high))/mean(th_zmat(pa_low)) ELSE zmat(j, i) = nodata
        endFOR 
     ENDFOR 
  ENDIF
     
  mis_ind = where(finite(zmat) EQ 0 , mis_cnt)
  IF mis_cnt GT 0 THEN zmat(mis_ind) = nodata
          
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; defines the y axis values and title for the different plot options 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  utitle = sat_names(control.sc_id)+' '+sat_info(chan_ind(0)).instrument+' '+sat_info(chan_ind(0)).species


  CASE pl_type OF
     1:BEGIN                    ; all pitch angles, one energy
        yarr = pa_rng
        energy_title = strcompress(sat_info(chan_ind).ek_low, /remove_all)+ $
                    '-'+ strcompress(sat_info(chan_ind).ek_high, /remove_all)+' KeV'
        uytitle = 'Pitch angle (deg)'
        utitle = utitle+'!C'+energy_title
        yst=yarr(0,0)*0.99 & yen=yarr(nbins-1,1)+1.01
        yscl(panel(0), 3) = 0   ; overrides choice from panel, always use linear scale
        
     END 
     2:BEGIN       ; all energies, one pitch angle
        yarr = [[sat_info(chan_ind).ek_low], [sat_info(chan_ind).ek_high]]

        pitch_title = strmid(strcompress(string(pa_arr(0)), /remove_all), 0, 4)+ $
                    '-'+ strmid(strcompress(string(pa_arr(1)), /remove_all), 0, 4)+' deg'
        utitle = utitle + '!C'+pitch_title
        uytitle = 'Energy (KeV)'
        yst=yarr(0,0)*0.99 & yen=yarr(n_ch-1,1)+1.01
     END 
     3:BEGIN   ; all pitch angles, selected energies
        yarr = pa_rng
        energy_title =  strcompress(sat_info(min(chan_ind)).ek_low, /remove_all)+ $
                    '-'+ strcompress(sat_info(max(chan_ind)).ek_high, /remove_all)+' KeV'
        utitle = utitle+'!C'+energy_title
        uytitle = 'Pitch angle (deg)'
        yst=yarr(0,0)*0.99 & yen=yarr(nbins-1,1)+1.01
        yscl(panel(0), 3) = 0  ; overrides choice from panel, always use linear scale
     end 
     4:BEGIN   ; all energies, pitch angle range
        yarr = [[sat_info(chan_ind).ek_low],[ sat_info(chan_ind).ek_high]]
        pitch_title = strmid(strcompress(string(pa_arr(0)), /remove_all), 0, 4)+ $
                    '-'+ strmid(strcompress(string(pa_arr(1)), /remove_all), 0, 4)+' deg'
        utitle = utitle+'!C'+pitch_title
        uytitle ='Energy (KeV)'
        yst=yarr(0,0)*0.99 & yen=yarr(n_ch-1,1)+1.01
     end 
     5:BEGIN ; anisotropy spectrogram
        yarr = [[sat_info(chan_ind).ek_low],[ sat_info(chan_ind).ek_high]]
         energy_title =  strcompress(sat_info(min(chan_ind)).ek_low, /remove_all)+ $
                    '-'+ strcompress(sat_info(max(chan_ind)).ek_high, /remove_all)+' KeV'
        pitch_title = strcompress(string(long(control.pa_bin2)), /remove_all)+ ' deg'+ $
                    '/'+ strcompress(string(long(control.pa_bin1)), /remove_all)+' deg'
        utitle = utitle+'!C'+pitch_title
        uytitle = 'Energy (KeV)'
        IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN   uztit = 'Anisotropy' ELSE uztit = 'Streaming'
        yst = yarr(0, 0)*0.99 &  yen = yarr(n_ch-1, 1)*1.01

                  
     END 
     6:BEGIN ; line plot, selected energies and pitch angles
        energy_title =  strcompress(sat_info(min(chan_ind)).ek_low, /remove_all)+ $
                    '-'+ strcompress(sat_info(max(chan_ind)).ek_high, /remove_all)+' KeV'
        pitch_title = strcompress(string(pa_arr(0)), /remove_all)+ $
                    '-'+ strcompress(string(pa_arr(1)), /remove_all)+'deg'
        utitle = utitle +'!c'+energy_title +'!c '+pitch_title
        IF control.cal EQ 0 THEN uytitle = 'Counts [s!e-1!n]' ELSE uytitle = 'Flux [cm!e-2!ns!e-1!nsr!e-1!nkeV!e-1!n]'
        zmat = reform(zmat, ndat) ;line plot 
        zmat = rotate(zmat, 4)
        ch_label = sat_info(chan_ind).label
        yst = min(zmat)*0.99 &  yen = max(zmat)*1.01
        control.graph = 0 ;line plot
     end 
     7:BEGIN ; anisotropy line plot for selected energy and pitch angles
        energy_title =  strcompress(sat_info(min(chan_ind)).ek_low, /remove_all)+ $
                    '-'+ strcompress(sat_info(max(chan_ind)).ek_high, /remove_all)+' KeV'
        pitch_title = strcompress(string(long(control.pa_bin1)), /remove_all)+ 'deg'+ $
                    '/'+ strcompress(string(long(control.pa_bin2)), /remove_all)+' deg'
        utitle = utitle+'!c'+energy_title +'!c '+pitch_title
        IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN uytitle = 'Anisotropy' ELSE uytitle = 'Streaming'
        zmat = reform(zmat, ndat) ;line plot
        zmat = rotate(zmat, 4)
 ;       ch_label = sat_info(chan_ind(0)).label
        IF control.pa_bin2 GT 60 AND control.pa_bin2 LT 120 THEN   uztit = 'Anisotropy' ELSE uztit = 'Streaming'
        control.graph = 0 ;line plot
        yst = min(zmat)*0.99 &  yen = max(zmat)*1.01
     END  

  ENDCASE
zmat = rotate(zmat, 4)

;check if any filtering or PSD is needed - get ephemeris data first -----------
IF control.L_SELECT_RANGE OR control.L_STAR_SELECT_RANGE OR $
   control.MLT_SELECT_RANGE OR control.MLAT_SELECT_RANGE OR $
   control.cal EQ 2 THEN BEGIN

    COMMON lanl_geo_eph_3, vv1, vv2, vv3, lanl_geo_eph_control
    COMMON geo_eph_data, eph_header, eph_data

    eph_plotinfo = papco_getplotinfostruct()
    eph_control = lanl_geo_eph_control
    eph_control.SC_ID = control.SC_ID
    eph_control.ORIG = 1 ;mag model coords
    eph_control.L_CALC = control.L_CALC
    eph_control.EXTERNAL_MODEL = control.EXTERNAL_MODEL
    eph_control.INTERNAL_MODEL = control.INTERNAL_MODEL
    eph_plotinfo.USR_PTR1 = ptr_new(eph_control)

    r_geo_eph, eph_plotinfo, VERBOSE=VERBOSE
    IF get_err_no NE 0 THEN BEGIN 
       message,'problem getting ephemeris data, cannot filter', /cont
       goto, no_restrict
    ENDIF 
    tag_idx = where(TAG_NAMES(eph_data) EQ eph_header.model_name)
    tag_idx = tag_idx(0)

    filt_t = eph_data.TAI-tai_1
    filt_lbl = ''

ENDIF

;filter the data using the ephemeris info -------------------------------------
;Handle different time bases - set non-filtered data to nodata flag.

;filter for L
IF control.L_SELECT_RANGE THEN BEGIN
       
    filt_var = eph_data.(tag_idx).L
    v1 = control.L_SELECT_FROM_VAL
    v2 = control.L_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.L_SELECT_RANGE_FRMT)+'!C'

ENDIF

;filter for L_STAR
IF control.L_STAR_SELECT_RANGE THEN BEGIN
       
    filt_var = eph_data.(tag_idx).LSTAR
    v1 = control.L_STAR_SELECT_FROM_VAL
    v2 = control.L_STAR_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.L_STAR_SELECT_RANGE_FRMT)+'!C'

ENDIF

;filter for MLT
IF control.MLT_SELECT_RANGE THEN BEGIN

    filt_var = eph_data.(tag_idx).MLT
    v1 = control.MLT_SELECT_FROM_VAL
    v2 = control.MLT_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)

    ;convert mlt to x,y, convert back. Handles midnight crossing.
    x=cos(filt_var*!pi/12)
    y=sin(filt_var*!pi/12)
    n_mlt=atan(y/x)

    idx=where((x lt 0) and y gt 0, c) ;2nd quadrant
    if c ne 0 then n_mlt(idx)=n_mlt(idx)+!pi
    idx=where((x lt 0) and y lt 0, c) ;3nd quadrant
    if c ne 0 then n_mlt(idx)=n_mlt(idx)+!pi
    idx=where((x gt 0) and y lt 0, c) ;4nd quadrant
    if c ne 0 then n_mlt(idx)=2*!pi+n_mlt(idx)
    n_mlt=n_mlt*12/!pi

    ;now select out range needed
    if v1 gt v2 then begin ;crosses over midnight
        idx=where( (n_mlt lt v1) AND (n_mlt gt v2), c)
    endif else begin
        idx=where( (n_mlt lt v1) OR (n_mlt gt v2), c)      
    ENDELSE 

    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.MLT_SELECT_RANGE_FRMT)+'!C'
ENDIF

;filter for MLAT
IF control.MLAT_SELECT_RANGE THEN BEGIN
       
    filt_var = eph_data.(tag_idx).MLAT
    v1 = control.MLAT_SELECT_FROM_VAL
    v2 = control.MLAT_SELECT_TO_VAL
    ;interpolate filt_var onto time base of data preserving bad data flags.
    filt_var = papco_interpol(filt_var, filt_t, time, NODATA = nodata)
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    filt_lbl = filt_lbl+string(v1, v2, $
                               format=control.MLAT_SELECT_RANGE_FRMT)+'!C'

ENDIF
no_restrict:

IF output GE  2 THEN BEGIN
   xarr = data.tai
   message,'plot data constructed',/cont
   IF output EQ 2 THEN return
   GOTO, write_output
ENDIF 
; build time array for plotting
timeDim=N_ELEMENTS(time)
xarr=DBLARR(timeDim-1,2)
xarr(*,0)=time(0:timeDim-2)
xarr(*,1)=time(1:timeDim-1)
timeDiff=DBLARR(timeDim-1)
FOR i=0,N_ELEMENTS(timeDiff)-1 DO timeDiff(i)=time(i+1)-time(i)
index=WHERE(timeDiff GE 1199.0,count) ; if time step is more than double the normal 600 sec (old value: 
IF (count GT 0) THEN FOR i=0,count-1 DO xarr(index,1)=xarr(index,0)+600 ; 86.0


; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data
;
; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling

if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1) & zmax=zscl(panel(0),2)
endif else BEGIN
    papco_autorange, zmat, zmin, zmax, LOG = zscl(panel(0),3), $
      NODATA = nodata, exclude=0
    zscl(panel(0),1)=zmin & zscl(panel(0),2)=zmax
endelse
;yst=yarr(0,0)*0.99 & yen=yarr(n_channels-1,1)+1.01
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
ENDELSE 

; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO.
extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), ztitle:uztit, $
               xtickformat:'noticks',ztickformat:'papco_color_bar_log_ticks'}

; add keyword structure set locally and the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)

;do color plot and color bar plot. nodata specifies the "nodata" flag
;in your data. Here we do not plot axis and data seperate, as
;papco_plot_colorspec cannot be used as an overplot!


resx=10 & resy=0.01
IF control.pl_type EQ 6 OR control.pl_type EQ 7  THEN control.graph = 0 ELSE control.graph = 1
IF control.graph EQ 0 THEN BEGIN 
   nodat = where(zmat EQ nodata, nod)
   IF nod GT 0 THEN zmat(nodat) = !values.f_nan
   lanl_geo_gap_oplot, time, zmat, _extra=extra_plotPar
   
ENDIF ELSE BEGIN 

   IF yscl(panel(0),3) EQ 1 THEN BEGIN ; if logaritmic yaxis
      papco_plot_colorspec,zmat,xarr,yarr,nodata=nodata,$
      xstyle=5, ystyle=5, _extra=extra_plotPar, /VERBOSE,resx=resx, resy=resy
   ENDIF ELSE BEGIN ; linear y axis
      papco_plot_colorspec,zmat,xarr,yarr,nodata=nodata,$
      xstyle=5, ystyle=5, _extra=extra_plotPar, /VERBOSE, resx=resx, resy=resy,dy=0.5
   ENDELSE
   papco_color_bar, _extra=extra_plotPar
enDELSE 

; store the coordinate information into common block
plotted_x = !x & plotted_y = !y

; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this routine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar     


  
; Plot S/C Potential
;IF (scPot EQ 1) THEN BEGIN
;    plot,[0, 1], _extra=extra_plotPar,xstyle=5,ystyle=5, /nodata
;    lanl_geo_gap_oplot, time, scPot_arr, _extra={color:1,psym:0}
;ENDIF

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label, panel, uytitle,/rot90



; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
IF control.graph EQ 0 THEN rot90 = 0 ELSE rot90 = 1

right_side_label,panel,utitle,rot90 = rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
write_output:  
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file', /cont
    description='lanl_geo_pitch '+utitle
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
endif  
   
return
  
;jump to here if no data to plot 
noplot: 
message,'No data to plot',/cont
plotted_x = !x  &  plotted_y = !y  

END

