;******************************************************************************
;*  NAME:        r_lanl_geo
;*
;*  DESCRIPTION: Routine to read lanl_geo files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       plotinfo structure only
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     Rewrite July 2003 with control structure, simplified,
;*               R Friedel
;******************************************************************************
PRO r_lanl_geo, plotInfo, VERBOSE=VERBOSE

COMMON lanl_geo_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

FORWARD_FUNCTION  r_lanl_geo_cal

get_err_no=0  &  get_err_msg='lanl_geo read succesfully' 

; check VERBOSE and keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
   
time=mjdt_start  &  no_files_read=0  &  day_count=0.0  
 
WHILE mjdt_end.mjd GE time.mjd DO BEGIN
    utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
    datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)    
    r_lanl_geo_one, datestr, plotInfo, VERBOSE=VERBOSE
     
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         input_data_buf=temporary(input_data)
       endif else begin                   ;otherwise concatonate arrays
         input_data_buf=[temporary(input_data_buf),temporary(input_data)]
       endelse     
       input_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
    endif
    next_day:
    time.mjd=time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) THEN break
    day_count=day_count+1
endwhile
   
; check if any files were read   
if (no_files_read ne 0) then begin
    get_err_no=0
    get_err_msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
endif else begin
    get_err_no=1 & get_err_msg='no files found for timerange'
    return
endelse      

end

;******************************************************************************
;*  NAME:        r_lanl_geo_one, datestr, plotInfo
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       datestr - date to read
;*               plotinfo - module info structure
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        July 2003
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_lanl_geo_one, datestr, plotInfo, VERBOSE=VERBOSE
  
COMMON lanl_geo_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents
  
get_err_no=0  &  get_err_msg='lanl_geo one read succesfully'

sep = papco_get_pathsep()

control = *plotinfo.USR_PTR1
data_type =  control.type
sat_id = sat_names(control.sc_id)
year_str = strmid(datestr, 0, 4)
message,'Reading for day '+datestr+'  '+sat_id ,/cont

CASE data_type OF 
    0: BEGIN 
        IF NOT (papco_check_data_env('LANL_EP', PATH=data)) THEN return 
        ext = '.text_idl' & sub_dir = 'cpa_sopa_esp'+sep+year_str+sep
        merged = 0 &  pitch = 0
    END
    1: BEGIN 
        IF NOT (papco_check_data_env('LANL_EP', PATH=data)) THEN return 
        ext = '.merged_idl' & sub_dir = 'merged'+sep+year_str+sep
        merged = 1 &  pitch = 0
    END
ENDCASE

; papco uses full 4 letter years for date! All idl savesets are stored
; with 8 letter dates yyyymmdd in a subdirectory for pitch, merged,
; cpa_sopa_esp.
year = strmid(datestr,0,4)
fln = datestr+'_'+sat_id+ext

on_ioerror, read_original
restore, data+sub_dir+fln
dealloc_lun, /quiet    ;frees logical units used by save/restore
IF keyword_set(VERBOSE) THEN message, fln, /cont
return

read_original:
r_lanl_geo_one_original, datestr, plotInfo, VERBOSE = 1

END 
  
;******************************************************************************
;*  NAME:        r_lanl_geo_one_original, datestr, plotInfo
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*  INPUT:       datestr - date to read
;*               plotinfo - module info structure
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        July 2003
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_geo_one_original, datestr, plotInfo, $
                                 VERBOSE = VERBOSE, NOSAVE = NOSAVE, $
                                 PITCH = PITCH,  MERGED = MERGED
  
COMMON lanl_geo_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg  
COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents
      
get_err_no=0  &  get_err_msg='lanl_geo_one_orignal read succesfully'

sep = papco_get_pathsep()

control = *plotinfo.USR_PTR1
data_type =  control.type
sat_id = sat_names(control.sc_id)
message,'Reading original ascii for day '+datestr+'  '+sat_id ,/cont

year = string(strmid(datestr,0,4),format="(i4.4)")
yyyy_date = datestr
yy_date = strmid(datestr,2,6)

CASE data_type OF 
    0: BEGIN 
        IF NOT (papco_check_data_env('LANL_EP', PATH=data)) THEN return 
        IF NOT (papco_check_data_env('LANL_EP_ARCHIVE', PATH=arch)) THEN return
        file_ext = '.text' & sub_dir = 'cpa_sopa_esp'+sep+year+sep
        merged = 0 &  pitch = 0
    END
    1: BEGIN 
        IF NOT (papco_check_data_env('LANL_EP', PATH=data)) THEN return 
        IF NOT (papco_check_data_env('LANL_MERGED_ARCHIVE', PATH=arch)) THEN return 
        file_ext = '.merged' & sub_dir = 'merged'+sep+year+sep
        merged = 1 &  pitch = 0
    END

ENDCASE


;------------------------------------------------------------------------------
; read the original data file. The header of each file contains the
; names of the columns in the file. The same names are used as tags
; for the data structure. TIME is given in decimal hours - convert to TAI

yyyy_fln = arch+year+sep+yyyy_date+'_'+sat_id+file_ext
yy_fln   = arch+year+sep+yy_date  +'_'+sat_id+file_ext

fln=findfile(yyyy_fln+'*', count=c) ;4 letter year date
IF c EQ 0 THEN BEGIN                ;not found, look for 2 letter year
    fln = findfile(yy_fln+'*', count = c)
    IF c EQ 0 THEN BEGIN 
        get_err_no = 9 & get_err_msg = 'No file found for '+yyyy_fln
        message, get_err_msg, /cont & return
    ENDIF 
ENDIF 

;Got here if file found. Now check for gzipped
papco_sepdir, fln(0), dir, file, ext
IF strmid(ext, strlen(ext)-2, 2) EQ 'gz' THEN BEGIN
    ext = strmid(ext, 0, strlen(ext)-3)
    unzip_dir = papco_getenv('PAPCO_DOWNLOAD')
    unzipfile = unzip_dir+file+ext
    message,'Unzipping '+fln(0)+ ' to '+ unzipfile, /cont 
    spawn,'gunzip -fc '+dir+file+ext+'.gz >! '+unzipfile 
    zipflag = 1
    fln = unzipfile
ENDIF ELSE zipflag = 0

header=''
on_ioerror, out  
openr,unit,fln(0),/get_lun

; clear data arrays for reuse
input_header=0  &  input_data=0  
   
input_header={lanl_geo_header, column_id:'',n_ch:0, sat_id:'',file_ext:''}
input_header.sat_id=sat_id
input_header.file_ext=file_ext
    
;read header and make data structures.
header = ''
WHILE header EQ '' DO readf,unit,header
input_header.column_id = header

;get the channel information for this bird. 
channels = r_lanl_geo_cal(sat_id, MERGED=merged)

n_ch = n_elements(channels)
IF merged THEN BEGIN
    header_list = strarr(n_ch+6)
    header_list(0)='TIME' & header_list(1)='LATITUDE'
    header_list(2)='LONGITUDE' & header_list(3)='RADIUS'
    header_list(4)='DST' & header_list(5)='KP'
    ch_st = 6
ENDIF ELSE BEGIN 
    header_list = strarr(n_ch+4)
    header_list(0)='TIME' & header_list(1)='LATITUDE'
    header_list(2)='LONGITUDE' & header_list(3)='RADIUS'
    ch_st = 4
ENDELSE 

nme = 0 & nmp = 0 
nce = 0 & ncp = 0 
nsp = 0 & nse = 0 & nee = 0 & nel = 0 & npl = 0
FOR i = 0, n_ch-1 DO BEGIN
     CASE channels(i).INSTRUMENT OF
         'MPA': BEGIN
             IF channels(i).SPECIES EQ 'E' THEN BEGIN
                 nme = nme+1 
                 header_list(i+ch_st) = 'MPA_E_'+varprt(nme)
             ENDIF ELSE BEGIN
                 nmp = nmp+1 
                 header_list(i+ch_st) = 'MPA_P_'+varprt(nmp)
             ENDELSE 
         END                  
         'CPA':BEGIN
             IF channels(i).SPECIES EQ 'E' THEN BEGIN
                 nce = nce+1
                 header_list(i+ch_st) = 'CPA_E_'+varprt(nce)
             ENDIF ELSE begin
                 ncp = ncp+1
                 header_list(i+ch_st) = 'CPA_P_'+varprt(ncp)
             ENDELSE 
         END
         'SOPA':BEGIN
             IF channels(i).SPECIES EQ 'E' THEN BEGIN
                 nse = nse+1
                 header_list(i+ch_st) = 'SOPA_E_'+varprt(nse)
             ENDIF ELSE begin
                 nsp = nsp+1
                 header_list(i+ch_st) = 'SOPA_P_'+varprt(nsp)
             ENDELSE 
         END
         'ESP':BEGIN
             nee = nee+1
             header_list(i+ch_st) = 'ESP_E_'+varprt(nee)
         END
         'MERGED':BEGIN 
             IF channels(i).SPECIES EQ 'E' THEN BEGIN
                nel = nel+1
                header_list(i+ch_st) = 'MERGED_E_'+varprt(nel)
            ENDIF ELSE begin
                npl = npl+1
                header_list(i+ch_st) = 'MERGED_P_'+varprt(npl)
            ENDELSE 
         END
     ENDCASE
ENDFOR    

;make the "create_struct" function call needed to make the structure!
initval=fltarr(n_ch+ch_st)
call_prm="'"+header_list(0)+"',"+ string(initval(0),format="(f3.1)")
result=EXECUTE('dummy=create_struct('+call_prm+')')
FOR i=1, n_ch+ch_st-1 DO BEGIN
    call_prm="dummy,'"+header_list(i)+"',"+string(initval(i),format="(f3.1)")
    result=EXECUTE('dummy=create_struct('+call_prm+')')
ENDFOR

dummy2=create_struct(dummy, 'TAI',0.0d)

;make enough data array for one day of 10 sec data
input_data=replicate(dummy2, 21000)

i=0  &  in_str=''
year=fix(strmid(datestr,0,4))
mon=fix(strmid(datestr,4,2))
day=fix(strmid(datestr,6,2))

;pad in_str to be long enough for any read
pad = fltarr(15)
pad_str = string(pad, format = "(' ',f8.3)")

WHILE NOT eof(unit) DO BEGIN ;read in data and store in input_data
    readf, unit, in_str
    IF in_str EQ '' THEN GOTO, end_file ;check for empty lines at end of file
    ;error check reading the input line. Ignore "wrong" reads
    error_status=0
    catch, error_status
    IF error_status NE 0 THEN BEGIN
       continue
    ENDIF
    reads, in_str+pad_str, dummy
    catch, /cancel 
    ;add TAI time to structure
    dummy2=create_struct(name='3'+sat_id+file_ext, dummy, 'TAI',0.0d)
    hr=fix(dummy.time) & min=(dummy.time-hr)*60.0
    sec=fix((min-fix(min))*60.0) & min=fix(min)
    IF i LT 10 THEN IF dummy.time GT 23.0 THEN dd = -1 ELSE dd = 0

    tai=UTC2TAI({YEAR:year, MONTH:mon, DAY:day+dd, HOUR:hr, $
                 MINUTE:min, SECOND:sec, MILLISECOND:0})

    ;check for monotonically increasing time
    IF i GT 1 THEN IF input_data(i-1).tai GT tai THEN continue
    dummy2.tai = tai
    input_data(i)=dummy2
    i=i+1     
ENDWHILE 
end_file:
IF i EQ 0 THEN BEGIN
    IF zipflag THEN spawn,'rm -f '+unzipfile
    get_err_no=5 &  get_err_msg='!CNo Data in File'
    message,get_err_msg,/cont
    return
ENDIF
IF zipflag THEN spawn,'rm -f '+unzipfile
   
input_data=input_data(0:i-1)
close,unit & free_lun,unit

;check for monotonically increasing time
IF i GE 2 THEN BEGIN 
    diff_time = input_data(1:i-1).tai-input_data(0:i-2).tai
    index = where(diff_time LT 0, c, COMPLEMENT = idx)
    IF c NE 0 THEN input_data = input_data(idx)
ENDIF


IF i EQ 0 THEN BEGIN 
    get_err_no=6  &  get_err_msg='No data found in file'
    message,get_err_msg, /cont  &  return
ENDIF 

; end of user written section
;------------------------------------------------------------------------------
   
IF keyword_set(NOSAVE) THEN return

; save read data with correct time. Rename data to *_idl
fln = datestr+'_'+sat_id+file_ext
path = data+sub_dir
flname = path+fln+'_idl'

;test if subdir exists, create if necessary
papco_create_dir, path

save, filename=flname, input_header, input_data, /COMPRESS
IF keyword_set(VERBOSE) THEN message,'Saving saveset '+flname, /cont
dealloc_lun,/quiet              ;frees logical units used by save/restore
return
  
out:
; if there was an error, report it.
IF zipflag THEN spawn,'rm -f '+unzipfile
get_err_no=!ERROR  &  get_err_msg='!C '+!ERR_STRING
message, get_err_msg, /cont
close, /all

END 


;******************************************************************************
;*  NAME:        r_lanl_geo_pitch_one_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_geo_pitch_one_original, sat_id, VERBOSE=VERBOSE, EVENT=EVENT
  
  common lanl_geo_pitch_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
   
  get_err_no=0 & get_err_msg='lanl_geo_pitch_one_orignal read succesfully'
  
  if not (papco_check_data_env('LANL_GEO_PITCH', path=PATH)) then return
   
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file. For lanl gep pitch, assemble all data
; into ONE idl saveset as energy / pitch angle matrices
  
; define the header and data structures. Edit to suit your data!
  if keyword_set(EVENT) then begin
    
    year=strmid(event,0,4)
    cmonth=strmid(event,4,7)
    files=findfile(path+'archive/'+year+'/event/'+cmonth+'/*',count=c)
    if c ne 0 then begin
      fln=files(0)  &  pos=strpos(fln,'.dat_')
      ext=strmid(fln,pos,strlen(fln))
      daystr=strmid(fln,pos+5,strlen(fln))  &  days=fix(str_sep(daystr,'-'))
      message, 'Found files for event in '+cmonth+' '+year+ $
        ' for days '+varprt(days(0))+' to '+varprt(days(1)),/cont
      file_root='archive/'+year+'/event/'+cmonth+'/'+sat_id+'_'+year+'_'
    endif else message,'No files found for event in '+cmonth+' '+year ,/cont
  endif
    
  time=0.0d
  sopa_e=fltarr(20)
  sopa_e_arr=fltarr(10, 20)  &  sopa_e_arr(*)=-99
  sopa_p_east=fltarr(20)
  sopa_p_west=fltarr(20)
  sopa_p_arr=fltarr(12, 20)  &  sopa_p_arr(*)=-99
  esp_all=fltarr(40)
  esp1_arr=fltarr(6, 20)     &  esp1_arr(*)=-99
  esp2_arr=fltarr(6, 20)     &  esp2_arr(*)=-99
  
  ;specify input format for ephemeris files
  ephem=fltarr(27)
  
  max_no_rec=9000               ;the maximum number time records 
  dummy={lanl_geo_pitch_data_6, $
         time:0.0d, LATITUDE:0.0, LONGITUDE:0.0, RADIUS:0.0, $
         use_fit:0, $
         sopa_e:sopa_e_arr, $
         sopa_p:sopa_p_arr, $
         esp1:esp1_arr, $
         esp2:esp2_arr }
  input_data=replicate(dummy, max_no_rec)
  
  on_ioerror, skip_sopa_e
  ;first, read all electron sopa channels. File format is colums of:
  ;dec.day,  20 bins of -1 -> +1 ,step 0.1 of cosine of pitch angle.
  ;and there are 10 energies in 10 seperate files. 
  n=-1
  for ch=1,10 do begin
    n=-1
    file=path+file_root+'sopa_e'+varprt(ch)+'_pa'
    if ch eq 10 then file=path+file_root+'sopa_e'+varprt(ch)+'pa'
    openr,unit,file+ext,/get_lun
    message,'Reading '+file+ext,/cont
    while not eof(unit) do begin   ;read in data and store in input_data
      readf,unit,time,sopa_e
      n=n+1
      input_data(n).sopa_e(ch-1,*)=sopa_e
    endwhile
    close,unit  &  free_lun,unit
  endfor
  goto, read_sopa_p
  
  skip_sopa_e:
  message,'Problem reading SOPA electrons',/cont
  
  read_sopa_p: on_ioerror, skip_sopa_p
  ;now read the proton sopa channels. File format is colums of:
  ;dec.day,  20 bins of -1 -> +1 ,step 0.1 of cosine of pitch angle.
  ;and there are 12 energies in 12 seperate files. 
  n=-1
  for ch=1,12 do begin
    n=-1
    file=path+file_root+'sopa_p'+varprt(ch)+'_pa'
    openr,unit,file+ext,/get_lun
    message,'Reading '+file+ext,/cont
    while not eof(unit) do begin   ;read in data and store in input_data
      readf,unit,time,sopa_p_east,sopa_p_west
      sopa_p=(sopa_p_east+sopa_p_west)/2.0
      n=n+1
      input_data(n).sopa_p(ch-1,*)=sopa_p
    endwhile
    close,unit  &  free_lun,unit
  endfor  
  goto, read_sopa_esp
  
  skip_sopa_p:
  message,'Problem reading SOPA protons',/cont
  
  read_sopa_esp:  &  on_ioerror, skip_esp  
  ;now read the esp channels. File format is colums of:
  ;dec.day,  20 bins of -1 -> +1 ,step 0.1 of cosine of pitch angle.
  ;and there are 6 energies in 12 seperate files. 
  ;data comes in pairs (esp1, esp2) for each pa bin
  n=-1  &  index_1=indgen(20)*2  &  index_2=indgen(20)*2+1
  for ch=1,6 do begin
    n=-1
    file=path+file_root+'sopa_esp_e'+varprt(ch)+'_pa'
    openr,unit,file+ext,/get_lun
    message,'Reading '+file+ext,/cont
    while not eof(unit) do begin   ;read in data and store in input_data
      readf,unit,time,esp_all
      n=n+1
      input_data(n).esp1(ch-1,*)=esp_all(index_1)
      input_data(n).esp2(ch-1,*)=esp_all(index_2)
    endwhile
    close,unit  &  free_lun,unit
  endfor  
  goto, read_eph
  
  skip_esp:
  message,'Problem reading ESP electrons',/cont
  
  read_eph: on_ioerror, skip_eph
  ;now read the eph & time info. File format is colums of:
  ;specify input format for ephemeris file
  ephem={pitch_eph, time:0.0d, xlat:0.0, xlng:0.0, xalt:0.0, $
         effx:0.0, effy:0.0, el:0.0, xmlong:0.0, blocal:0.0, bmin:0.0, $
         field:fltarr(4), t:fltarr(3,2), iflag:0, e:fltarr(3),dt_max:0.0, $
         maxidx:0,use_fit:0}
  
  n=-1  &  index_1=indgen(20)*2  &  index_2=indgen(20)*2+1
  file=path+file_root+'ephemeris_pa'
  openr,unit,file+ext,/get_lun
  message,'Reading '+file+ext,/cont
  while not eof(unit) do begin   ;read in data and store in input_data
    readf,unit,ephem, format="(x,f12.6,2(x,f9.4),x,f9.6,3(x,f9.6),x,f9.4,2(x,e10.3),4(x,f9.4),6(x,f11.4),x,i1,3(x,e10.3),x,e10.4,i1,x,i2)"
    n=n+1
    input_data(n).time=ephem.time
    input_data(n).LATITUDE=ephem.xlat
    input_data(n).LONGITUDE=ephem.xlng
    input_data(n).RADIUS=ephem.xalt
    input_data(n).use_fit=ephem.use_fit
  endwhile
  close,unit  &  free_lun,unit
  goto,read_ok
  
  skip_eph:
  message,'Problem reading Ephemeris',/cont
  return
  
  read_ok:
  input_data_all=input_data(0:n)
  input_data.time=(input_data.time-fix(input_data.time))*24.0d
  
; make header infor - read channel assignment 
  r_lanl_geo_header,'yyddmm_'+sat_id+'.pitch', header, [0,0,0,2]
  input_header={lanl_geo_pitch_header_2, header:header, sat_id:sat_id, ndat:n}
  
; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
; for event, save one file per day!  
  if keyword_set(event) then begin
    first=convert_date_to_t90(year=fix(year), doy=days(0), /mjdt)
    last= convert_date_to_t90(year=fix(year), doy=days(1), /mjdt)
    time=first
    message,'Writing data to savsets for days '+daystr,/cont
    on_ioerror, out
    for i=first.mjd, last.mjd do begin
      time.mjd=i  &   day=days(0)+i-first.mjd
      convert_t90_to_Date, time, yr, day_oy, hour, minute, second, $
        month, cmonth, day_om, /MJDT
      date=string(yr,month,day_om,format="(i4.4,i2.2,i2.2)")
      fln=date+'_'+sat_id+'.pitch.idl'
      index=where(fix(input_data_all.time) eq day,c)
      print,fln, day, c 
      if c ne 0 then begin
        input_data=input_data_all(index)
        input_data.time=(input_data.time-fix(input_data.time))*24.0d
        input_header.ndat=c
        save, filename=path+'/savesets/'+year+'/'+fln, $
          input_header,input_data, VERBOSE=VERBOSE
      endif
    endfor  
  endif  
  
  return
  
  out:
  ; if there was an error, report it. 
  get_err_no=!ERROR
  get_err_msg='!C '+!ERR_STRING
  message, get_err_msg,/cont
  
end

pro r_lanl_geo_pitch_one_original_batch
  
  sats=['1990-095','1994-084','lanl-97a', '1991-080']
  
  for i=0,n_elements(sats)-1 do begin
    r_lanl_geo_pitch_one_original, sats(i), event='1997may', /verb
    r_lanl_geo_pitch_one_original, sats(i), event='1998aug', /verb
    r_lanl_geo_pitch_one_original, sats(i), event='1998jun', /verb
    r_lanl_geo_pitch_one_original, sats(i), event='1998mar', /verb
    r_lanl_geo_pitch_one_original, sats(i), event='1998may', /verb
    r_lanl_geo_pitch_one_original, sats(i), event='1998oct', /verb
    r_lanl_geo_pitch_one_original, sats(i), event='1998sep', /verb
  endfor
  
end

;******************************************************************************
;*  NAME:        r_lanl_geo_flux_converison
;*
;*  DESCRIPTION: Routine to read lanl_geo flux conversion factors
;*
;*  INPUT:       sat         name of satellite
;*
;*  OUTPUT:      structure containing energy channels and converion factors.
;*
;*               To convert counts/s -> Flux do the following;
;*               if  C = counts/s  and F = flux
;*          
;*               then  F[i] = C[i]/FCF[i]
;*          
;*               I.e. you divide the Counts/s by the FCFs returned.
;*          
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore
;*                           comments
;*               MERGED   -  read from merged files
;*
;*  DATE:        March 2002
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
function r_lanl_geo_flux_converison, sat,  MERGED = MERGED, VERBOSE = VERBOSE

IF keyword_set(VERBOSE) THEN verbose = VERBOSE ELSE verbose = 0

root = papco_getenv('PAPCO_MODULES')
path = 'lanl_geo/FluxConversionFactors/'

;make standard structure for each satellite to contain conv. info
channel_info = {channel_info, $
                instrument:'', species:'', label:'', $
                ek_low:0.0, ek_high:0.0, fcf:0.0}       
channels = replicate(channel_info, 100)

IF keyword_set(MERGED) THEN BEGIN
    fln2 = 'EnergyRanges_merged.txt'
    n_ch = 100
    GOTO, merged
ENDIF

fln1 = 'FluxConversionFactors.txt'
fln2 = 'EnergyRanges.txt'

;first, read flux conversion factors
openr, u, root+path+fln1, /get_lun
instr = '' & result = ['', '']

WHILE NOT eof(u) DO BEGIN
    readf, u, instr & IF instr EQ '' THEN continue
    pos =  strpos(instr, sat)
    IF pos NE -1 THEN BEGIN
        IF verbose THEN $
          message, 'Reading Conversion Factors for '+instr, /cont

        WHILE result(0) NE 'Type' DO BEGIN
            readf, u, instr & result = strsplit(instr, ': ', /extract)
        ENDWHILE
        instrument = result(1)

        WHILE result(0) NE 'nE' DO BEGIN
            readf, u, instr & result = strsplit(instr, ': ', /extract)
        ENDWHILE
        e_ch = fix(result(1))

        FOR i = 0, e_ch-1 DO BEGIN
            readf, u, instr
            channels(i).instrument = instrument
            channels(i).species = 'electrons'
            channels(i).fcf = float(instr)
        ENDFOR  
        n_ch = e_ch

        WHILE result(0) NE 'nP' DO BEGIN
            readf, u, instr & result = strsplit(instr, ': ', /extract)
        ENDWHILE
        p_ch = fix(result(1))

        FOR i = n_ch, n_ch+p_ch-1 DO BEGIN
            readf, u, instr
            channels(i).instrument = instrument
            channels(i).species = 'ions'
            channels(i).fcf = float(instr)
        ENDFOR  
        n_ch = n_ch+p_ch
        
        ;check for ESP channels present
        WHILE strmid(instr, 0, 4) NE 'Bird' DO BEGIN
            readf, u, instr 
            IF instr EQ 'Type: ESP' THEN BEGIN
                WHILE result(0) NE 'nE' DO BEGIN
                    readf, u, instr & result = strsplit(instr, ': ', /extract)
                ENDWHILE
                esp_e = fix(result(1))                
                
                FOR i = n_ch, n_ch+esp_e-1 DO BEGIN
                    readf, u, instr
                    channels(i).instrument = 'ESP'
                    channels(i).species = 'electrons'
                    channels(i).fcf = float(instr)
                ENDFOR  
                n_ch = n_ch+esp_e

               WHILE result(0) NE 'nP' DO BEGIN
                    readf, u, instr & result = strsplit(instr, ': ', /extract)
                ENDWHILE
                esp_p = fix(result(1))


                FOR i = n_ch, n_ch+esp_p-1 DO BEGIN
                    readf, u, instr
                    channels(i).instrument = 'ESP'
                    channels(i).species = 'ions'
                    channels(i).fcf = float(instr)
                ENDFOR  
                n_ch = n_ch+esp_p

                break
            ENDIF 
        ENDWHILE        

    ENDIF

ENDWHILE

close, u & free_lun, u

merged:

;now, read energy range information. Return all energies in keV!
unit_str=['MEV','KEV','EV','MEV']  &  units=[1000,1,0.001,1000]

openr, u, root+path+fln2, /get_lun
instr = ''

WHILE NOT eof(u) DO BEGIN
    readf, u, instr & IF instr EQ '' THEN continue
    pos =  strpos(instr, sat)
    IF pos NE -1 THEN BEGIN
        IF verbose THEN $
          message, 'Reading Energy Channel Ranges for '+instr, /cont
        pos =  strpos(instr, 'ch')
        IF pos NE -1 THEN n_ch = fix(strmid(instr, pos+3, 3))
        readf, u, instr
        instr = strupcase(strcompress(instr))
        ch = strsplit(instr, ' ', /extract)
        FOR i = 0, n_ch-1 DO BEGIN
            ch_str=strmid(ch(i),1,strlen(ch(i)))
            FOR  j=0,2 DO  BEGIN 
                pos=strpos(ch_str,unit_str(j))
                if pos ne -1 then break
            ENDFOR
            ch_str=strmid(ch_str,0,pos)
            ch_str=str_sep(ch_str,'-')
            channels(i).ek_low = ch_str(0)*units(j)
            channels(i).ek_high = ch_str(1)*units(j)
            channels(i).label = ch(i)
        ENDFOR  
        break
    ENDIF

ENDWHILE

close, u & free_lun, u

IF keyword_set(MERGED) THEN BEGIN
    channels = channels(0:n_ch-1)
    channels(*).INSTRUMENT = 'MRG'
    FOR i = 0, n_ch-1 DO BEGIN
        pos = strpos(channels(i).LABEL, 'P')
        IF pos NE -1 THEN channels(i).SPECIES = 'ions' ELSE $
          channels(i).SPECIES = 'electrons'
    ENDFOR 
ENDIF

idx=where(channels.INSTRUMENT EQ 'ESP',c) 
;IF c NE 0 THEN channels(idx).FCF = channels(idx).FCF*1e3

channels = channels(0:n_ch-1) 
return, channels

END

;******************************************************************************
;*  NAME:        r_lanl_geo_coverage
;*
;*  DESCRIPTION: Routine to make a file of geo coverage
;*
;*  INPUT:       none
;*
;*  OUTPUT:      a file of coverage data
;*          
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        March 2002
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
PRO m_lanl_geo_coverage, HAVE_TAI = HAVE_TAI, NOREAD = NOREAD, $
                         SATRANGE = SATRANGE

COMMON lanl_geo, sat_names
COMMON lanl_geo_data, input_header, input_data
COMMON lanl_geo_coverage, tai
COMMON get_error, get_err_no, get_err_msg  

papco, /set_only

; get env. vars
lanl_geo_check_env_vars, data, arch, ftp, MERGED=merged, ORIGINAL=org

years = 1976+indgen(28)
yy = strarr(n_elements(years))
yy(0:23) = '19' & yy(24:27) = '20'

IF NOT keyword_set(SATRANGE) THEN BEGIN
    from = 0 & to = n_elements(sat_names)
ENDIF ELSE BEGIN
    from = satrange(0) & to = satrange(1)
ENDELSE 

message, 'Finding coverage for ', /cont
FOR i = from, (to < n_elements(sat_names))-1 DO print, sat_names(i)


FOR i = from, (to < n_elements(sat_names))-1 DO BEGIN
    sat = sat_names(i)
    files = ['']

    IF keyword_set(HAVE_TAI) THEN BEGIN
        IF keyword_set(NOREAD) THEN GOTO, jump
        file = findfile(data+'cpa_sopa_esp/'+sat+'_tai.idl', count = c)
        IF c EQ 1 THEN BEGIN 
           restore, file(0), /verb
           GOTO, jump
        ENDIF
    ENDIF 

    ;assemble list of all files found for this satellite.    
    FOR j = 0, n_elements(years)-1 DO BEGIN
        cd, arch+varprt(years(j)), current = current
        result= findfile('*'+sat+'*.text.gz', count = n)
        IF n NE 0 THEN BEGIN
            IF NOT keyword_set(files) THEN files = yy(j)+result ELSE $
              files = [files, yy(j)+result]
        ENDIF
        print, arch+varprt(years(j)), n
        cd,  current
    ENDFOR

    print, ''
    n_files = n_elements(files)
    message, 'Found '+varprt(n_files)+' files for '+sat, /cont
    print, ''

    tai = dblarr(n_files*1400*6) ; max # of data points of 10s data
    count = 0l

    plotInfo = papco_getplotinfostruct()
    plotInfo.ioptions(3) = 0
    plotInfo.ioptions(10) = i

    ;loop through files and read the time array.
    FOR k = 0, n_elements(files)-1 DO BEGIN
        datestr = strmid(files(k), 0, 8)
        r_lanl_geo_one, datestr, plotInfo, /VERBOSE
        IF get_err_no NE 0 THEN continue
        ndat = n_elements(input_data)
        tai(count:count+ndat-1) = input_data.tai
        count = count+ndat
        message, varprt(100.0*k/n_files)+'% done', /cont
    ENDFOR
    tai = tai(0:count-1)
    save,file=data+'cpa_sopa_esp/'+sat+'_tai.idl',tai, /compress,/verb 

jump:
    count = n_elements(tai)
    
    ;now search through time array to find data intervals.
    ;ignore gaps in the data shorter than 10 minutes. 
    gap = 600.0
    tai_diff = tai(1:count-1)-tai(0:count-2)
    index = where(tai_diff gt gap, c)
 
    n_intervals = c+2
    intervals = dblarr(n_intervals, 2)
   
    IF c EQ 0 THEN BEGIN
                
    ENDIF ELSE BEGIN
        intervals(0, 0) = tai(0)
        intervals(0, 1) = tai(index(0))

        FOR j = 1, n_intervals-3 DO BEGIN
            intervals(j, 0) = tai(index(j-1)+1)
            intervals(j, 1) = tai(index(j))
        ENDFOR 

        intervals(n_intervals-1, 0) = tai(index(c-1))
        intervals(n_intervals-1, 1) = tai(count-1)
       
    ENDELSE

    ;save time ranges of data to a papco_events file
    path = papco_getenv('PAPCO_HOME')+'papco_products/event_times/'
    openw, u, path+'lanl_geo_'+sat+'_coverage.event', /get_lun
    FOR k = 0, n_intervals-1 DO BEGIN
        t1 = tai2utc(intervals(k, 0), /ext)
        t2 = tai2utc(intervals(k, 1), /ext)
        out_str = $
          string(t1.month, t1.day, t1.year, t1.hour, t1.minute, t1.second, $
             t2.month, t2.day, t2.year, t2.hour, t2.minute, t2.second, $
             format = "(i2.2,'/',i2.2,'/',i4.4,' ',i2.2,':',i2.2,':',i2.2,' - ',i2.2,'/',i2.2,'/',i4.4,' ',i2.2,':',i2.2,':',i2.2)")
        printf, u, out_str
    ENDFOR
    close, u & free_lun, u
    message, 'Wrote '+path+'lanl_geo_'+sat+'_coverage.event', /cont

    tmp = temporary(files) &  tmp = ''

ENDFOR

END

;******************************************************************************
;*  NAME:        r_lanl_geo_merged_header
;*
;*  DESCRIPTION: reads header info of lanl geo files and makes a
;*               seperate file of energy range info for all satellites.
;*
;*  INPUT:       none
;*
;*  OUTPUT:      a file of coverage data
;*          
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore
;                            comments
;*               MERGED   -  reads headers for merged files
;*
;*  DATE:        January 2004
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_lanl_geo_headers, MERGED = MERGED

COMMON lanl_geo                 ;see lanl_geo_commons for details

temp = papco_getenv('PAPCO_DOWNLOAD')
out_dir = papco_getenv('PAPCO_MODULES')+'lanl_geo/FluxConversionFactors/'

path = '/n/projects/cda/lanl/geo/ep_archive'
out_fln = 'EnergyRanges_sopa_esp_2.txt'

IF keyword_set(MERGED) THEN BEGIN
    path = '/n/projects/cda/lanl/geo/merged'
    out_fln = 'EnergyRanges_merged.txt'
ENDIF 

openw, u1, out_dir+out_fln, /get_lun

unit_str=['MEV','KEV','EV']  &  units=[1000,1,0.001]

;loop through all satellites in sat_names

FOR i = 0, n_elements(sat_names)-1 DO BEGIN

    files = file_search(path+'/????/*'+sat_names(i)+'*.gz', count = c)

    print, sat_names(i), c
    printf, u1, sat_names(i)
    
    IF c NE 0 THEN BEGIN
        file = files(0)
        dir = file_dirname(file) 
        fln = file_basename(file)
        ;gunzip the file to a temp location
        print, 'gunzip -c '+file+' >! '+temp+'temp.txt'
        spawn, 'gunzip -c '+file+' >! '+temp+'temp.txt'

        in_str = ''
        openr, u, temp+'temp.txt', /get_lun
        readf, u, in_str &  print, in_str
        close, u & free_lun, u

        file_delete, temp+'temp.txt', /verb

        ;make an array of headers
        headers = strsplit(strcompress(in_str), /EXTRACT)

        ;energy channel descriptors end with "eV"
        pos = strpos(headers, 'eV') & idx = where(pos NE -1, c)
        IF c EQ 0 THEN continue
        old_headers = headers(idx)
        headers = strupcase(headers(idx))
    
        FOR ch = 0, c-1 DO BEGIN

            ;fix for missing "P" in last entry of 1989-046
            IF sat_names(i) EQ '1989-046' THEN BEGIN
                IF strmid(headers(ch), 0, 1) EQ '7' THEN $
                  headers(ch) = 'P'+headers(ch)
            ENDIF
    
            ;get species string
            species = strmid(headers(ch), 0, 1)

            ;get units string
            k = 0 & pos = -1
            WHILE pos EQ -1 DO BEGIN 
                pos = strpos(headers(ch),unit_str(k)) & k = k+1
            ENDWHILE 
            k = k-1

            range_str = strmid(headers(ch), 1, pos-1)
            ;check for integral channels, change
            IF strmid(range_str, 0, 1) EQ '>' THEN $
                range_str = strmid(range_str, 1, strlen(range_str)-1)+'-99.999'
                   
            IF range_str EQ '3.94.8' THEN range_str = '3.9-4.8'

            range = strsplit(range_str, '-', /EXTRACT)
            IF range(1) EQ '1160' THEN range(1) = '11.6'

            e1 = float(range(0))*units(k) & e2 = float(range(1))*units(k)

            out_str = strformat(old_headers(ch), 20, /LEFT)+ $
                   strformat(species,5, /CENTER)+ $
                   string(e1, e2, format = "(2(f15.5))")+ '   keV' 
            print, out_str
            printf, u1, out_str

        ENDFOR 

    ENDIF 
    printf, u1, ''

ENDFOR

close, u1 & free_lun, u1

END


;******************************************************************************
; reads lanl geo cal file
FUNCTION r_lanl_geo_cal, sat, MERGED = MERGED, VERBOSE = VERBOSE

;all units are in keV!!!

in_dir = papco_getenv('PAPCO_MODULES')+'lanl_geo/FluxConversionFactors/'
in_fln = 'cal_cpa_sopa_esp.txt'

IF keyword_set(MERGED) THEN BEGIN
    in_fln = 'cal_merged.txt'
ENDIF

lbl = '' & inst = '' & spec = '' & ek_low = 0.0 & ek_high = 0.0 & fcf = 0.0

channel_info = {cals, $
                label:'', $
                instrument:'', $
                species:'', $
                ek_low:0.0, ek_high:0.0, $
                units:'keV', $
                fcf:0.0}       

openr, u, in_dir+in_fln, /get_lun
in_str = '' & ch_count = 0

WHILE NOT eof(u) DO BEGIN
    
    readf, u, in_str 
    IF in_str EQ ';' THEN CONTINUE
    
    IF in_str EQ sat THEN BEGIN
        readf, u, in_str 
        WHILE in_str NE ';' DO BEGIN 
            reads, in_str, lbl, inst, spec, ek_low, ek_high, fcf, $
              format = "(a30, a8, a5, 2(f15.5), f15.4)"
            IF keyword_set(verbose) THEN $
              print, lbl, inst, spec, ek_low, ek_high, fcf, $
                format = "(a30, a8, a5, 2(f15.5), f15.4)"
            channel_info.label = strcompress(lbl, /REMOVE_ALL)
            channel_info.instrument = strcompress(inst, /REMOVE_ALL)
            channel_info.species = strcompress(spec, /REMOVE_ALL)
            channel_info.ek_low = ek_low
            channel_info.ek_high = ek_high
            channel_info.fcf = fcf
            IF ch_count EQ 0 THEN ch = channel_info ELSE ch = [ch, channel_info]
            ch_count= ch_count+1
            readf, u, in_str
        ENDWHILE 
        BREAK
    ENDIF     
    
ENDWHILE 

close, u & free_lun, u

return, ch

END 


;******************************************************************************
FUNCTION sopa_cpa_esp_cal_def, sat

;all units are in keV!!!

dummy = {G:0.0,l1:'',l2:'',e1:0.0,e2:0.0,fcf:0.0}

;set CPA electrons - they are all the same
cpa_e_ch = replicate(dummy, 12)
cpa_e_ch[0] = {G:3.6e-3,l1:'LoE',l2:'CPA',e1:  30.0,e2:  45.0,fcf:-1.0} 
cpa_e_ch[1] = {G:3.6e-3,l1:'LoE',l2:'CPA',e1:  45.0,e2:  65.0,fcf:-1.0} 
cpa_e_ch[2] = {G:3.6e-3,l1:'LoE',l2:'CPA',e1:  65.0,e2:  95.0,fcf:-1.0} 
cpa_e_ch[3] = {G:3.6e-3,l1:'LoE',l2:'CPA',e1:  95.0,e2: 140.0,fcf:-1.0} 
cpa_e_ch[4] = {G:3.6e-3,l1:'LoE',l2:'CPA',e1: 140.0,e2: 200.0,fcf:-1.0} 
cpa_e_ch[5] = {G:3.6e-3,l1:'LoE',l2:'CPA',e1: 200.0,e2: 300.0,fcf:-1.0} 

cpa_e_ch[6] = {G:1.8e-2,l1:'HiE',l2:'CPA',e1: 200.0,e2: 300.0,fcf:-1.0}
cpa_e_ch[7] = {G:1.8e-2,l1:'HiE',l2:'CPA',e1: 300.0,e2: 400.0,fcf:-1.0}
cpa_e_ch[8] = {G:1.8e-2,l1:'HiE',l2:'CPA',e1: 400.0,e2: 600.0,fcf:-1.0}
cpa_e_ch[9] = {G:1.8e-2,l1:'HiE',l2:'CPA',e1: 600.0,e2: 900.0,fcf:-1.0}
cpa_e_ch[10]= {G:1.8e-2,l1:'HiE',l2:'CPA',e1: 900.0,e2:1400.0,fcf:-1.0}
cpa_e_ch[11]= {G:1.8e-2,l1:'HiE',l2:'CPA',e1:1400.0,e2:2000.0,fcf:-1.0}
 
CASE sat OF
    
    '1976-059':BEGIN

        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:145.0,e2:160.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:160.0,e2:175.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:175.0,e2:190.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:190.0,e2:215.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:215.0,e2:245.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:245.0,e2:290.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:290.0,e2:340.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:340.0,e2:400.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:400.0,e2:480.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:480.0,e2:560.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:400.0,e2:500.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:500.0,e2:630.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:630.0,e2:800.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   800.0,e2:  1000.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1000.0,e2:  1300.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1300.0,e2:  1700.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1700.0,e2:  2700.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2700.0,e2:  4700.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4700.0,e2:  9000.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  9000.0,e2: 13000.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 13000.0,e2: 26000.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 26000.0,e2: 39000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 39000.0,e2: 61000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 61000.0,e2: 92000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 92000.0,e2:146000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:146000.0,e2:220000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)
        
    END 
    
    '1977-007':BEGIN
        
        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:147.0,e2:162.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:162.0,e2:170.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:170.0,e2:195.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:195.0,e2:220.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:220.0,e2:250.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:250.0,e2:295.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:295.0,e2:340.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:340.0,e2:405.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:405.0,e2:490.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:490.0,e2:580.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:450.0,e2:540.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:540.0,e2:680.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:680.0,e2:830.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   830.0,e2:  1100.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1100.0,e2:  1400.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1400.0,e2:  1750.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1750.0,e2:  2800.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2800.0,e2:  4800.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4800.0,e2:  9000.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  9000.0,e2: 14000.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 14000.0,e2: 25000.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 25000.0,e2: 38000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 38000.0,e2: 61000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 61000.0,e2: 92000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 92000.0,e2:144000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:144000.0,e2:210000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)

    END         

    '1979-053':BEGIN

        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:105.0,e2:133.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:133.0,e2:156.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:156.0,e2:187.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:187.0,e2:225.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:225.0,e2:265.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:265.0,e2:320.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:320.0,e2:390.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:390.0,e2:460.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:460.0,e2:555.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:555.0,e2:670.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:390.0,e2:480.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:480.0,e2:630.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:630.0,e2:790.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   790.0,e2:  1000.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1000.0,e2:  1300.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1300.0,e2:  1700.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1700.0,e2:  2700.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2700.0,e2:  4500.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4500.0,e2:  8600.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  8600.0,e2: 13000.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 13000.0,e2: 26000.0,fcf:-1.0}
        ch[33]= {G:8.0e-3,l1:'HiP',l2:'CPA',e1: 26000.0,e2: 38000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 38000.0,e2: 60000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 60000.0,e2: 91000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 91000.0,e2:145000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:145000.0,e2:200000.0,fcf:-1.0}
        
        ch.fcf = ch.g * (ch.e2-ch.e1)

    END 

    '1981-025':BEGIN

        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:107.0,e2:133.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:133.0,e2:157.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:157.0,e2:197.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:197.0,e2:236.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:236.0,e2:295.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:295.0,e2:340.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:340.0,e2:420.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:420.0,e2:500.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:500.0,e2:610.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:610.0,e2:750.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:400.0,e2:470.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:470.0,e2:620.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:620.0,e2:780.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   780.0,e2:  1000.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1000.0,e2:  1300.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1300.0,e2:  1700.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1700.0,e2:  2600.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2600.0,e2:  4400.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4400.0,e2:  8000.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  8000.0,e2: 12000.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 12000.0,e2: 24000.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 24000.0,e2: 43000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 43000.0,e2: 68000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 68000.0,e2:110000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:110000.0,e2:160000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:160000.0,e2:210000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)

    END 

    '1982-019':BEGIN
        
        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 95.0,e2:110.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:110.0,e2:130.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:130.0,e2:150.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:150.0,e2:195.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:195.0,e2:230.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:230.0,e2:255.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:255.0,e2:320.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:320.0,e2:380.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:380.0,e2:460.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:460.0,e2:600.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:370.0,e2:520.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:520.0,e2:640.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:640.0,e2:800.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   800.0,e2:  1000.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1000.0,e2:  1300.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1300.0,e2:  1700.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1700.0,e2:  2700.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2700.0,e2:  4400.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4400.0,e2:  8300.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  8300.0,e2: 13000.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 13000.0,e2: 24000.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 24000.0,e2: 41000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 41000.0,e2: 64000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 64000.0,e2:100000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:100000.0,e2:160000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:160000.0,e2:230000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)

    END 

    '1984-037':BEGIN
        
        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 70.0,e2: 83.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 83.0,e2:110.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:110.0,e2:130.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:130.0,e2:160.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:160.0,e2:190.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:190.0,e2:235.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:235.0,e2:290.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:290.0,e2:350.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:350.0,e2:450.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:450.0,e2:560.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:380.0,e2:490.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:490.0,e2:680.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:680.0,e2:870.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   870.0,e2:  1050.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1050.0,e2:  1450.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1450.0,e2:  1800.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1800.0,e2:  2900.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2900.0,e2:  4800.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4800.0,e2:  9000.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  9000.0,e2: 14000.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 14000.0,e2: 24000.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 24000.0,e2: 39000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 39000.0,e2: 60000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 60000.0,e2: 90000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 90000.0,e2:145000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:145000.0,e2:200000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)

    END 

    '1984-129':BEGIN

        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 72.0,e2: 91.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 91.0,e2:104.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:104.0,e2:125.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:125.0,e2:153.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:153.0,e2:190.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:190.0,e2:235.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:235.0,e2:292.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:292.0,e2:365.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:365.0,e2:457.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:457.0,e2:573.0,fcf:-1.0} 
        
        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:360.0,e2:480.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:480.0,e2:630.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:630.0,e2:800.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   800.0,e2:  1000.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1000.0,e2:  1400.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1400.0,e2:  1800.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1800.0,e2:  2800.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2800.0,e2:  4800.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4800.0,e2:  9100.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  9100.0,e2: 14500.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 14500.0,e2: 25000.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 25000.0,e2: 37000.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 37000.0,e2: 57000.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 57000.0,e2: 88000.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 88000.0,e2:145000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:145000.0,e2:210000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)

    END 

    '1987-097':BEGIN
        
        ch = replicate(dummy, 38)

        ch[0:11] = cpa_e_ch

        ch[12]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 75.0,e2: 88.0,fcf:-1.0} 
        ch[13]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1: 88.0,e2:106.0,fcf:-1.0} 
        ch[14]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:106.0,e2:129.0,fcf:-1.0} 
        ch[15]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:129.0,e2:157.0,fcf:-1.0} 
        ch[16]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:157.0,e2:194.0,fcf:-1.0} 
        ch[17]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:194.0,e2:241.0,fcf:-1.0} 
        ch[18]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:241.0,e2:299.0,fcf:-1.0} 
        ch[19]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:299.0,e2:376.0,fcf:-1.0} 
        ch[20]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:376.0,e2:472.0,fcf:-1.0} 
        ch[21]= {G:3.9e-3,l1:'LoP',l2:'CPA',e1:472.0,e2:590.0,fcf:-1.0} 

        ch[22]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:370.0,e2:480.0,fcf:-1.0} 
        ch[23]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:480.0,e2:670.0,fcf:-1.0} 
        ch[24]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:670.0,e2:850.0,fcf:-1.0} 
        ch[25]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:   850.0,e2:  1030.0,fcf:-1.0}
        ch[26]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1030.0,e2:  1330.0,fcf:-1.0}
        ch[27]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1330.0,e2:  1720.0,fcf:-1.0}
        ch[28]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  1720.0,e2:  2600.0,fcf:-1.0}
        ch[29]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  2600.0,e2:  4740.0,fcf:-1.0}
        ch[30]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  4740.0,e2:  8960.0,fcf:-1.0}
        ch[31]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1:  8960.0,e2: 13600.0,fcf:-1.0}
        ch[32]= {G:4.4e-2,l1:'HiP',l2:'CPA',e1: 13600.0,e2: 23900.0,fcf:-1.0}
        ch[33]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 23900.0,e2: 36400.0,fcf:-1.0}
        ch[34]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 36400.0,e2: 58300.0,fcf:-1.0}
        ch[35]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 58300.0,e2: 83800.0,fcf:-1.0}
        ch[36]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1: 83800.0,e2:147000.0,fcf:-1.0}
        ch[37]= {G:8.0e-2,l1:'HiP',l2:'CPA',e1:147000.0,e2:192000.0,fcf:-1.0}

        ch.fcf = ch.g * (ch.e2-ch.e1)
	 
    END 
	
    '1989-046':BEGIN 

        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:1095.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 522.0}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf: 237.0}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf: 144.0}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf: 101.0}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  51.1}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  40.0}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  24.2}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  21.0}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:9438.0}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 479.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 320.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 170.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf: 108.0}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  59.3}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  27.7}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  13.0}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  11.4}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  6.26}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  3.99}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  2.81}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf: 0.179}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)

    END 

    '1990-095':BEGIN 

        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:1919.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 470.0}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf: 235.0}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf: 153.0}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf:  78.8}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  62.5}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  34.1}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  24.5}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  16.5}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:8661.0}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 685.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 235.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 135.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf:  89.9}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  53.3}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  26.4}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  12.9}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  13.5}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  7.36}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  4.31}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  2.50}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf: 0.302}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)

    END

    '1991-080':BEGIN

        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:2631.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 508.0}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf: 297.0}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf: 178.0}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf: 104.0}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  68.4}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  44.6}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  30.0}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  24.3}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:11340.0}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 439.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 253.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 162.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf: 108.0}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  61.8}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  26.1}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  12.7}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  11.2}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  7.24}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  3.74}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  3.98}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf: 0.304}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)

    END 

    '1994-084':BEGIN 

        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:1428.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 234.0}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf: 194.0}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf: 134.0}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf: 103.0}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  51.0}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  37.0}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  25.0}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  18.0}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:  25.8}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 350.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 188.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 128.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf: 110.0}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  62.0}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  25.0}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  16.0}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  17.0}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  9.00}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  5.20}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  3.00}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf: 0.366}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)

    END 

    'LANL-97A':BEGIN

        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:1280.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 108.9}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf:  95.8}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf:  64.2}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf:  48.9}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  38.0}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  22.1}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  19.0}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  14.5}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:  15.8}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 350.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 236.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 130.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf:  95.6}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  49.4}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  20.6}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  11.8}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  9.91}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  5.71}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  3.29}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  2.49}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf:  0.15}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)       

    END

    'LANL-01A':BEGIN
                
        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:1280.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 108.9}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf:  95.8}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf:  64.2}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf:  48.9}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  38.0}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  22.1}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  19.0}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  14.5}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:  15.8}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 350.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 236.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 130.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf:  95.6}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  49.4}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  20.6}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  11.8}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  9.91}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  5.71}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  3.29}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  2.49}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf:  0.15}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)

    END

    'LANL-02A':BEGIN

        ch = replicate(dummy, 31)

        ch[0] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf:1280.0}
        ch[1] = {G:-1.0,l1:'E',l2: 'SOPA',e1:   75.0,e2:  105.0,fcf: 108.9}
        ch[2] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  105.0,e2:  150.0,fcf:  95.8}
        ch[3] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  150.0,e2:  225.0,fcf:  64.2}
        ch[4] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  225.0,e2:  315.0,fcf:  48.9}
        ch[5] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  315.0,e2:  500.0,fcf:  38.0}
        ch[6] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  500.0,e2:  750.0,fcf:  22.1}
        ch[7] = {G:-1.0,l1:'E',l2: 'SOPA',e1:  750.0,e2: 1100.0,fcf:  19.0}
        ch[8] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1100.0,e2: 1500.0,fcf:  14.5}
        ch[9] = {G:-1.0,l1:'E',l2: 'SOPA',e1: 1500.0,e2:99999.0,fcf:  15.8}

        ch[10]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   50.0,e2:   75.0,fcf: 350.0}
        ch[11]= {G:-1.0,l1:'P',l2: 'SOPA',e1:   75.0,e2:  113.0,fcf: 236.0}
        ch[12]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  113.0,e2:  170.0,fcf: 130.0}
        ch[13]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  170.0,e2:  250.0,fcf:  95.6}
        ch[14]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  250.0,e2:  400.0,fcf:  49.4}
        ch[15]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  400.0,e2:  670.0,fcf:  20.6}
        ch[16]= {G:-1.0,l1:'P',l2: 'SOPA',e1:  670.0,e2: 1200.0,fcf:  11.8}
        ch[17]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1200.0,e2: 1900.0,fcf:  9.91}
        ch[18]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 1900.0,e2: 3100.0,fcf:  5.71}
        ch[19]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 3100.0,e2: 5000.0,fcf:  3.29}
        ch[20]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 5000.0,e2: 7700.0,fcf:  2.49}
        ch[21]= {G:-1.0,l1:'P',l2: 'SOPA',e1: 7700.0,e2:50000.0,fcf:  0.15}

        ch[22]= {G:-1.0,l1:'E',l2: 'ESP', e1:  700.0,e2: 1800.0,fcf: 110.0}
        ch[23]= {G:-1.0,l1:'E',l2: 'ESP', e1: 1800.0,e2: 2200.0,fcf:  41.0}
        ch[24]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2200.0,e2: 2700.0,fcf:  51.0}
        ch[25]= {G:-1.0,l1:'E',l2: 'ESP', e1: 2700.0,e2: 3500.0,fcf:  82.0}
        ch[26]= {G:-1.0,l1:'E',l2: 'ESP', e1: 3500.0,e2: 4500.0,fcf: 102.0}
        ch[27]= {G:-1.0,l1:'E',l2: 'ESP', e1: 4500.0,e2: 6000.0,fcf: 153.0}
        ch[28]= {G:-1.0,l1:'E',l2: 'ESP', e1: 6000.0,e2: 7800.0,fcf: 185.0}
        ch[29]= {G:-1.0,l1:'E',l2: 'ESP', e1: 7800.0,e2:10800.0,fcf: 310.0}
        ch[30]= {G:-1.0,l1:'E',l2: 'ESP', e1:10800.0,e2:25800.0,fcf:1535.0}

        ch(0:21).fcf = 1 / (0.160 * ch(0:21).fcf)
  
    END
    
ENDCASE

return, ch

END

;******************************************************************************
PRO write_sopa_cpa_sp_cal

COMMON lanl_geo                 ;see lanl_geo_commons for details

out_dir = papco_getenv('PAPCO_MODULES')+'lanl_geo/FluxConversionFactors/'
out_fln = 'cal_cpa_sopa_esp.txt'

openw, u, out_dir+out_fln,/GET_LUN

printf, u, ';Calibration file for CPA, SOPA and ESP data for all GEO birds'
printf, u, ';'
printf, u, ';Format is satellite name followed by channel descriptions:'
printf, u, ';Label, Instrument, species,e low,e high,fcf'
printf, u, ';'
printf, u, ';Use read format:'
printf, u, ';format = "(a30, a8, a5, 2(f15.5), f15.4)"'

FOR i = 0, n_elements(sat_names)-1 DO BEGIN

    sat = sat_names(i)

    print, ';' & print, sat & printf, u, ';' & printf, u, sat

    ch = sopa_cpa_esp_cal_def(sat)

    FOR j = 0, n_elements(ch)-1 DO BEGIN 

        CASE 1 OF
            (ch(j).e1 LT 1): BEGIN ;fomat label as eV
                e1 = ch(j).e1*1e3
                e2 = ch(j).e2*1e3
                unit = 'eV'

            END
            (ch(j).e1 GT 1000.0): BEGIN ;fomat label as MeV
                e1 = ch(j).e1/1e3
                e2 = ch(j).e2/1e3
                unit = 'MeV'
            END 
            ELSE: BEGIN ;format as keV
                e1 = ch(j).e1
                e2 = ch(j).e2
                unit = 'keV'
            END 
        ENDCASE
        lbl = ch(j).l2+strlowcase(ch(j).l1)+'_'+ $
              string(e1,e2, format = "(f8.3,'-',f8.3)") + '_'+unit
        lbl = strcompress(lbl, /remove_all)
        lbl = str_replace(lbl, '.000', '.0')
        lbl = str_replace(lbl, '.000', '.0')
        lbl = str_replace(lbl, '.00', '.0')
        lbl = str_replace(lbl, '.00', '.0')
        lbl = str_replace(lbl, '00-', '-')
        lbl = str_replace(lbl, '00_', '_')
        lbl = str_replace(lbl, '0-', '-')
        lbl = str_replace(lbl, '0_', '_')
        lbl = str_replace(lbl, '.-', '-')
        lbl = str_replace(lbl, '._', '_')
        lbl = str_replace(lbl, '_', '')
        lbl = str_replace(lbl, '_', '')

        out_str = strformat(lbl, 30, /left) + $
                  strformat(ch(j).l2, 8,/LEFT)+strformat(ch(j).l1, 5,/LEFT)+ $
                  string(ch(j).e1, ch(j).e2, format = "(2(f15.5))")+ $
                  string(ch(j).fcf, format = "(f15.4)")
        print, out_str
        printf, u, out_str
    ENDFOR 

ENDFOR 
printf, u, ';'

close, u & free_lun, u

END


