;******************************************************************************
;*  NAME:        r_lanl_geo_pitch
;*
;*  DESCRIPTION: Routine to read sopa pitch data files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                papco read philosophy
;*
;*  INPUT:       PlotInfo 
;*                  gives info on satellite, what species to read
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        July 2003
;*
;*  AUTOR:       Arne Aasnes at LANL
;*
;******************************************************************************
pro r_lanl_geo_pitch, PlotInfo, VERBOSE=VERBOSE

COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON lanl_geo               ;see lanl_geo_set_commons for contents
COMMON geo_pitch_time,  geo_pitch_year, geo_pitch_month, geo_pitch_day_om, geo_sat
COMMON get_error, get_err_no, get_err_msg

; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; empty panel with the error message get_err_msg is plotted.

control = *PlotInfo.USR_PTR1
message,/reset & get_err_no=0 & get_err_msg='lanl_geo_pitch read succesfully'

; check for env. variable pointing to data directories
if not (papco_check_data_env('LANL_GEO_PITCH', PATH=path)) then return
if not (papco_check_data_env('LANL_GEO_PITCH_BIN', PATH=bin_path)) then return
  
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  
; clear data arrays for reuse
lanl_geo_pitch_header=0  &  lanl_geo_pitch_data=0  
; Assign satellite name
sc=sat_names(control.sc_id)
geo_sat = sc
;sat_codes = ['l9', 'l0', 'l1', 'l4', 'l7', 'a1', 'a2']

; loop through from start to end time concatonating data files into one
; array. Here we assume that the data is stored in files per day, and
; that the time info in each file is seconds of day. papco standard is
; to hand over data to the plot routines with the time running in
; seconds from the number of seconds of the startday.
  
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
  
while mjdt_end.mjd ge time.mjd do begin
                                ; assemble the data file from the time range requested
     convert_t90_to_date, time, year, day_oy, hour, minute, second,$
                         month, cmonth, day_om, /mjdt   
    IF day_om LE 9 THEN daystr='0'+STRTRIM(STRING(day_om),2) $
       ELSE daystr=STRTRIM(STRING(day_om),2)
    IF month LE 9 THEN monthstr = '0'+STRTRIM(STRING(month),2) $
       ELSE monthstr=STRTRIM(STRING(month),2)
    yearstr = STRTRIM(STRING(year),2)
    datestr=daystr+'-'+STRLOWCASE(cmonth)+'-'+yearstr


;    IF control.ek1 LT 2^10 THEN species = 'e' $    
;       ELSE IF control.ek1 GE 2^10 THEN species = 'p'

    ;; construct the filename
    
    fln=path+strlowcase(sc)+'_'+yearstr+'_sopa_e1_angle' ; the first energy channel to be read
    bin_fln =  strlowcase(sc)+'_'+datestr+'_sopa_pa'

    bin_fln = bin_path+bin_fln
    
    geo_pitch_year = year
    geo_pitch_month = month
    geo_pitch_day_om = day_om
      
    ; read one day of data
    r_lanl_geo_pitch_onefile,fln,bin_fln, VERBOSE=VERBOSE, DAYCOUNTS = day_count
        
    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
       if no_files_read eq 0 then begin   ;start the array buffer
         lanl_geo_pitch_data_buf=temporary(lanl_geo_pitch_data)
       endif else begin                   ;otherwise concatonate arrays
         lanl_geo_pitch_data_buf=[temporary(lanl_geo_pitch_data_buf),temporary(lanl_geo_pitch_data)]
       endelse        
       lanl_geo_pitch_data=0                     ;clear array for reuse
       no_files_read=no_files_read+1
    endif
;    day_count=day_count+1
    time.mjd=time.mjd+1                  ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) THEN break
    day_count=day_count+1
ENDWHILE    
   
; check if any files were read   
if (no_files_read ne 0) then begin
;    input_data=0  &  input_data=input_data_buf  &  geo_data_buf=0
   lanl_geo_pitch_data = 0 &  lanl_geo_pitch_data = lanl_geo_pitch_data_buf & geo_data_buf = 0
    get_err_no = 0
ENDIF 
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0
    get_err_msg='% r_lanl_geo_pitch: '+string(no_files_read,format="(i2)")+ $
      ' files read succesfully'
    ;clear arrays and load buffer into output array
    lanl_geo_pitch_data=0  &  lanl_geo_pitch_data=lanl_geo_pitch_data_buf  &  lanl_geo_pitch_data_buf=0
endif else begin
    get_err_no=1
    get_err_msg='% r_lanl_geo_pitch:!C' +'  One of necessary datafiles not found!'
    return
endelse

; remove invalid data points
;idx=where(lanl_geo_pitch_data.key2 EQ 0, c1, COMPLEMENT = idx2, NCOMPLEMENT = c2)
;if c1 ne 0 then begin
;    message, 'Removing '+varprt(c2)+' invalid data points.', /cont
;    lanl_geo_pitch_data=lanl_geo_pitch_data(idx)
;ENDIF

END 

;******************************************************************************
;*  NAME:        r_lanl_geo_pitch_onefile,fln,bin_fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures lanl_geo_pitch_header and
;*               lanl_geo_pitch_data
;*
;*  INPUT:       fln     - string of original data filename
;*               bin_fln - string of IDL binary data filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        May 2003
;*
;*  AUTOR:       Arne Aasnes at LANL
;*
;*  MODIFIED:    A.Aa, july 2004 following changes made by Yue
;******************************************************************************
pro r_lanl_geo_pitch_onefile, fln, bin_fln, VERBOSE=VERBOSE, DAYCOUNTS = daycounts
  
  COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data
  COMMON get_error, get_err_no, get_err_msg
  
  IF keyword_set(DAYCOUNTS) THEN daycounts = DAYCOUNTS ELSE daycounts = 0 
  get_err_no=0 & get_err_msg='geo onefile read succesfully'
   
; clear data arrays for reuse
  lanl_geo_pitch_header=0  &  lanl_geo_pitch_data=0

;  openr, 11, bin_fln+'.idl', error = err ;--cy--030904 
;  close, 11 
;     IF err NE 0 THEN GOTO, read_original
  on_ioerror, read_original
  restore, bin_fln+'.idl', VERBOSE=VERBOSE
  dealloc_lun,/quiet            ;frees logical units used by save/restore
  message,'saveset read:', /cont
  print, '  ',bin_fln+'.idl'

; due to a renaming of the common block variables we have to check for
; old savesets where data is stored in input_header/input_data
  IF n_elements(input_data) NE 0 THEN BEGIN 
     lanl_geo_pitch_header = input_header &  input_header = 0
     lanl_geo_pitch_data = input_data &  input_data = 0
  ENDIF 
  return

read_original:
; if the .idl save file could not be found, then try to read the original
  r_lanl_geo_pitch_onefile_original, fln, bin_fln, VERBOSE=VERBOSE, DAYCOUNTS = daycounts
      
END    
  
;******************************************************************************
;*  NAME:        r_lanl_geo_pitch_onefile_original,fln,bin_fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln     - string of original data filename
;*               bin_fln - string of IDL binary data filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        May 2003
;*
;*  AUTOR:       Arne Aasnes at LANL
;******************************************************************************
pro r_lanl_geo_pitch_onefile_original, fln, bin_fln, VERBOSE=VERBOSE, DAYCOUNTS = daycounts, read_all = read_all
  
  COMMON lanl_geo_data, lanl_geo_pitch_header, lanl_geo_pitch_data
  COMMON lanl_mpa_pitch_data,  mpa_pitch_header,  mpa_pitch_data, mpa_pitch_sat
  COMMON mjdt, mjdt_start, mjdt_end
  COMMON geo_pitch_time, geo_pitch_year, geo_pitch_month, geo_pitch_day_om, geo_sat
  COMMON get_error, get_err_no, get_err_msg 
  COMMON lanl_geo               ;see lanl_geo_pitch_set_commons for contents

  IF keyword_set(DAYCOUNTS) THEN daycounts = DAYCOUNTS ELSE daycounts = 0 ;--cy--030804
  print,  'daycounts=  ',  daycounts

;  IF strmid(fln, 7, 1, /reverse_offset) EQ 'e' THEN spec = 0 ELSE spec = 1
  
  get_err_no=0 & get_err_msg='geo_pitch_onefile_original read succesfully'
  
  verbose = 1
  
; clear data arrays for reuse
  lanl_geo_pitch_header=0  &  lanl_geo_pitch_data=0
  
;------------------------------------------------------------------------------
; read the original data file
  

  max_no_rec=200                ;the maximum number time records in your data
  pol_det = 3                   ;telescopes on SOPA
  azi_det = 32                  ;azimuthal spin records



                                ; find the labels of energy channels for this specific satellite

  channels = r_lanl_geo_flux_converison(geo_sat)
  e_OR_p = strmid(channels.label, 0, 1) ; array of strings, 'E' or 'P'
  en_chan = n_elements(e_OR_p)
  e_esp = where(channels.species EQ 'electrons' AND channels.instrument EQ 'ESP' )
  p_esp = where(channels.species EQ 'ions' AND channels.instrument EQ 'ESP') 
  e_sopa = where(channels.species EQ 'electrons' AND channels.instrument NE 'ESP')
  p_sopa = where(channels.species EQ 'ions' AND channels.instrument NE 'ESP' )
; define the header and data structures. Edit to suit your data!

  lanl_geo_pitch_header = {lanl_geo_pitch_header, tag1:'Time in decimals of hours from beginning of day', $
; tag2:'Binned raw counts (time,energy,pitch angles)'
 tag2:'Unbinned counts', $
 tag3:'unbinned pitch angles for sopa',  $
 tag4:'unbinned pitch angles for esp', $
; tag5:'Pitch angle limits of data', 
 tag5:'modified julian day time', $
 tag6:'satellite id', tag7:'magnetic field direction, in theta, phi, satellite coords', $
 tag8:'B determination flag', tag9:'tai time', tag10:'data (yyyymmdd)'} 
  dummy = {uthours:0d,$ ; binned_data:fltarr(en_chan, bin_nr),
          counts:fltarr(pol_det, azi_det, en_chan), $
          alpha:fltarr(pol_det, azi_det), $
          alpha_esp:fltarr(pol_det, azi_det), $
          ; pitch_angles:pitch_lims, $
          mjdt_time:0., csat_id:geo_sat, mag_dir:fltarr(2), B_flag:0e, tai:0d, date:'' }

  lanl_geo_pitch_data = replicate(dummy, max_no_rec)


  ext='.dat'                    ;file extention of your dat files
  idl_ext = '.idl'              ;file extension of your idl files
  montharr=['JAN', 'FEB', 'MAR', 'APR', 'MAY', 'JUN', 'JUL',$
          'AUG', 'SEP', 'OCT', 'NOV', 'DEC']
;lanl_geo_pitch_header={lanl_geo_pitch_header,tag1:'',tag2:'', tag3:'', tag4:'', $
;       tag5:'', tag6:''}
  mjdt = {MJDT, MJD:0l, T:0.0}
  tai = 0.0d

  message, 'Reading ascii file(s): (slow) or idl savesets: (less slow)', /cont

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Data files come in one file per energy, we read all of them even
;; though only a few energies may be requested for plotting.
;; We got to loop through to get all the energies, we construct the
;; file names from the information in "channels"
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  slash_pos = strpos(fln, '/', /reverse_search)
  fln_1 = strmid(fln, 0, slash_pos+21) ;first part
  fln_2 = strmid(fln, slash_pos+22, 6) ;last part

  extra = make_array(en_chan, value = '') 
  IF e_esp(0) NE -1 THEN extra(e_esp) = 'sp_e'
  IF p_esp(0) NE -1 THEN extra(p_esp) = 'sp_p'
  num_arr = [strcompress(string(e_sopa-min(e_sopa)+1), /remove_all), $
             strcompress(string(p_sopa-min(p_sopa)+1), /remove_all)]  
  IF e_esp(0) NE -1 THEN num_arr = [num_arr, strcompress(string(e_esp-min(e_esp) +1), /remove_all)]
  IF p_esp(0) NE -1 THEN num_arr = [num_arr, strcompress(string(p_esp-min(p_esp) +1), /remove_all)]
  fln_arr = strmid(fln, 0, slash_pos+20)+strlowcase(e_OR_p)+extra+num_arr+fln_2


; we make save files of the original 1 year ascii data files
  if not (papco_check_data_env('LANL_GEO_PITCH_BIN', PATH=bin_path)) then BEGIN 
     print, 'Path for savesets not properly configured!'
     return   
  endIF  
  bin_fln_1 =bin_path+strmid(fln, slash_pos+1, 18)
  bin_fln_2 = fln_2
  bin_all_fln = bin_fln_1+'_all_pa' ;path for idl saveset of all energy channels, all year
  bin_all_fln_gdd = bin_all_fln+'_gdd'
  save_read_all = 0

  res = findfile(bin_all_fln_gdd, count = gottime)
  IF gottime EQ 1 THEN BEGIN 
     restore, bin_all_fln_gdd, /verbose ; geo_decimal_doy
;     convert_t90_to_date, mjdt_start.t90, year, day_oy
     mjd2date, mjdt_start.mjd, year, month, day
     strdate = strtrim(day, 2)+'/'+strtrim(month, 2)+'/'+strtrim(year, 2)
     doy = convert_datetoday(strdate) &  day_oy = doy.day+daycounts
     got_data = where(fix(geo_decimal_doy) EQ day_oy, got_cnt)
     IF got_cnt EQ 0 THEN BEGIN 
        print, 'No geo data for this day!'
        GOTO, out
     ENDIF 
  endIF 

; read the mpa data before restoring the whole geo data block to save
; time if there is no data!

;  a_lanl_mpa_pitch, SAT_ID = geo_sat, T1 = mjdt_start,  T2 = mjdt_end, /MJDT

;    returns the mpa pitch data in common block lanl_mpa_pitch_data
;    all we need is the magnetic field direction as deduced from the
;    gyrotropy of the mpa electrons (or sometimes ions). Time in this
;    data set is given in tai and uthours

;  IF n_elements(mpa_pitch_data) LE 1 THEN GOTO, out
  ON_IOERROR, go_on1
  restore, bin_all_fln, /verbose
  IF keyword_set(verbose) THEN print, 'Read concatenated idl saveset!'
  save_read_all = 1

go_on1:
  unique = 0
                                ; if the data was not previously made
                                ; into idl saveset, read the ascii files
  IF n_elements(geo_counts) EQ 0 THEN BEGIN
     
     FOR en_chan_loop = 1, en_chan-3 DO BEGIN
        
        
        fln = fln_arr(en_chan_loop-1) ;fln_1+chan_str+fln_2
           
        r = findfile(fln+ext, count = c)

           
;   reading ascii data file and putting the data in the structure data
;   So far we are reading only one file containing one energy. ESP
;   data files have only one polar detector, at 90 deg!
           IF c NE 0 THEN BEGIN  

              on_ioerror, out   
              
              IF keyword_set(verbose) THEN print, 'Reading '+fln+ext
              data = readascii(fln+ext, VERBOSE = verbose)
           
;    data is a structure containing data.decimal_doy (time in decimals
;    of day of year) and data.counts (3*32 dimensional float array
;    with SOPA counts for the 3 polar detectors sampled 32 times in
;    azimuthal direction)

;   IF get_err_no NE 0 THEN GOTO, out
           ENDIF ELSE print, 'File not found:'+fln+ext 
           
           sz = size(data, /type) ; check that data really is of type structure
           IF size(data, /type) EQ 8 THEN BEGIN 
           
              IF en_chan_loop EQ 1 OR n_elements(geo_counts) EQ 0  THEN BEGIN  
                 
                 n_el_geo = n_elements(data.decimal_doy)
                 geo_counts = fltarr(pol_det, azi_det, en_chan, n_el_geo) &  geo_counts(*) = -1e31      
              ENDIF 
        
              
;; sometimes there are a varying amount of data for different energies
;; as we want to put all data into one array this causes problems...
              this_n_el_geo = n_elements(data.decimal_doy)
              IF this_n_el_geo NE n_el_geo  OR (NOT unique AND en_chan_loop EQ en_chan) THEN BEGIN 
                 IF keyword_set(verbose) THEN print, 'Varying number of elements in arrays: sorting...'
                 
                 all_times = [geo_decimal_doy, data.decimal_doy] 
                 unique_times = uniq(all_times, sort(all_times))
                 

                 new_geo_decimal_doy= all_times(unique_times) ; need to use this as the reference array!!
                 new_el_geo = n_elements(new_geo_decimal_doy)    
                 
;; uniq function picks the index of the latter unique value in the
;; array. To find the positions of geo_decimal_doy in
;; new_geo_decimal_doy we see where uniq returns indices over new_el_geo 


                 times_old = [new_geo_decimal_doy, geo_decimal_doy]
                 unique_old = uniq(times_old, sort(times_old))

; unique_old consists of numbers lt new_n_el_geo (defining positions
; only found in the new data) and numbers ge new_n_el_geo(defining
; positions found in the old data)
                 ind = where(unique_old LT new_el_geo, cnt)
                 IF cnt GT 0 THEN BEGIN  
                    test_arr_new = intarr(new_el_geo) &   test_arr_new(ind) = 1 ; where only new data
                    pos_old_data = where(test_arr_new EQ 0) ;picks out all indices with old data
                 ENDIF ELSE pos_old_data = findgen(new_el_geo)
                 
                 times_new = [new_geo_decimal_doy, data.decimal_doy]
                 unique_new = uniq(times_new, sort(times_new))

; unique_new consists of numbers lt new_n_el_geo (defining positions
; only found in the old data) and numbers ge new_n_el_geo(defining
; positions found in the new data)

          
              
                 ind = where(unique_new LT new_el_geo, cnt)
                 IF cnt GT 0 THEN BEGIN 
                    test_arr_old = intarr(new_el_geo) &   test_arr_old(ind) = 1 ; where only new data
                    pos_new_data = where(test_arr_old EQ 0) ;picks out all indices with new data
                 endIF ELSE pos_new_data = findgen(new_el_geo)
              
              
              

;              IF new_el_geo EQ n_el_geo THEN new_geo_counts = geo_counts ELSE $
                 new_geo_counts = fltarr(pol_det, azi_det, en_chan, new_el_geo) ; the new array for data

              
                 ind = where(unique_old GE new_el_geo, aa) &  IF aa GT 0 THEN $
                              old_data = unique_old(ind)-new_el_geo
                 ind = where(unique_new GE new_el_geo, bb) &  IF bb GT 0 THEN $
                              new_data = unique_new(ind)-new_el_geo

              

        ; get the indices of the unique times        
                 old_uniq = uniq(geo_decimal_doy, sort(geo_decimal_doy)) 
                 new_uniq = uniq(data.decimal_doy, sort(data.decimal_doy))

                 IF aa GT 0 THEN  new_geo_counts(*, *, *, pos_old_data) = geo_counts(*, *, *, old_data)      
                  
                 IF bb GT 0 THEN BEGIN 
                    IF strpos(fln, '_esp_') NE -1 THEN new_geo_counts(1, *, en_chan_loop-1, pos_new_data) = data.counts(*, new_data) $
                    ELSE new_geo_counts(*, *, en_chan_loop-1, pos_new_data) = data(new_data).counts
                 ENDIF 
                 geo_counts = new_geo_counts & new_geo_counts = 0
                 geo_decimal_doy = new_geo_decimal_doy &  new_geo_decimal_doy = 0
                 n_el_geo = new_el_geo
                 unique = 1
                 
              ENDIF      ELSE BEGIN
                 
                 IF n_elements(geo_decimal_doy) EQ 0 THEN geo_decimal_doy = data.decimal_doy
              
                 IF strpos(fln, '_esp_') NE -1 THEN  geo_counts(1, *, en_chan_loop-1,* ) = data.counts ELSE $
                                                     geo_counts(*, *, en_chan_loop-1,* ) =  data.counts
              ENDELSE 
              
;           IF NOT save_read THEN save, data, filename = e_bin_fln+idl_ext, VERBOSE = verbose
           ENDIF    
           data = 0
          
        ENDFOR    
        IF n_elements(geo_counts) EQ 0 THEN GOTO, out

        ; set all the negative values to the default missing data value
        nd_geo = where(geo_counts LT 0 AND geo_counts NE -1e31, nd_cnt)
        IF nd_cnt GT 0 THEN geo_counts(nd_geo) = -1e31
        
; finally test that geo_decimal_doy is
; monotonically increasing and unique

        sind = sort(geo_decimal_doy)
        geo_decimal_doy = geo_decimal_doy(sind)
        unique = uniq(geo_decimal_doy)
        geo_decimal_doy = geo_decimal_doy(unique)
        geo_counts = geo_counts(*, *, *, sind)
        geo_counts = geo_counts(*, *, *, unique)

        IF NOT save_read_all THEN save,geo_counts, filename = bin_all_fln, /compress
     ENDIF         
     IF NOT gottime THEN save, geo_decimal_doy, filename = bin_all_fln_gdd, /verbose
     IF n_elements(geo_counts) EQ 0 THEN GOTO, out
          tmp_startmjdt = mjdt_start & tmp_endmjdt = mjdt_end
     tmp_mjdt1 = {mjd:mjdt_start.mjd, t:mjdt_start.t}
     tmp_mjdt2 = {mjd:tmp_mjdt1.mjd+1,   t:0.0}
     IF daycounts GT 0 THEN BEGIN 
      tmp_mjdt1.mjd = tmp_mjdt1.mjd+daycounts
      tmp_mjdt1.t = 0.
      tmp_mjdt2 = {mjd:tmp_mjdt1.mjd+1,   t:0.0}
      IF tmp_mjdt1.mjd GE mjdt_end.mjd THEN BEGIN 
       tmp_mjdt1.mjd = mjdt_end.mjd
       tmp_mjdt2.mjd = mjdt_end.mjd
       tmp_mjdt2.t   = mjdt_end.t
      ENDIF
      IF tmp_mjdt2.mjd GT mjdt_end.mjd THEN BEGIN
       tmp_mjdt2.mjd = mjdt_end.mjd
       tmp_mjdt2.t   = mjdt_end.t
      ENDIF
     ENDIF ELSE BEGIN 
      IF tmp_mjdt2.mjd GE mjdt_end.mjd THEN BEGIN 
       tmp_mjdt2.mjd = mjdt_end.mjd
       tmp_mjdt2.t = mjdt_end.t
      ENDIF
     ENDELSE    

     a_lanl_mpa_pitch, SAT_ID = geo_sat, T1 = tmp_mjdt1,  T2 = tmp_mjdt2, /MJDT
     mjdt_start = tmp_startmjdt & mjdt_end = tmp_endmjdt  ;restore the common block

     IF n_elements(mpa_pitch_data) LE 1 THEN GOTO, out
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Converts time to a common format and selects the mpa data points
;; within each period of the geo data points. Use an average
;; of the magnetic field direction as determined from mpa for each
;; time step of geo data
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     mjdt_start0 = tmp_mjdt1 & mjdt_end0 = tmp_mjdt2

     mjdt_find_start = {mjd:mjdt_start0.mjd, time:mjdt_start0.t}
     mjdt_find_end = {mjd:mjdt_end0.mjd, time:mjdt_end0.t}
     
     mpa_start_doy = utc2doy(mjdt_find_start)
     mpa_end_doy = utc2doy(mjdt_find_end)
     IF mpa_end_doy gt mpa_start_doy+1 THEN stop
     mpa_decimal_doy = mpa_start_doy+(mpa_pitch_data.uthours)/24.
;geo_decimal_doy = data.decimal_doy

     mpa_elements = n_elements(mpa_decimal_doy)
     mpa_mag_dir = mpa_pitch_data.mag_dir
     mpa_mag_theta = mpa_mag_dir(0, *)
     mpa_mag_phi = mpa_mag_dir(1, *)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; We pick out the times when the derived magnetic field direction
;; deviates significantly from the mean, as we can only accept
;; "smooth" variations when we take the mean direction
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; first finding the invalid points
  mag_dir_good = where(abs(mpa_mag_theta) LT !dpi AND  abs(mpa_mag_phi) LT !dpi, good) 
  IF good GT 0 THEN BEGIN 
     mpa_decimal_doy = mpa_decimal_doy(mag_dir_good)
     mpa_mag_theta = mpa_mag_theta(mag_dir_good)
     mpa_mag_phi = mpa_mag_phi(mag_dir_good)
     mpa_elements = good
  ENDIF ELSE stop               ;GOTO, out

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Remove the magnetic field directions that deviate significantly
;; from the neighbouring directions... Take the difference between a
;; boxcar smoothed direction and the real direction and remove the
;; data points that deviate by more than a given limit. Then,
;; interpolate the array without the deviating data points back to
;; it's original dimension and again remove data points that deviate
;; from this interpolated line. 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 

  lim = 0.1

  sm_mpa_mag_theta = smooth(mpa_mag_theta, 3,/edge_truncate)
  first_it_keep = where(abs(sm_mpa_mag_theta - mpa_mag_theta) LT lim, keep_pts)
  IF keep_pts LT mpa_elements THEN BEGIN 
     sm_mpa_mag_theta = smooth(mpa_mag_theta(first_it_keep), 3, /edge_truncate)
     sm_interp_mag_theta = interpol(sm_mpa_mag_theta, first_it_keep, indgen(mpa_elements))
     sec_it_rem_theta = where(abs(sm_interp_mag_theta-mpa_mag_theta) GT lim,  rem_theta_pts)
  ENDIF ELSE rem_theta_pts = 0
  
  sm_mpa_mag_phi = smooth(mpa_mag_phi, 3,/edge_truncate)
  first_it_keep = where(abs(sm_mpa_mag_phi - mpa_mag_phi) LT lim, keep_pts)
  IF keep_pts LT mpa_elements THEN BEGIN 
     sm_mpa_mag_phi = smooth(mpa_mag_phi(first_it_keep), 3, /edge_truncate)
     sm_interp_mag_phi = interpol(sm_mpa_mag_phi, first_it_keep, indgen(mpa_elements))
     sec_it_rem_phi = where(abs(sm_interp_mag_phi-mpa_mag_phi) GT lim,  rem_phi_pts)
  ENDIF ELSE rem_phi_pts = 0

;; use null_array to find the union of sec_it_rem_theta and
;; sec_it_rem_phi, as we need to remove bad data points in either array

  null_arr = make_array(mpa_elements, value = 0)
  IF rem_theta_pts GT 0 THEN null_arr(sec_it_rem_theta) = 1
  IF rem_phi_pts GT 0 THEN null_arr(sec_it_rem_phi) = 1
  keep = where(null_arr EQ 0, good)
  IF good GT 0 THEN BEGIN
     mpa_decimal_doy = mpa_decimal_doy(keep)
     mpa_mag_theta = mpa_mag_theta(keep)
     mpa_mag_phi = mpa_mag_phi(keep)
     mpa_elements = good
  ENDIF 
  





;; sopa data files got data for a whole year, so we want to restrict
;; our data to the given times, this is included when reading the mpa
;; data and we can anyhow only use the geo data when we got mpa data
;; available

  n_el_geo = n_elements(geo_decimal_doy)
  
  IF n_el_geo LE 2 THEN BEGIN 
     print, 'Not enough data in file!'
     GOTO, out
  ENDIF 

  geo_ind = where(geo_decimal_doy GE mpa_decimal_doy(0) AND $
                 geo_decimal_doy LE mpa_decimal_doy(mpa_elements-1), geo_elements)

  

  IF geo_elements LE 1 THEN BEGIN 
     print, 'The geo data files does not cover the selected date!!!'
     GOTO, out
  ENDIF


               ; often, at the beginning of days
               ; there are two equal times, so we
               ; remove one of them... (this should not happen anymore)
;  IF geo_elements GT 1 THEN BEGIN ;--cy--022704,somtimes only 1 pt in the day..
     IF geo_decimal_doy(geo_ind(0)) EQ geo_decimal_doy(geo_ind(1)) THEN BEGIN 
        geo_ind = geo_ind(1:geo_elements-1)
        geo_elements = geo_elements-1
     ENDIF 
;  ENDIF 
  geo_decimal_doy = geo_decimal_doy(geo_ind)

  geo_counts = geo_counts(*, *, *, min(geo_ind):max(geo_ind))
  mag_theta = fltarr(geo_elements)
  mag_phi = fltarr(geo_elements)
  resolution = 1d/(24*6) ; the time resolution of data
  FOR i = 0, geo_elements-1 DO BEGIN 

     mpa_ind = where(mpa_decimal_doy GE geo_decimal_doy(i) AND mpa_decimal_doy $
                    LE geo_decimal_doy(i)+resolution , mpa_dat)
   IF mpa_dat GT 0 THEN BEGIN 
     
      mag_theta(i) =mean(mpa_mag_theta(mpa_ind))
      mag_phi(i) =  mean(mpa_mag_phi(mpa_ind))

   ENDIF ELSE BEGIN 
      mag_theta(i) = -1e31
      mag_phi(i) = -1e31
   ENDELSE 
      
ENDFOR

   ; pick out the times when we got a magnetic field direction from mpa
   mpa_inc_ind = where(mag_theta NE -1e31,inc_mpa )
   IF inc_mpa LT geo_elements THEN BEGIN
      geo_decimal_doy = geo_decimal_doy(mpa_inc_ind)
      geo_elements = inc_mpa
      geo_counts = geo_counts(*, *, *, mpa_inc_ind)
      mag_theta = mag_theta(mpa_inc_ind)
      mag_phi = mag_phi(mpa_inc_ind)
   ENDIF 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; setting up the detector angles, look in lanl_pitch_geo_panel.help for
;; details
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
geo_theta = [30, 90, 120]/!radeg
geo_phi = ([findgen(32)*11.25]+5.625)/!radeg+!dpi/2d 
; the 0 deg azimuth in mpa is defined 90 deg off the sopa 0 deg  azimuth

;*******************************************************************************************
;* Apparently the differense in azimuthal directions between SOPA and 
;* MPA is not exactly 90 degrees. We correct for this by a visual
;* determination of the offset by plotting PA versus counts for the 3
;* detectors (see lanl_geo_pitch_tools, plot_PA ). The values chosen
;* seem to remove the "butterfly" distrubutions very well
;********************************************************************************************

;one correcion value for each spacecraft
;                                                   l9   l0   l1  l4  l7  a1  a2                                       
corr_not_done = $
               [1,   1,   1,   1,   1,   1,   1,   1,   0,   0,   0,   0,   0,   0,   0]
phi_corr =     [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 4.0, 3.5, 5.0, 4.0, 4.0, 4.0, 4.0] 
esp_phi_corr = [90., 90., 90., 90., 90., 90., 90., 90., 150., 135., 132., 130., 123., 130., 130.]
sat_ind = where(sat_names EQ geo_sat)

set_val  = corr_not_done(where(sat_names EQ geo_sat))
IF set_val(0) EQ 1 THEN print, $
        'Corrections for any offset between axis definition in MPA and SOPA has ' + $
        'not been done yet for this satellite!!!'

IF min(mag_phi) LT 0 THEN mag_phi(where(mag_phi LT 0)) = mag_phi(where(mag_phi LT 0))+2*!dpi
; expand all arrays to do the pitch angle calculations   , dimensions
; are [geo_elements,  pol_det, azi_det]    

mag_theta = rebin(reform(mag_theta,  1, 1,  geo_elements), $
                  pol_det,  azi_det,  geo_elements)
mag_phi = rebin(reform(mag_phi, 1, 1,  geo_elements), $
                  pol_det,  azi_det,  geo_elements)

;mag_phi_esp = mag_phi -180.0/!radeg

geo_theta = rebin(reform(geo_theta,  pol_det, 1,  1), $
                 pol_det,  azi_det,  geo_elements  )


geo_phi = rebin(reform(geo_phi,  1, azi_det,  1), $
                 pol_det,  azi_det,  geo_elements  )
phi_corr = phi_corr(sat_ind)/!radeg & phi_corr = phi_corr(0)
esp_phi_corr = esp_phi_corr(sat_ind)/!radeg & esp_phi_corr = esp_phi_corr(0)
geo_phi = geo_phi-phi_corr 
geo_phi_esp = geo_phi+esp_phi_corr

cosalpha = sin(geo_theta)*cos(geo_phi)*sin(mag_theta)*cos(mag_phi)+ $
           sin(geo_theta)*sin(geo_phi)*sin(mag_theta)*sin(mag_phi)+ $
           cos(geo_theta)*cos(mag_theta)
cosalpha_esp = sin(geo_theta)*cos(geo_phi_esp)*sin(mag_theta)*cos(mag_phi)+ $
           sin(geo_theta)*sin(geo_phi_esp)*sin(mag_theta)*sin(mag_phi)+ $
           cos(geo_theta)*cos(mag_theta)
alpha = acos(cosalpha)*!radeg ; converting from radians to degrees
alpha_esp = acos(cosalpha_esp)*!radeg ;

;alpha_st = reform(alpha, pol_det*azi_det, geo_elements)
;geo_counts_st = reform(geo_counts, pol_det*azi_det,en_chan, geo_elements)

; array to store the pa binned counts
;geo_ordered_counts = fltarr(bin_nr, en_chan,geo_elements )


decimal_time = geo_decimal_doy-floor(geo_decimal_doy)
hr = floor(decimal_time*24.) 
mn = floor((decimal_time*24.-hr)*60.)
sec = floor(((decimal_time*24.-hr)*60.-mn)*60)
uthours = hr+mn/60.+sec/3600.
IF geo_pitch_month LT 10 THEN strmn = '0'+strcompress(string(geo_pitch_month), /remove_all) ELSE $
                               strmn = strcompress(string(geo_pitch_month), /remove_all)
IF geo_pitch_day_om LT 10 THEN strdom = '0'+strcompress(string(geo_pitch_day_om), /remove_all) ELSE $
                                strdom = strcompress(string(geo_pitch_day_om), /remove_all)
strdate = strcompress(string(geo_pitch_year), /remove_all)+strmn+strdom



; binning the data into specified bins (not done after 2003/07/28)

FOR i = 0, geo_elements-1 DO BEGIN
   
;   geo_pitch_hist = histogram(alpha_st(*, i), min = bin_min, max = bin_max, nbins = bin_nr,  $
;                        reverse_indices = r)
;      FOR k = 0, bin_nr-1 DO IF r(k+1) GT r(k) THEN $
;            geo_ordered_counts(k, *, i) = $
;           reform(rebin(geo_counts_st(r(r(k):r(k+1)-1), *, i ), 1, en_chan), en_chan) ELSE $
;            geo_ordered_counts(k, *, i)= -1e31 
 
   dummy.uthours = uthours(i)
;   dummy.binned_data = geo_ordered_counts( *, *, i)
   dummy.counts = geo_counts(*, *, *, i)
   dummy.alpha = alpha(*, *, i)
   dummy.alpha_esp = alpha_esp(*, *, i)
   dummy.tai = UTC2TAI({YEAR:geo_pitch_year, MONTH:geo_pitch_month, DAY:geo_pitch_day_om, $
                            HOUR:hr(i),  MINUTE:mn(i), SECOND:sec(i), MILLISECOND:0})
   dummy.mag_dir = [mag_theta(0, 0, i), mag_phi(0, 0, i)]
   dummy.B_flag = 1
   dummy.date = strdate
   lanl_geo_pitch_data(i) = dummy
   
ENDFOR 
lanl_geo_pitch_data = lanl_geo_pitch_data(0:geo_elements-1) 

; end reading original data
;------------------------------------------------------------------------------
; save read data with correct time. Rename data to *.idl
save, filename = bin_fln+'.idl', lanl_geo_pitch_header, lanl_geo_pitch_data, $
  /compress, /verbose  
dealloc_lun,/quiet              ;frees logical units used by save/restore

IF keyword_set(verbose) THEN print, 'Data files read and saved!'

geo_counts = 0 &  geo_decimal_doy = 0 & mag_theta = 0 &  mag_phi = 0 & alpha = 0 & cosalpha = 0

return
  
out:
; if there was an error, report it. Use the IDL error variables. Error
; 171 is for "file not fond" which is used to invoke the remote get.  
;get_err_no=!ERROR

get_err_no = 1
get_err_msg='r_lanl_geo_pitch_onefile:!C '+'one of necessary datafiles not found!'
   
END
