;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  lanl_geo_eph_panel, plotinfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      plotinfo        a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*      TICKS           switch to indicate ticks only are required.
;*     
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from plotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(plotinfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION lanl_geo_eph_panel, plotinfo, GROUP=group, ACTION=ACTION,  $
                                       TICKS = TICKS

COMMON lanl_geo_eph_panel_data, widgetData
COMMON lanl_geo_eph

;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(lanl_geo_eph_control)
control = *plotinfo.USR_PTR1
;control now contains the user defiend control structure for this module.

IF keyword_set(TICKS) THEN ticks = 1 ELSE ticks = 0
control.ticks = ticks

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
panel_base=papco_WIDGET_BASE(title=plotinfo.panelKind, row=1, $
                             GROUP_LEADER=group, /MODAL)
panel_left=widget_base(panel_base, column=1)
panel_right=widget_base(panel_base,column=1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
title = 'none'

sat_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
sat_base=CW_BGROUP(sat_outer, sat_names, ROW=3, /EXCLUSIVE, $
                   LABEL_TOP='LANL Geosynchronous Satellite ID', $
                   UVALUE='spb_sat', $
                   SET_VALUE=control.sc_id)


names = papco_make_choice_names(control.orig_info, TITLE = title)
orig_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
orig_base = CW_BGROUP(orig_outer, names , ROW=1, /EXCLUSIVE, $
                      LABEL_LEFT=title, UVALUE='spb_orig', $
                      SET_VALUE=control.orig)

eph_outer = widget_base(panel_left, ROW=1, /ALIGN_CENTER, /FRAME)
IF TICKS EQ 0 THEN BEGIN
    eph_base = CW_BGROUP(eph_outer, eph_names, ROW=1, /EXCLUSIVE, $
                         LABEL_LEFT='Original Ephemeris:', $
                         UVALUE='spb_eph', $
                         SET_VALUE=control.product)
ENDIF ELSE BEGIN
    n_ch = n_elements(eph_names) & set=bytarr(n_ch)
    FOR i = 0, n_ch-1 DO IF (control.product AND 2^i) EQ 2^i THEN set(i) = 1
    eph_base = CW_BGROUP(eph_outer, eph_names, ROW=1, /NONEXCLUSIVE, $
                         LABEL_LEFT='Original Ephemeris', $
                         UVALUE='spb_eph_ticks', $
                         SET_VALUE = set)
ENDELSE

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;

;--- The next panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    lanl_geo_eph. If you do not want to use them, just comment them out.

;--- add Mag Field model choices ----------------------------------------------

IF ticks EQ 0 THEN BEGIN
    widgets = papco_panel_Add_MagMod_Widgets(panel_left, control)
ENDIF ELSE BEGIN
    widgets = papco_panel_Add_MagMod_Widgets(panel_left, control, /TICKS)
ENDELSE 
mag_base = widgets.MAG_MODEL_OUTER_BASE
additionalWData=create_struct(widgets, additionalWData)

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
;  widgets=papco_panel_Addcolorbar_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)


IF ticks EQ 0 THEN BEGIN 

;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_left, plotinfo, 0)
additionalWData=create_struct(widgets, additionalWData)

;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(panel_left, plotinfo)
additionalWData=create_struct(widgets, additionalWData)

ENDIF 

;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
;  widgets=papco_panel_AddZScale_Widgets(panel_right, plotinfo,/VERTICAL)
;  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_left, Action, plotinfo, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in lanl_geo_eph example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'lanl_geo_eph_panel.help', $
                 ticks:ticks, $
                 orig_base:orig_base, $
                 mag_base:mag_base, $
                 eph_base:eph_base}

; add control struct, papco_panel structs
widgetData=papco_panel_CreateWidgetData(additionalWData, thisWidgetsData)
widgetData=papco_panel_CreateWidgetData(widgetData, control)

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData
lanl_geo_eph_refresh, /ALL
papco_xmanager, 'lanl_geo_eph_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified }

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_geo_eph_refresh
;* 
;* DESCRIPTION:  
;*      refresh rotuine for lanl_geo_eph_panel, called by event handler
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO lanl_geo_eph_refresh, ALL = ALL, COORDS = COORDS

COMMON lanl_geo_eph_panel_data, widgetData

IF keyword_set(ALL) THEN BEGIN
    COORDS = 1
ENDIF

IF keyword_set(COORDS) THEN BEGIN
    IF widgetData.ticks THEN BEGIN
        widget_control, widgetData.orig_base,  SENSITIVE = 0
        widget_control, widgetData.eph_base,  SENSITIVE = 1
        widget_control, widgetData.mag_base,  SENSITIVE = 1
        GOTO, done_coords    
    ENDIF ELSE widget_control, widgetData.orig_base,  SENSITIVE = 1
    IF widgetData.orig EQ 0 THEN BEGIN 
        widget_control, widgetData.eph_base,  SENSITIVE = 1
        widget_control, widgetData.mag_base,  SENSITIVE = 0
   ENDIF ELSE BEGIN
        widget_control, widgetData.eph_base,  SENSITIVE = 0
        widget_control, widgetData.mag_base,  SENSITIVE = 1
   ENDELSE 
   done_coords:
END


END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_geo_eph_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO lanl_geo_eph_panel_event, event
  
COMMON lanl_geo_eph_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF
    'spb_sat' : widgetData.sc_id = event.value
    'spb_orig': BEGIN
        widgetData.orig = event.value
        lanl_geo_eph_refresh, /COORDS
    END 
    'spb_eph' : widgetData.product = event.value
    'spb_eph_ticks' : BEGIN 
        WIDGET_CONTROL, widgetData.eph_base, GET_VALUE = ch
        widgetData.product = 0 &  n_ch = n_elements(ch)
        FOR i = 0, n_ch-1 DO IF ch(i) THEN $
          widgetData.product = widgetData.product+2^i  
    END
    ELSE: BEGIN
        widgetData=papco_panel_Event(event, widgetData)
    END
ENDCASE 
    
END
