;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_geo_eph, panel, PlotInfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behavior is controlled by the vector panel
;*	which controls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*              - if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale or zscale   
;*
;* INPUTS:       
;*      panel   three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      PlotInfo    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an existing panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*    PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_lanl_geo_eph, panel, PlotInfo, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
PRO p_lanl_geo_eph, panel, PlotInfo, OUTPUT=OUTPUT, OVERPLOT=OVERPLOT, $
                    PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                    SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end               ;time limit in mjdt
COMMON yscale, yscl                             ;man/auto yscaling
COMMON shift_label, down                        ;for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  ;info on coords

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_geo_eph. 
COMMON geo_eph_data, input_header, input_data
COMMON lanl_geo_eph_tick, time, yray, mag_model, tick_plotinfo
COMMON lanl_geo_eph
COMMON papco_color_names
  
IF keyword_set(OUTPUT) THEN output = OUTPUT ELSE output = 0

;restrict data to actual time range requested - this makes for faster zooming
;input data is in TAI. For plotting, always start at zero to get maximum
;resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.tai ge tai_1) AND (input_data.tai le tai_2),c)
if c ne 0 then data=input_data(index) ELSE GOTO, noplot
xut1=0  &  xut2=tai_2-tai_1 & time = data.tai-tai_1

control = *plotinfo.USR_PTR1
sat_id = sat_names(control.sc_id)

mag =  plotinfo.ioptions(2)
type = plotinfo.typevector(1) 
utitle='LANL geo eph !C'+sat_id

;select data to plot
Re=6371.0
;set bad data flag
nodata = -99 & n_ch = 0
idx = where(data.rad eq 0, cc) 
IF cc EQ n_elements(data) THEN GOTO, noplot

CASE control.orig OF 
    0: BEGIN                    ;original epehemeris data
        CASE control.product OF 
            0: BEGIN
                yray = data.LAT
            END 
            1: BEGIN
                yray = data.LON
            END 
            2: BEGIN
                yray = data.RAD
            END 
        ENDCASE
        IF cc NE 0 THEN yray(idx) = nodata
        utitle=utitle+'!C'+eph_names(control.product)
        uytitle = yunits(control.product)
    END 
    1: BEGIN                    ;Model ephemeris / mag field data
        nodata = -1e31
        model_num = varprt(control.internal_model)+'_'+$
                             varprt(control.external_model)

        IF control.MAG_DYN THEN dyn_str = '_DYNA' ELSE dyn_str = '_STAT'
        mag_model = 'MODEL_'+model_num+dyn_str

        tags = TAG_NAMES(input_data(0))
        mag_idx = where(tags EQ mag_model, c)
        IF c EQ 0 THEN BEGIN 
            message, 'Magnetic model data for '+mag_model+' not found', /cont
            GOTO, noplot
        ENDIF 

        mag_coord_names = papco_make_choice_names(control.MAG_COORDS_INFO)
        mag_coord_units = papco_make_choice_names(control.MAG_COORDS_UNITS)

        CASE 1 OF
            (control.mag_coords EQ 0): BEGIN ;L coords are indep.of PA
                yray = data.(mag_idx(0)).L(0)
                yray = transpose(yray)
            END  
            (control.mag_coords EQ 1): BEGIN ;L* coords are per 9 PA's
                pitch_names = papco_make_choice_names(control.pitch_proc_info)
                pa_idx = control.pitch_plot
                mag_model = mag_model+'_'+varprt(pitch_names(pa_idx))
                yray = data.(mag_idx(0)).LSTAR(pa_idx)
                yray = transpose(yray)
            END  
            (control.mag_coords GE 2) AND (control.mag_coords LE 6): BEGIN
                yray = data.(mag_idx(0)).(control.mag_coords)
            END 
            (control.mag_coords EQ 7): BEGIN ;Btot, derived
                yray = sqrt(data.(mag_idx(0)).b1^2 + $
                            data.(mag_idx(0)).b2^2 + $
                            data.(mag_idx(0)).b3^2)
            END 
            (control.mag_coords EQ 8): BEGIN ;Btot/Bmin
                btot = sqrt(data.(mag_idx(0)).b1^2 + $
                            data.(mag_idx(0)).b2^2 + $
                            data.(mag_idx(0)).b3^2)
                bmin =  data.(mag_idx(0)).bmin
                idx = where(bmin NE 0, c)
                yray = fltarr(n_elements(data)) & yray(*) = nodata
                IF c NE 0 THEN yray(idx) = btot(idx)/bmin(idx)
            END 
        ENDCASE 
        uytitle = mag_coord_units(control.mag_coords)
        utitle=utitle+'!C' + $
               mag_model+' '+mag_coord_names(control.mag_coords)
        idx = where(yray EQ nodata, cc)
        IF cc NE 0 THEN yray(idx) = nodata
        tick_label = mag_coord_names(control.mag_coords) + $
                     '('+uytitle+':'+model_num+')'
    END
ENDCASE

IF output EQ 2 THEN return

panelset,panel	         		;sets the panel position viewport

if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    papco_ylim,time,yray,yst,yen, LOG=yscl(panel(0),3), BAD_DATA=nodata
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
endelse

; use auto y-labeling routine to get "good" y-lables
papco_y_label, yst, yen, LOG=yscl(panel(0),3)  

extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
               ylog:yscl(panel(0),3), xtickformat:'noticks'}
   
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; use papco routine to draw time axis. This checks for bottom plot
; and uses the user's  xtickformat if it is something other than 'noticks'.
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

; now do the actual data plot, data only, no axis
IF n_ch EQ 0 THEN BEGIN         ;ephemeris coords

    papco_gap_plot, time, yray, nodata, xstyle=5, ystyle=5, $
                                        _extra=extra_plotPar

ENDIF ELSE BEGIN                ;mag field data
    
    FOR i = 0, n_ch-1 DO BEGIN
        IF n_ch NE 1 then extra_plotPar.color = colors(i)
        papco_gap_plot, time, yray(idx2(i), *), nodata, $
              xstyle=5, ystyle=5, _extra=extra_plotPar
    ENDFOR 
    ; plot channel info inside of panel.
    papco_in_side_label, panel, in_label, NCOL = colors, _extra=extra_plotPar

ENDELSE

; draw y-axis label
IF NOT keyword_set(overplot) THEN left_side_label, panel, uytitle, /ROT90

; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y

; plot info right of the panel. Multiple panel labels are handled.
papco_rs_lbl, panel, utitle, /LINE, _extra=extra_Par
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
  
IF KEYWORD_SET(OUTPUT) THEN BEGIN 
    message, 'Writing plot data out to file', /info
    description='lanl geo eph data'
    time_label='Seconds since start of day'
    ;channels=yarr
    y_label=uytitle
    ;z_label=uztit
    rs_label=utitle
    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, $
                      RS_LABEL=RS_LABEL
ENDIF   

return

;jump to here if no data to plot 
noplot: 
message,'No data to plot', /info
plotted_x = !x  &  plotted_y = !y

END 
