;******************************************************************************
;*  NAME:        r_geo_eph
;*
;*  DESCRIPTION: Routine to read geo_eph files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
PRO r_geo_eph, plotinfo, AVERAGE = AVERAGE, VERBOSE=VERBOSE

COMMON geo_eph_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg

get_err_no=0  &  get_err_msg='geo_eph read succesfully'

; check env. variable pointing to data directory
if not (papco_check_data_env('GEO_EPH_DATA', path=PATH)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     

; clear data arrays for reuse
input_header=0  &  input_data=0

; loop through from start to end time concatonating data files into one
; array.
   
time=mjdt_start  &  no_files_read=0  &  day_count=0.0
    
WHILE mjdt_end.mjd GE time.mjd DO BEGIN

    r_geo_eph_one, plotinfo, time, AVERAGE = AVERAGE, VERBOSE=VERBOSE

    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin 
      if no_files_read eq 0 then begin   ;start the array buffer
        input_data_buf=temporary(input_data)
      endif else begin                   ;otherwise concatonate arrays
        input_data.t=input_data.t+86400*day_count
        input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse        
      input_data=0                     ;clear array for reuse
      no_files_read=no_files_read+1
    endif
    next_day:
    time.mjd=time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) THEN BREAK
    day_count=day_count+1

ENDwhile
  
; check if any files were read   
if (no_files_read ne 0) then begin
    msg=string(no_files_read,format="(i2)")+ ' files read succesfully'
    get_err_no=0  &  get_err_msg=msg
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
endif else begin
    msg='no files found for timerange'
    get_err_no=1  &  get_err_msg=msg
endelse      
message,msg,/cont

END 

;******************************************************************************
;*  NAME:        r_geo_eph_one,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       sat_id, date  - string of sate name, date yyyymmdd
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_geo_eph_one, plotinfo, time, AVERAGE = AVERAGE, VERBOSE = VERBOSE, $
                                   SAVE = SAVE 
  
COMMON geo_eph_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON lanl_geo_eph
   
get_err_no=0 & get_err_msg='geo_eph one read succesfully'
   
; check keywords and set
if keyword_set(VERBOSE) then VERBOSE=1 else VERBOSE=0

; check env. variable pointing to data directory
if not (papco_check_data_env('GEO_EPH_DATA', path=PATH)) then return

;assemble the data file from the date requested
utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)
  
control = *plotinfo.USR_PTR1
sat_id = sat_names(control.sc_id)

file_path = path+strmid(datestr, 0, 4)+papco_get_pathsep()
fln = file_path+datestr+'_'+sat_id+'_eph.idl'
eph_struct_flname = file_path+'eph_struct.idl'

;test if subdir exists, create if necessary
papco_create_dir, file_path

IF keyword_set(SAVE) THEN BEGIN
    message, 'Saving', /info
    print, '  ', fln
    save, file = fln, input_header, input_data, /COMPRESS
    return
ENDIF

; clear data arrays for reuse
input_header=0  &  input_data=0

;read existing file, or make it
filethere=findfile(fln, count = c)

IF c EQ 0 THEN BEGIN
    get_err_no=2 & get_err_msg = '..'+strright(fln+' not found', 60)
    message, get_err_msg, /cont
    ;then read original data, extract ephemeris
    r_geo_eph_one_original, plotinfo, datestr, flname, AVERAGE = AVERAGE

    IF get_err_no EQ 0 THEN BEGIN     
        ;if no gps_eph_struct exists, write the original one
        result = findfile(eph_struct_flname, count = c)
        IF c EQ 0 THEN BEGIN
            eph_struct = input_data(0)
            save, file = eph_struct_flname, eph_struct
        ENDIF
    ENDIF ELSE BEGIN
        return
    ENDELSE     
ENDIF ELSE BEGIN
    restore, fln,  VERBOSE = VERBOSE
    message, 'idl saveset read succesfully', /cont
    print, '  ', fln
ENDELSE

;now make/add/check mag coords in input_data.
;need to "hand over" time in TAI,coord system info, actual eph coords. 

;GEO eph is in geographic lat/long
tai = input_data.tai
ndat =  n_elements(input_data)
pos = dblarr(3, ndat)

rad = input_data.RAD*6371.0d
lon = input_data.LON
colat = 90.0d - input_data.LAT

cor_type='CAR' & cor_sys='GEO'
pos(0, *) = rad*sin(colat/!radeg)*cos(lon/!radeg)
pos(1, *) = rad*sin(colat/!radeg)*sin(lon/!radeg)
pos(2, *) = rad*cos(colat/!radeg)

papco_add_mag, control, cor_type, cor_sys, $
               eph_struct_flname, new_struct_name, tai, pos, input_data, error

input_header = {model_name:new_struct_name}

IF error EQ 2 THEN return ;no new structures added to input_data

;save the new eph data to file
message, 'Writing new', /cont
print, '  ', fln
save, file = fln, input_header, input_data, /COMPRESS

;save the new eph structure to file
message, 'Writing new structure contents to file', /cont
eph_struct = input_data(0)
save, file = eph_struct_flname, eph_struct
      
END 
  
;******************************************************************************
;*  NAME:        r_geo_eph_one_original, plotinfo, date, flname
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_geo_eph_one_original, plotinfo, date, flname, VERBOSE=VERBOSE, $
                            AVERAGE = AVERAGE
    
COMMON geo_eph_data, geo_eph_header, geo_eph_data
COMMON lanl_geo_data, geo_header, geo_data
COMMON lanl_mpa_data, mpa_header, mpa_data, input_sat
COMMON get_error, get_err_no, get_err_msg    
COMMON lanl_geo                 ;see lanl_geo_commons.pro for contents
COMMON lanl_mpa, mpa_sat_names, sat_codes, mpa_select_names, mpa_select_frmts, mpa_dat_names, lanl_mpa_control
 
;read lanl geo data file, extract ephemeris info.

control = *plotinfo.USR_PTR1
new_plotinfo = papco_getplotinfostruct()
new_control = lanl_geo_control
new_control.SC_ID = control.SC_ID
new_plotinfo.USR_PTR1 = ptr_new(new_control)

r_lanl_geo_one, date, new_plotInfo, VERBOSE=VERBOSE

IF get_err_no NE 0 THEN BEGIN   ;try to get eph data from mpa files
    IF NOT keyword_set(mpa_sat_names) THEN GOTO, no_mpa
    sat_id = sat_names(control.sc_id)
    idx = where(strupcase(mpa_sat_names) EQ strupcase(sat_id))
    PlotInfo.typevector(0) = idx(0)
    r_lanl_mpa, PlotInfo
    IF get_err_no EQ 0 THEN BEGIN 
        ndat = n_elements(mpa_data)
        dmy = {time:0.0d, Latitude:0.0d, Longitude:0.0d, Radius:0.0d}
        geo_data = replicate(dmy, ndat)
        rect = transpose([[mpa_data.XGEO], [mpa_data.YGEO], [mpa_data.ZGEO]])
        res = cv_coord(from_rect = rect,  /TO_SPHERE,  /DEGREES,  /DOUBLE)
        geo_data.longitude =  transpose(res(0, *))
        geo_data.latitude = transpose(res(1, *))
        geo_data.radius = transpose(res(2, *))/6371.0
        geo_data.time = mpa_data.UTHOURS
    ENDIF
ENDIF 

no_mpa:

;if no data read, fill with nodata values.

;now eph data is read. Average as required
if keyword_set(AVERAGE) then av=AVERAGE else av=300 ;deault 5 min
message, 'averaging to '+varprt(av)+' sec resolution', /cont

yr=fix(strmid(date,0,4))  & mo=fix(strmid(date,4,2)) & dy=fix(strmid(date,6,2))

;make new structures for ascii ephemeris data only
dummy={T:0.0, LAT:-99.0, LON:-99.0, RAD:-99.0, TAI:0.0d}
ndat=fix(86400.0 / av)
input_data=replicate(dummy, ndat)
  
for j=0l,ndat-1 do begin
    
    t=j*av+av/2  &  t1=t-av/2  &  t2=t+av/2
    input_data(j).t=t

    hour=fix(t/3600.0)  &  min=fix((t mod 3600.0)/60.0)
    sec = t-hour*3600l-min*60l
    input_data(j).tai=UTC2TAI({YEAR:yr, MONTH:mo, DAY:dy, HOUR:hour, $
                               MINUTE:min, SECOND:sec, MILLISECOND:0})


    IF get_err_no NE 0 THEN continue

    idx=where((geo_data.time ge t1/3600.0) and (geo_data.time lt t2/3600.0),cc)
    if cc ne 0 then BEGIN
        idx2 = where(geo_data(idx).Radius GT 6, c1)
        IF c1 NE 0 THEN BEGIN
            input_data(j).Lat=total(geo_data(idx(idx2)).Latitude)  / c1
            input_data(j).Lon=total(geo_data(idx(idx2)).Longitude) / c1
            input_data(j).Rad=total(geo_data(idx(idx2)).Radius)    / c1
        endif      
    endif 
endfor

get_err_no = 0

geo_eph_data = input_data

return  
    
read_error:
close,unit & free_lun,unit
msg='Problem reading file'  &  message,msg,/cont
get_err_no=3  &  get_err_msg=msg 
spawn,'rm '+scratch_dir+file
      
END 


;------------------------------------------------------------------------------
;program to compute lanl footpoints
;
;Keywords:
;          HIGHRES     draw high resolution maps
;          ONEDAY      draw geo footprint for full day of date
;          LONG        geographic longitude of geostationary satellite
;          DATE        year day for footpoints
;          KP_LIMITS   array of kp values for footpoints
;          PLACES      indicated list of places on map
;          LIMIT       map projection limits, [lat1,long1,lat2,long2]
;          PS          1 make postscript, 2 sent to printer
;          OUTPUT      write footpoints coords to file
;          POLAR       make polar view of map. Used with NORTH_SOUTH
;          NORTH_SOUTH plot lcoal noon / midnight for all longitudes
;
; Used calls to this program and reason:
; For Borovsky, roadmap mission website, 15 March 2002
; 

pro m_lanl_footpoints, PS=PS, VERB=VERB, LIMIT=LIMIT, NO_FOOT=NO_FOOT, $
                       KP_LIMITS=KP_LIMITS, HIGHRES=HIGHRES, DATE=DATE, $
                       ONEDAY = ONEDAY, LONG=LONG, FILE = FILE, $
                       RIVERS = RIVERS, NORTH_SOUTH = NORTH_SOUTH,  $
                       PLACES = PLACES,  INSIDE = INSIDE, $
                       OUTPUT = OUTPUT, POLAR = POLAR, DASI = DASI, IRIS = IRIS

papco, /set_only
  
COMMON papco_color_names
  
cols=[black,red,dark_green,magenta,burgundy,olive,teal,royal_blue,violet]
  
; setup some gepographical points
dummy={pl, str:'', lat:0.0, lon:0.0}  &  place=replicate(dummy,15)
; City                          latitude               eastlongitude  
place(0).str='Fairbanks'   &  place(0).lat=64.983  &  place(0).lon=-147.67
place(1).str='Dawson'      &  place(1).lat=64.067  &  place(1).lon=-139.42
place(2).str='Yellowknife' &  place(2).lat=62.500  &  place(2).lon=-114.17
place(2).str=''            &  place(2).lat=62.500  &  place(2).lon=-114.17
place(3).str='Churchhill'  &  place(3).lat=58.750  &  place(3).lon=-94.083
place(4).str='GreatWhale'  &  place(4).lat=55.333  &  place(4).lon=-77.667
place(5).str='Tromso'      &  place(5).lat=69.667  &  place(5).lon=19.000
place(6).str='GooseBay'    &  place(6).lat=53.250  &  place(6).lon=-60.333
place(7).str='Kiruna'      &  place(7).lat=67.833  &  place(7).lon=20.333
place(8).str='Oulu'        &  place(8).lat=65.017  &  place(8).lon=25.483
place(9).str='Sodankyla'   &  place(9).lat=67.370  &  place(9).lon=26.630
place(10).str='Uppsala'    &  place(10).lat=59.900 &  place(10).lon=17.350
place(11).str='Kev'        &  place(11).lat=69.760 &  place(11).lon=27.010
place(12).str='Lovozero'   &  place(12).lat=67.970 &  place(12).lon=35.080
place(13).str='Helsinki'   &  place(13).lat=60.250 &  place(13).lon=25.050  
place(14).str='EISCAT'     &  place(14).lat=69.590 &  place(14).lon=19.230 
 
if keyword_set(HIGHRES) then highres=HIGHRES else HIGHRES=0
if keyword_set(KP_LIMITS) then kp_limits=KP_LIMITS else kp_limits=[0]
kp_lim_str = ''
FOR i = 0, n_elements(kp_limits)-1 DO $
    kp_lim_str = kp_lim_str+', '+varprt(kp_limits(i))
if keyword_set(DATE) then date=DATE else date='19950321'
if keyword_set(LONG) then long=(LONG) else long=0.0d0
  result=size(long) 
if result(1) eq 7 then begin
    result=strpos(place.str, long)
    index=where(result ne -1,c)
    if c eq 1 then long= place(index(0)).lon else begin
      message,'Place: '+long+' not found',/cont
      return
    endelse  
endif  
long=double(long)  
IF keyword_set(INSIDE) THEN dd = 0.185 ELSE dd = 0

filename='/u/friedel/lanl_foot_map_'+varprt(fix(long))+'_'+date
 
if not keyword_set(PS) then begin
    set_plot,'X'
    th = 1
    window, 20, xsize=600, ysize=600, xpos=0
    wset,20
endif else begin
    set_plot,'PS'
    th = 1
    fln=filename+'.ps'
    device,/inches, filename=fln, $
      xsize=8, ysize=8, yoffset=1, xoffset=0.25, /color, bits=8
 endelse  
  
papco_setcolortable

!p.position=[0.1,0.0,0.9,0.8]
if keyword_set(LIMIT) then limit=LIMIT else limit=[50,-30,85,70]
if strupcase(string(LIMIT(0))) eq 'SKANDINAVIA' then limit=[56,4,72,32]

IF keyword_set(POLAR) THEN $
    map_set, 90, 20, 0, /azimuthal, LIMIT= [50,-180,90,180] ELSE $
    map_set, /cyl,  LIMIT=limit, color=black
    
map_continents, /countries, hires=HIGHRES, color=black, mlinethick=th/2
map_continents, /coasts,    hires=HIGHRES, color=royal_blue, mlinethick=th
IF keyword_set(RIVERS) THEN $
    map_continents, /rivers,    hires=HIGHRES, color=cyan, mlinethick=th/2

map_grid, LONDEL=10, LATDEL=5, LATLAB = -121, LONLAB = 72, $
    label = 1, color=black
  
if keyword_set(NO_FOOT) then return  
  
year=long(strmid(date,0,4))
month=long(strmid(date,4,2))
day=long(strmid(date,6,2))
  
;loop calls to m_foot

mdate={zdat, iyear:year, imonth:month, iday:day,$
             ihour:0L, imin:0L, secs:0.0d0, amjd:   0.0d0}
Re = 6371.0d  
mpos = {zgeo, radius:6.6d*Re, colat:90.0d0, elong:long}

IF keyword_set(FILE) THEN BEGIN
    in_str = ''
    openr, u, '/u/friedel/Weiss_geo_foot.txt', /get_lun
    readf, u, in_str
    head = strcompress(in_str)
    head = strsplit(head, ' ', /extract)
    dummy = create_struct(head(0), 0.0)
    FOR i = 1, n_elements(head)-1 DO dummy = create_struct(dummy, head(i), 0.0)
    data = replicate(dummy, 200)
    ndat = 0
    WHILE not eof(u) DO BEGIN
        readf, u, dummy
        data(ndat) = dummy
        ndat = ndat+1
    ENDWHILE 

    data = data(0:ndat-1)
    close, u & free_lun, u
  
    ;plot required data
    FOR i = 0, ndat-1 DO BEGIN
        plots,data(i).T0_DMSPLONG,data(i).T0_DMSPLAT, $
          /data
        plots,data(i).T1_DMSPLONG,data(i).T1_DMSPLAT, $
          /data, color=teal, /continue,  thick = th*1.5
        plots,data(i).T0_DMSPLONG,data(i).T0_DMSPLAT, $
          /data
        plots,data(i).T2_DMSPLONG,data(i).T2_DMSPLAT, $
          /data, color=teal, /continue,  thick = th*1.5
        plots,data(i).T2_DMSPLONG,data(i).T2_DMSPLAT, $
          /data, psym=6, symsize = 0.25, color=violet,  thick = th

        plots,data(i).T0_DMSPLONG,data(i).T0_DMSPLAT, $
          /data, psym=2, symsize = 3, color=black, thick = th
        plots,data(i).T1_DMSPLONG,data(i).T1_DMSPLAT, $
          /data, psym=6, symsize = 0.25, color=violet,  thick = th
    ENDFOR

ENDIF
  
if keyword_set(ONEDAY) then BEGIN

    count = 0
    n_footpoints = 1440/(120/(1+highres*7)) + 1
    foot = fltarr(n_footpoints, n_elements(kp_limits), 3)

    
    xyouts,0.1,0.96,'Geostationary Northern Footprint for '+ $
      varprt((fix(long))) + '!Uo!N  longitude',charsize=1.5, /norm, color=black
    xyouts,0.1,0.93,'For date: '+date,$
      charsize=1.5, /norm, color=black
    xyouts,0.1,0.90,'IGRF Internal, T89 external model',$
      charsize=1.2, /norm, color=black
    xyouts,0.1,0.88,'Kp'+kp_lim_str, charsize=1.2, /norm, color=black    

    for j=0, n_elements(kp_limits)-1 do begin   ;loop all KP's
      kp=double(kp_limits(j)) &  col=cols(1+count) 
      xyouts,0.1+count*0.09,0.84,'Kp = '+varprt(fix(kp)),$
      charsize=1.2, /norm, color=col, charthick=2
      ;loop for footprints during one day at given longitude.
      for i=0l,1440,120/(1+highres*7) do begin
        hour=(i / 60l) mod 24
        min=i mod 60l
        mdate.ihour=hour  &  mdate.imin=min
      
        ;new, uses uxidl under LINUX
        COMMON mag_foot, n_colat, n_elong, s_colat, s_elong
        m_foot, MDATE = MDATE, MPOS = MPOS, KP = kp

        if i eq 0 then plots,n_elong,90-n_colat,/data else $
          plots,n_elong,90-n_colat, /data, /continue, thick=3, color=col 
             
        IF keyword_set(VERBOSE) THEN $
          print,  i/(120/(1+highres*7)), n_elong, 90-n_colat
        foot(i/(120/(1+highres*7)), j, 0) = n_elong
        foot(i/(120/(1+highres*7)), j, 1) = 90-n_colat
        foot(i/(120/(1+highres*7)), j, 2) = i

        ;IF min EQ 0 THEN BEGIN
        ;    xyouts, n_elong,90-n_colat, varprt(hour), /data, color = black, $
        ;      align = 0.5,  charsize = charsize
        ;ENDIF 

     endfor
      count=count+1
    endfor
endif  
  
IF keyword_set(OUTPUT) THEN BEGIN
      out_fln = filename+'.txt'
      message, 'writing footpoint coords to file: '+out_fln, /cont
      openw, u, out_fln, /get_lun
      head_str = 'KP     lon       lat       min'
      printf, u, head_str
      FOR i = 0, n_elements(kp_limits)-1 DO BEGIN
          FOR j = 0,n_footpoints-1 DO BEGIN
              printf, u, kp_limits(i), foot(j,i,0), foot(j,i,1), foot(j,i,2), $
                format = "(i2,'  ',2(f8.4,'  '),'  ',i4)"
          ENDFOR 
      ENDFOR
      close, u & free_lun, u
ENDIF


IF keyword_set(NORTH_SOUTH) THEN BEGIN 
  ;do northern / southern limit of footprint
  for j=0, n_elements(kp_limits)-1 do begin   ;loop all KP's
    kp=double(kp_limits(j)) &  col=cols(1+count) 
    xyouts,0.13+count*0.09,0.84-dd,'Kp = '+varprt(fix(kp)),$
      charsize=1.2, /norm, color=col, charthick=2

    xyouts,0.13,0.96-dd,'Geostationary Northern Footprint for all longitudes',$
      charsize=1.5, /norm, color=black
    xyouts,0.13,0.93-dd,'at local noon and local midnight',$
      charsize=1.5, /norm, color=black
    xyouts,0.13,0.90-dd,'For date: '+date, $
      charsize=1.5, /norm, color=black
    xyouts,0.13,0.87-dd,'IGRF Internal, T89 external model',$
      charsize=1.2, /norm, color=black
    
    ;do for local midnight at each longitude
    for i=0,360,5 do begin
      mpos.elong=double(i)        ;longitude
      loct_midnight=24.0-(mpos.elong/15.0)

      mdate.ihour=long(loct_midnight)
      mdate.imin=long((loct_midnight-double(mdate.ihour))*60.0)
  
      status=CALL_EXTERNAL(ext_lib,'unilib_get_footpoint_', $
                           mdate, mpos, intmod, extmod, kp, $
                           n_colat, n_elong, s_colat, s_elong,$
                           ifail)
      
      if i eq 0 then plots,n_elong,90-n_colat,/data else $
        plots,n_elong,90-n_colat, /data, /continue, color=col
      ;xyouts,n_elong,90-n_colat,'M',/data, align=0.5
      if keyword_set(VERB) then $
        print, mpos.elong, 90-n_colat,n_elong,mdate.ihour,mdate.imin
  
    endfor
    
    ;do for local noon at each longitude
    for i=0,360,5 do begin
      mpos.elong=double(i)        ;longitude
      loct_midnight=24.0-(mpos.elong/15.0)

      mdate.ihour=long((loct_midnight+12.0) mod 24)
      mdate.imin=long(((loct_midnight+12.0)-long(loct_midnight+12.0))*60.0)
  
      status=CALL_EXTERNAL(ext_lib,'unilib_get_footpoint_', $
                           mdate, mpos, intmod, extmod, kp, $
                           n_colat, n_elong, s_colat, s_elong,$
                           ifail)
    
      if i eq 0 then plots,n_elong,90-n_colat,/data else $
        plots,n_elong,90-n_colat, /data, /continue, color=col
      ;xyouts,n_elong,90-n_colat,'N',/data, align=0.5
      if keyword_set(VERB) then $
        print, mpos.elong, 90-n_colat,n_elong,mdate.ihour,mdate.imin
    
    endfor    
    count=count+1
  endfor
ENDIF

IF keyword_set(PLACES) THEN BEGIN 
    ;plot places
    charsize = 1
    for i=0,n_elements(place)-1 do begin
        plots,place(i).lon,place(i).lat, /data, psym = 4, noclip = 0, $
          color=dark_green, symsize = 1,  thick = th
        IF i MOD 2 NE 0 THEN $
        xyouts,place(i).lon,place(i).lat,' '+place(i).str, /data, noclip=0, $
          color=dark_green, charthick=th, charsize = charsize $
        ELSE $
        xyouts,place(i).lon,place(i).lat,place(i).str+' ', /data, noclip=0, $
          color=dark_green, charthick=th, charsize = charsize, align = 1.0
    endfor  
ENDIF

IF keyword_set(DASI) THEN BEGIN
    lon_box = [13.5, 27.0, 27.0, 13.5, 13.5]
    lat_box = [67.6, 67.6, 72.5, 72.5, 67.6]
    FOR i = 0, n_elements(lat_box) -1 DO BEGIN
        IF i eq 0 then plots, lon_box(i), lat_box(i), /data else $
          plots, lon_box(i), lat_box(i), /data, /continue, $
            thick=2, color=black, linestyle = 2
    ENDFOR 
    xyouts, lon_box(3), lat_box(3), 'DASI FOW ', $
      charthick=th, charsize = charsize, align = 1.0, color=violet
ENDIF
         
IF keyword_set(IRIS) THEN BEGIN
    lon_box = [17.6, 23.5, 23.5, 17.6, 17.6]
    lat_box = [67.8, 67.8, 70.2, 70.2, 67.8]
    FOR i = 0, n_elements(lat_box) -1 DO BEGIN
        IF i eq 0 then plots, lon_box(i), lat_box(i), /data else $
          plots, lon_box(i), lat_box(i), /data, /continue, $
            thick=2, color=black, linestyle = 2
    ENDFOR 
    xyouts, lon_box(3), lat_box(3), 'IRIS FOW ', $
      charthick=th, charsize = charsize, align = 1.0, color=violet
ENDIF
  
if keyword_set(PS) then begin
    device,/close
    message, 'PS file: '+fln, /cont
    case ps of
      1: begin
      end
      2: begin
        print, 'lpr -P n114rgb '+fln
        spawn, 'lpr -P n114rgb '+fln
      end  
      3: begin
        print, 'lpr -P w108abw '+fln
        spawn, 'lpr -P w108abw '+fln
      end  
      4: begin
        print, 'lpr -P n114rgb_t '+fln
        spawn, 'lpr -P n114rgb_t '+fln
      end  
      else: begin
      end
    endcase
ENDIF 

END 

;------------------------------------------------------------------------------
pro steinberg_q
  
  for lon=-180,180,10 do begin
    
    pos={radius:6.6,$
         latitude:0.0,$
         longitude:lon}
    
    time_date={year:1999, month:01, day:01, $
                   hour:10, min:0, secs:0}
    
    f77_unilib_get_BL, time_date, pos, fbm, flm, xmlt, xlat, $
      INTMOD=0, EXTMOD=5, PITCH=90.0
    
    print, lon, flm, xmlt, xlat
    
  endfor  
    
end


;******************************************************************************
PRO r_lanl_geo_eph, plotinfo, AVERAGE = AVERAGE, VERBOSE=VERBOSE

r_geo_eph, plotinfo, AVERAGE = AVERAGE, VERBOSE=VERBOSE

END 
;******************************************************************************
