;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  lanl_gps_panel, plotinfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      plotinfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from plotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(plotinfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION lanl_gps_panel, plotinfo, GROUP=group, ACTION=ACTION, $, $
                         SIZE_ONLY = SIZE_ONLY, SCROLL = scroll, $
                         X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize, $
                         _EXTRA=extra
                         
COMMON lanl_gps_panel_data, widgetData
COMMON lanl_gps
  
;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;***NEW PAPCO 10.5*** 
;All the user's widget info is now stored in a structure,
;and the only thing stored in the plotinfostruct is the pointer to
;that structure. This allows unlimited freedom in variables for the
;module. The structure papco_cdf_control is defined in lanl_geo_eph_commons.pro

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(lanl_gps_control)
control = *plotinfo.USR_PTR1
;control now contains the user defined control structure for this module.

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
base=papco_WIDGET_BASE(title=plotinfo.panelKind, COLUMN=1, $
                       GROUP_LEADER=group, /MODAL)

panel_base = widget_base(base, COLUMN=1, _EXTRA=extra)

panel_top =    widget_base(panel_base, column = 1, /base_align_center)

panel_middle = widget_base(panel_base, ROW = 1, SCROLL = scroll, $
                           X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize)

panel_left=widget_base(panel_middle,column=1, /frame)
panel_right=widget_base(panel_middle,column=1,/frame)

panel_bottom = widget_base(panel_base,column = 1, /base_align_center)

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_top, plotinfo)
additionalWData=create_struct(widgets, additionalWData)

;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;--- satellite choice -- sat_id is set in lanl_gps_set_commons ----------------
sats_base=CW_BGROUP(panel_left, sat_id, /FRAME, ROW=1, /EXCLUSIVE, $
                    UVALUE='mpb_sats' ,SET_VALUE=control.sc_id)
  
;--- plot style choice -- style_names is set in lanl_gps_set_commons  ---------
style_base=CW_BGROUP(panel_left, style_names, ROW=1, /EXCLUSIVE, $
                     UVALUE='spb_style', SET_VALUE=control.plot_style) 
    
;--- buttons to chose more options --------------------------------------------
opti_base=WIDGET_BASE(panel_left, ROW=1)
  
average_base=CW_BGROUP(opti_base, ['seg.','av.'], /FRAME, ROW=1, /EXCLUSIVE, $
                       UVALUE='spb_average', SET_VALUE=control.seg_av) 
reload_base=CW_BGROUP(opti_base,['Relaod','Adopt'], /FRAME,ROW=1,/EXCLUSIVE, $
                      UVALUE='spb_reload', SET_VALUE=control.reload)
spec_base=CW_BGROUP(opti_base, ['line','spec'], /FRAME, ROW=1, /EXCLUSIVE, $
                    UVALUE='spb_spec', SET_VALUE=control.spec)
sngl_base=CW_BGROUP(opti_base,['normal','singles'],/FRAME, ROW=1, /EXCLUSIVE, $
                    UVALUE='spb_sngl', SET_VALUE=control.singles)

;--- buttons to chose type of calibration to be used --------------------------
flux_cal_base=CW_BGROUP(panel_left, flux_cal_names, ROW=2,/EXCLUSIVE, $
                        UVALUE='spb_flux_cal', SET_VALUE=control.cal)
  
;--- couple of single toggle buttons for more choices -------------------------
toggle_base=WIDGET_BASE(panel_left, /FRAME, COLUMN=3, /NONEXCLUSIVE)

spb_dead_time=WIDGET_BUTTON(toggle_base, UVALUE='spb_dead_time', $
                            VALUE='dead time correction (*) ')
WIDGET_CONTROL, spb_dead_time, SET_BUTTON=control.dead_time 
  
spb_chanlab=WIDGET_BUTTON(toggle_base,UVALUE='spb_chanlab', $
                          VALUE='channel labels ')
WIDGET_CONTROL, spb_chanlab, SET_BUTTON=control.chanlab
  
spb_eqcor=WIDGET_BUTTON(toggle_base, UVALUE='spb_eqcor', $
                        VALUE='correct to equator ')
WIDGET_CONTROL, spb_eqcor, SET_BUTTON=control.eqcor
  
spb_diag=WIDGET_BUTTON(toggle_base, UVALUE='spb_diag', $
                       VALUE='Turn on Diagnostics')
WIDGET_CONTROL, spb_diag, SET_BUTTON=control.diag

spb_bckg=WIDGET_BUTTON(toggle_base, UVALUE='spb_bckg', $
                       VALUE='Take out background')
WIDGET_CONTROL, spb_bckg, SET_BUTTON=control.bckg
  
;--- pitch angle sin^n choice -------------------------------------------------
ef_pad_n=CW_FIELD(panel_left, TITLE='Pitch Angle Dist. fit n:  ' ,$
                  UVALUE='ef_mu', /FLOATING, XSIZE = 12, $
                  VALUE=control.pad_n)

;--- button to chose "select" variable ----------------------------------------
filter_outer = widget_base(panel_left, COLUMN=1, /ALIGN_CENTER, /FRAME)
filter_names = 'select ' + select_names

flt_outer = widget_base(filter_outer, COLUMN=1, /ALIGN_CENTER)
filter_base=CW_BGROUP(flt_outer, filter_names, ROW=1,  /EXCLUSIVE, $
                  LABEL_LEFT='Filter variable', $
                  UVALUE='spb_filter', SET_VALUE=control.filter)

;-- select orginal or model coords - ------------------------------------------
names = ['Original Ephemeris', 'Mag Model Coords']
orig_base = CW_BGROUP(flt_outer, names , ROW=1, /EXCLUSIVE, $
                   LABEL_LEFT='Coord Source:', $
                   UVALUE='spb_orig', SET_VALUE=control.orig)

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(filter_outer, control, /MODEL_ONLY)
additionalWData=create_struct(widgets, additionalWData)

;--- Add sliders to filter data - L, MLT, MLAT, etc ---------------------------
;    Multiple map bases
select_outer_base=WIDGET_BASE(filter_outer, COLUMN=1, /ALIGN_CENTER)
select_base=WIDGET_BASE(select_outer_base, /align_center)

;--- make select widgets for all variables in "filter_names" ------------------
;    use papco_panel compound widget for this!

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L', $
            control.L_SELECT_RANGE, $
            control.L_SELECT_FROM_VAL, $
            control.L_SELECT_TO_VAL, 4.0, 20.0, 0.1, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L_STAR', $
            control.L_STAR_SELECT_RANGE, $
            control.L_STAR_SELECT_FROM_VAL, $
            control.L_STAR_SELECT_TO_VAL, 4.0, 20.0, 0.1, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLT', $
            control.MLT_SELECT_RANGE, $
            control.MLT_SELECT_FROM_VAL, $
            control.MLT_SELECT_TO_VAL, 0.0, 24.0, 0.5, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLAT', $
            control.MLAT_SELECT_RANGE, $
            control.MLAT_SELECT_FROM_VAL, $
            control.MLAT_SELECT_TO_VAL, -60, 60.0, 2, '(f5.1)')
additionalWData=create_struct(widgets, additionalWData)

;--- buttons for choosing channels. Any combination of buttons may be pressed.
;    default is all chosen. Not all choices are mapped together!
ch_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
ch_base=WIDGET_BASE(ch_outer_base, /align_center)  
  
;--- choose energy channels of ns08, ns10 to plot -----------------------------
energy_base_1=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=energy_names_1 & n_ch=n_elements(energy_names)
for i=0,n_ch-1 do $
    energy_names(i)=strformat(energy_names(i),14,/center)

energy_set=bytarr(n_ch) 
for i=0,n_ch-1 do begin
    if (control.product and 2^i) eq 2^i then energy_set(i)=1 $
    else energy_set(i)=0
endfor
  
mpb_typ1=CW_BGROUP(energy_base_1, energy_names, COLUMN=4,/NONEXCLUSIVE, $
    LABEL_TOP='Select energy channels (nominal MeV for ns08, ns10)', $
    UVALUE='mpb_typ1', SET_VALUE=energy_set)    
  
;--- choose FIT channels of ns08, ns10 to plot --------------------------------
fit_base_1=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=fit_energy_names_1 & n_ch=n_elements(energy_names)
for i=0,n_ch-1 do $
    energy_names(i)=strformat(energy_names(i),14,/center)

energy_set=bytarr(n_ch) 
for i=0,n_ch-1 do begin
    if (control.product and 2^i) eq 2^i then energy_set(i)=1 $
    else energy_set(i)=0
endfor

mpb_typ2=CW_BGROUP(fit_base_1, energy_names, COLUMN=4,/NONEXCLUSIVE, $
    LABEL_TOP='Select FIT energy channels (nominal MeV for ns08, ns10)', $
                     UVALUE='mpb_typ2', SET_VALUE=energy_set)   
  
;--- choose energy values to evaluate FIT function ----------------------------
fit_energy_base=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_entry=CW_FIELD(fit_energy_base, /COLUMN, /STRING, $ 
    title='Enter list of energy channels to evaluate (MeV, space delimited)',$
    UNAME= 'energy_entry', VALUE=control.fit_ch)
  
;--- choose energy channels of ns18,ns24,ns33,ns39 to plot --------------------
energy_base_2=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=energy_names_2 & n_ch=n_elements(energy_names)
for i=0,n_ch-1 do energy_names(i)=strformat(energy_names(i),14,/center)

energy_set=bytarr(n_ch)  
for i=0,n_ch-1 do begin
    if (control.product and 2^i) eq 2^i then energy_set(i)=1 $
    else energy_set(i)=0
endfor

mpb_typ3=CW_BGROUP(energy_base_2, energy_names, COLUMN=4,/NONEXCLUSIVE, $
    LABEL_TOP='Select energy channels '+$
              '(nominal MeV for ns18,ns24,ns28,ns33,ns39)', $ 
    UVALUE='mpb_typ3', SET_VALUE=energy_set)  
  
;--- choose FIT energy channels of ns33 to plot -------------------------------
fit_base_2=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=fit_energy_names_2 & n_ch=n_elements(energy_names)
for i=0,n_ch-1 do energy_names(i)=strformat(energy_names(i),14,/center)

energy_set=bytarr(n_ch)   
for i=0,n_ch-1 do begin
    if (control.product and 2^i) eq 2^i then energy_set(i)=1 $ 
    else energy_set(i)=0
endfor

mpb_typ5=CW_BGROUP(fit_base_2, energy_names, COLUMN=4,/NONEXCLUSIVE, $
    LABEL_TOP='Select FIT energy channels (nominal MeV for ns33', $
    UVALUE='mpb_typ5', SET_VALUE=energy_set)   
  
;--- choose singles channels to plot ------------------------------------------
energy_base_3=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=singles_names_1 & n_ch=n_elements(energy_names) 
set=bytarr(n_ch) 
for i=0,n_ch-1 do energy_names(i)=strformat(energy_names(i),14,/center)
for i=0,n_ch-1 do if (control.sngl_prod and 2^i) eq 2^i then set(i)=1 $
    else set(i)=0
mpb_typ4=CW_BGROUP(energy_base_3, energy_names, COLUMN=4,/NONEXCLUSIVE, $
                   LABEL_TOP='Select single channels ', $
                   UVALUE='mpb_typ4', SET_VALUE=set)  
  
;--- choose energy channels of ns41 to plot -----------------------------------
energy_base_41=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=energy_names_41 & n_ch=n_elements(energy_names)
for i=0,n_ch-1 do energy_names(i)=strformat(energy_names(i),14,/center)

energy_set=bytarr(n_ch) 
for i=0,n_ch-1 do $
    if (control.product and 2^i) eq 2^i then energy_set(i)=1 $ 
      else energy_set(i)=0

mpb_typ41=CW_BGROUP(energy_base_41, energy_names, COLUMN=4,/NONEXCLUSIVE, $
    LABEL_TOP='Select energy channels (for ns41)', $
    UVALUE='mpb_typ41', SET_VALUE=energy_set)   

;--- choose energy channels of ns54 to plot -----------------------------------
energy_base_54=WIDGET_BASE(ch_base, column=1, /frame, /align_center)
energy_names=energy_names_54 & n_ch=n_elements(energy_names)
  for i=0,n_ch-1 do energy_names(i)=strformat(energy_names(i),14,/center)

energy_set=bytarr(n_ch) 
for i=0,n_ch-1 do $
    if (control.product and 2^i) eq 2^i then energy_set(i)=1 $
      else energy_set(i)=0

mpb_typ54=CW_BGROUP(energy_base_54, energy_names, COLUMN=4,/NONEXCLUSIVE, $
    LABEL_TOP='Select energy channels (for ns54)', $
    UVALUE='mpb_typ54', SET_VALUE=energy_set)   
 
;--- sets up choices for PSD plots --------------------------------------------
psd_base=WIDGET_BASE(ch_base, /FRAME, COLUMN=1, /align_center) 
fld_mu = CW_FIELD(psd_base, /ROW,  XSIZE=46, $
                  TITLE = 'Enter mu1, mu2, ... (MeV/G):', $
                  UVALUE='fld_mu', VALUE = control.psd_mu)

;--- set up energy choices for coherent plots ---------------------------------
coherent_base = WIDGET_BASE(ch_base, /FRAME, COLUMN=1, /align_center) 
fld_coherent_base = CW_FIELD(coherent_base, /COLUMN,  XSIZE=46, $
     TITLE = 'Enter list of energy channels to interpolate to ' + $
              '(MeV, space delimited)', $
     UVALUE='fld_coherent', VALUE = control.coh_ch)

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (plotinfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    lanl_gps. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;  widgets=papco_panel_AddFile_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for specifying own plot description label -----
widgets=papco_panel_AddDescr_Widgets(panel_right, plotinfo, /VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the color bar for color plots ----
;  widgets=papco_panel_Addcolorbar_Widgets(panel_left, plotinfo)
;  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
widgets=papco_panel_add_PlotStyleButtons(panel_right, plotinfo, 0,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
Scale_Widget_base=widget_base(panel_right, ROW=1,/align_center)
   
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_panel_AddYScale_Widgets(Scale_Widget_base, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_panel_AddZScale_Widgets(Scale_Widget_base, plotinfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_bottom, Action, plotinfo, $
                                     /CONFIG, ROW=1)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the lanl_gps example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'lanl_gps_panel.help', $
                   mpb_typ1:mpb_typ1, $
                   mpb_typ2:mpb_typ2, $
                   mpb_typ3:mpb_typ3, $
                   mpb_typ41:mpb_typ41, $
                   mpb_typ54:mpb_typ54, $
                   mpb_typ4:mpb_typ4, $
                   mpb_typ5:mpb_typ5, $
                   energy_base_1:energy_base_1, $
                   energy_base_2:energy_base_2, $
                   energy_base_3:energy_base_3, $
                   energy_base_41:energy_base_41, $
                   energy_base_54:energy_base_54, $
                   fit_base_1:fit_base_1, $
                   fit_base_2:fit_base_2, $
                   fit_energy_base:fit_energy_base, $
                   energy_entry:energy_entry, $
                   psd_base:psd_base, $
                   fld_mu:fld_mu, $
                   coherent_base:coherent_base, $
                   fld_coherent_base:fld_coherent_base, $
                   ef_pad_n:ef_pad_n}

; add control struct, papco_panel structs
widgetData=papco_PANEL_CreateWidgetData(control, thisWidgetsData)
widgetData=papco_PANEL_CreateWidgetData(widgetData, additionalWData)

;--- return size of the window only here --------------------------------------
IF keyword_set(SIZE_ONLY) THEN BEGIN 
    size_only = WIDGET_INFO(panel_base, /GEOMETRY) & return, 1
ENDIF

;--- realize the window -------------------------------------------------------
lanl_gps_panel_event_refresh, /ALL
papco_Panel_Realize, base, widgetData
papco_xmanager, 'lanl_gps_panel', base
  
;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified}

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_gps_panel_event_refresh 

;* DESCRIPTION:  
;*      Takes refreesh actions on panel widgets controlled by keywords.
;*
;* INPUTS:       
;*      none
;*                  
;* KEYWORDS:
;*      
;* 
;* CALLING SEQUENCE:
;*      lanl_gps_panel_event_refresh, /keywords
;******************************************************************************
PRO lanl_gps_panel_event_refresh, ALL=ALL, $
                                  CHANNEL=CHANNEL, $
                                  EPH_LIMITS = EPH_LIMITS
  
COMMON lanl_gps_panel_data, widgetData
  
IF keyword_set(ALL) THEN BEGIN 
    PLOT_STYLE=1 & CHANNEL=1 & EPH_LIMITS = 1
ENDIF  

IF keyword_set(CHANNEL) THEN BEGIN 

    wid_ids = [widgetData.energy_base_1, widgetData.energy_base_2, $
               widgetData.energy_base_3, widgetData.energy_base_41, $
               widgetData.energy_base_54, widgetData.fit_base_1, $
               widgetData.fit_base_2, widgetData.fit_energy_base, $
               widgetData.psd_base, widgetData.coherent_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid)
    plot_style = widgetData.plot_style

    ;select by plotstyle
    CASE 1 OF                   
        ;ch./t; ch./L,t (stk); ch./L,t (3D); psd/t; psd/L,t;  Trap. Bound.
        ;  0         1              2          3        4         5
        (plot_style EQ 0) OR (plot_style EQ 1) OR  $
          (plot_style EQ 2) OR (plot_style EQ 5): BEGIN ;all not PSD
            id=widgetData.sc_id ; satellite id
            cal_type=widgetData.cal ; calibration type
            ;c/s (ch), int.flux (bt), diff.flux (bt), fit parm, diff.flux (fit)
            ;   0           1              2             3            4
            CASE 1 OF           ;select by spacecraft
                ;ns08 or ns10
                (id eq 0) or (id eq 1): BEGIN 
                    map = [1, 0, 0, 0, 0, 0, 0, 0, 0, 0]
                END             
                ;ns18, ns24, ns28, ns39
                (id eq 2) or (id eq 3) or (id eq 4) or (id eq 6): BEGIN
                    CASE cal_type OF ;select calibration type
                        3:    map = [0, 0, 0, 0, 0, 0, 1, 0, 0, 0] 
                        4:    map = [0, 0, 0, 0, 0, 0, 0, 1, 0, 0] 
                        6:    map = [0, 0, 0, 0, 0, 0, 0, 0, 0, 1]
                        7:    map = [0, 0, 0, 0, 0, 0, 0, 0, 0, 1]
                        ELSE: map = [0, 1, 0, 0, 0, 0, 0, 0, 0, 0]
                    ENDCASE                        
                END            
                ;ns33
                (id eq 5): BEGIN
                    CASE cal_type OF ;select calibration type
                        3:    map = [0, 0, 0, 0, 0, 0, 1, 0, 0, 0] 
                        4:    map = [0, 0, 0, 0, 0, 0, 0, 1, 0, 0] 
                        6:    map = [0, 0, 0, 0, 0, 0, 0, 0, 0, 1]
                        7:    map = [0, 0, 0, 0, 0, 0, 0, 0, 0, 1]
                        ELSE: map = [0, 1, 0, 0, 0, 0, 0, 0, 0, 0]
                    ENDCASE    
                END    
                ;ns41
                (id eq 7): BEGIN   
                    map = [0, 0, 0, 1, 0, 0, 0, 0, 0, 0]
                END 
                ;ns54
                (id eq 8): BEGIN   
                    CASE cal_type OF ;select calibration type
                        3:    map = [0, 0, 0, 0, 0, 0, 1, 0, 0, 0] 
                        4:    map = [0, 0, 0, 0, 0, 0, 0, 1, 0, 0] 
                        ELSE: map = [0, 0, 0, 0, 1, 0, 0, 0, 0, 0]
                    ENDCASE    
                END                           
                
            ENDCASE 
        END  

        (plot_style EQ 3) OR (plot_style EQ 4): BEGIN ;PSD
            map = [0, 0, 0, 0, 0, 0, 0, 0, 1, 0]
        END

    ENDCASE

    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)    
    papco_panel_Refresh, widgetData, /yscale

ENDIF

; --- Filter Variable multiple bases ------------------------------------------
IF keyword_set(EPH_LIMITS) THEN BEGIN
    wid_ids = [widgetData.l_select_base, widgetData.l_star_select_base, $
               widgetData.mlt_select_base, widgetData.mlat_select_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid) 
    map(widgetData.filter) = 1 
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
ENDIF 

END  

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_gps_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO lanl_gps_panel_event, event
  
COMMON lanl_gps_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id
  
;test uvalue for structure - select variable sliders return structure uvals.
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval
    uval = uval.uvalue
ENDIF

CASE uval OF

   'mpb_sats' : BEGIN 
       widgetData.sc_id=event.value
       lanl_gps_panel_event_refresh, /CHANNEL
   end
   'spb_sngl': BEGIN 
       widgetData.singles=event.value
       lanl_gps_panel_event_refresh, /CHANNEL     
   END 
   'spb_average'   : widgetData.seg_av=event.value
   'spb_reload'    : widgetData.reload=event.value
   'spb_spec'      : widgetData.spec=event.value
   'spb_chanlab'   : widgetData.chanlab=event.select
   'spb_eqcor'     : widgetData.eqcor=event.select
   'spb_dead_time' : widgetData.dead_time=event.select
   'spb_diag'      : widgetData.diag=event.select
   'spb_bckg'      : widgetData.bckg=event.select
   'spb_flux_cal' : begin
     widgetData.cal=event.value
     lanl_gps_panel_event_refresh, /CHANNEL
   end
   'spb_style' : begin
     widgetData.plot_style=event.value
     lanl_gps_panel_event_refresh, /CHANNEL
   end
   'spb_vars' : begin
     widgetData.filter=event.value
     lanl_gps_panel_event_refresh, /PLOT_STYLE
   EnD

    'spb_orig': widgetData.orig = event.value

    'spb_filter' : BEGIN 
        widgetData.filter = event.value
        lanl_gps_panel_event_refresh, /EPH_LIMITS
    END 

   'mpb_typ1' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ1,GET_VALUE=channel_event
     widgetData.product=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.product=widgetData.product+2^i  
   END
   'mpb_typ2' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ2,GET_VALUE=channel_event
     widgetData.product=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.product=widgetData.product+2^i  
   END
   'mpb_typ3' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ3,GET_VALUE=channel_event
     widgetData.product=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.product=widgetData.product+2^i  
   END
   'mpb_typ41' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ41,GET_VALUE=channel_event
     widgetData.product=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.product=widgetData.product+2^i   
    END
   'mpb_typ54' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ54,GET_VALUE=channel_event
     widgetData.product=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.product=widgetData.product+2^i   
    END
   'mpb_typ4' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ4,GET_VALUE=channel_event
     widgetData.sngl_prod=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.sngl_prod=widgetData.sngl_prod+2^i  
   END
   'mpb_typ5' : BEGIN
     WIDGET_CONTROL,widgetData.mpb_typ5,GET_VALUE=channel_event
     widgetData.product=0
     for i=0,n_elements(channel_event)-1 do if channel_event(i) then $
       widgetData.product=widgetData.product+2^i  
   END
   
   ELSE:  BEGIN
       WIDGET_CONTROL, widgetData.ef_pad_n, GET_VALUE=x
       widgetData.pad_n = x
       WIDGET_CONTROL, widgetData.fld_mu, GET_VALUE=x
       widgetData.psd_mu = x
       WIDGET_CONTROL, widgetData.energy_entry, GET_VALUE=x
       widgetData.fit_ch = x
       WIDGET_CONTROL, widgetData.fld_coherent_base, GET_VALUE=x
       widgetData.coh_ch = x
       widgetData=PAPCO_panel_Event(event, widgetData)
   END 
   
ENDCASE

END
