;******************************************************************************
;*  NAME:        eval_rel_maxw
;*
;*  DESCRIPTION: Routine evaluate f, j for a relativistic maxwellian
;*               distribution given that distribution's density n and 
;*               temperature T
;*
;*  INPUT:       n    density of distribution      cm-3
;*               T    temperature of distribution, MeV
;*
;*  OUTPUT:      returns a structure
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        October 1999
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
function mBessel2, x
  common rel_maxw, alpha
  f=exp(-1.0*alpha*x) * (1+x) * sqrt(x*(2+x))
  return, f 
end

;******************************************************************************
function eval_rel_maxw, n, T, ENERGIES=ENERGIES
  
  common rel_maxw, alpha
  
; define some constants  
  Mo   = double(9.11e-31)               ;electron rest mass, kg 
  c    = double(3e10)                   ;speed of light, cm/s
  eV   = double(1.6e-19)                ;one electron-volt, Joule
  k    = double(1.38e-23)               ;Boltz.an's constant, J/deg kelvin
  Moc2 = double(0.511)                  ;rest energy of electon in MeV
  
  
; define useful combinations of constants  
  alpha = Moc2 / T              ;dimensionless if T is in MeV
  
  if keyword_set(ENERGIES) then  E=ENERGIES else begin
    ; make array of energies in MEV from 0.1 to 10 MeV
    E=fltarr(41)
    for i=0,40 do E(i)=10^(-1+i/20.0)
  endelse
  
; check for "good" n, T
  if (n eq 0) OR (T eq 0) then begin
    J=E  &  J(*)=-99
    return,{E:e,J:j}
  endif
    
; perform equivalent integral to evaluate bessel function. Integral is
; from zero to infinity: uses idl function qromo
  
  ;keep iterating upper limit until changes are less than 0.5%
  oldval=0.0 &  upper=0.05  &  iterate=1
  while iterate do begin
    I=qromo('mBessel2',0,upper)
    if abs(oldval-I)/I lt 0.005 then iterate=0
    oldval=I  &  upper=upper*2
    if check_math() ne 0 then stop
  endwhile  
      
  A = n * c / (4* !pi * Moc2 * I)
    
  ;print,T,alpha,I,A,upper
    
  J = A * exp(-1*E/T) * (E / Moc2) * (2 + E / Moc2)
  
  index=where(j eq 0,c)
  if c ne 0 then j(index)=-1
    
  return,{E:e,J:j}
  
end
