;******************************************************************************
;* PROCEDURE:     
;*      p_lanl_gps_line, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* MODIFICATION HISTORY:       
;*     written December 1995, Reiner Friedel, at Goddard
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_lanl_gps_line, panel, plotinfo, OUTPUT=OUTPUT, $
                     OVERPLOT=OVERPLOT, $
                     PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                     SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON mjdt, mjdt_start, mjdt_end       ;common time limit in mjdt
COMMON yscale, yscl		        ;man/auto yscaling
COMMON zscale, zscl			;man/auto zscaling
COMMON shift_label, down                ;common for x-axis label shifting
COMMON coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

COMMON lanl_gps_data, input_header, input_data
COMMON lanl_gps_singles_data, singles_header, singles_input_data

COMMON lanl_gps
COMMON slice_lanl_gps, control, time, data, yray, yarr, extra_plotPar, $
                       utitle, uytitle, uztit, tai_1

COMMON papco_color_names

if keyword_set(OUTPUT) then output=OUTPUT else output=0
  
;get control variables for this module
control = *plotinfo.USR_PTR1
sat_name = sat_id(control.sc_id)   ;GPS satellite name, string
nodata = -99.0
  
; if singles are chosen, write the singles data into the input_data array  
if control.singles eq 1 then begin
    n=n_elements(singles_input_data)
    for j=1,11 do begin ;fold eph data
      temp=interpol(input_data.(j),input_data.time,singles_input_data.time)
      input_data(0:n-1).(j)=temp
    endfor  
    ;copy time, data
    input_data(0:n-1).TIME=  singles_input_data.time
    input_data(0:n-1).COUNTS(0)=singles_input_data.COUNTS(0)
    input_data(0:n-1).COUNTS(1)=singles_input_data.COUNTS(1)
    input_data=input_data(0:n-1)
endif  

utitle='LANL-GPS '+sat_name
uytitle='s!e-1'
  
; select correct channel names into energy_names
CASE 1 OF 
    ;earlier birds names handled by calibration routine. 
    (sat_name eq 'ns41'): energy_names=energy_names_41
    (sat_name eq 'ns54'): BEGIN
        CASE control.cal OF 
            3: BEGIN            ;fit channels
            END
            4: BEGIN            ;evaluate spectrum, M. Kippen's Fit 
                energy_names=strtrim(control.fit_ch,2)
                energy_names=(str_sep(strcompress(energy_names),' '))
            END 
            ELSE : energy_names=energy_names_54
        ENDCASE 
    END     
    ELSE: BEGIN &  END 
ENDCASE
   
if control.singles eq 1 then energy_names=singles_names_1
n_ch=n_elements(energy_names)

; restrict data to actual time range requested - this makes for faster zooming
; rememember input data is in TAI. For plotting, always start at zero
; to get maximum resolution (plot works in float only, large values
; get rounded! 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
index=where((input_data.time GE tai_1) AND (input_data.time LE tai_2),c)
if c ne 0 then data=input_data(index)

xut1=0  &  xut2=tai_2-tai_1  & time=data.time-tai_1
ndat = n_elements(data)  

;--- limit data by filter ranges if needed ------------------------------------
lanl_gps_select, control, data, nodata, utitle
      
;--- deadtime correction on counts --------------------------------------------
IF control.dead_time EQ 1 THEN BEGIN
    lanl_gps_deadtime_correction, data, sat_name & uytitle=uytitle+' (*)'
ENDIF

;--- do required calibrations -------------------------------------------------
lanl_gps_calib    ;seperate routine, also used by p_lanl_gps_lspec

;--- make index of channels to plot - from binary number in control.product ---
n_ch=n_elements(energy_names) & to_plot=bytarr(n_ch)
CASE 1 OF
    (control.cal EQ 4): BEGIN   ;calculate fit @ set MeV's required
        FOR i=0, n_ch-1 DO to_plot(i)=1
        pl_idx=where(to_plot EQ 1, nplot)
    END      
    (control.cal EQ 6) OR (control.cal EQ 7): BEGIN ;coherent channels
        FOR i=0, n_ch-1 DO to_plot(i)=1
        pl_idx=where(to_plot EQ 1, nplot)
    END 
    ELSE: BEGIN 
        FOR i=0,n_ch-1 DO IF (control.product and 2^i) EQ 2^i THEN to_plot(i)=1
        pl_idx=where(to_plot eq 1, nplot)
    END
ENDCASE

IF nplot eq 0 THEN GOTO, no_plot

;--- see if we need to correct to equator -------------------------------------
IF control.eqcor THEN BEGIN
    lanl_gps_equ_corr, data
    utitle=utitle+'!CCorr. Equ.'
    utitle=utitle+'!CPAD s = '+string(control.pad_n, format="(f3.1)")
ENDIF

;--- select / make the data required ------------------------------------------
case 1 of 

    (control.plot_style eq 0): begin         ;ch./t
        CASE control.cal OF
            4: BEGIN                    ;evaluate FIT
                yray=data.counts(0:n_ch-1)
            END
            ELSE: yray=data.counts
        ENDCASE       
    END

    ; psd/t  find psd at constant mu
    (control.plot_style eq 3) or (control.plot_style eq 4): BEGIN 
        papco_m_mu_arr, control.psd_mu, 1, 10000, 20, $
                                mus, mus_str, ylow, yhigh
        n_ch = n_elements(mus) & psd_mat = fltarr(ndat, n_ch)

        IF control.eqcor THEN BEGIN
            mag_bt = data.BEQU
        ENDIF ELSE mag_bt = data.BSAT
        ;B is in nT, from Tom its Gauss. Times 1e5.
        mag_bt = mag_bt*1e5

        yminarr = energy_ch(pl_idx, 0) * 1000.0 ;MeV-> keV
        ymaxarr = energy_ch(pl_idx, 1) * 1000.0 ;MeV-> keV
        ;make diff energy data array,  needs to be in diff flux/keV
        yray = transpose(data.counts(pl_idx)) / 1000.0
        ;now call the mu / psd calc routine...
        psd_mat = fltarr(ndat, n_ch)

        FOR i = 0, n_ch-1 DO BEGIN
            mu = mus(i)
            papco_mu_calculation, yminarr, ymaxarr, mu, yray, mag_bt, PSD,/GEM 
            psd_mat(*, i) = PSD
        ENDFOR 
        yray = psd_mat
        idx = where(finite(psd_mat) EQ 0, c)
        IF c NE 0 THEN yray(idx) = nodata
        
        yminarr = ylow & ymaxarr = yhigh
    END 
    (control.plot_style eq 5): begin         ;trapping boundary/t
        ;we have thresholds marked by -99: attempt to find trapping boundary
        concur = 2
        control.diag = 0
        
        lanl_gps_trapping_boundary, plotinfo, data, to_plot, $
          trap_b, thresholds, concur, DIAG=control.diag

        index=where(trap_b.count GE concur, c)
        yray=trap_b.L_STEP(0) & yray(*) = -99

        IF c NE 0 THEN BEGIN
            FOR i = 0, c-1 DO BEGIN
                idx = index(i)
                index2 = where(trap_b(idx).L_STEP NE 0, c2)
                yray(idx) = total(trap_b(idx).L_STEP(index2))/c2
            ENDFOR
        ENDIF
        time=trap_b.time - tai_1
        channels = ['[1 -2]', '[1 -2]']

    END   
ENDCASE   

yarr=energy_ch(pl_idx, *)
IF output EQ 2 THEN return

panelset, panel

;--- set the extra_plotPar keywords per type of plot, do y/z scaling ----------
CASE  control.plot_style OF

    0: BEGIN                    ;ch./t
        lanl_gps_scale, pl_idx, yray, st, en
        IF control.spec EQ 0 THEN BEGIN ;line plot
            yst = st & yen = en
            zst=0  &  zen=0
        ENDIF ELSE BEGIN ;spec plot
            yarr=energy_ch(pl_idx, *)
            yst = yarr(0,0) & yen = yarr(nplot-1,1)
            zst=st & zen = en
        ENDELSE     
        extra_plotPar = {xrange:[xut1,xut2], yrange:[yst,yen], $
                         ylog:yscl(panel(0),3), xtickformat:'noticks'}
    END 

    3: BEGIN                    ;psd/t psd at constant mu
        papco_autorange, yray, st, en, $
              log=zscl(panel(0),3), exclude=0, nodata=-99 
        IF control.spec EQ 0 THEN BEGIN ;line plot
            yst = st & yen = en
            zst=0  &  zen=0        
        ENDIF ELSE BEGIN
            yarr=fltarr(n_ch,2)
            yarr(*,0)=yminarr & yarr(*,1)=ymaxarr
            yst = yarr(0,0) & yen = yarr(n_ch-1,1)
            zst=st & zen = en
        ENDELSE     
        extra_plotPar = {xrange:[xut1,xut2], yrange:[yst,yen], $
             ylog:yscl(panel(0),3), xtickformat:'noticks'}
        uytitle='Hilmer PSD s!U3!N km!U-6!N'
        uytitle = 'Hilmer PSD c!u3!N / MeV!u3!N cm!u3!N'
        utitle=utitle+'!C!4l!3 (MeV/G), G:'+varprt(gain)
    END 
 
    5: BEGIN                    ;trapping boundary
        yst = 4.0 & yen = 12.0  
        zst=0  &  zen=0        
        extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], $
                       ylog:0, xtickformat:'noticks'}
    END 

ENDCASE  

;set y, z scaling
IF (zscl(panel(0),0) eq 1) THEN BEGIN 
    zst=zscl(panel(0),1)  &  zen=zscl(panel(0),2)
ENDIF ELSE BEGIN 
    zscl(panel(0),1)=zst  &  zscl(panel(0),2)=zen
ENDELSE    

IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
    yst=yscl(panel(0),1)  &   yen=yscl(panel(0),2)
ENDIF ELSE BEGIN
    yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
ENDELSE       

extra_plotPar.yrange = [yst, yen]

; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
     
color_indx=[black, red, green, blue, magenta, cyan, $
            burgundy, olive, dark_green, teal, royal_blue, violet]

plotted_x = !x  &  plotted_y = !y

; check if we need to do plot or just return arrays to slice.
IF output eq 2 then BEGIN
    message, 'plot data constructed', /cont & return
ENDIF

;IF output EQ 3 THEN GOTO, do_output
down=0
   
uztit=''
; now do the actual data plot, data only, no axis
CASE control.plot_style OF
    0: BEGIN                   ;plot energy channel, multiples
        papco_y_label, yst, yen, log=yscl(panel(0),3)  
        ; check how many chanels to plot first
        n=0 & count=0
    
        IF control.spec EQ 0 THEN BEGIN ;line plot
            papco_draw_time_axis, panel, OVERPLOT=OVERPLOT,_extra=extra_plotPar
            
            IF nplot EQ 1 THEN $
              utitle=utitle+'!C'+energy_names(pl_idx(0))
            nm = ''
            IF keyword_set(overplot) THEN $
               FOR i=0, PLOTS_ATTHISPOSITION-1 DO nm = nm+'!C'
            FOR i = 0, nplot-1 DO BEGIN
                idx = pl_idx(i)
                extra_plotPar.color=color_indx(i mod n_elements(color_indx))
                c=count/5 & mult=count mod 5 
                name = energy_names(idx)
                IF count eq 0 then channels=[energy_names(i)] ELSE $
                  channels=[channels,energy_names(i)]
                IF control.chanlab AND (nplot NE 1) THEN  BEGIN 
                    FOR j=1,c DO  name='!C'+name
                    xyouts,!P.position(0)+0.01+0.135*mult, $
                           !P.position(3)-0.015, nm+name, /normal, $
                           charsize=!p.charsize, color=extra_plotPar.color
                ENDIF   
                count=count+1
             
                index=where(yray(idx,*) lt 0,c)
                if c ne 0 then yray(idx,index)=-99
 
                lanl_gps_gap_plot, time, yray(i,*), AVERAGE=control.seg_av, $
                                   _extra=extra_plotPar
                
            ENDFOR   

        ENDIF ELSE BEGIN       ;spec plot

            zmat=fltarr(ndat, nplot)
            FOR i = 0, nplot-1 do  BEGIN
                idx = pl_idx(i) & zmat(*,i) = yray(idx,*) 
            ENDFOR 
            
            extra_plotPar=create_struct(extra_plotPar,'zrange',[zst,zen], $
                          'zlog',zscl(panel(0),3),'ztitle',uytitle, $
                          'ztickformat','papco_color_bar_log_ticks')
            uytitle='Energy (MeV)'

            papco_y_label, yst, yen, log=yscl(panel(0),3)  
            papco_draw_time_axis, panel, OVERPLOT=OVERPLOT,_extra=extra_plotPar

            ; do spec plot
            resx=20  & resy=0.05    
            papco_plot_colorspec, zmat, time, yarr, nodata=-99, $
              resx=resx, resy=resy, _extra=extra_plotPar
            papco_color_bar, _extra=extra_plotPar    
        ENDELSE    
        if nplot ne 1 then extra_Par.color=1
    END   

    3: BEGIN  ; psd/t psd at constant mu, multiples
        IF control.spec EQ 0 THEN BEGIN ;line plot
            papco_draw_time_axis, panel, OVERPLOT=OVERPLOT,_extra=extra_plotPar
            papco_y_label, yst, yen, log=yscl(panel(0),3)  
            count=0
            if n_ch eq 1 then utitle = utitle+'!C'+mus_str(0)
            FOR i=0, n_ch-1 do BEGIN
                y = yray(*, i) & idx = where(y NE nodata, c)
                IF c EQ 0 THEN continue
                IF n_ch NE 1 THEN $
                  extra_plotPar.color=color_indx(i mod n_elements(color_indx))
                c=count/5 & mult=count mod 5 & name = mus_str(i)
                if count eq 0 then channels=[name] else $
                    channels=[channels, name]
                IF control.chanlab AND (n_ch NE 1) then begin ; label channels
                    for j=1,c do name='!C'+name
                    xyouts,!P.position(0)+0.01+0.135*mult,$
                           !P.position(3)-0.015,  name, /normal, $
                           charsize=!p.charsize,color=extra_plotPar.color
                ENDIF  
                count=count+1

                lanl_gps_gap_plot, time, yray(*, i), AVERAGE=control.seg_av, $
                                   _extra=extra_plotPar
            ENDFOR             
            if n_ch ne 1 then extra_plotPar.color=1
        ENDIF ELSE BEGIN        ;spec plot
            
            extra_plotPar=create_struct(extra_plotPar,'zrange',[zst,zen], $
                          'zlog',zscl(panel(0),3),'ztitle',uytitle, $
                          'ztickformat','papco_color_bar_log_ticks')
            uytitle='!C!4l!3 (MeV/G)'

            papco_y_label, yst, yen, log=yscl(panel(0),3)  
            papco_draw_time_axis, panel, OVERPLOT=OVERPLOT,_extra=extra_plotPar
            papco_color_bar, _extra=extra_plotPar                

            ; do spec plot
            resx=20  & resy=0.05    
            papco_plot_colorspec, yray, time, yarr, nodata=nodata, $
              resx=resx, resy=resy, _extra=extra_plotPar
            papco_color_bar, _extra=extra_plotPar                

        ENDELSE 
    END 

    5: BEGIN                     ;trapping boundary
        papco_draw_time_axis, panel, OVERPLOT=OVERPLOT,_extra=extra_plotPar
        uytitle='L-value'
        utitle=utitle+'!CTrapping Boundary'
        lanl_gps_gap_plot, time, yray, _extra=extra_plotPar
    END  
ENDCASE  
   
; store the coordinate information into common block for slices
plotted_x = !x  &  plotted_y = !y   

; plot y-axis label at left of plot. Use scalable routine!  
IF NOT keyword_set(OVERPLOT) THEN left_side_label,panel,uytitle,/rot90       
  
; plot info right of the panel. Multiple panle lebals are handled
IF control.spec THEN papco_rs_lbl, panel, utitle, /ROT90, _extra=extra_Par $
                ELSE papco_rs_lbl, panel, utitle, /LINE, _extra=extra_Par

do_output:
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
IF KEYWORD_SET(OUTPUT) THEN BEGIN
    time = time+tai_1
    message,'Writing plot data out to file',/cont
    description='lanl_gps data'
    time_label='TAI'
    n_ch=n_elements(channels)
    ch_arr = fltarr(n_ch, 2)
    FOR i = 0, n_ch-1 DO BEGIN
        pos1 = strpos(channels(i),'[')
        pos2 = strpos(channels(i),']')
        a = strmid(channels(i), pos1+1, pos2-pos1-1)
        res = strsplit(a, '-', /extract)
        IF n_elements(res) NE 2 THEN BEGIN
            res = strsplit(a, '>', /extract)
            IF n_elements(res) NE 1 THEN break 
            ch_arr(i, 0) = res(0) & ch_arr(i, 1) = 99999
        ENDIF ELSE BEGIN
            ch_arr(i, 0) = res(0) & ch_arr(i, 1) = res(1)
        ENDELSE 
        ;print, a, '  ', ch_arr(i, 0), ch_arr(i, 1),  pos1,  pos2
    ENDFOR
    channels=ch_arr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=-99
    CASE control.plot_style OF
        3: yray = yray 
        5: yray = yray 
        ELSE:   yray = transpose(yray(0:n_ch-1,*))
    ENDCASE

    papco_write_data, time, yray, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      Y_LABEL=y_label, z_label=z_label, nodata=nodata, $
                      RS_LABEL=RS_LABEL, channels= channels
ENDIF   

return

no_plot:
message, 'nothing to plot!', /cont
plotted_x = !x  &  plotted_y = !y

END  

;------------------------------------------------------------------------------
; routine to automatically scale gps data. Input is the channels chosen,
; yray; outputs are the y-axis start and end value. Bad data value expected
; is -99, extra room at top of range for labels.
PRO lanl_gps_scale, pl_idx, yray, yst, yen
  
nplot=n_elements(pl_idx)  &  channelscale=fltarr(nplot,2)   
  
for i=0,nplot-1 do begin
    idx=pl_idx(i)
    nozeroarr=yray(idx,*)
    ;find min by taking out < zeros first - remove bad data points!
    index=where(nozeroarr gt 0,c)
    if c ne 0 then begin
      nozeroarr=nozeroarr(index)
      channelscale(i,0)= min(nozeroarr)
      channelscale(i,1)= max(yray(i,*))
    endif else begin
      message,'no points to scale', /cont
    endelse  
endfor

index=where(channelscale ne 0,count)
if count ne 0 then begin
    channelscale=channelscale(index) 
    yst=min(channelscale)  &  yen=max(channelscale)
    logdiff=abs(alog10(yen)-alog10(yst))
    newlogyen=alog10(yen)+logdiff*0.15
    newlogyst=alog10(yst)-logdiff*0.05
    yen=10^newlogyen  &  yst=10^newlogyst
endif else begin
    yst=1  &  yen=1e4
endelse  

if yst eq 0 then yst = yen /1e3
message,'st: '+varprt(yst)+' en: '+varprt(yen), /cont
  
END
 
;------------------------------------------------------------------------------
; routine to find channel threshold, subtract out and mark with -99
pro lanl_gps_bad, to_plot, input, thresholds
  
message,'taking out threshold', /cont
idx = where(to_plot EQ 1, n_ch)
ndat = n_elements(input.counts(0))
yray = fltarr(n_ch, ndat)
yray(0:n_ch-1, 0:ndat-1)=input.counts(idx)
thresholds = fltarr(n_ch)
factor=2.0 

for i=0,n_ch-1 do begin
    data=yray(i,*) 
    data=data(where(data gt 0))

    ;loop up from 0.01 count/sec level and find the level which gives the
    ;lowest standard deviation below it!
    level = 0.01
    WHILE 1 DO begin
        index=where(data le level,c)
        if c lt 10 then goto, next_level else begin
          res=moment(data(index))
          ;print,level,c,res(0),sqrt(res(1))
          if res(1) gt res(0)*factor then break
        endelse
        next_level:
        level = level*2
    ENDWHILE 

    data=yray(i,*) 
    index=where(data lt res(0),c)
    if c ne 0 then begin
        yray(i,*)=yray(i,*)-res(0)  ;subtract out background level
        yray(i,index) = -99   
    endif  
    ;store the level subtracted out
    print, res(0)    
    thresholds(i) = res(0)
ENDFOR 
FOR i = 0, n_ch-1 DO input.counts(idx(i)) = transpose(yray(i, *))

END   
 
;------------------------------------------------------------------------------
; routine to limit data by ephemeris range. Here eph and data have SAME
; time stamps...  mark everything outside as "bad" -99
PRO lanl_gps_select, control, data, nodata, utitle
  
COMMON mjdt, mjdt_start, mjdt_end       ;common time limit in mjdt
COMMON gps_eph, gps_eph_sat_id, gps_eph_names, gps_eph_select_names, $
                      gps_eph_control
COMMON gps_eph_data, gps_eph_header, gps_eph_data
COMMON get_error, get_err_no, get_err_msg

;check if any filtering or PSD is needed - get ephemeris data first -----------
IF control.L_SELECT_RANGE OR control.L_STAR_SELECT_RANGE OR $
   control.MLT_SELECT_RANGE OR control.MLAT_SELECT_RANGE OR $
   (control.plot_style eq 3) OR (control.plot_style eq 4) THEN BEGIN

    new_plotinfo = papco_getplotinfostruct()
    new_control = gps_eph_control
    new_control.SC_ID = control.SC_ID
    new_control.ORIG = 1 ;mag model coords
    new_control.L_CALC = control.L_CALC
    new_control.EXTERNAL_MODEL = control.EXTERNAL_MODEL
    new_control.INTERNAL_MODEL = control.INTERNAL_MODEL
    new_control.MAG_DYN = control.MAG_DYN
    new_plotinfo.USR_PTR1 = ptr_new(new_control)

    r_gps_eph, new_plotinfo, VERBOSE=VERBOSE
    IF get_err_no NE 0 THEN BEGIN 
        message,'problem getting ephemeris data, cannot filter', /cont
        goto, no_restrict
    ENDIF 
    
    IF new_control.mag_dyn THEN dyn_str = 'DYNA' ELSE dyn_str = 'STAT'
    model_name = 'MODEL_'+varprt(control.INTERNAL_MODEL)+'_'+ $
                          varprt(control.EXTERNAL_MODEL)+'_'+dyn_str
    tag_idx = where(TAG_NAMES(gps_eph_data) EQ model_name)
    tag_idx = tag_idx(0)

    tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
    tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

    idx=where((gps_eph_data.time ge tai_1) AND (gps_eph_data.time le tai_2),c)
    if c ne 0 then eph_data=gps_eph_data(idx)

ENDIF

;filter the data using the ephemeris info -------------------------------------
n_filt = 0
;filter for L
IF control.L_SELECT_RANGE THEN BEGIN
    message, 'Filtering L: '+model_name, /info
    filt_var = eph_data.(tag_idx).L
    v1 = control.L_SELECT_FROM_VAL
    v2 = control.L_SELECT_TO_VAL
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN data(idx).COUNTS(*) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.L_SELECT_RANGE_FRMT)
    n_filt = n_filt + 1
ENDIF

;filter for L_STAR
IF control.L_STAR_SELECT_RANGE THEN BEGIN
    message, 'Filtering L*: '+model_name, /info
    filt_var = eph_data.(tag_idx).LSTAR(0)
    v1 = control.L_STAR_SELECT_FROM_VAL
    v2 = control.L_STAR_SELECT_TO_VAL
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN data(idx).COUNTS(*) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.L_STAR_SELECT_RANGE_FRMT)
    n_filt = n_filt + 1
ENDIF

;filter for MLT
IF control.MLT_SELECT_RANGE THEN BEGIN
    message, 'Filtering MLT: '+model_name, /info
    filt_var = eph_data.(tag_idx).MLT
    v1 = control.MLT_SELECT_FROM_VAL
    v2 = control.MLT_SELECT_TO_VAL
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN data(idx).COUNTS(*) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.MLT_SELECT_RANGE_FRMT)
    n_filt = n_filt + 1
ENDIF

;filter for MLAT
IF control.MLAT_SELECT_RANGE THEN BEGIN
    message, 'Filtering MLAT: '+model_name, /info
    filt_var = eph_data.(tag_idx).MLAT
    v1 = control.MLAT_SELECT_FROM_VAL
    v2 = control.MLAT_SELECT_TO_VAL
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN data(idx).COUNTS(*) = nodata
    utitle = utitle+'!C'+string(v1, v2, $
                               format=control.MLAT_SELECT_RANGE_FRMT)
    n_filt = n_filt + 1
ENDIF

IF n_filt NE 0 THEN utitle = utitle+'!C'+model_name
no_restrict:

END 

;------------------------------------------------------------------------------
; routine to plot lines, testing for -99 and leaving gaps there!
; if keyword average is set then average all point in a given valid
; range together, then plot.
pro lanl_gps_gap_plot, time, yray,  AVERAGE=AVERAGE, NODATA = NODATA, _EXTRA=E
  
IF keyword_set(NODATA) THEN nodata =  NODATA ELSE nodata = -99

;re-establish plot axis coords
plot, time, time, _EXTRA=E, xstyle=5, ystyle=5, /NODATA
;make keywords for plots routine
gap_extra_Par={PSYM:E.PSYM, $
               COLOR:E.COLOR, $
               THICK:E.THICK, $
               LINESTYLE:E.LINESTYLE}
  
  n=0 & sn=0 & sd=0 & st=0 & c=0
  for i=0l,n_elements(time)-1 do begin
    if yray(i) eq nodata then begin
      if keyword_set(AVERAGE) then begin
        if c ne 0 then begin
          if sn eq 0 then begin
            plots, st/c, sd/c, _EXTRA=E
            sn=sn+1
          endif else begin
            plots, st/c, sd/c, _EXTRA=E,  /CONTINUE, NOCLIP=0
          endelse  
        endif  
      endif  
      n=0 & sd=0 & st=0 & c=0
      goto,next
    endif  
    ; got here if first data point found
    if n eq 0 then begin
      if keyword_set(AVERAGE) then begin
        st=st+time(i)
        sd=sd+yray(i)
        c=c+1
      endif else begin 
        plots, time(i), yray(i), _EXTRA=E
      endelse  
      n=n+1
    endif else begin 
      if keyword_set(AVERAGE) then begin
        st=st+time(i)
        sd=sd+yray(i)
        c=c+1        
      endif else begin 
        plots, time(i), yray(i), _EXTRA=E,  /CONTINUE, NOCLIP=0
      endelse  
    endelse  
    next:                        
  endfor   

END   

;------------------------------------------------------------------------------
; routine to perform dead-time correction on counts in LE1-LE5.
PRO lanl_gps_deadtime_correction, data, sat
  
CASE sat OF 
    'ns08': begin
      do_correct=0
    end   
    'ns10': begin
      do_correct=0
    end    
    'ns18': begin
      do_correct=1
    end
    'ns24': begin
      do_correct=1
    end
    'ns28': begin
      do_correct=1
    end
    'ns33': begin
      do_correct=1
    end
    'ns39': begin
      do_correct=0
    end
    else : begin
      do_correct=0
    end  
ENDCASE    
  
if do_correct then begin
    dt=1e-5                     ;deadtime is given: 1e-5 seconds
    n=n_elements(data)
    sum=fltarr(n,6)
    for i=0,4 do for j=i,4 do sum(*,i)=sum(*,i)+data.counts(j)
    for i=0,4 do begin $
        index_bad_data=where(data.counts(i) eq -99.0,countz) 
        data.counts(i)= data.counts(i) / ( (1-dt*sum(*,i)) *(1-dt*sum(*,i+1)) )
        if countz ne 0 then data(index_bad_data).counts(i) =-99.0
    ENDFOR 
    message,'applied',/cont
ENDIF   

END 

;------------------------------------------------------------------------------
; routine to do the DIPOLE transformation of a magnetic field strength
; along a field line to the equator: given B at a known latitutde
; along L, this computes B at the equator for the same L.
; Magnetic latitude mlat must be in radians
; B_mlat is the field strength at latitde mlat on the field line
function find_bequ, B_mlat, mlat
  
  bequ =  B_mlat * cos(mlat)^6 / sqrt(4-3*cos(mlat)^2)
  return, bequ
  
end

;------------------------------------------------------------------------------
; routine to calculate relativistic PSD based on mcadams paper
function find_psd, flux, E
  
  Er=0.511 ;electron rest mass, in MeV
  psd = flux / (E * ( 1+ E/(2*Er) ))
  return,psd
  
end

;------------------------------------------------------------------------------
; routine to calculate mu based on the mcadams paper approximation. 
; E is the energy in MeV, B the field strength in nT
function find_mu, E, B
  
  Er=0.511 ;electron rest mass, in MeV
  mu = E * ( 1+ E/(2*Er) ) / B
  return, mu
  
end

;------------------------------------------------------------------------------
;do the converse function as well - given mu, find E. Solve the quadratic...
function find_MeV, mu, BnT
  
  a=1/(2*0.511)  &  b=1  &  c=-mu*BnT
  d=sqrt(b*b -4*a*c)
  r1=(-1*b + d) / (2*a)
  return, r1

end

;------------------------------------------------------------------------------
; routine to interpolate fluxes to a given mu given an electron
; spectrum from GPS. Input is the differential spectrum measured at
; GPS (transformed to the equator), the magnetic field value at GPS
; and the magnetic latitude at GPS,  and the mu value required.
; We find the corresponding equatorial B first, fit a spectrum to the
; GPS data, convert that spectrum to mu and the find the nearest mu to
; the requested mu. (Will use Tom's fit here eventually!!)
function get_mu_flux, spec, B_mlat, mlat, mu, DIAG=DIAG
  
  common lanl_gps

  ;define energies for channels - energy_ch is in common lanl_gps
  x=energy_ch(0:4,0)
  
  ;restrict finding spectra to those with AT LEAST 4 data points
  index=where(spec gt 0, c)
  if c lt n_elements(spec)-1 then return, [-99]
  x2=x(index) & spec2=spec(index)
  
  ;get mag value at equator - dipole formulae
  bequ=find_bequ(B_mlat, mlat/!RADEG)
  
  ;initial guess for fit, weights
  A=[max(spec2),-0.5, 0]  &  weights=x2/x2
  
  ;compute curve fit paramters
  message, /reset_error
  !quiet=1
  yfit=curvefit(x2,spec2, weights, A, SIGMA, FUNCTION_NAME='exp_funct', ITMAX=40)
  !quiet=0
  
  ;now calculate energy corresponding to requested mu's
  n_mu=n_elements(mu)  &  psd=fltarr(n_mu)
  for i=0, n_mu-1 do begin
  
    MeV=find_MeV(mu(i),bequ*1e5)  ;B is in nT, from Tom itsGauss. Times 1e5.
    exp_funct, MeV, A, Flux       ;find the flux for that energy from fit!
    psd(i)=find_psd(flux, MeV)    ;find the Phase space density for that energy
  
    if keyword_set(DIAG) then begin
      x3=[0.1, 0.5, 0.75, 1.0, 1.5, 3, 5]  ;define energies for fit
      erase  &  x3=rebin(x3,10*n_elements(x3))  &  x3=x3(0:n_elements(x3)-10)
      exp_funct, x3, A, F
      plot,x3,F,/xlog,/ylog,xr=[0.1,5], yr=[1e2,5e7]  
      oplot,x2, spec, color=1
      xyouts,MeV, Flux, 'FIT', /data, color=1
    endif
  
  endfor
  
  return,psd
  
end

;------------------------------------------------------------------------------
; routine defining the function to be used in fitting the spectrum. We
; use an exponential here. - F(x) = a * exp(b*x) + c
pro exp_funct, X, A, F, pder
  
  bx = exp(a(1) * X)  
  F = a(0) * bx ;+A(2)
  if n_params() ge 4 then $
    pder=[[bx],[a(0)*x* bx],[replicate(1.0, n_elements(x))]]
  
END 

;------------------------------------------------------------------------------
; routine to correct GPS fluxes to equator.
; Uses model MLAT info and Michel Tuszewski's formula
PRO lanl_gps_equ_corr, data

COMMON  lanl_gps_control, control

message,'Calculating correction factors to equator',/cont
L_arr=data.LT89
s=control.pad_s
cor_fac= (L_arr/4.2)^(3.7*s)

FOR i=0, n_elements(data)-1 DO BEGIN 
    data(i).counts = data(i).counts * cor_fac(i)
    idx = where(data(i).counts LT 0, c)
    IF c NE 0 THEN data(i).counts(idx) = -1
ENDFOR  

END 
