;******************************************************************************
;*  NAME:        r_lanl_gps
;*
;*  DESCRIPTION: Routine to read lanl_gps files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       satid   the gps satellite id
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               FIT      -  read the fitted data files    
;*
;*  DATE:        December 1998
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_gps, plotinfo, satid, VERBOSE=VERBOSE

COMMON lanl_gps_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
COMMON lanl_gps
COMMON r_lanl_gps, bdd_dir, cxd_dir, sve_dir

get_err_no=0  &  get_err_msg='lanl_gps read succesfully'

; check for env. variable pointing to data archive directory
if not (papco_check_data_env('LANL_GPS_BDD_ARCH', PATH=bdd_dir)) then return
if not (papco_check_data_env('LANL_GPS_CXD_ARCH', PATH=cxd_dir)) then return
if not (papco_check_data_env('LANL_GPS_SAVESETS', PATH=sve_dir)) then return

; check VERBOSE keyword 
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0
    
; clear data arrays for reuse
input_header=0  &  input_data=0   
     
;get control variables for this module
control = *plotinfo.USR_PTR1
IF (control.cal GE 3) AND (control.cal LE 5) THEN FIT=1 else FIT=0

; make list of GPS files to read. GPS files are per week...
if VERBOSE then message,'searching for files...',/cont
res = size(satid, /stru)
sat_name = sat_id(control.sc_id)
satid = control.sc_id

fln_2=strarr((mjdt_end.mjd+7 - mjdt_start.mjd-7)/7 +2)
fln_date=intarr(n_elements(fln_2))
fln_got=intarr(n_elements(fln_2))
fln_1=fln_2
ss=mjdt_start  &  ss.mjd=ss.mjd-7
count=0
while ss.mjd le mjdt_end.mjd DO BEGIN 
    convert_t90_to_Date, ss, year,doy,hr,min,sec,mon,cmon,doym, /mjdt
    
    IF keyword_set(FIT) THEN BEGIN 
        CASE sat_name OF 
            'ns54': BEGIN
                gps_path = cxd_dir
                ext = '.kp4'
                ver = 'h'
            END 
            ELSE: BEGIN 
                ext='.fit' &  subdir=sat_name+'_smooth/FIT/'
                gps_path=bdd_dir+subdir  & ver='g'
                IF sat_name eq 'ns33' then ver='h' ;for ns33     
            END 
        ENDCASE 
    ENDIF ELSE BEGIN 
        CASE sat_name OF 
            'ns08': BEGIN 
                ext='.kp0' &  subdir=sat_name+'/'
                gps_path=bdd_dir+subdir+string(year,format="(i4.4,'/')")  
                ver='g'
            END 
            'ns10': BEGIN 
                ext='.kp0' &  subdir=sat_name+'/'
                gps_path=bdd_dir+subdir+string(year,format="(i4.4,'/')")  
                ver='g'
            END 
            ELSE: BEGIN 
                ext='.kp2' &  subdir=sat_name+'/'
                gps_path=bdd_dir+subdir+string(year,format="(i4.4,'/')")  
                ver='h'
            END 
        ENDCASE 
    ENDELSE 
    
    if year ge 2000 then yr=year-2000 else yr=year-1900
    fln=sat_name+ver+ string(yr,mon,doym, format="(i2.2,i2.2,i2.2)")+ext
    fln_got=intarr(n_elements(fln_2))
    
    found=findfile(gps_path+fln,count=nfs)
    if nfs eq 0 then $ ;check if savesets are there
      found=findfile(sve_dir+fln+'.idl', count=nfs) 
    if nfs eq 1 then begin
      fln_2(count)=found(0)
      papco_sepdir,found(0),pp,ff
      fln_1(count)=ff
      fln_got(count)=1
      fln_date(count)=year
      count=count+1
      if VERBOSE then message,'found '+fln,/cont
    endif
    if count ge 1 then ss.mjd=ss.mjd+7 else ss.mjd=ss.mjd+1
endwhile

if count eq 0 then begin
    get_err_no=1 & get_err_msg='No files found for timerange requested'
    message,get_err_msg,/cont & return
ENDIF

; loop from startfile to endfile to read all data, concatonate if needed
no_files_read=0
for i=0,count-1 do begin
    IF VERBOSE then message,'Reading file: '+fln_2(i),/cont
    ; read one file of data
    r_lanl_gps_onefile, fln_1(i), fln_2(i), satid, FIT=FIT, VERBOSE=VERBOSE
    ; concatonate data arrays
    if get_err_no eq 0 then begin          ;a file was read succesfully
        if no_files_read eq 0 then begin     ;start the array buffer
            input_data_buf=input_data
        endif else begin		     ;otherwise concatonate arrays
            input_data_buf=[input_data_buf,input_data]
        endelse 
        no_files_read=no_files_read+1
    ENDIF 
    next_file:
ENDFOR 
   
; check if any files were read   
if (no_files_read ne 0) and (get_err_no eq 0) then begin
    get_err_msg=string(no_files_read,format="(i2)")+ $
      ' file(s) read succesfully'
    ;clear arrays and load buffer into output array
    input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
endif else begin
    get_err_no=1 & get_err_msg='no files found for timerange' & return
endelse         
  
; check for time interval required. Convert PAPCO time to GPS time and
; select. Beware end of year days in gps data files!

;tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
;tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
;index=where((input_data.time ge tai_1) AND (input_data.time le tai_2), c)

;if c eq 0 then begin
;    input_data=0 & get_err_no=2  &  get_err_msg='no data found for timerange'
;    message,get_err_msg, /cont & return
;ENDIF

message, get_err_msg, /cont
  
END 

;******************************************************************************
;*  NAME:        r_lanl_gps_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
PRO r_lanl_gps_onefile, fln_1, fln_2, satid, FIT=FIT, VERBOSE=VERBOSE
  
COMMON lanl_gps_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON r_lanl_gps, bdd_dir, cxd_dir, sve_dir
   
get_err_no=0 & get_err_msg='lanl_gps_onefile read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
ext='.idl'
on_ioerror, read_original
pos=strpos(fln_1,'.idl')
if pos ne -1 then fln=strmid(fln,0,pos)
if VERBOSE then message,'Reading '+fln_1, /cont
restore, sve_dir+fln_1+ext, VERBOSE=VERBOSE
dealloc_lun,/quiet    ;frees logical units used by save/restore
message,'idl saveset read succesfully', /cont
print,'  ',sve_dir+fln_1+ext
return

read_original:
; if the .idl save file could not be found, then try to read the original
r_lanl_gps_onefile_original, fln_1, fln_2, satid, FIT=FIT, VERBOSE=VERBOSE

END  
  
;******************************************************************************
;*  NAME:        r_lanl_gps_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_gps_onefile_original, fln_1, fln_2, satid, FIT=FIT, VERBOSE=VERBOSE
  
common lanl_gps_data, input_header, input_data
common get_error, get_err_no, get_err_msg
common lanl_gps
COMMON r_lanl_gps, bdd_dir, cxd_dir, sve_dir
   
get_err_no=0 & et_err_msg='lanl_gps_onefile_orignal read succesfully'

; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BY USER!!!!
sat_name=sat_id(satid)
   
; see if file is actually there
filethere=findfile(fln_2)
if filethere(0) eq '' then begin
    get_err_no=2 & get_err_msg='File '+fln_2+' not found'
    print,get_err_msg & return
ENDIF
   
message,'reading ascii format (slow): ',/cont & print,'  ',fln_2

; open file and report problems
openr, unit, fln_2, /get_lun,ERROR=err
if err ne 0 then begin
    get_err_no=3 & get_err_msg='Problem opening '+fln & return
ENDIF
   
start_year=fix(strmid(fln_1,5,2))
if start_year gt 50 then start_year=start_year+1900 else $
      start_year=start_year+2000

; read ascii data
    
junk='' & ct=0
eph_in=fltarr(12)
input_header={lanl_gps_header, sat_id:sat_name, start_year:start_year}

dmy1={time:0.0d, lat:0.0, long:0.0, rad:0.0, rho:0.0, z:0.0, mlong:0.0, $
      Lt89:0.0, BBo:0.0, loct:0.0, Bsat:0.0, Bequ:0.0}

;make data structures for common block
IF keyword_set(FIT) THEN BEGIN 
    CASE sat_name OF 
        'ns33' : BEGIN 
            dmy2=create_struct(name='lanl_gps_fitted_a', dmy1, $
                               'counts', fltarr(13))    
        END 
        'ns54' : BEGIN ;cxd, fit params from M. Kippen
            dmy2=create_struct(name='lanl_gps_fitted_cxd', dmy1, $
                               'counts', fltarr(65)) 
        END 
        ELSE : BEGIN 
            dmy2=create_struct(name='lanl_gps_fitted_b', dmy1, $
                               'counts', fltarr(15))
        END 
    ENDCASE 
ENDIF ELSE BEGIN  
    CASE sat_name OF  
        'ns41' : n_ch = 16
        'ns54' : n_ch = 16
        ELSE : n_ch = 11
    ENDCASE 
    dmy2=create_struct(name='lanl_gps_'+sat_name, dmy1,'counts',fltarr(n_ch))
ENDELSE 

;now read data into structures  
input_data=replicate(dmy2,6300)
while not eof(unit) do begin
    readf, unit, dmy2
    if ct gt 6299 then input_data=[input_data,replicate(dmy2,6300)]
    input_data(ct) = dmy2
    ct = ct+1 
endwhile
    
close,unit & free_lun,unit
   
input_data=input_data(0:ct-1)

; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
on_ioerror, out
  
;reassign time tag in array. Use TAI - best for continuity!
;files at the end of the year have day numbers that go on beyond 365/366.
;need to check anf correct for that...
year=start_year
year_days=utc2doy({YEAR:year, MONTH:12, DAY:31, HOUR:0, MINUTE:0, $
                   SECOND:0, MILLISECOND:0})
  
message,'Converting time to TAI (slow)',/cont
for i=0, ct-1 do begin
    doy=fix(input_data(i).time)
    sec=(input_data(i).time-doy)*86400000.0
    if doy gt year_days then begin
      doy=doy-year_days
      year=start_year+1
    endif  
    int=doy2utc(doy,year)
    int.time=long(sec)
    utc=int2utc(int)
    IF VERBOSE EQ 2 then print,utc
    input_data(i).time=utc2tai(utc)
ENDFOR   

save, filename=sve_dir+fln_1+'.idl', input_header, input_data, $
      VERBOSE=VERBOSE,  /compress
message,'written idl saveset: ',/cont & print,'  ',sve_dir+fln_1+'.idl'
dealloc_lun,/quiet              ;frees logical units used by save/restore
return
  
out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_lanl_gps_onefile:!C '+!ERR_STRING
   
end
