;******************************************************************************
;*  NAME:        r_lanl_gps_singles
;*
;*  DESCRIPTION: Routine to read lanl_gps_singles files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       satid   the gps satellite id
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        December 1998
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_gps_singles, satid, VERBOSE=VERBOSE

  common lanl_gps_singles_data, input_header, input_data
  common mjdt, mjdt_start, mjdt_end
  common get_error, get_err_no, get_err_msg
  common lanl_gps_sats, gps_sats, gps_dirs
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

  get_err_no=0  &  get_err_msg='lanl_gps_singles read succesfully'
   
; check env. variable pointing to data directory
  env_var='LANL_GPS_SINGLES_DATA_ARCHIVE'
  if not (papco_check_data_env(env_var, path=PATH)) then return
  
; make sure gps_sats, gps_dirs are there, find gps sat name
  if n_elements(gps_sats) eq 0 then g_lanl_gps_singles_sats
  sat_name=gps_sats(satid)
  
; check VERBOSE keyword and set
  if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
  ;verbose=1
    
; clear data arrays for reuse
  input_header=0  &  input_data=0   
   
; make list of GPS singles files to read   
  if VERBOSE then message,'searching for files...', /cont
  fln_to_read=strarr((mjdt_end.mjd+7 - mjdt_start.mjd-7)/7 +2)
  fln_got=intarr(n_elements(fln_to_read))
  fln_idl=fln_to_read
  ss=mjdt_start  &  ss.mjd=ss.mjd-7
  count=0
  ext='.wes1'
  while ss.mjd le mjdt_end.mjd do begin
    convert_t90_to_Date, ss, year,doy,hr,min,sec,mon,cmon,doym, /mjdt
    year_str=string(year,format="(i4.4,'/')")
    fln=sat_name+'*'+string(year-1900,mon,doym, $
                            format="(i2.2,i2.2,i2.2)")
    found=findfile(path+sat_name+'/'+year_str+fln+ext,count=nfs)
   
    if nfs ne 0 then begin
        fln_to_read(count)=found(0)
        papco_sepdir,found(0),pp,ff
        fln_idl(count)=ff
        count=count+1
        if VERBOSE then message,'found '+fln,/cont
        skipp:
    endif
      
    if count ge 1 then ss.mjd=ss.mjd+7 else ss.mjd=ss.mjd+1
  endwhile
  index=where(fln_to_read ne '',count)
     
  if count ne 0 then fln_to_read=fln_to_read(index) else begin
    get_err_no=1
    get_err_msg='No files found for timerange requested'
    message,get_err_msg,/cont
    return
  endelse
  
; loop from startfile to endfile to read all data, concatonate if needed
  no_files_read=0
  for i=0,n_elements(fln_to_read)-1 do begin
     if VERBOSE then message,'Reading file: '+fln_to_read(i), /cont
     ; read one file of data
     r_lanl_gps_singles_onefile, fln_idl(i), fln_to_read(i), satid, $
                                 VERBOSE=VERBOSE
     ; concatonate data arrays
     if get_err_no eq 0 then begin          ;a file was read succesfully
       if no_files_read eq 0 then begin     ;start the array buffer
         input_data_buf=input_data
       endif else begin		            ;otherwise concatonate arrays
         input_data_buf=[input_data_buf,input_data]
       endelse 
       no_files_read=no_files_read+1
     endif
     next_file:
  endfor
   
; check if any files were read   
  if (no_files_read ne 0) and (get_err_no eq 0) then begin
     get_err_msg=string(no_files_read,format="(i2)")+ $
                        ' files read succesfully'
     ;clear arrays and load buffer into output array
     input_data=0  &  input_data=input_data_buf  &  input_data_buf=0
  endif else begin
     get_err_no=1
     get_err_msg='no files found for timerange'
     return
  endelse         
  
; reduce array to time interval required. Convert PAPCO time to GPS time and
; select. Beware end of year days in gps data files!
  
  convert_t90_to_Date, mjdt_start, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  dc_start=doy+(hr*3600l+min*60l+sec)/86400.0
  convert_t90_to_Date, mjdt_end, year,doy,hr,min,sec,mon,cmon,doym, /mjdt 
  dc_end=doy+(hr*3600l+min*60l+sec)/86400.0
  
  if year ne input_header.start_year then begin
    day1=julday(1,1,year,0,0,0)
    day2=julday(1,1,year+1,0,0,0)
    daysinyear=day2-day1
    dc_start=dc_start+daysinyear
    dc_end=dc_end+daysinyear
  endif  
  
  index=where((input_data.time ge dc_start) and $
               (input_data.time le dc_end),count)
  if count ne 0 then input_data=input_data(index)
    
; Change time to start at seconds since start day, then running seconds. 
; Beware that the start file for multiple file data may be missing!
  offset=0
  while fix(dc_start) lt fix(input_data(0).time) do begin
      offset=offset+1
      dc_start=dc_start+1
      ;print,offset,dc_start,input_data(0).time
  endwhile

  ts=fix(input_data(0).time)
  for i=0l,n_elements(input_data)-1 do begin
      t=input_data(i).time - ts
      input_data(i).time=t*86400.0
  endfor
  input_data.time=input_data.time + offset*86400

  print, get_err_msg
  
end

;******************************************************************************
;*  NAME:        r_lanl_gps_singles_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_gps_singles_onefile, fln, fln_archive, satid, VERBOSE=VERBOSE
  
   common lanl_gps_singles_data, input_header, input_data
   common get_error, get_err_no, get_err_msg
   
   get_err_no=0
   get_err_msg='lanl_gps_singles_onefile read succesfully'
   
; check for env. variable pointing to data (idl binaries) directory
  data_dir='LANL_GPS_SINGLES_DATA'
  path=papco_getenv(data_dir)
  if strlen(path(0)) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif
  
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
   ext='.idl'
   on_ioerror, read_original
   pos=strpos(fln,'.idl')
   if pos ne -1 then fln=strmid(fln,0,pos)
   if VERBOSE then message,'Reading '+fln, /cont
   restore,path+fln+ext, VERBOSE=VERBOSE
   dealloc_lun,/quiet    ;frees logical units used by save/restore
   message,'idl saveset read succesfully', /cont
   print,'  ',path+fln+ext
   return

   read_original:
   ; if the .idl save file could not be found, then try to read the original
   r_lanl_gps_singles_onefile_original, fln, fln_archive, satid, $
                                FIT=FIT, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_lanl_gps_singles_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_lanl_gps_singles_onefile_original, fln, fln_archive, satid, $
                                 FIT=FIT, VERBOSE=VERBOSE
  
  common lanl_gps_singles_data, input_header, input_data
  common get_error, get_err_no, get_err_msg  
  common lanl_gps_sats, gps_sats, gps_dirs
   
  get_err_no=0
  get_err_msg='lanl_gps_singles_onefile_orignal read succesfully'
   
; check env. variable pointing to data directory
  env_var='LANL_GPS_SINGLES_DATA'
  if not (papco_check_data_env(env_var, path=PATH)) then return
     
; clear data arrays for reuse
   input_header=0  &  input_data=0   
   
;------------------------------------------------------------------------------
; read the original data file - THIS NEEDS TO BE REWRITTEN BU USER!!!!
   
; see if file is actually there
  filethere=findfile(fln_archive)
  if filethere(0) eq '' then begin
    get_err_no=2
    get_err_msg='File '+fln_archive+' not found'
    print,get_err_msg 
    return
  endif
    
  message,'reading ascii format (slow): ',/cont
  print,'  ',fln_archive
  
; open file and report problems
  openr,unit,fln_archive,/get_lun,ERROR=err
    if err ne 0 then begin
      get_err_no=3
      get_err_msg='Problem opening '+fln
      return
    endif
    
    start_year=fix(strmid(fln,5,2))+1900

; read ascii data
    sat_id=gps_sats(satid)
    junk='' & ct=0
    eph_in=fltarr(12)
    input_header={lanl_gps_singles_header,sat_id:sat_id,start_year:start_year}
   
    dummy1={time:0.0, lat:0.0, long:0.0, rad:0l, data:fltarr(2)}
    dummy3=create_struct(name='lanl_gps_singles1', $
                         'time',0.0, 'COUNTS',fltarr(2))
    input_data=replicate(dummy3,6300)
    
    junk=''
    readf,unit,junk
    
    while not eof(unit) do begin ;both files have same length!
      readf,unit,dummy1, format="(f12.6,2(f10.4),i10,2(e10.4))"
      if dummy1.data(0) eq 1.0 then  dummy1.data(*)=-1.0
      dummy3.time=dummy1.time
      dummy3.COUNTS(0)=dummy1.data(0)
      dummy3.COUNTS(1)=dummy1.data(1)   
      input_data(ct)=dummy3
      ct=ct+1
    endwhile   
      
  close,unit  &  free_lun,unit
   
  input_data=input_data(0:ct-1)

  
; end of user written section
;------------------------------------------------------------------------------
   
; save read data with correct time. Rename data to *.idl
  on_ioerror, out
  save,filename=path+fln+'.idl',input_header,input_data, VERBOSE=VERBOSE  
  message,'written idl saveset: ',/cont
  print,'  ',path+fln+'.idl'
  dealloc_lun,/quiet              ;frees logical units used by save/restore
  return
  
  out:
  ; if there was an error, report it. Use the IDL error variables. Error
  ; 171 is for "file not fond" which is used to invoke the remote get.  
  get_err_no=!ERROR
  get_err_msg='r_lanl_gps_singles_onefile:!C '+!ERR_STRING
   
end

