;***************************************************************************
;* FUNCTION:     
;*      FUNCTION  lanl_mpa_lt_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$PAPCO_PATH/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the PAPCO_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type PAPCO_PANEL_EDITOR_RETURN. Though this
;*	type is nowhere defined, I like to think of it as virtually existing
;*	data-type. 
;*	All the functions in $PAPCO_PATH/papco.pro that edit plots or add new
;*	ones, expect the opened editor-windows to return a structure containing
;*	at last some fields. Those fields might make up a structure, but as no
;*	program defines the structure-type PAPCO_PANEL_EDITOR_RETURN, this 
;*	poor structure is not alive but only virtual.
;*	The return-values of plot-editors have to expand this structure - I 
;*	like to think, that they inherit from that structure - and contain at
;*	least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		channel:0, $
;*	    		typeVector:[0,0,0,0], $ 
;*	    		manualYScaling:0 $
;*	    		yScaleMin:LONG(1), $
;*	    		yScaleMax:LONG(1), $
;*	    		panelKind:'some string'}
;*
;*      Additionally, some Panel-editors may return these fields in
;*      structure:
;*               {$
;*		        color:1, $
;*			plotStyle:1}
;*			
;*      For an enhanced description read $PAPCO_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=PAPCO_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     September 1998, written by Haje Korth
;* Modified August 2002, R. Friedel, for PAPCO >10
;***************************************************************************
FUNCTION  lanl_mpa_lt_panel, aPlotInfo, GROUP=group, ACTION=ACTION
   
COMMON lanl_mpa_lt_panel_DATA, widgetData

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

dirpath=papco_getenv('MPA_EVENT_DATA')
filter='.event'

; the routines used by papco_panel.pro define tags for a structure. Those
; tags will be stored in the structure 'additionalWData'. When you realize the
; window, you have to add your own window-specific data to that structure 
; and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window --------------------------------------------------
panel_base=WIDGET_BASE(title=aPlotInfo.panelKind,$
	               column=1, GROUP_LEADER=group, /MODAL)

;--- the first part of the window will consist in the widgets ----------------
;    that control the height of that plot
   widgets=PAPCO_panel_AddHeightWidgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)


; --- Choose plot type------------------------------------------------------ 
   sc_base=WIDGET_BASE(panel_base, /FRAME, COLUMN=1)
   sc_label=WIDGET_LABEL(sc_base,VALUE='Select Satellite(s):')
   sc_button_base=WIDGET_BASE(sc_base, ROW=1, /NONEXCLUSIVE)

   scNames=['1989-046', $
            '1990-095', $
            '1991-080', $
            '1994-084', $
            'LANL-97a']
   spb_type=lonarr(N_ELEMENTS(scNames))

   FOR i=0, N_ELEMENTS(scNames)-1 DO BEGIN
      spb_type(i)=WIDGET_BUTTON(sc_button_base, $
  			VALUE=scNames(i), $
			UVALUE='spb_type'+strtrim(string(i), 2))

      WIDGET_CONTROL, spb_type(i), $
          SET_BUTTON=(aPlotInfo.typevector(1) AND LONG(2)^i) EQ LONG(2)^i
   ENDFOR

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this.
   widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo,$
                                       DIRPATH=dirpath, FILTER=filter)
   additionalWData=create_struct(widgets, additionalWData)      
 
;  add ioptions variable since PAPCO_paneladd_PlotStyleButtons() is not used.
   additionalWData=create_struct({ioptions:intarr(3)}, additionalWData)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=PAPCO_panel_AddActionButtons(panel_base, Action, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
   thisWidgetsData={helpfile:'lanl_mpa_lt_panel.help', $
                    typevector:aPlotInfo.typeVector, $
                    spb_type:spb_type}

   widgetData=PAPCO_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
PAPCO_Panel_Realize, panel_base, widgetData
xmanager, 'lanl_mpa_lt_panel', panel_base

; the returned data consists of the standard-data inherited from
; PAPCO_PANEL.PRO and your new data - e.g. the typeVector-tag.
   returnData={typeVector:widgetData.typeVector, $
     ioptions:widgetData.ioptions}

   additionalRData=PAPCO_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO lanl_mpa_lt_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, PAPCO_PANEL_Event
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     September 1998, written by Haje Korth
;*
;***************************************************************************
PRO lanl_mpa_lt_panel_event, event
   
COMMON lanl_mpa_lt_panel_DATA, widgetData
   
WIDGET_CONTROL,GET_UVALUE=uval, event.id

   CASE strleft(uval, 8) OF
     'spb_type': BEGIN
         bit= 2^fix(strfrom(uval, 8))
         set= event.select              
         widgetData.typeVector(1)=(widgetData.typeVector(1) AND NOT(bit) ) + $
           bit*set
     END

     'eventTable': BEGIN
     END

     ELSE: BEGIN
         widgetData=PAPCO_panel_Event(event, widgetData)
     END
   ENDCASE
   
END
