;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  lanl_mpa_pitch_panel, PlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      PlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      ioptions:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, ioptions and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          ioptions      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*      panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(plotInfo, ACTION=, GROUP=)
;*
;* Written May 2003, Arne Aasnes at LANL (modified from lanl_mpa_spec_panel)
;******************************************************************************
FUNCTION  lanl_mpa_pitch_panel, PlotInfo, GROUP=group, ACTION=ACTION

COMMON lanl_mpa_pitch_panel_data, widgetData
COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents
 
;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;test if the plotinfo pointer tag points to an 'INT' (means not
;assigned to a structure). If so, load in the control structure.
r = size(*plotinfo.USR_PTR1, /ST)
IF r.TYPE_NAME EQ 'INT' THEN plotinfo.USR_PTR1 = ptr_new(lanl_mpa_control)
control = *plotinfo.USR_PTR1
;control now contains the user defined control structure for this module.

;--- create the base-window ---------------------------------------------------
panel_base=papco_widget_base(TITLE=PlotInfo.panelKind, $
                             COLUMN=1, GROUP_LEADER=group, /MODAL)
panel_top =    WIDGET_BASE(panel_base, ROW=1, /ALIGN_CENTER)
panel_middle = widget_base(panel_base, ROW = 1, SCROLL = scroll, $
                           X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize)
panel_left =   WIDGET_BASE(panel_middle, COLUMN=1, /ALIGN_CENTER)
panel_right =  WIDGET_BASE(panel_middle, COLUMN=1, /ALIGN_CENTER)
panel_bottom = WIDGET_BASE(panel_base, ROW=1, /ALIGN_CENTER)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_top, PlotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;--- select satellite to use ------------------------------------------------  
sat_base = widget_base(panel_left, ROW=1, /ALIGN_CENTER) 
spb_sat = CW_BGROUP(sat_base, sat_names, ROW = 2, /EXCLUSIVE,/FRAME, $
                    LABEL_TOP='Select Satellite:', UVALUE='spb_sat', $
                    SET_VALUE=control.sc_id)



;--- select species ---------------------------------------------------------

names = papco_make_choice_names(control.species_info)
spb_spc = CW_BGROUP(sat_base, names, ROW = 2, /EXCLUSIVE, /FRAME, $
                    LABEL_TOP='Select Species:', UVALUE='spb_spc', $
                    SET_VALUE=control.species)

choice_base = widget_base(sat_base, ROW=2, /ALIGN_CENTER, /FRAME) 
;---choose removal of photo electrons---------------------------------------
names = ['No', 'Yes']
spb_rem_ph = CW_BGROUP(choice_base, names, ROW = 1, /EXCLUSIVE, $
                      LABEL_LEFT = 'Filter ph. el:', UVALUE = 'spb_rem_ph', $
                      SET_VALUE = control.rem_ph)

;---choose interpolation of mag. field--------------------------------------
names = ['No', 'Yes']
spb_int_mag = CW_BGROUP(choice_base, names, ROW = 1, /EXCLUSIVE,  $
              LABEL_LEFT = 'intrp. magdir:', UVALUE = 'spb_mag_corr', $
              SET_VALUE = control.mag_corr)


;--- select options ---------------------------------------------------------
opt_base = widget_base(panel_left, ROW = 1, /ALIGN_CENTER) 
names = papco_make_choice_names(control.pl_type_info)
spb_opt = CW_BGROUP(opt_base, names, COLUMN = 4, /EXCLUSIVE, /FRAME, $
                    LABEL_top='Data type:', UVALUE='spb_opt', $
                    SET_VALUE=control.pl_type)
;--- select pitch angle resolution -----------------------------------------
fld_base = WIDGET_BASE(panel_left, /align_center,/frame,  ROW = 1, xsize = 500)

fld_pa = CW_FIELD(fld_base, /ROW, $
                  TITLE = 'Pitch angle res(deg):', $
                  UVALUE= 'fld_pa', VALUE = control.pa_res)

;--- select magnetic field component to plot---------------------------------
names = papco_make_choice_names(control.mf_comp_info)
spb_mf = CW_BGROUP(fld_base, names, ROW = 1, /EXCLUSIVE, /FRAME,  $
                   LABEL_LEFT = 'MF component:', UVALUE = 'spb_mf', $
                   SET_VALUE = control.mf_comp)

;--- select energy channel and pitch angle (range)---------------------------

pitch_base=WIDGET_BASE(panel_left, /FRAME, /align_center, ROW = 2)
pitch_base1=WIDGET_BASE(pitch_base,/align_center, ROW = 1)
pitch_base2 = WIDGET_BASE(pitch_base,/align_center, ROW = 1)

;ek_base = WIDGET_BASE(pitch_base1, ROW = 1)
;ek_label = WIDGET_LABEL(ek_base, /Align_center, XSIZE=200, VALUE = '')
sl_ek=WIDGET_SLIDER(pitch_base1, XSIZE = 200, /DRAG, $
                         TITLE=strcenter('Energy Channel 1', 50), $
                         VALUE=control.ek1 > 0 < 39, $
                         UVALUE='sl_ek', $
                         MINIMUM=0, MAXIMUM=39)

;pa_base = WIDGET_BASE(pitch_base, ROW = 1)
;pa_label = WIDGET_LABEL(pa_base, /Align_center, XSIZE=150, VALUE = '')
sl_pa=WIDGET_SLIDER(pitch_base1, XSIZE = 295, /DRAG, $
                         TITLE=strcenter('Pitch angle bin 1', 40), $
                         VALUE=control.pa_bin1 > 0 < 180, $
                         UVALUE='sl_pa', $
                         MINIMUM=0, MAXIMUM=180)

;ek_base2 = WIDGET_BASE(pitch_base2)
;ek_label2 = WIDGET_LABEL(ek_base2, /Align_center, XSIZE=200, VALUE = '')
sl_ek2 = WIDGET_SLIDER(pitch_base2,XSIZE = 200, /DRAG, $
                         TITLE=strcenter('Energy Channel 2', 50), $
                         VALUE=control.ek1 > 0 < 39, $
                         UVALUE='sl_ek2', $
                         MINIMUM=0, MAXIMUM=39)


;pa_base2 = WIDGET_BASE(pitch_base2)
;pa_label2 = WIDGET_LABEL(pa_base2, /Align_center, XSIZE=150, VALUE = '')
sl_pa2=WIDGET_SLIDER(pitch_base2,XSIZE = 295, /DRAG,  $
                         TITLE=strcenter('Pitch angle bin 2', 40), $
                         VALUE=control.pa_bin2 > 0 < 180, $
                         UVALUE='sl_pa2', $
                         MINIMUM=0, MAXIMUM=180)

;--- button to chose "select" variable ----------------------------------------
filter_outer = widget_base(panel_left, COLUMN = 2, /ALIGN_CENTER)
filter_names = 'select ' + select_names
filter_base=CW_BGROUP(filter_outer, filter_names, ROW = 2, /EXCLUSIVE, /FRAME,$
                  LABEL_TOP='Select filter variable', $
                  UVALUE='spb_filter', SET_VALUE=control.filter)
;--- Add sliders to filter data - L, MLT, MLAT, etc ---------------------------
;    Multiple map bases
select_outer_base=WIDGET_BASE(filter_outer, COLUMN=1, /align_center)
select_base=WIDGET_BASE(select_outer_base, /align_center)

;--- make select widgets for all variables in "select_names ------------------
;    use papco_panel compound widget for this!
widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L_STAR', $
            control.L_STAR_SELECT_RANGE > 6 < 7.5, $
            control.L_STAR_SELECT_FROM_VAL > 6 < 7.5, $
            control.L_STAR_SELECT_TO_VAL, 6.0, 7.5, 0.05, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'L', $
            control.L_SELECT_RANGE > 6 < 7.5, $
            control.L_SELECT_FROM_VAL > 6 < 7.5, $
            control.L_SELECT_TO_VAL, 6.0, 7.5, 0.05, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLT', $
            control.MLT_SELECT_RANGE, $
            control.MLT_SELECT_FROM_VAL, $
            control.MLT_SELECT_TO_VAL, 0.0, 24.0, 0.5, '(f4.1)')
additionalWData=create_struct(widgets, additionalWData)

widgets = papco_panel_AddVar_Sel_Widgets(select_base, 'MLAT', $
            control.MLAT_SELECT_RANGE, $
            control.MLAT_SELECT_FROM_VAL, $
            control.MLAT_SELECT_TO_VAL, -15, 15.0, 1, '(f5.1)')
additionalWData=create_struct(widgets, additionalWData)

;--- add Mag Field model choices ----------------------------------------------
widgets=papco_panel_Add_MagMod_Widgets(panel_left, control, /MODEL_ONLY)
additionalWData=create_struct(widgets, additionalWData)

;-- map different bases according to what type of data is chosen.  
type_outer_base=WIDGET_BASE(panel_left, COLUMN=1, /align_center)
type_base=WIDGET_BASE(type_outer_base, /align_center)
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (PlotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    mpa. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;   widgets=papco_panel_AddFile_Widgets(panel_base, PlotInfo)
;   additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for specifying own plot description label -----
widgets=papco_panel_AddDescr_Widgets(panel_right, PlotInfo, /vertical)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the color bar for color plots ----
widgets=papco_panel_Addcolorbar_Widgets(panel_left, PlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
   widgets=papco_panel_add_PlotStyleButtons(panel_left, PlotInfo, 0)
   additionalWData=create_struct(widgets, additionalWData)


panel_scl =   WIDGET_BASE(panel_right, ROW=1, /ALIGN_CENTER)

;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_PANEL_AddYScale_Widgets(panel_scl, PlotInfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_PANEL_AddZScale_Widgets(panel_scl, PlotInfo,/VERTICAL)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_bottom, Action, PlotInfo, $
                                     ROW = 1, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create 
;    the helpfile in the papco_interface directory of your module (a template
;    for this help file is in the mpa example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'lanl_mpa_pitch.help', $
                 base:panel_base, $
                 spb_mf:spb_mf, $
;                 spb_rem_ph:spb_rem_ph, $
                 sl_pa:sl_pa,$ ; pa_label:pa_label, $
                 sl_ek:sl_ek,$; ek_label:ek_label, $
                 sl_pa2:sl_pa2,$; pa_label2:pa_label2, $
                 sl_ek2:sl_ek2, $;ek_label2:ek_label2, $
                 fld_pa:fld_pa, $
                 sat_base:sat_base, $
                 opt_base:opt_base}


; add control struct, papco_panel structs
widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)
widgetData=papco_panel_CreateWidgetData(widgetData, control)



;--- realize the window
;    -------------------------------------------------------

papco_Panel_Realize, panel_base, widgetData
WIDGET_CONTROL, panel_base, HOURGLASS=1
lanl_mpa_pitch_refresh, /ALL

;--- As this is a function that will return all changes made by the
;    user, the xmanager has to be called with the keyword MODAL. If we
;    wouldn't specify /MODAL, no changes would be reported but the 
;    data contained in PlotInfo would be returned.
papco_xmanager, 'lanl_mpa_pitch_panel', panel_base

;now return data. Since all the widget's info in in a control structure,
;only load it's pointer into USR_PTR1 and return that.

;control is part of the widgetData structure. Extract it
struct_assign, widgetData, *plotinfo.USR_PTR1

;work out if there has been a change in the control structure
plotinfo.modified = plotinfo.modified + $
                    papco_compare_struct(control, *plotinfo.USR_PTR1)

returnData={USR_PTR1:plotinfo.USR_PTR1, $
            MODIFIED:plotinfo.modified}

additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_mpa_pitch_refresh, ALL=ALL, DATA_TYPE=DATA_TYPE
;*
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* CALLING SEQUENCE:
;*      have a look at the event-routine lanl_mpa_line_panel_event
;*
;* MODIFICATION HISTORY:       
;*      May 2003, written by Arne Aasnes at LANL  
;******************************************************************************
PRO lanl_mpa_pitch_refresh, ALL=ALL, SAT = SAT,  DATATYPE = DATATYPE,  FILTER = FILTER

COMMON lanl_mpa_pitch_panel_data, widgetData
COMMON lanl_mpa_pitch_refresh, warn_flag

IF KEYWORD_SET(ALL) THEN BEGIN
    SAT = 1 &  DATATYPE = 1 &  FILTER = 1 
 ENDIF

; --- Filter Variable multiple bases ------------------------------------------
IF keyword_set(FILTER) THEN BEGIN
    wid_ids = [widgetData.l_select_base, widgetData.l_star_select_base, $
               widgetData.mlt_select_base, widgetData.mlat_select_base]
    n_wid = n_elements(wid_ids) & map = bytarr(n_wid) 
    map(widgetData.filter) = 1 
    FOR i = 0, n_wid-1 DO WIDGET_CONTROL, wid_ids(i), MAP = map(i)
ENDIF

IF KEYWORD_SET(SAT) THEN BEGIN
    IF WidgetData.sc_id EQ 5 THEN BEGIN
        IF NOT keyword_set(warn_flag) THEN $
          result = MessageBox(['*** Warning: *** ', '', $
                               'LANL-01A fluxes subject to', $
                               'intermittent noise contamination'], $
                               ['Continue'], $
                               TITLE = 'LANL-01A Warning', /CENTER)
        warn_flag = 1
    ENDIF 
ENDIF 

IF keyword_set(DATATYPE) THEN BEGIN 
       wid_ids = [widgetData.sl_ek, widgetData.sl_pa, $
               widgetData.sl_ek2, widgetData.sl_pa2, widgetData.spb_mf]
    CASE WidgetData.pl_type OF 
 
        0:BEGIN ;All Pitch Angles @ one Energy
            set_plot = [1, 0] & set_sens = [1, 0, 0, 0, 0]
            widgetData.ylinlog = 0
        END 
        1:BEGIN ;All Energies @ one Pitch Angle
            set_plot = [1, 0] & set_sens = [1, 1, 0, 0, 0]
            widgetData.ylinlog = 1
        END 
        2:BEGIN ;All Pitch Angles for an Energy Range
            set_plot = [1, 0] & set_sens = [1, 0, 1, 0, 0]
            widgetData.ylinlog = 0
        END 
        3:BEGIN                 ;All Energies for a Pitch Angle Range
            set_plot = [1, 0] & set_sens = [1, 1, 0, 1, 0]
            widgetData.ylinlog = 1
        END 
        4:BEGIN ;anisotropy spectrum plot
           set_plot = [1, 0] &  set_sens = [1, 1, 0, 1, 0]
           widgetData.ylinlog = 1
        END   
        5:BEGIN ;Pitch Angle Range @ an Energy Range (line plot)
            set_plot = [0, 1] & set_sens = [1, 1, 1, 1, 0]
;            widgetData.ylinlog = 0
        END 
        6:BEGIN ;anisotropy (line plot)
           set_plot = [0, 1] &  set_sens = [1, 1, 0, 1, 0]
        END  
        7:BEGIN ;magnetic field component (line plot)
            set_plot = [0, 1] & set_sens = [0, 0, 0, 0, 1]
            widgetData.ylinlog = 0
         END 
         
    ENDCASE
;    FOR i = 0,1 DO BEGIN
;        widget_control, widgetData.plot_ids(i), sensitive = set_plot(i)
;        widget_control, widgetData.plot_base, set_value = set_plot(i)
;    ENDFOR 
    FOR i = 0,4 DO BEGIN
        widget_control, wid_ids(i), sensitive = set_sens(i)
    ENDFOR 
    papco_panel_refresh, widgetData, /YSCALE
ENDIF

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_mpa_pitch_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO lanl_mpa_pitch_panel_event, event
  
COMMON lanl_mpa_pitch_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval
    uval = uval.uvalue
ENDIF

CASE uval OF
    'spb_sat': BEGIN
        WidgetData.sc_id = event.value
        lanl_mpa_pitch_refresh, /SAT
    END 
    'spb_mf': BEGIN 
       widgetData.mf_comp = event.value
    END 
    'spb_spc': BEGIN
        WidgetData.species = event.value
    END 
    'spb_rem_ph':BEGIN
       WidgetData.rem_ph = event.value
    END 
    'spb_mag_corr':BEGIN
       WidgetData.mag_corr = event.value
    END  

    'spb_opt': BEGIN
        WidgetData.pl_type = event.value
        lanl_mpa_pitch_refresh, /DATATYPE
     END

    'sl_ek': BEGIN
       WidgetData.ek1 = event.value
    END
    'sl_ek2': BEGIN
       WidgetData.ek2 = event.value
    END
    
     'sl_pa':BEGIN
        WidgetData.pa_bin1 = event.value
     END
     'sl_pa2':BEGIN
        WidgetData.pa_bin2 = event.value
     END
    'spb_filter' : BEGIN 
        widgetData.filter = event.value
        lanl_mpa_pitch_refresh, /FILTER
    END 

    ELSE : BEGIN
       WIDGET_CONTROL, widgetData.fld_pa, GET_VALUE = Y
       widgetData.pa_res = y
       widgetdata=papco_panel_event(event,WidgetData)        
    END
ENDCASE      

END
