;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  lanl_mpa_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a panelplot.
;*	Its window is constructed using the methods defined in the file
;*	$papco_XX/papco_panel.pro
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_olotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_PANEL_EDITOR_RETURN. Though this
;*      type is nowhere defined, I like to think of it as virtually existing
;*      data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Those fields might make up a structure, but as no
;*      program defines the structure-type papco_PANEL_EDITOR_RETURN, this 
;*      poor structure is not alive but only virtual.
;*      The return-values of plot-editors have to expand this structure - I 
;*      like to think, that they inherit from that structure - and contain at
;*      least those fields:
;*   		{$
;*			canceled:1, $
;*	    		panelHeight:1, $
;*	    		typeVector:[0,0,0,0], $ 
;*                      ioptions:[0,0,0], $
;*	                fileName:'', $        	 	
;*	    		panelKind:'some string'}
;*
;*      The fields typeVector, ioptions and fileName are meant for the user to
;*      use and set in his/her panel editor.
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          ioptions      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      These tags can be used for ANYTHING really, these are just
;*      suggestions. You are, however, limited to these three tags, and only
;*      these tags will get stored when you use the "SAVE" function.
;*
;*      NB! If you need more variables to commincate between your
;*      panel editor and your plot routine then use your own common block
;*
;*      Additionally, some Panel-editors may return more fields, depending on
;*      which papco supplied widgets were used:
;*              {$
;*	                manualYScaling:0, $
;*	                manualZScaling:0, $
;*	                yscaleMin:0.0, $
;*	                yscaleMax:0.0, $
;*		        zscaleMin:0.0, $
;*	                zscaleMax:0.0, $
;*	                yLinLog:1,$
;*	                zLinLog:1,$
;*	                channel:0, $	      
;*		        color:1, $
;*			plotStyle:1}
;*
;*      papco internally uses some more tags which the user need not worry
;*      about:
;*              {$
;*	                panelPosition:0,$
;*	                overplot:0, $          	  	 	
;*	                xut1:0.0, $                       
;*                      xut2:0.0}                           
;*
;*                  
;*      For an enhanced description read 
;*                          $papco_XX/papco_doc/papco_variables.text
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;*
;* Modified August 2002, R. Friedel, for PAPCO >10
;******************************************************************************
FUNCTION  lanl_mpa_spec_panel, aPlotInfo, GROUP=group, ACTION=ACTION

COMMON lanl_mpa_spec_panel_data, widgetData
COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents
 
;--- if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

;--- set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
panel_base=papco_widget_base(TITLE=aPlotInfo.panelKind, $
                             COLUMN=1, GROUP_LEADER=group, /MODAL)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    control the height of that panel. This is needed for all panel editors.
widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;

;--- select satellite to use ------------------------------------------------  
sat_base = widget_base(panel_base, ROW=1, /ALIGN_CENTER) 
spb_sat = CW_BGROUP(sat_base, sat_names, ROW = 2, /EXCLUSIVE, /FRAME, $
                    LABEL_TOP='Select Satellite:', UVALUE='spb_sat', $
                    SET_VALUE=aPlotInfo.typevector(0))

;--- select species ---------------------------------------------------------
choice_base = widget_base(panel_base, ROW=1, /ALIGN_CENTER) 
names = ['Electrons', 'Ions']
spb_spc = CW_BGROUP(choice_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                    LABEL_TOP='Select Data Category:', UVALUE='spb_spc', $
                    SET_VALUE=aPlotInfo.typevector(1))

;--- select options ---------------------------------------------------------
opt_base = widget_base(panel_base, ROW=1, /ALIGN_CENTER) 
names = ['Energy Flux', 'Differential Flux', 'Phase Space Density']
spb_opt = CW_BGROUP(opt_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                    LABEL_TOP='Select Option:', UVALUE='spb_opt', $
                    SET_VALUE=aPlotInfo.typevector(2))

;--- select S/C Potential --------------------------------------------------
names = ['No', 'Yes ']
spb_scp = CW_BGROUP(choice_base, names, ROW = 1, /EXCLUSIVE, /FRAME, $
                    LABEL_TOP='Plot S/C Potential:', UVALUE='spb_scp', $
                    SET_VALUE=aPlotInfo.typevector(3))

;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    mpa. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
;   widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo)
;   additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
widgets=papco_panel_Addcolorbar_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
;   widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
;   additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
widgets=papco_PANEL_AddZScale_Widgets(panel_base, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo, $
                                     ROW = 2, /CONFIG)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create 
;    the helpfile in the papco_interface directory of your module (a template
;    for this help file is in the mpa example module) and describe 
;    your data and your choices in your widget. Put in here also the ID's of 
;    the widgets you create so that you can perform widget_control actions on
;    them them in the events routine or in a refresh routine if needed.
thisWidgetsData={helpfile:'lanl_mpa_line_panel.help', $
                 typevector:aPlotInfo.typevector, $
                 ioptions:aPlotInfo.ioptions}
widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, widgetData

;--- As this is a function that will return all changes made by the
;    user, the xmanager has to be called with the keyword MODAL. If we
;    wouldn't specify /MODAL, no changes would be reported but the 
;    data contained in aPlotInfo would be returned.
papco_xmanager, 'lanl_mpa_spec_panel', panel_base

; --- the returned data consists of the standard-data inherited from
;     papco_PANEL.PRO and your new data - the typeVector-tag and ioptions-tag 
;     must always be included, but you can add more. However, if possible stick
;     to the tags provided in aPlotInfo, it makes life easier. 
returnData={typevector:widgetData.typevector, $
            ioptions:widgetData.ioptions}
additionalRData=papco_panel_ReturnData(widgetData)

return, create_struct(ReturnData, additionalRData)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_mpa_spec_refresh, ALL=ALL, DATA_TYPE=DATA_TYPE
;*
;* DESCRIPTION:  
;*      update the panelplot-editor display
;* 
;* INPUTS:       
;*      none
;* CALLING SEQUENCE:
;*      have a look the the event-routine lanl_mpa_line_panel_event
;*
;* MODIFICATION HISTORY:       
;*      August 2002, written R Friedel  
;******************************************************************************
PRO lanl_mpa_spec_refresh, ALL=ALL, SAT = SAT

COMMON lanl_mpa_spec_panel_data, widgetData
COMMON lanl_mpa_spec_refresh, warn_flag

IF KEYWORD_SET(ALL) THEN BEGIN
    
ENDIF

IF KEYWORD_SET(SAT) THEN BEGIN
    IF WidgetData.typevector(0) EQ 5 THEN BEGIN
        IF NOT keyword_set(warn_flag) THEN $
          result = MessageBox(['*** Warning: *** ', '', $
                               'LANL-01A fluxes subject to', $
                               'intermittent noise contamination'], $
                               ['Continue'], $
                               TITLE = 'LANL-01A Warning', /CENTER)
        warn_flag = 1
    ENDIF 
ENDIF 

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO lanl_mpa_spec_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO lanl_mpa_spec_panel_event, event
  
COMMON lanl_mpa_spec_panel_data, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF
    'spb_sat': BEGIN
        WidgetData.typevector(0) = event.value
        lanl_mpa_spec_refresh, /SAT
    END 
    'spb_spc': BEGIN
        WidgetData.typevector(1) = event.value
    END 
    'spb_opt': BEGIN
        WidgetData.typevector(2) = event.value
    END 
    'spb_scp': BEGIN
        WidgetData.typevector(3) = event.value
    END 
    ELSE : BEGIN
        widgetdata=papco_panel_event(event,WidgetData)        
    END
ENDCASE      

END
