;******************************************************************************
;* PROCEDURE:     
;*      PRO mpa_user_drawslice_lanl_mpa_spec, panelNr, seconds, yValue
;* 
;* DESCRIPTION:  
;*      Draw an energy flux slice for mpa spectrum plots
;*      calling a separate window for the plot.
;*
;* INPUTS:       
;*      panelNr     an integer, the number of the panel for which the slice 
;*                  is to be drawn
;*      seconds     time of mouse click, a float
;*      yvalue      y-value of mouse click, a float
;*      
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by papco_DrawSLice (in $papco_PATH/papco_cursor.pro)
;*
;* MODIFICATION HISTORY:       
;*      September 1998, written by Haje Korth
;******************************************************************************
PRO mpa_user_drawslice_lanl_mpa_spec, panelNr, seconds, yValue, sltype

  COMMON MPA_DRAWSLICE, sliceWidgetData
  COMMON MPA_USERSLICE_WIDGETDATA, pb_fit, pb_toFile
  COMMON MPA_USERSLICEDATA, slicePlotData,extra_plotPar,extra_axisPar, fitPlotPar
  COMMON MPA_USERSLICE_SYSVAR, spbangx, spbangy
  COMMON PLOT_COMPOSER, widgetData
  COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  COMMON lanl_mpa_data, input_header, input_data, input_sat
  COMMON papco_color_names
  COMMON lanl_mpa_slice, utitle
  
  COMMON lanl_mpa               ;see lanl_mpa_set_commons.pro for contents

  WIDGET_CONTROL,/HOURGLASS
  
;ensure correct data set is loaded in mpa_data
  panSat=widgetData.plotInfos(panelNr).typevector(0)
  currPlotinfo=widgetData.plotInfos(panelNr)
  IF (panSat NE input_sat) THEN r_lanl_mpa, currPlotInfo
  utitle = strsubst(currPlotInfo.DESCR_STR, '!C', ', ')
  
;save !x !y parameter
  bangx=!x & bangy=!y
  
  IF((slicePlotData.overPlot EQ 1) AND $
   (slicePlotData.noSliceYet EQ 0)) THEN BEGIN
     !x=spbangx & !y=spbangy
  ENDIF
  
  slicePlotData.fitsuccess=0
  slicePlotData.fitx=-1.0
  slicePlotData.fity=-1.0
  slicePlotData.fitden=-1.0
  slicePlotData.fittemp=-1.0
  slicePlotData.fitchi2=-1.0
  slicePlotData.fitprob=-1.0
  slicePlotData.fitkappa=-1.0
  
  IF (slicePlotData.numslice GT 5) THEN BEGIN
     info1='Sorry, no more than 5 (five) slices'
     info2='allowed in this routine! :-('
     GOTO, MESSED_UP
  ENDIF
  
; use the same typevector, switch etc as is used in the call to your plot
; routine. thisPlot contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
  thisPlot=widgetData.plotsDrawn(panelNr)
  type=thisPlot.typeVector
  CASE type(2) OF
     0: pltype='enflux'
     1: pltype='diffflux'
     2: pltype='phspdens'
  ENDCASE
  
; set the panel window for the slice to full window
  !p.position=[0.125,0.1,0.9,0.9]
  
; select the index for the time chosen out of the time array
  time=input_data.tai
  tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
  time = time - tai_1
  
  DiffTimes=ABS(time - seconds) ; this sets timeIndex to the index with
  result=MIN(DiffTimes, timeIndex) ; t90(index) is closest to 'seconds'
  slicePlotData.timeIndex=timeIndex

; erase plot window if appropriate
  IF (slicePlotData.overPlot EQ 0) THEN BEGIN
     WIDGET_CONTROL, sliceWidgetData.dw_slice,  GET_VALUE=drawIndex
     IF !D.NAME EQ 'X' THEN  WSET, drawIndex
     ERASE
     slicePlotData.noSliceYet=1
     slicePlotData.labdown=0
     slicePlotData.labcol=0
     slicePlotData.numslice=1
     slicePlotData.slicearr=-1
     slicePlotData.fit_str_arr=''
     
     slicePlotData.xstart=thisPlot.yscl_min
     slicePlotData.xstop=thisPlot.yscl_max
     slicePlotData.dxstart=thisPlot.yscl_min
     slicePlotData.dxstop=thisPlot.yscl_max
     slicePlotData.ystart=thisPlot.zscl_min
     slicePlotData.ystop=thisPlot.zscl_max
     slicePlotData.dystart=thisPlot.zscl_min
     slicePlotData.dystop=thisPlot.zscl_max
  ENDIF
  
  time_str=papco_cursor_time(seconds,panelNr)
  time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)

  scpot_str=STRTRIM(STRING(input_data(timeIndex).sc_poten,FORMAT='(f7.1)'),2)   
  info1=string(time_str,scpot_str, $
             format="('Time: ',A18,'    S/C Potential: ',A7,' V')")
  info2=''

  IF (type(1) EQ 0) THEN BEGIN
     yarr=REVERSE(input_data(timeIndex).ave_flux_e)
     xarr=REVERSE(input_data(timeIndex).ece4)
     
    IF (slicePlotData.scpcorrect EQ 1) THEN BEGIN
       corxarr=xarr+ABS(input_data(timeIndex).sc_poten)
       index=WHERE(corxarr GT 0,count)
       IF (count NE 0) THEN BEGIN
          corxarr=corxarr(index)
          xarr=xarr(index)
          yarr=yarr(index)
       ENDIF
       yarr=yarr*xarr/corxarr
       xarr=corxarr
       slicePlotData.xstart=MIN(xarr)
       slicePlotData.xstop=MAX(xarr)
    ENDIF
    
    IF (sltype EQ 'phspdens') THEN factor=8.09e-32 ELSE factor=1.0
    IF (slicePlotData.numslice EQ 1) THEN slicePlotData.sliceType=0
    sliceType=0
 ENDIF ELSE BEGIN
    yarr=REVERSE(input_data(timeIndex).ave_flux_i)
    xarr=REVERSE(input_data(timeIndex).ecp4)
    
    IF (slicePlotData.scpcorrect EQ 1) THEN BEGIN
       corxarr=xarr-ABS(input_data(timeIndex).sc_poten)
       index=WHERE(corxarr GT 0,count)
       IF (count NE 0) THEN BEGIN
          corxarr=corxarr(index)
          xarr=xarr(index)
          yarr=yarr(index)
       ENDIF
       yarr=yarr*xarr/corxarr
       xarr=corxarr
       slicePlotData.xstart=MIN(xarr)
       slicePlotData.xstop=MAX(xarr)
    ENDIF
    
    IF (sltype EQ 'phspdens') THEN factor=2.72e-25 ELSE factor=1.0
    IF (slicePlotData.numslice EQ 1) THEN slicePlotData.sliceType=1
    sliceType=1
 ENDELSE
 
 errorarr=MPA_ERRCALC(yarr,type)
 IF (slicePlotData.errorPlot EQ 1) THEN $
      slicePlotData.errorPlotarr(slicePlotData.numslice-1)=1 $
    ELSE $
      slicePlotData.errorPlotarr(slicePlotData.numslice-1)=0

 index=WHERE(yarr GT 1,count)
 IF (count NE 0) THEN BEGIN
    xarr=xarr(index)
    yarr=yarr(index)
    errorarr=errorarr(index)
 ENDIF ELSE RETURN
 
 xtitle='Energy (eV)'
 
 CASE sltype OF
    'enflux' : BEGIN
       ytitle='Energy Flux (cm!E-2!Ns!E-1!Nsr!E-1!N)'
            yarr=yarr*xarr
            errorarr=errorarr*xarr
         END
         'diffflux' : BEGIN
            ytitle='Diff. Flux (cm!E-2!Ns!E-1!Nsr!E-1!NeV!E-1!N)'
         END
         'phspdens' : BEGIN
            ytitle='Phase Space Density (cm!E-6!Ns!E3!N)'
            yarr=factor*yarr/xarr
            errorarr=factor*errorarr/xarr
         END
         
      ENDCASE
      
      IF (slicePlotData.sliceType EQ sliceType) THEN BEGIN
         slicePlotData.slicearr(5*slicePlotData.numslice-5,0:count-1)=xarr
         slicePlotData.slicearr(5*slicePlotData.numslice-4,0:count-1)=yarr
         slicePlotData.slicearr(5*slicePlotData.numslice-3,0:count-1)=errorarr
         slicePlotData.time_str_arr(slicePlotData.numslice-1)=time_str
         IF (slicePlotData.yAuto EQ 1) THEN BEGIN
            slicePlotData.ystart=thisPlot.zscl_min
            slicePlotData.ystop=thisPlot.zscl_max
         ENDIF
      ENDIF ELSE BEGIN
         info1="Please don't give me a hard time..."
         info2='...and choose only ONE particle species for this plot.'
         GOTO, MESSED_UP
      ENDELSE
      
      IF (sltype NE pltype) AND (slicePlotData.noSliceYet EQ 1) AND $
      (slicePlotData.yAuto EQ 1) THEN BEGIN
         
         dummye_arr=input_data.ave_flux_e
         energye_arr=input_data.ece4
         dummyi_arr=input_data.ave_flux_i
         energyi_arr=input_data.ecp4
         index=WHERE((dummye_arr GE 0.0) AND (dummye_arr LE 1.0), count)
         IF (count NE 0) THEN dummye_arr(index)=-1
         index=WHERE((dummyi_arr GE 0.0) AND (dummyi_arr LE 1.0), count)
         IF (count NE 0) THEN dummyi_arr(index)=-1
         
         IF (type(1) EQ 0) THEN BEGIN
            rngarr=dummye_arr
            energy_arr=energye_arr
            IF (slicePlotData.scpcorrect EQ 1) THEN BEGIN
               esize=SIZE(energy_arr)
               poten_arr=DBLARR(esize(1),esize(2))
               FOR i=0,esize(2)-1 DO poten_arr(*,i)=ABS(input_data(i).sc_poten)
               energy_arr=energy_arr+poten_arr
            ENDIF
         ENDIF ELSE BEGIN
            rngarr=dummyi_arr
            energy_arr=energyi_arr
            IF (slicePlotData.scpcorrect EQ 1) THEN BEGIN
               esize=SIZE(energy_arr)
               poten_arr=DBLARR(esize(1),esize(2))
               FOR i=0,esize(2)-1 DO poten_arr(*,i)=ABS(input_data(i).sc_poten)
               energy_arr=energy_arr-poten_arr
            ENDIF
         ENDELSE
         
         index=WHERE(energy_arr LE 0,count)
         IF (sltype EQ 'enflux') THEN rngarr=rngarr*energy_arr
         IF (sltype EQ 'phspdens') THEN rngarr=factor*rngarr/energy_arr
         IF (count NE 0) THEN rngarr(index)=-1
         
         index=WHERE(rngarr LT 0.0, count)
         IF (count NE 0) THEN rngarr(index)=-1
         yrng=rngarr
         
         logscl=widgetData.plotInfos.zlinlog
         logscl=logscl(panelNr)
         IF (logscl EQ 1) THEN BEGIN
            papco_autorange,yrng,ymin,ymax,log=1,exclude=0,nodata=-1
            mpa_autorange,[ymin,ymax],yntick,ytickv,ytickl,log=1,ntickmax=4
         ENDIF ELSE BEGIN
            papco_autorange,yrng,ymin,ymax,log=0,exclude=0,nodata=-1
            mpa_autorange,[ymin,ymax],yntick,ytickv,ytickl,log=0,ntickmax=4
         ENDELSE     
         slicePlotData.ystart=ymin
         slicePlotData.ystop=ymax
         slicePlotData.dystart=ymin
         slicePlotData.dystop=ymax
      ENDIF
      
      xlog=slicePlotData.xlog
      ylog=slicePlotData.ylog
      
      IF (slicePlotData.xAuto EQ 1) THEN BEGIN
         xrng=[slicePlotData.xstart,slicePlotData.xstop]
         mpa_autorange,xrng,xntick,xtickv,xtickl,log=xlog,ntickmax=4
         extra_PlotPar={xticks:xntick-1,xtickv:xtickv,xtickname:xtickl,xstyle:5,xlog:xlog}
         extra_AxisPar={xticks:xntick-1,xtickv:xtickv,xtickname:xtickl,xlog:xlog, $
                    xtitle:xtitle}
      ENDIF ELSE BEGIN
         extra_PlotPar={xrange:[slicePlotData.xstart,slicePlotData.xstop], $
                    xticks:4,xstyle:5,xlog:xlog}
         extra_AxisPar={xrange:[slicePlotData.xstart,slicePlotData.xstop], $
                    xticks:4,xstyle:1,xlog:xlog, xtitle:xtitle}
      ENDELSE

      IF (slicePlotData.yAuto EQ 1) THEN BEGIN
         yrng=[slicePlotData.ystart,slicePlotData.ystop]     
         mpa_autorange,yrng,yntick,ytickv,ytickl,log=ylog,ntickmax=4
         extra_par={yticks:yntick-1,ytickv:ytickv,ystyle:5,ylog:ylog}
         extra_axpar={yticks:yntick-1,ytickv:ytickv,ylog:ylog,ytitle:ytitle}
         extra_plotPar=create_struct(extra_plotPar, extra_par)
         extra_axisPar=create_struct(extra_axisPar, extra_axpar)
      ENDIF ELSE BEGIN
         extra_par={yrange:[slicePlotData.ystart,slicePlotData.ystop], $
                    yticks:4,ystyle:5,ylog:ylog}
         extra_axpar={yrange:[slicePlotData.ystart,slicePlotData.ystop], $
                    yticks:4,ystyle:1,ylog:ylog,ytitle:ytitle}
         extra_plotPar=create_struct(extra_plotPar, extra_par)
         extra_axisPar=create_struct(extra_axisPar, extra_axpar)
      ENDELSE

      fitPlotPar=extra_plotPar
      
      symData=FINDGEN(17)*!pi*2./16.
      USERSYM,COS(symData),SIN(symData),/FILL
      
      labdown=slicePlotData.labdown
      colorarr=[black,red,blue,dark_green,yellow]
      labcol=colorarr(slicePlotData.labcol)
      
      IF (slicePlotData.nosliceYet EQ 1) THEN BEGIN
         IF (extra_axisPar.xlog EQ 1) THEN BEGIN
            !x.minor=0
            xlogmin=FLOOR(ALOG10(MIN(xarr)))
            xlogmax=CEIL(ALOG10(MAX(xarr)))
            xAxis=[10.0^xlogmin,10.0^xlogmax]
            extra_axisPar.xticks=0
         ENDIF ELSE xAxis=xarr
     
     IF (extra_axisPar.ylog EQ 1) THEN BEGIN
       !y.minor=0
       ylogmin=FLOOR(ALOG10(MIN(yarr)))
       ylogmax=CEIL(ALOG10(MAX(yarr)))
       yAxis=[10.0^ylogmin,10.0^ylogmax]
       extra_axisPar.yticks=0
     ENDIF ELSE yAxis=yarr

     title = utitle+', ' +strmid(info1, 0, 26)
     title = strsubst(title, '!C', ', ')

     PLOT, xAxis, yAxis, title = title,  color=1, psym=8, /nodata, $
       _extra=extra_axisPar
     spbangx=!x & spbangy=!y
    
     slicePlotData.nosliceYet=0
   ENDIF

   OPLOT,xarr,yarr,color=labcol,psym=-8

   IF (slicePlotData.errorPlot EQ 1) THEN SPOPLOTERR,xarr,yarr,errorarr,COLOR=labcol

IF (slicePlotData.overPlot EQ 1) THEN BEGIN
    satstr=sat_names(input_sat)
    extra_labelPar={COLOR:labcol,CHARSIZE:0.8,NORMAL:1}
    FITF_XYOUTS,0.65,0.9,time_str+satstr,bangccnt=labdown,_extra=extra_labelPar
    slicePlotData.labdown=slicePlotData.labdown+1
ENDIF

slicePlotData.labcol=slicePlotData.labcol+1
slicePlotData.numslice=slicePlotData.numslice+1
slicePlotData.fitOK=1

IF (slicePlotData.autoFit EQ 1) THEN BEGIN
    mpa_fit

    denstr='Den.:'+STRTRIM(STRING(slicePlotData.fitden(0),FORMAT='(G8.2)'),2)+' '+ $
            STRING(177B)+' '+STRTRIM(STRING(slicePlotData.fitden(1),FORMAT='(G8.2)'),2)
    tempstr='Temp.:'+STRTRIM(STRING(slicePlotData.fittemp(0),FORMAT='(G8.2)'),2)+' '+ $
            STRING(177B)+' '+STRTRIM(STRING(slicePlotData.fittemp(1),FORMAT='(G8.2)'),2)
    chi2str='Chi2:'+STRTRIM(STRING(slicePlotData.fitchi2,FORMAT='(G8.2)'),2)+ $
             ', Prob.:'+STRTRIM(STRING(slicePlotData.fitprob,FORMAT='(G8.2)'),2)
    info2=denstr+', '+tempstr+', '+chi2str
    IF (slicePlotData.fitkappa(0) NE -1) THEN BEGIN
       kappastr='Kappa:'+STRTRIM(STRING(slicePlotData.fitkappa(0),FORMAT='(G8.2)'),2)+' '+ $
                STRING(177B)+' '+STRTRIM(STRING(slicePlotData.fitkappa(1),FORMAT='(G8.2)'),2)
       info2=info2+', '+kappastr
ENDIF ELSE kappastr=''

    slicePlotData.fit_str_arr(0,slicePlotdata.numslice-2)=denstr
    slicePlotData.fit_str_arr(1,slicePlotdata.numslice-2)=tempstr
    slicePlotData.fit_str_arr(2,slicePlotdata.numslice-2)=chi2str
    slicePlotData.fit_str_arr(3,slicePlotdata.numslice-2)=kappastr

    IF (slicePlotData.fitsuccess EQ 1) THEN BEGIN
       IF (slicePlotData.overPlot EQ 1) THEN BEGIN
         colorarr=[black,red,blue,dark_green,yellow]
         labcol=colorarr(slicePlotData.labcol-1)
         extra_labelPar={COLOR:labcol,CHARSIZE:0.8,NORMAL:1}
         FITF_XYOUTS,0.65,0.9,denstr,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
         slicePlotData.labdown=slicePlotData.labdown+1
         FITF_XYOUTS,0.65,0.9,tempstr,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
         slicePlotData.labdown=slicePlotData.labdown+1
         FITF_XYOUTS,0.65,0.9,chi2str,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
         slicePlotData.labdown=slicePlotData.labdown+1
         IF (kappastr NE '') THEN BEGIN
           FITF_XYOUTS,0.65,0.9,kappastr,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
           slicePlotData.labdown=slicePlotData.labdown+1
         ENDIF
       ENDIF ELSE BEGIN
         WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
       ENDELSE
     ENDIF ELSE BEGIN
       info2='Sorry, I did not manage to fit this curve... :-('
       WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
     ENDELSE

     slicePlotData.fitOK=0
  ENDIF 

;reset system variables !x and !y
!x=bangx & !y=bangy

; Replace these labels and include your plot! 

; user defined routine for plotting the slice included here. An example is in 
; $papco_MODULES/crres_MEB/papco_interface/crres_slices.pro
; You typically need to include the commons from the read routine, a way
; of finding the index of the data you want to plot using the time "seconds"
; plus a simple plot statement to plot the data.
; Two strings, info1 and info2, can be used to annotate the slice window
; by providing additional information, ie ephemeris data, etc.

; these calls put your info strings in the provided labels of the slice widget
   MESSED_UP:

WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
IF (slicePlotData.fitOK EQ 1) AND (slicePlotData.sltype EQ 'phspdens') THEN $
     WIDGET_CONTROL,pb_fit,SENSITIVE=1 $
  ELSE WIDGET_CONTROL,pb_fit,SENSITIVE=0

END   


;******************************************************************************
PRO mpa_slice_options_event, event

COMMON MPA_SLICEOPTION_WIDGETDATA,Slice_Option_Base,xscaleInput_Base,yscaleInput_Base, $
           pb_xscaleAuto,pb_xscaleManual,pb_yscaleAuto,pb_yscaleManual, $
           ef_xscaleFrom,ef_xscaleTo,ef_yscaleFrom,ef_yscaleTo, $
           pb_xLog,pb_xLin,pb_yLog,pb_yLin,pb_overPlot,pb_errorPlot, $
           pb_scpcorrect,ef_fitFrom,ef_fitTo,ofilename,ofile_button
COMMON MPA_DRAWSLICE, sliceWidgetData
COMMON MPA_USERSLICEDATA, slicePlotData, extra_plotPar, extra_axisPar, fitPlotPar
COMMON MPA_USERSLICE_WIDGETDATA, pb_fit, pb_toFile

WIDGET_CONTROL,Event.Id,GET_UVALUE=Ev

CASE Ev OF
    'pb_overPlot' : BEGIN
       IF (event.select EQ 1) THEN slicePlotData.overplot=1 $
         ELSE slicePlotData.overPlot=0
    END

    'pb_errorPlot' : BEGIN
       IF (event.select EQ 1) THEN slicePlotData.errorplot=1 $
         ELSE slicePlotData.errorPlot=0
    END

    'pb_scpcorrect' : BEGIN
       IF (event.select EQ 1) THEN slicePlotData.scpcorrect=1 $
         ELSE slicePlotData.scpcorrect=0
    END

    'pb_maxwell' : BEGIN
       slicePlotData.fitType='maxwell'
    END

    'pb_kappa' : BEGIN
      slicePlotData.fitType='kappa'
    END

    'pb_autoFit' : BEGIN
       IF (event.select EQ 1) THEN slicePlotData.autoFit=1 $
         ELSE slicePlotData.autoFit=0
    END

    'ef_fitFrom' : BEGIN
       WIDGET_CONTROL,ef_fitFrom,GET_VALUE=fitstart
       slicePlotData.fitstart=fitstart
    END

    'ef_fitTo' : BEGIN
       WIDGET_CONTROL,ef_fitTo,GET_VALUE=fitstop
       slicePlotData.fitstop=fitstop
    END

    'ofile_button': BEGIN
        IF (slicePlotData.filename NE '') THEN BEGIN
            WIDGET_CONTROL,pb_toFile,SENSITIVE=0
            close, slicePlotData.fileunit
            free_lun, slicePlotData.fileunit
            slicePlotData.fileunit=-1
            slicePlotData.filename=''
        ENDIF ELSE BEGIN
            result = dialog_pickfile(group=Slice_Option_Base, /write, get_path=path, $
                             filter = '*.data', /fix_filter)
            IF (result NE '') THEN BEGIN
                IF (strpos(result, '/') EQ -1) THEN BEGIN
                    filename = path+result 
                ENDIF ELSE BEGIN
                    filename = result
                ENDELSE
                IF (strpos(result, '.data') EQ -1) THEN BEGIN
                    filename = filename+'.data'
                    ret = dialog_message('Filename must have '+$
                                         '".data" extention, '+$
                                         'extension apended. ', $
                                         dialog_parent=Slice_Option_Base)
                ENDIF 
                fileExist=findfile(filename,count=count)
                openw, /get_lun, unit, filename, error = error, /append
                IF (error NE 0) THEN BEGIN
                    print, 'Error opening file.', !err_string
                    ret = widget_message(/error, 'Error opening file: '+$
                                         !err_string, $
                                         dialog_parent=Slice_Option_Base)
                ENDIF ELSE BEGIN
                    slicePlotData.filename=filename
                    slicePlotData.fileunit=unit
                    IF (count EQ 0) THEN BEGIN
                      PRINTF,unit,'* MPA Slice Extract V1.0'
                      PRINTF,unit,'*'
                      PRINTF,unit,'* File format:'
                      PRINTF,unit,'* 1. Slice data: Energy.'
                      PRINTF,unit,'* 2. Slice data: Energy flux, diff. flux or phase space density.'
                      PRINTF,unit,'* 3. Error: Calculated error of 2.'
                      PRINTF,unit,'* 4. Fit data: Energy.'
                      PRINTF,unit,'* 5. Fit data: Energy flux, diff. flux or phase space density.'
                      PRINTF,unit,'* 6. Fit data: Density and error.'
                      PRINTF,unit,'* 7. Fit data: Temperature and error.'
                      PRINTF,unit,'* 8. Fit data: Chi^2 value.'
                      PRINTF,unit,'* 9. Fit data: Probability value.'
                      PRINTF,unit,'* 10. Fit data: Kappa value and error. For Maxwell fit kappa is infinity.'
                      PRINTF,unit,'*'
                    ENDIF
                ENDELSE
            ENDIF ELSE BEGIN
                print, 'Error in file name.'
                ret = widget_message(/error, 'Error in file name.', $
                                     dialog_parent=Slice_Option_Base)
            ENDELSE
            WIDGET_CONTROL,pb_toFile,SENSITIVE=1
        ENDELSE

        len=strlen(slicePlotData.filename)
        widget_control, ofilename, set_value=slicePlotData.filename
        widget_control, ofilename, set_text_select = [0, len]
        widget_control, ofilename, set_text_select = [len, len]
        IF (slicePlotData.filename EQ '') THEN BEGIN
            widget_control, ofile_button, set_value='Open File'
        ENDIF ELSE BEGIN
            widget_control, ofile_button, set_value='Close File'
        ENDELSE
     END
  
    'pb_xLin' : BEGIN
       slicePlotData.xLog=0
    END

    'pb_xLog' : BEGIN
       slicePlotData.xLog=1
    END

    'pb_yLin' : BEGIN
       slicePlotData.yLog=0
    END

    'pb_yLog' : BEGIN
       slicePlotData.yLog=1
    END

    'pb_xscaleManual' : BEGIN
       slicePlotData.xAuto=0
       WIDGET_CONTROL,xscaleInput_Base,SENSITIVE=1
       WIDGET_CONTROL,ef_xscaleFrom,GET_VALUE=xstart
       slicePlotData.xstart=xstart
       WIDGET_CONTROL,ef_xscaleTo,GET_VALUE=xstop
       slicePlotData.xstop=xstop
    END

    'pb_xscaleAuto' : BEGIN
       slicePlotData.xAuto=1
       WIDGET_CONTROL,xscaleInput_Base,SENSITIVE=0
       xstart=slicePlotData.dxstart
       xstop=slicePlotData.dxstop
       slicePlotData.xstart=xstart
       slicePlotData.xstop=xstop
       WIDGET_CONTROL,ef_xscaleFrom,SET_VALUE=xstart
       WIDGET_CONTROL,ef_xscaleTo,SET_VALUE=xstop
    END

    'ef_xscaleFrom' : BEGIN
       WIDGET_CONTROL,ef_xscaleFrom,GET_VALUE=xstart
       slicePlotData.xstart=xstart
    END

    'ef_xscaleTo' : BEGIN
       WIDGET_CONTROL,ef_xscaleTo,GET_VALUE=xstop
       slicePlotData.xstop=xstop
    END

    'pb_yscaleManual' : BEGIN
       slicePlotData.yAuto=0
       WIDGET_CONTROL,yscaleInput_Base,SENSITIVE=1
       WIDGET_CONTROL,ef_yscaleFrom,GET_VALUE=ystart
       slicePlotData.ystart=ystart
       WIDGET_CONTROL,ef_yscaleTo,GET_VALUE=ystop
       slicePlotData.ystop=ystop
    END

    'pb_yscaleAuto' : BEGIN
       slicePlotData.yAuto=1
       WIDGET_CONTROL,yscaleInput_Base,SENSITIVE=0
       ystart=slicePlotData.dystart
       ystop=slicePlotData.dystop
       slicePlotData.ystart=ystart
       slicePlotData.ystop=ystop
       WIDGET_CONTROL,ef_yscaleFrom,SET_VALUE=ystart
       WIDGET_CONTROL,ef_yscaleTo,SET_VALUE=ystop
    END

    'ef_yscaleFrom' : BEGIN
       WIDGET_CONTROL,ef_yscaleFrom,GET_VALUE=ystart
       slicePlotData.ystart=ystart
    END

    'ef_yscaleTo' : BEGIN
       WIDGET_CONTROL,ef_yscaleTo,GET_VALUE=ystop
       slicePlotData.ystop=ystop
    END

    'pb_default' : BEGIN
       slicePlotData.overPlot=slicePlotData.doverPlot
       slicePlotData.errorPlot=slicePlotData.derrorPlot
       slicePlotData.scpcorrect=slicePlotData.dscpcorrect
       slicePlotData.fitType=slicePlotData.dfitType
       slicePlotData.autoFit=slicePlotData.dautoFit
       slicePlotData.xAuto=slicePlotData.dxAuto
       slicePlotData.yAuto=slicePlotData.dyAuto
       slicePlotData.xLog=slicePlotData.dxLog
       slicePlotData.yLog=slicePlotData.dyLog
       slicePlotData.xstart=slicePlotData.dxstart
       slicePlotData.xstop=slicePlotData.dxstop
       slicePlotData.ystart=slicePlotData.dystart
       slicePlotData.ystop=slicePlotData.dystop

       WIDGET_CONTROL,ef_xscaleFrom,SET_VALUE=slicePlotData.dxstart
       WIDGET_CONTROL,ef_xscaleTo,SET_VALUE=slicePlotData.dxstop
       WIDGET_CONTROL,ef_yscaleFrom,SET_VALUE=slicePlotData.dystart
       WIDGET_CONTROL,ef_yscaleTo,SET_VALUE=slicePlotData.dystop

       IF (slicePlotData.overPlot EQ 1) THEN $
         WIDGET_CONTROL, pb_overPlot, SET_BUTTON=1 $
       ELSE $
         WIDGET_CONTROL, pb_overPlot, SET_BUTTON=0

       IF (slicePlotData.errorPlot EQ 1) THEN $
         WIDGET_CONTROL, pb_errorPlot, SET_BUTTON=1 $
       ELSE $
         WIDGET_CONTROL, pb_errorPlot, SET_BUTTON=0

       IF (slicePlotData.scpcorrect EQ 1) THEN $
         WIDGET_CONTROL, pb_scpcorrect, SET_BUTTON=1 $
       ELSE $
         WIDGET_CONTROL, pb_scpcorrect, SET_BUTTON=0

       CASE slicePlotData.fitType OF
         'maxwell': WIDGET_CONTROL,pb_maxwell,SET_BUTTON=1
         'kappa': WIDGET_CONTROL,pb_kappa,SET_BUTTON=1
       ENDCASE

       IF (slicePlotData.autoFit EQ 1) THEN $
         WIDGET_CONTROL, pb_autoFit, SET_BUTTON=1 $
       ELSE $
         WIDGET_CONTROL, pb_autoFit, SET_BUTTON=0

       IF (slicePlotData.xAuto EQ 1) THEN BEGIN
         WIDGET_CONTROL,pb_xscaleAuto,SET_BUTTON=1
         WIDGET_CONTROL,xscaleInput_Base,SENSITIVE=0
       ENDIF ELSE BEGIN
         WIDGET_CONTROL,pb_xscaleManual,SET_BUTTON=1
         WIDGET_CONTROL,xscaleInput_Base,SENSITIVE=1
       ENDELSE

       IF (slicePlotData.yAuto EQ 1) THEN BEGIN
         WIDGET_CONTROL,pb_yscaleAuto,SET_BUTTON=1
         WIDGET_CONTROL,yscaleInput_Base,SENSITIVE=0
       ENDIF ELSE BEGIN
         WIDGET_CONTROL,pb_yscaleManual,SET_BUTTON=1
         WIDGET_CONTROL,yscaleInput_Base,SENSITIVE=1
       ENDELSE

       IF (slicePlotData.xLog EQ 1) THEN $
         WIDGET_CONTROL, pb_xLog, SET_BUTTON=1 $
       ELSE $
         WIDGET_CONTROL, pb_xLin, SET_BUTTON=1

       IF (slicePlotData.yLog EQ 1) THEN $
         WIDGET_CONTROL, pb_yLog, SET_BUTTON=1 $
       ELSE $
         WIDGET_CONTROL, pb_yLin, SET_BUTTON=1
    END

    'pb_done' : BEGIN
       slicePlotData.noSliceYet=1
       slicePlotData.labdown=0
       slicePlotData.labcol=0
       slicePlotData.numslice=1
       slicePlotData.slicearr=-1
       panelNr=slicePlotData.panelNr
       seconds=slicePlotData.seconds
       yValue=slicePlotData.yValue
       sltype=slicePlotData.sltype
       WIDGET_CONTROL, sliceWidgetData.dw_slice,  GET_VALUE=drawIndex
       IF !D.NAME EQ 'X' THEN  WSET, drawIndex
       ERASE
       mpa_user_drawslice_lanl_mpa_spec, panelNr, seconds, yValue, sltype
       WIDGET_CONTROL, event.top, /DESTROY
    END
    ELSE:BEGIN
    END
ENDCASE

END

;******************************************************************************
PRO mpa_slice_options, group

  COMMON MPA_SLICEOPTION_WIDGETDATA,Slice_Option_Base,xscaleInput_Base,yscaleInput_Base, $
           pb_xscaleAuto,pb_xscaleManual,pb_yscaleAuto,pb_yscaleManual, $
           ef_xscaleFrom,ef_xscaleTo,ef_yscaleFrom,ef_yscaleTo, $
           pb_xLog,pb_xLin,pb_yLog,pb_yLin,pb_overPlot,pb_errorPlot, $
           pb_scpcorrect,ef_fitFrom,ef_fitTo,ofilename,ofile_button
  COMMON MPA_USERSLICEDATA, slicePlotData, extra_plotPar, extra_axisPar, fitPlotPar
  COMMON PLOT_COMPOSER, widgetData

  IF XREGISTERED('mpa_slice_options') THEN RETURN

  thisPlot=widgetData.plotsDrawn(slicePlotData.panelNr)

  Slice_Option_Base=WIDGET_BASE(title='MPA Slice Options',$
                                column=1, $
                                GROUP_LEADER=group)

  button_Base=WIDGET_BASE(Slice_Option_Base, ROW=1, /FRAME, /align_center, $
                /NONEXCLUSIVE)
  pb_overPlot=WIDGET_BUTTON(button_Base,VALUE='Over Plot',UVALUE='pb_overPlot')
  pb_errorPlot=WIDGET_BUTTON(button_Base,VALUE='Plot Errorbars',UVALUE='pb_errorPlot')
  pb_autoFit=WIDGET_BUTTON(button_Base,VALUE='Auto Fit',UVALUE='pb_autoFit')
  pb_scpcorrect=WIDGET_BUTTON(button_Base,VALUE='S/C Pot. Correction',UVALUE='pb_scpcorrect')

  fit_Base=WIDGET_BASE(Slice_Option_Base, COLUMN=1, /FRAME, /align_center)
  fitChoice=WIDGET_BASE(fit_Base, ROW=1, /ALIGN_CENTER)
  fitLabel=WIDGET_LABEL(fitChoice,VALUE='Fit Type:')
  fitUse_Base=WIDGET_BASE(fitChoice,ROW=1,/FRAME,/EXCLUSIVE)
  pb_maxwell=WIDGET_BUTTON(fitUse_Base,VALUE='Maxwell',UVALUE='pb_maxwell')
  pb_kappa=WIDGET_BUTTON(fitUse_Base,VALUE='Kappa',UVALUE='pb_kappa')
  fitInput_Base=WIDGET_BASE(fit_Base, ROW=1, /ALIGN_CENTER)
  fitInputLabel=WIDGET_LABEL(fitInput_Base,VALUE='Fit Limits:  ')
  ef_fitFrom=CW_FIELD(fitInput_Base,TITLE='Low:  ',$
		VALUE=slicePlotData.fitstart, $
                UVALUE='ef_fitFrom', $
                /FLOATING,/FRAME,/ALL_EVENTS)
  ef_fitTo=CW_FIELD(fitInput_Base,TITLE='High: ', $
		VALUE=slicePlotData.fitstop, $
                UVALUE='ef_fitTo', $
                /FLOATING,/FRAME,/ALL_EVENTS)
  ofile_base = widget_base(Slice_Option_Base, /column, /frame, /base_align_center)
  diag_label = widget_label(ofile_base, value = 'Output filename')
  ofilename = widget_text(ofile_base, uvalue='ofilename', xsize=75)
  ofile_button=widget_button(ofile_base,uvalue='ofile_button')
  xscale_Base=WIDGET_BASE(Slice_Option_Base, COLUMN=1, /FRAME, /align_center)
  xscaleChoice=WIDGET_BASE(xscale_Base, Row=1,/ALIGN_CENTER)
  xscaleLabel=WIDGET_LABEL(xscaleChoice,VALUE='X-scaling: ')
  xscaleUse_Base=WIDGET_BASE(xscaleChoice, ROW=1, /FRAME, /EXCLUSIVE)
  pb_xscaleAuto=WIDGET_BUTTON(xscaleUse_Base, VALUE='Automatic',$
			      UVALUE='pb_xscaleAuto')
  pb_xscaleManual=WIDGET_BUTTON(xscaleUse_Base, VALUE='Manual   ',$
			        UVALUE='pb_xscaleManual')
  linlog_Base=WIDGET_BASE(xscaleChoice, ROW=1, /FRAME, /EXCLUSIVE)
  pb_xLog=WIDGET_BUTTON(linlog_Base, VALUE='Logarithmic',$
			UVALUE='pb_xLog')
  pb_xLin=WIDGET_BUTTON(linlog_Base, VALUE='Linear     ',$
			UVALUE='pb_xLin')
  xscaleInput_Base=WIDGET_BASE(xscale_Base, ROW=1, /ALIGN_CENTER)
  xInputLabel=WIDGET_LABEL(xscaleInput_Base,VALUE='Limits:  ')
  ef_xscaleFrom=CW_FIELD(xscaleInput_Base, TITLE='Low:  ',$
		         VALUE=slicePlotData.xstart, $
                         UVALUE='ef_xscaleFrom', $
                         /FLOATING, /FRAME,/ALL_EVENTS)
  ef_xscaleTo=CW_FIELD(xscaleInput_Base, TITLE='High: ', $
	               VALUE=slicePlotData.xstop, $
                       UVALUE='ef_xscaleTo', $
                       /FLOATING, /FRAME, /ALL_EVENTS)

  yscale_Base=WIDGET_BASE(Slice_Option_Base, COLUMN=1, /FRAME, /align_center)
  yscaleChoice=WIDGET_BASE(yscale_Base, ROW=1,/ALIGN_CENTER)
  yscaleLabel=WIDGET_LABEL(yscaleChoice,VALUE='Y-scaling: ')
  yscaleUse_Base=WIDGET_BASE(yscaleChoice, ROW=1, /FRAME, /EXCLUSIVE)
  pb_yscaleAuto=WIDGET_BUTTON(yscaleUse_Base, $
			VALUE='Automatic',$
			UVALUE='pb_yscaleAuto')
  pb_yscaleManual=WIDGET_BUTTON(yscaleUse_Base, $
			VALUE='Manual   ',$
			UVALUE='pb_yscaleManual')
  linlog_Base=WIDGET_BASE(yscaleChoice, ROW=1, /FRAME, /EXCLUSIVE)
  pb_yLog=WIDGET_BUTTON(linlog_Base, $
			VALUE='Logarithmic',$
			UVALUE='pb_yLog')
  pb_yLin=WIDGET_BUTTON(linlog_Base, $
			VALUE='Linear     ',$
			UVALUE='pb_yLin')
  yscaleInput_Base=WIDGET_BASE(yscale_Base, ROW=1, /ALIGN_CENTER)
  yInputLabel=WIDGET_LABEL(yscaleInput_Base,VALUE='Limits:  ')
  ef_yscaleFrom=CW_FIELD(yscaleInput_Base, $
		TITLE='Low:  ',$
		VALUE=slicePlotData.ystart, $
                UVALUE='ef_yscaleFrom', $
                /FLOATING, /FRAME, /ALL_EVENTS)
  ef_yscaleTo=CW_FIELD(yscaleInput_Base, $
		TITLE='High: ', $
		VALUE=slicePlotData.ystop, $
                UVALUE='ef_yscaleTo', $
                /FLOATING, /FRAME, /ALL_EVENTS)

  pb_default=WIDGET_BUTTON(Slice_Option_Base,VALUE='Default Values', $
                           UVALUE='pb_default')

  pb_done=WIDGET_BUTTON(Slice_Option_Base,VALUE='Done',UVALUE='pb_done')

;***Set Widgets
  IF (slicePlotData.overPlot EQ 1) THEN $
    WIDGET_CONTROL, pb_overPlot, SET_BUTTON=1

  IF (slicePlotData.errorPlot EQ 1) THEN $
    WIDGET_CONTROL, pb_errorPlot, SET_BUTTON=1

  IF (slicePlotData.autoFit EQ 1) THEN $
    WIDGET_CONTROL, pb_autoFit, SET_BUTTON=1
  IF (slicePlotData.sltype NE 'phspdens') THEN WIDGET_CONTROL,pb_autoFit,SENSITIVE=0

  IF (slicePlotData.scpcorrect EQ 1) THEN $
    WIDGET_CONTROL, pb_scpcorrect, SET_BUTTON=1

  CASE slicePlotData.fitType OF
    'maxwell': WIDGET_CONTROL,pb_maxwell,SET_BUTTON=1
    'kappa': WIDGET_CONTROL,pb_kappa,SET_BUTTON=1
  ENDCASE

  IF (slicePlotData.fileunit EQ -1) THEN BEGIN
    WIDGET_CONTROL, ofile_button, SET_VALUE='Open File'
  ENDIF ELSE BEGIN
    WIDGET_CONTROL, ofilename, SET_VALUE=slicePlotData.filename
    WIDGET_CONTROL, ofile_button, SET_VALUE='Close File'
  ENDELSE

  IF (slicePlotData.xAuto EQ 1) THEN BEGIN
    WIDGET_CONTROL, pb_xscaleAuto, SET_BUTTON=1
    WIDGET_CONTROL, xscaleInput_Base, SENSITIVE=0
  ENDIF ELSE BEGIN
    WIDGET_CONTROL, pb_xscaleManual, SET_BUTTON=1
    WIDGET_CONTROL, xscaleInput_Base, SENSITIVE=1
  ENDELSE

  IF (slicePlotData.yAuto EQ 1) THEN BEGIN
    WIDGET_CONTROL, pb_yscaleAuto, SET_BUTTON=1
    WIDGET_CONTROL, yscaleInput_Base, SENSITIVE=0
  ENDIF ELSE BEGIN
    WIDGET_CONTROL, pb_yscaleManual, SET_BUTTON=1
    WIDGET_CONTROL, yscaleInput_Base, SENSITIVE=1
  ENDELSE

  IF (slicePlotData.xLog EQ 1) THEN $
    WIDGET_CONTROL, pb_xLog, SET_BUTTON=1 $
  ELSE $
    WIDGET_CONTROL, pb_xLin, SET_BUTTON=1
  IF (slicePlotData.yLog EQ 1) THEN $
    WIDGET_CONTROL, pb_yLog, SET_BUTTON=1 $
  ELSE $
    WIDGET_CONTROL, pb_yLin, SET_BUTTON=1

;***Realize Widgets
  WIDGET_CONTROL,Slice_Option_Base,/REALIZE
  XMANAGER, 'mpa_slice_options', Slice_Option_Base, /JUST_REG

END

;******************************************************************************
;* PROCEDURE:
;*      PRO USER_MPA_Energy_Flux_Slice_mpa_spec_Event, Event
;*
;* DESCRIPTION:
;*	The slice-window displays a slice through the data,
;*	when the user clicks in the drawing-widget.
;*	This procedure handles events for that window.
;* INPUTS:
;*	Event	 	an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     September 1998, written by Haje Korth
;******************************************************************************
PRO USER_MPA_Slice_Event, Event

COMMON LANL_MPA_DATA, input_header, input_data,input_sat
COMMON MPA_DRAWSLICE, sliceWidgetData
COMMON MPA_USERSLICEDATA, slicePlotData,extra_plotPar,extra_axisPar,fitPlotPar
COMMON MPA_USERSLICE_WIDGETDATA, pb_fit, pb_toFile
COMMON PLOT_COMPOSER, widgetData
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON slice_calls, slice_calls
COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
                            white, burgundy, olive, dark_green, teal, $
                            royal_blue, violet, dark_grey, grey, $
                            foreground, background

WIDGET_CONTROL, Event.Id, GET_UVALUE=Ev

CASE Ev OF
    'pb_done' : BEGIN
        IF (slicePlotData.filename NE '') THEN BEGIN
            close, slicePlotData.fileunit
            free_lun, slicePlotData.fileunit
            slicePlotData.fileunit=-1
            slicePlotData.filename=''
        ENDIF

        WIDGET_CONTROL, sliceWidgetData.base, /DESTROY
    END

    'pb_erase': BEGIN
       WIDGET_CONTROL, sliceWidgetData.dw_slice,  GET_VALUE=drawIndex
       IF !D.NAME EQ 'X' THEN  WSET, drawIndex
       ERASE
       slicePlotData.noSliceYet=1
       slicePlotData.labdown=0
       slicePlotData.labcol=0
       slicePlotData.numslice=1
       slicePlotData.slicearr=-1
       slicePlotData.fit_str_arr=''
       info1=''
       info2=''
       WIDGET_CONTROL, sliceWidgetData.lbl_slice1, SET_VALUE=info1
       WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
    END

    'pb_help': BEGIN
       dir=getenv('PAPCO_MODULES')
       dir=dir+'/lanl_mpa/papco_interface/'
       xdisplayfile, dir+'lanl_mpa_slice.help', $
       title='papco Online-Help', group=event.top, /modal
    END

    'pb_print': BEGIN
        papco_printmenu_event, slice_calls(0)
    END

    'pb_toFile': BEGIN
       IF (slicePlotData.fileunit NE -1) THEN BEGIN
         unit=slicePlotData.fileunit
         numslice=slicePlotData.numslice-1
         PRINTF,unit,slicePlotData.time_str_arr(numslice-1)
         PRINTF,unit,slicePlotData.slicearr(5*numslice-5,*),FORMAT='(40F12.5)'
         PRINTF,unit,slicePlotData.slicearr(5*numslice-4,*),FORMAT='(40E15.6)'
         PRINTF,unit,slicePlotData.slicearr(5*numslice-3,*),FORMAT='(40E15.6)'
         PRINTF,unit,slicePlotData.fitx,FORMAT='(40F12.5)'
         PRINTF,unit,slicePlotData.fity,FORMAT='(40E15.6)'
         PRINTF,unit,slicePlotData.fitden
         PRINTF,unit,slicePlotData.fittemp
         PRINTF,unit,slicePlotData.fitchi2 
         PRINTF,unit,slicePlotData.fitprob
         PRINTF,unit,slicePlotData.fitkappa
       ENDIF
    END

    'pb_options': BEGIN
       mpa_slice_options, event.top
    END
     
    'pb_previous': BEGIN
       slicePlotData.overPlot=0
       slicePlotData.labdown=0
       slicePlotData.labcol=0
       slicePlotData.numslice=1
       slicePlotData.slicearr=-1
       slicePlotData.fit_str_arr=''
       slicePlotData.timeIndex=slicePlotData.timeIndex-1
       time=input_data.tai
       tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
       time = time - tai_1
       timeIndex=slicePlotData.timeIndex
       slicePlotData.seconds=time(timeIndex)
       panelNr=slicePlotData.panelNr
       seconds=slicePlotData.seconds
       yValue=slicePlotData.yValue
       sltype=slicePlotData.sltype
       mpa_user_drawslice_lanl_mpa_spec, panelNr, seconds, yValue, sltype
       mpa_strike,seconds
     END
     
     'pb_next': BEGIN
       slicePlotData.overPlot=0
       slicePlotData.labdown=0
       slicePlotData.labcol=0
       slicePlotData.numslice=1
       slicePlotData.slicearr=-1
       slicePlotData.fit_str_arr=''
       slicePlotData.timeIndex=slicePlotData.timeIndex+1
       time=input_data.tai
       tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
       time = time - tai_1
       timeIndex=slicePlotData.timeIndex
       slicePlotData.seconds=time(timeIndex)
       panelNr=slicePlotData.panelNr
       seconds=slicePlotData.seconds
       yValue=slicePlotData.yValue
       sltype=slicePlotData.sltype
       mpa_user_drawslice_lanl_mpa_spec, panelNr, seconds, yValue, sltype
       mpa_strike,seconds
    END

    'pb_fit': BEGIN
       mpa_fit
       denstr='Den.:'+STRTRIM(STRING(slicePlotData.fitden(0),FORMAT='(G8.2)'),2)+' '+ $
              STRING(177B)+' '+STRTRIM(STRING(slicePlotData.fitden(1),FORMAT='(G8.2)'),2)
       tempstr='Temp.:'+STRTRIM(STRING(slicePlotData.fittemp(0),FORMAT='(G8.2)'),2)+' '+ $
              STRING(177B)+' '+STRTRIM(STRING(slicePlotData.fittemp(1),FORMAT='(G8.2)'),2)
       chi2str='Chi2:'+STRTRIM(STRING(slicePlotData.fitchi2,FORMAT='(G8.2)'),2)+ $
               ', Prob.:'+STRTRIM(STRING(slicePlotData.fitprob,FORMAT='(G8.2)'),2)
       info2=denstr+', '+tempstr+', '+chi2str
       IF (slicePlotData.fitkappa(0) NE -1) THEN BEGIN
         kappastr='Kappa:'+STRTRIM(STRING(slicePlotData.fitkappa(0),FORMAT='(G8.2)'),2)+' '+ $
                  STRING(177B)+' '+STRTRIM(STRING(slicePlotData.fitkappa(1),FORMAT='(G8.2)'),2)
         info2=info2+', '+kappastr
       ENDIF ELSE kappastr=''

       slicePlotData.fit_str_arr(0,slicePlotdata.numslice-2)=denstr
       slicePlotData.fit_str_arr(1,slicePlotdata.numslice-2)=tempstr
       slicePlotData.fit_str_arr(2,slicePlotdata.numslice-2)=chi2str
       slicePlotData.fit_str_arr(3,slicePlotdata.numslice-2)=kappastr

       IF (slicePlotData.fitsuccess EQ 1) THEN BEGIN
         IF (slicePlotData.overPlot EQ 1) THEN BEGIN
           colorarr=[black,red,blue,dark_green,yellow]
           labcol=colorarr(slicePlotData.labcol-1)
           extra_labelPar={COLOR:labcol,CHARSIZE:0.8,NORMAL:1}
           FITF_XYOUTS,0.65,0.9,denstr,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
           slicePlotData.labdown=slicePlotData.labdown+1
           FITF_XYOUTS,0.65,0.9,tempstr,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
           slicePlotData.labdown=slicePlotData.labdown+1
           FITF_XYOUTS,0.65,0.9,chi2str,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
           slicePlotData.labdown=slicePlotData.labdown+1
           IF (kappastr NE '') THEN BEGIN
             FITF_XYOUTS,0.65,0.9,kappastr,bangccnt=slicePlotData.labdown,_extra=extra_labelPar
             slicePlotData.labdown=slicePlotData.labdown+1
           ENDIF
         ENDIF ELSE BEGIN
           WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
         ENDELSE
       ENDIF ELSE BEGIN
         info2='Sorry, I did not manage to fit this curve... :-('
         WIDGET_CONTROL, sliceWidgetData.lbl_slice2, SET_VALUE=info2
       ENDELSE

       slicePlotData.fitOK=0
       WIDGET_CONTROL,pb_fit,SENSITIVE=0
    END
     
    ELSE : BEGIN
    END
ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO USER_MPA_Energy_Flux_Slice_mpa_spec, panelNr, seconds,
;*                                               yValue, CLOSE=CLOSE
;*
;* DESCRIPTION:
;*	The slice-window displays a slice through the data,
;*	when the user clicks in the drawing-widget.
;*	This procedure opens that window and calls other routines, that draw
;*	the slice.
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the slice is to be
;*			drawn
;*	seconds		a float
;*	yvalue		a float
;*			the time and the y-value, for that the slice shall
;*			be drawn
;*
;* KEYWORDS:
;* 	CLOSE		when specified, the window is destroyed
;*
;* CALLING SEQUENCE:
;*	USER_MPA_Energy_Flux_Slice_mpa_spec, aPanel, $
;*          convert_tzero_to_t90(3600, 1, 1990), 42.42
;*	USER_MPA_Energy_Flux_Slice_mpa_spec, /CLOSE
;*
;* MODIFICATION HISTORY:
;*     September 1998, written by Haje Korth
;******************************************************************************
PRO USER_MPA_Slice, panelNr, seconds, yValue, sltype, CLOSE=CLOSE

COMMON MPA_DRAWSLICE, sliceWidgetData
COMMON MPA_USERSLICEDATA, slicePlotData,extra_plotPar,extra_axisPar, fitPlotPar
COMMON MPA_USERSLICE_WIDGETDATA, pb_fit, pb_toFile

COMMON PLOT_COMPOSER, widgetData
COMMON ADDINS, plot_types
COMMON ISREAD, IsReadArr
 
drawID=widgetData.old_draw_id( widgetdata.no_draw_window ) 
  
;-- close the window ? --------------------------------------------------------
IF KEYWORD_SET(CLOSE) THEN BEGIN
    IF XREGISTERED('USER_MPA_slice') THEN $
        IF N_ELEMENTS(sliceWidgetData) GT 0 THEN BEGIN
	    IF sliceWidgetData.base NE 0 THEN $
  	       WIDGET_CONTROL, sliceWidgetData.base, /DESTROY
            sliceWidgetData.base=0
        ENDIF
    RETURN
ENDIF

thisPlot=widgetData.plotsDrawn(panelNr)

;-- if the window exists, bring it in front. ELSE, create it ------------------
IF NOT XREGISTERED('USER_MPA_slice') THEN BEGIN

    slicePlotData={xstart:thisPlot.yscl_min,xstop:thisPlot.yscl_max, $
           ystart:thisPlot.zscl_min,ystop:thisPlot.zscl_max, $
           dxstart:thisPlot.yscl_min,dxstop:thisPlot.yscl_max, $
           dystart:thisPlot.zscl_min,dystop:thisPlot.zscl_max, $
           fitstart:1000,fitstop:thisPlot.yscl_max, $
           xLog:widgetData.plotinfos(panelNr).ylinlog, $
           yLog:widgetData.plotinfos(panelNr).zlinlog, $
           dxLog:widgetData.plotinfos(panelNr).ylinlog, $
           dyLog:widgetData.plotinfos(panelNr).zlinlog, $
           xAuto:1,yAuto:1,dxAuto:1,dyAuto:1, $
           panelNr:panelNr,seconds:seconds,yValue:yValue,sltype:sltype, $
           noSliceYet:1,overPlot:0,doverPlot:0,timeIndex:0, $
           errorPlot:0,derrorPlot:0,errorPlotarr:INTARR(5), $
           scpcorrect:0,dscpcorrect:0, $
           fitx:DBLARR(40),fity:DBLARR(40),labdown:0,labcol:0,numslice:1, $
           slicearr:DBLARR(25,40),slicetype:0,time_str_arr:STRARR(5), $
           fit_str_arr:STRARR(4,5), $
           fitden:DBLARR(2),fittemp:DBLARR(2),fitchi2:0.0d,fitprob:0.0d, $
           fitkappa:DBLARR(2),fitsuccess:0, $
           autoFit:0,dautoFit:0,fitOK:0,fitType:'maxwell',dfitType:'maxwell', $
           filename:'',fileunit:-1}

    Slice_Base=WIDGET_BASE(COLUMN=1, GROUP_LEADER=drawID, $
		           TITLE='LANL MPA slice window')

;-- setup draw area -----------------------------------------------------------
    draw_base=WIDGET_BASE(Slice_base, COLUMN=1, /FRAME)
    dw_Slice  =WIDGET_DRAW(draw_base, XSIZE=600, YSIZE=500)

;-- set up label fields for user data -----------------------------------------
    user_label_base=WIDGET_BASE(slice_base, COLUMN=1, /FRAME)
    lbl_slice1=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',100))
    lbl_slice2=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',100))

;-- set up buttons for the Actions --------------------------------------------
    Butt_Base=WIDGET_BASE(Slice_Base, ROW=2, /FRAME)
    pb_previous=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Previous', 17), $
                             UVALUE='pb_previous')
      
    pb_next=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Next', 17), $
                             UVALUE='pb_next')
      
    pb_fit=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Fit', 17), $
                             UVALUE='pb_fit')
      
    pb_erase=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Erase', 17), $
                             UVALUE='pb_erase')

    pb_toFile=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Data 2 File', 17), $
                             UVALUE='pb_toFile')

    pb_print=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Print', 17), $
                             UVALUE='pb_print')

    pb_options=WIDGET_BUTTON(Butt_Base, $
                             Value=strcenter('Options', 17), $
                             UVALUE='pb_options')
      
    pb_help=WIDGET_BUTTON(Butt_Base, $
                              Value=strcenter('Help', 17), $
                            UVALUE='pb_help')
      
    pb_done=WIDGET_BUTTON(Butt_Base, $
                              Value=strcenter('Close', 37), $
			      UVALUE='pb_done')
    sliceWidgetData={PAPCO_DRAWSLICE_a, $
		       base:slice_base, $
	               dw_slice:dw_slice, $
		       lbl_slice1:lbl_slice1, $
                       lbl_slice2:lbl_slice2, $
                       slice_call:'', $
	               panelNr:panelNr, $
		       lastPanelDrawn:-1}

    WIDGET_CONTROL,pb_toFile,SENSITIVE=0

    WIDGET_CONTROL, Slice_Base, /REALIZE
      XMANAGER, 'USER_MPA_slice', Slice_Base, /JUST_REG
ENDIF ELSE BEGIN
    slicePlotData.seconds=seconds
ENDELSE

IF (sltype NE 'phspdens') THEN WIDGET_CONTROL,pb_fit,SENSITIVE=0

slicePlotData.panelNr=panelNr

;--- now, we need to get the data --------------------------------------------
;    but we will only read the data, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).
;    also check the ISREAD common, data might not need to be read again
;    anyway!

IF sliceWidgetData.lastPanelDrawn NE panelNr THEN BEGIN
    result=strpos(plot_types,strchminus(thisPlot.panelKind),0)
    typeindex=where(result eq 0,count)
    IF count ne 0 THEN BEGIN ; plot_type is add-in, check the ISREAD array
        IF IsReadArr(typeindex(0)) ne 1 THEN BEGIN
            print, 'I need to read the data:'
            WIDGET_CONTROL,/HOURGLASS
            r = EXECUTE(widgetData.plotsDrawn(panelNr).get_data_call)
            IF NOT r  THEN BEGIN
                dummy=messageBox(['Sorry, there seems to be an error',$
                                  'I cannot read the data for the panel.'], $
                                 ['Ok'])
                RETURN
            ENDIF
        ENDIF
    ENDIF
ENDIF

;--- prepare the drawing widget for plotting ----------------------------------
WIDGET_CONTROL, sliceWidgetData.dw_slice,  GET_VALUE=drawIndex
IF !D.NAME EQ 'X' THEN  WSET, drawIndex

;--- store the current coordinate system --------------------------------------
;   old_x = !x
;   old_y = !y
;   old_p = !p

; ---  Call add-in routine for slice  -----------------------------------------
result=strpos(plot_types,strchminus(thisPlot.panelKind),0)
typeindex=where(result eq 0,count)
IF count ne 0 THEN BEGIN
    drawslice_call='mpa_user_drawslice_'+strchminus(thisPlot.panelKind)
    drawslice_prmt='panelNr, seconds, yValue, sltype'
    slice_call=drawslice_call+', '+ $
        string(panelNr)+','+string(seconds)+','+string(yValue)
    sliceWidgetData.slice_call=slice_call
    callgood=execute(drawslice_call+', '+drawslice_prmt)
    IF callgood then goto, jumpslice
ENDIF

jumpslice:

; --- restore the coordinate-system -------------------------------------------
;   !x = old_x
;   !y = old_y
;   !p = old_p
   
sliceWidgetData.lastPanelDrawn = panelNr   

END


;******************************************************************************
PRO USER_MPA_Energy_Flux_Slice_lanl_mpa_spec, panelNr, seconds, yValue, $
    READDATA=READDATA, CLOSE=CLOSE
user_mpa_slice, panelNr, seconds, yValue, 'enflux', CLOSE=CLOSE
END

;******************************************************************************
PRO USER_MPA_Diff_Flux_Slice_lanl_mpa_spec, panelNr, seconds, yValue, $
    READDATA=READDATA, CLOSE=CLOSE
user_mpa_slice, panelNr, seconds, yValue, 'diffflux', CLOSE=CLOSE
END

;******************************************************************************
PRO USER_MPA_Phase_Space_Density_Slice_lanl_mpa_spec,panelNr,seconds,yValue, $
    READDATA=READDATA, CLOSE=CLOSE
user_mpa_slice, panelNr, seconds, yValue, 'phspdens', CLOSE=CLOSE
END
;*******************************************************************************

PRO USER_MPA_Energy_PA_Slice_lanl_mpa_pitch,panelNr,seconds,yValue, $
    READDATA=READDATA, CLOSE=CLOSE, SOPA = sopa, NOWIDGET = nowidget
;  user_mpa_slice, panelNr, seconds, yValue, 'pitchangle', CLOSE=CLOSE
  COMMON time, xut1, xut2
  COMMON get_error, get_err_no, get_err_msg  
  COMMON USER_Energy_PA_Slice_lanl_mpa_pitch, slice_widget
  COMMON PLOT_COMPOSER, widgetData
  COMMON papco_color_names
  
; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_cluster_rapid to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you switch panels, the new data is 
; read. 
  
  COMMON lanl_mpa
  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data
  COMMON lanl_mpa_pitch_slice, utitle, uytitle, uztit
  COMMON lanl_mpa_pitch_output,xarr, yarr, zmat
  COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  COMMON yscale, yscl           ;man/auto yscaling
  COMMON zscale, zscl           ;man/auto zscaling
  
; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
plotinfo =widgetData.plotinfos(panelNr)
plotdrawn=widgetData.plotsDrawn(panelNr)
control = *plotinfo.USR_PTR1
;if keyword_set(OVERPLOT) then overplot=OVERPLOT else overplot=0
IF NOT keyword_set(low_en) THEN low_en = 13

if keyword_set(NOWIDGET) then BEGIN
   slice_widget={resolution:fix(control.pa_res),$
                 low_en:0, $
;                  average:fix(control.mus), $
;                  pitch:fix(control.pa_ctr_3dd), $
;                  label:fix(control.pa_lbl_3dd), $
;                  have_pitch:0, $
                  overplot:overplot, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}    
    goto, plotonly
ENDIF

; see if slice window needs to be closed --------------------------------------
if keyword_set(CLOSE) then begin
    if xregistered('USER_Energy_PA_Slice_lanl_mpa_pitch') then $
      if n_elements(slice_widget) gt 0 then begin
	if slice_widget.base ne 0 then $
          widget_control, slice_widget.base, /destroy
        slice_widget.base=0
      endif
    return
ENDIF 

; if slice window exists, bring it in front. ELSE, create it ------------------
if not xregistered('USER_Energy_PA_Slice_lanl_mpa_pitch') then begin
    
    slice_base=widget_base(column=1, group=widgetdata.main_base, $
                           title='Energy_PA_Slice_lanl_mpa_pitch')
    
;-- setup draw area -----------------------------------------------------------
    draw_base=WIDGET_BASE(Slice_base, COLUMN=1, /FRAME)
    dw_Slice  =WIDGET_DRAW(draw_base, XSIZE=600, YSIZE=500)

;-- set up label fields for user data -----------------------------------------
    user_label_base=WIDGET_BASE(slice_base, COLUMN=1, /FRAME)
    lbl_slice1=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
    lbl_slice2=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
   
;-- set up some additional options for this slice -----------------------------
    Opt_base = WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    ef_res = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Pitch angle resolution', $
                     VALUE = control.pa_res, $
                     UVALUE='ef_res')
    ef_low_en = WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    ef_res = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Lowest energy channel:', $
                     VALUE = low_en, $
                     UVALUE='ef_low_en')

;-- set up some toggle switch options for this slice --------------------------
;    toggle_base=WIDGET_BASE(Slice_Base, /FRAME, ROW=1, /NONEXCLUSIVE)
;    spb_pitch = WIDGET_BUTTON(toggle_base, UVALUE='spb_pitch', $
;                             VALUE='Overlay Pitch Angle Contour')
;    WIDGET_CONTROL, spb_pitch, SET_BUTTON = fix(control.pa_ctr_3dd)

;-- set up some toggle switch options for this slice --------------------------
;    spb_label = WIDGET_BUTTON(toggle_base, UVALUE='spb_label', $
;                             VALUE='Label cells with Pitch Angle')
;    WIDGET_CONTROL, spb_label, SET_BUTTON = fix(control.pa_lbl_3dd)

    
;-- set up buttons for the Actions --------------------------------------------
    Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    pb_help=WIDGET_BUTTON(Butt_Base, Value=strcenter('Help', 12), $
                          UVALUE='pb_help')
    
    pb_print=WIDGET_BUTTON(Butt_Base, Value=strcenter('Print', 12), $
                           UVALUE='pb_print')
    pb_output=WIDGET_BUTTON(Butt_Base, Value=strcenter('Output', 12), $
                           UVALUE='pb_output')    

    pb_done=WIDGET_BUTTON(Butt_Base, Value=strcenter('Close', 12), $
                          UVALUE='pb_done')

    slice_widget={rsl_lines:0, $ 
                 resolution:fix(control.pa_res), $
                 low_en:0, $
 ;                 average:fix(control.mus), $
 ;                 pitch:fix(control.pa_ctr_3dd), $
 ;                 label:fix(control.pa_lbl_3dd), $
 ;                 have_pitch:0, $
 ;                 overplot:overplot, $
                  base:slice_base, $
                  dw_slice:dw_slice, $
		  lbl_slice1:lbl_slice1, $
                  lbl_slice2:lbl_slice2, $
                  pb_output:pb_output, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}
    
    widget_control, slice_base, /realize
    xmanager, 'USER_Energy_PA_Slice_lanl_mpa_pitch', slice_base, /JUST_REG
    
 ENDIF                          ; end construct slice window
    
 IF keyword_set(NOPLOT) THEN return
 
 WIDGET_CONTROL,/HOURGLASS

;--- now, we need to get the data --------------------------------------------
;    but we will only read the data, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).
 IF slice_widget.lastPanelDrawn NE panelNr THEN begin
    papco_get_slice_data, panelNr
    READDATA=1
 endif else READDATA=0  
  
;--- prepare the drawing widget for plotting ----------------------------------
 if !D.name eq 'X' then begin
    widget_control, slice_widget.dw_slice, get_value=drawindex
    wset, drawindex
 endif  
 
; set the panel window for the slice to full window
 !p.position=[0.1,0.1,0.75,0.95]  
 IF keyword_set(PANEL) THEN panel = panel ELSE panel = [0, 1, 1]
 panelset, panel
 
plotonly:
 
;--- store the current coordinate system --------------------------------------
 old_x = !x  &  old_y = !y  &  old_p = !p
 
; NOW do plot into draw window!!!
; ---------------------------------------------
 
; load the normal colortable
 papco_set_subtable, 0


; restrict data to actual time range requested - this makes for faster zooming
; mpa_pitch data is in TAI. For plotting, always start at zero to get max
; resolution (plot works in float only, large values get rounded! 
  tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
  index = where(mpa_pitch_data.tai GT tai_1) &  index = index(0)

;  tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
;  index=where((mpa_pitch_data.tai ge tai_1) AND (mpa_pitch_data.tai le tai_2) $
           ;   AND mpa_pitch_data.b_flag NE 4  ,c)
;  if c GT 2 then data=mpa_pitch_data(index) ELSE GOTO, no_plot
 
 
; select the index for the time chosen out of the time array
 
;make time of middle of averaged intervals:
 t = (xarr(*, 0)+xarr(*, 1))/2.0
 
 DiffTimes=ABS(t - seconds)     ; this sets timeIndex to the index with
 result=MIN(DiffTimes, timeIndex)  &  timeIndex = TimeIndex+index; time(timeIndex,0) closest to 'seconds'

;check for nearest time found. Needs to be within diff*2
 diff = 84

; should return the correct time.
 time_str=papco_cursor_time(seconds,panelNr)
 time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)
 
 if result gt diff*2 then BEGIN
    message, 'No data found at: '+time_str,/cont
    xl=!p.position(0)  &  xr=!p.position(2)
    yb=!p.position(1)  &  yt=!p.position(3)
    plots, [xl, xr, xr, xl, xl], [yb, yb, yt, yt, yb], COLOR=1, /NORMAL
    goto, no_plot
 endif  

 mpa_chan = 40-low_en
 

 sc_id = control.sc_id
 species = control.species

;IF NOT keyword_set(pa_res) THEN pa_res = 5


 utitle=sat_names(sc_id)


  mpa_energy_info  = r_lanl_mpa_channels(sc_id,species)
  mpa_center_en = reverse((mpa_energy_info(*, 0)+mpa_energy_info(*, 1))/2)

  en_chans = mpa_chan 
  syarr = mpa_energy_info(low_en:mpa_chan+low_en-1, *)

  IF species EQ 0 THEN BEGIN  ; electrons
     mpa_pa = mpa_pitch_data(timeIndex).e_mag(0:mpa_chan-1, *, *)
     mpa_data = mpa_pitch_data(timeIndex).e_unbinned(0:mpa_chan-1, *, *)
  ENDIF ELSE BEGIN  ; ions
     mpa_pa = mpa_pitch_data(timeIndex).p_mag(0:mpa_chan-1, *, *)
     mpa_data = mpa_pitch_data(timeIndex).p_unbinned(0:mpa_chan-1, *, *)
  ENDELSE 

  nodata = -1e31

;; convert mpa data to flux (from energy flux)
;     IF NOT keyword_set(en_pa) then  mpa_data = mpa_data/rebin(reform(mpa_center_en(0:mpa_chan-1), $
;                                                                   mpa_chan, 1, 1), mpa_chan, 6, 24)

  pa_rng = papco_pitch_bin(control.pa_res, pa_idx = pa_idx) 
  pa_bins = n_elements(pa_rng)/2
  pa_en_data = fltarr(en_chans,pa_bins)
;     pa_en_counts = fltarr(en_chans,n_elements(pa_rng))
  mpa_data = reform(mpa_data, mpa_chan,6*24)
  mpa_pa = reform(mpa_pa, mpa_chan,6*24)

  FOR i = mpa_chan-1,0, -1 DO BEGIN
     FOR j = 0, pa_bins-1 DO BEGIN 
        ind = where(mpa_pa(i, *) GT pa_rng(j, 0) AND mpa_pa(i, *) LE pa_rng(j, 1) $
                                                 AND mpa_pa NE nodata, in_box)
        IF in_box GT 0 THEN pa_en_data(i, j) = total(mpa_data(i, ind))/in_box ELSE pa_en_data(i, j) = nodata
     ENDFOR
  ENDFOR
   


;; can also include SOPA energy channels, but need to make sure the
;; right data is read
  IF keyword_set(SOPA) THEN BEGIN  
; find the geo time that is closest to the given (mpa) time
     geo_diff = min(abs(mpa_pitch_data(timeIndex).tai-geo_pitch_data.tai), geo_time)
     
;; get the channel information
     geo_sat_info = r_lanl_geo_flux_converison(sat_id)
     geo_center_en = (geo_sat_info.ek_low+geo_sat_info.ek_high)/2*1000. ; convert to eV
     

;; get the right species data
     IF species EQ 0 THEN geo_chan_ind = where(geo_sat_info.species EQ 'electrons' $
                                               AND geo_sat_info.instrument NE 'ESP', geo_chan) $
     ELSE geo_chan_ind = where(geo_sat_info.species EQ 'ions' $
                               AND geo_sat_info.instrument NE 'ESP', geo_chan) 
                                ; taking away the top two energy channels in sopa
     geo_chan_ind = geo_chan_ind(0:geo_chan-3)
     geo_chan = geo_chan-2

     syarr=[yarr,[[geo_sat_info.ek_low(geo_chan_ind)*1000],[geo_sat_info.ek_high(geo_chan_ind)*1000]]]
     
;; convert geo data to flux (from counts)
     
     flux_conv = fltarr(31) &  flux_conv(geo_chan_ind) = 1
     sopa_calibration, sat_id, flux_conv
     eflux_conv = flux_conv/geo_sat_info.fcf &  eflux_conv = eflux_conv(geo_chan_ind)
     
     geo_data = geo_pitch_data(geo_time).counts(*, *, geo_chan_ind)* $
             rebin(reform(eflux_conv, 1, 1, geo_chan), 3, 32, geo_chan)

     en_chans = mpa_chan+geo_chan 
     geo_pa = reform(geo_pa, 3*32)
     FOR j = 0, pa_bins-1 DO BEGIN
        ind = where(geo_pa(*) GT pa_rng(j, 0) AND geo_pa(*) LT pa_rng(j, 1), in_box)
        IF in_box GT 0 THEN FOR i = 0, geo_chan-1 DO pa_en_data(i, j) = $
          total(geo_data(ind, geo_chan_ind(i)))/in_box
     endFOR 

     pa_en_data = [pa_en_data, geo_pa]
  ENDIF 

     zmat = rotate(pa_en_data, 3)
     sxarr = pa_rng
     xlog = 0 &  ylog = 1
     utitle = utitle
     ztitle = 'Energy flux'
     xtitle = 'Pitch Angle (deg)'
     uytitle = 'Energy (eV)'
     xticks = 5
     xtickv = [0, 45, 90,135,180]
     

     zscl(panelNr,3) = plotdrawn.ZLINLOG


; use the old zscaling if available, else calculate
     IF control.pl_type LT 4 THEN BEGIN 
        zmin=plotdrawn.ZSCL_MIN  
        zmax=plotdrawn.ZSCL_MAX 
     ENDIF  ELSE BEGIN
;do automatic scaling for full dynamic range
        papco_autorange,zmat,zmin,zmax, log=zscl(panelNr,3), exclude=0, nodata=nodata

     ENDELSE  





;yst=plotdrawn.YSCL_MIN  &  yen=plotdrawn.YSCL_MAX
yst = syarr(0, 0) &  yen = syarr(n_elements(syarr(*, 0))-1, 1)
xst = sxarr(0, 0) &  xen = sxarr(n_elements(sxarr(*, 0))-1, 1)
papco_y_label, yst, yen, log=ylog
 
extra_plotPar={xrange:[xst,xen], yrange:[yst,yen], zrange:[zmin,zmax], $
               xlog:xlog, ylog:ylog, zlog:zscl(panelNr,3), $
               ztitle:ztitle, color:0,xstyle:1, ystyle:1, $
               xtickformat:'',ztickformat:'papco_color_bar_log_ticks', $
               xtitle:xtitle, xticks:xticks, xminor:0, xtickv:xtickv, yticks:0,$
                yminor:0, ytickv:0,ytickformat:''}

if keyword_set(NOWIDGET) then begin
    extra_plotPar.xtickformat='noticks'
    extra_plotPar.xtitle=''
    extra_plotPar=create_struct(extra_plotPar, {ytickformat:'noticks'})
endif ELSE erase 
  

; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
  
;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
;for all the possible options of papco_plot_colorspec.
;Hint: you might want to use the resx, resy keywords, which set the
;resolution to plot add. Without these, the routine will try to use
;device resolution, which for postscript may result in HUGE files!
;resx, resy are in data coordinates. If the axis is logarithmic, 
;resx, resy need to be too.
;resx=4000  &  resy=(alog10(yen)-alog10(yst))/n_channels

papco_plot_colorspec, zmat, sxarr, syarr, nodata=nodata, _extra=extra_plotPar  

papco_color_bar, _extra=extra_plotPar


no_plot:



if keyword_set(NOWIDGET) then begin
    if NOWIDGET eq 2 then begin
      panelset, PANEL  ;set to normal panel size!
      ;papco_color_bar, _extra=extra_plotPar
      right_side_label,panel,utitle,/rot90
    ENDIF
    ;restore the coordinate-system
    !x = old_x  &  !y = old_y  &  !p = old_p
    return
endif  
 

; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle,/rot90

  
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
right_side_label,panel,utitle,/rot90
  
; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the papco philosophy then the time axis is in 
; running seconds since the start of the start day.
   
info1='This string not used yet'
info2=string(time_str, yValue, format="('time: ',A20,'   Y-value: ',f9.2)")
   
; these calls put your info strings in the provided labels of the slice widget
WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2 
  
;--- restore the coordinate-system --------------------------------------------
!x = old_x  &  !y = old_y  &  !p = old_p 
  
;--- keep old panel number- used for detecting panel changed of mouse ---------
slice_widget.lastPanelDrawn = panelNr   

END 
;******************************************************************************
;* PROCEDURE:
;*      PRO USER_Energy_PA_Slice_lanl_mpa_pitch_Event, Event
;*
;* DESCRIPTION:
;*	This slice displays a Energy/PA slice through the data, when the user clicks
;*      in the drawing-widget. This procedure handles events for that window.
;*
;* INPUTS:
;*	Event	 	an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     written November 2003, Arne Aasnes
;******************************************************************************
PRO USER_MPA_Energy_PA_Slice_lanl_mpa_pitch_Event, Event

  COMMON USER_Energy_PA_Slice_lanl_mpa_pitch, slice_widget
  COMMON plot_composer, widgetdata
  COMMON slice_calls, slice_calls

  widget_control,event.id,get_uvalue=ev
  
  CASE ev OF 
     'ef_res': BEGIN 
        slice_widget.resolution = Event.value 
;        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
;        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
;        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
;        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue
   end 
     'ef_low_en':BEGIN 
        slice_widget.low_en = Event.value
;        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
;        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
;        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
;        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue
     end 
     'spb_pitch': slice_widget.pitch = event.select
     'spb_label': slice_widget.label = event.select
     'pb_done' :  widget_control, slice_widget.base, /destroy
     'pb_help':   xdisplayfile, papco_helpfile('papco_slice.help'), $
                  title='papco online-help', group=event.top
     'pb_print':  begin         ;loop through slices for all panels if needed
        slice_widget.rsl_lines=0
        for i=0,n_elements(slice_calls)-1 do begin
           close_dev=[i,n_elements(slice_calls)-1]
           papco_printmenu_event,slice_calls(i), CLOSE_DEV=close_dev
        endfor  
     end
     'pb_output': begin
                                ;establish error handler
        if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
        if error_status ne 0 then begin ;call error handler
           papco_error_handler, $
          error_status, proc_call, '', [0,0,0], /NO_ERR_DRAW
           error_status=0 & catch, /CANCEL ;disable error catching
           return
        endif
        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue,/OUTPUT
        catch, /CANCEL          ;disable error catching 
     end  
  else : begin & end
  ENDCASE 
END  
;*----------------------------------------------------------------------
;* 
;*
;*-----------------------------------------------------------------------

PRO USER_MPA_PA_Slice_lanl_mpa_pitch,panelNr,seconds,yValue, $
    READDATA=READDATA, CLOSE=CLOSE, SOPA = sopa, NOWIDGET = nowidget
;  user_mpa_slice, panelNr, seconds, yValue, 'pitchangle', CLOSE=CLOSE
  COMMON time, xut1, xut2
  COMMON get_error, get_err_no, get_err_msg  
  COMMON USER_PA_Slice_lanl_mpa_pitch, slice_widget
  COMMON PLOT_COMPOSER, widgetData
  COMMON papco_color_names
  
; add in here any other common blocks you would need to make your plot data
; available. The common block included here is a dummy for illustration.
; papco uses the the getdatacall you set in papco_add_cluster_rapid to read the
; data you need for the slice. As long as you stay in one panel with the 
; mouse, the data is not read again. When you switch panels, the new data is 
; read. 
  
  COMMON lanl_mpa
  COMMON lanl_mpa_pitch_data, mpa_pitch_header, mpa_pitch_data
  COMMON lanl_mpa_pitch_slice, utitle, uytitle, uztit
  COMMON lanl_mpa_pitch_output,xarr, yarr, zmat
  COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
  COMMON yscale, yscl           ;man/auto yscaling
  COMMON zscale, zscl           ;man/auto zscaling
  
; plotinfo contains the full papco structure of plot information for
; the panel in which the mouse was clicked in. 
plotinfo =widgetData.plotinfos(panelNr)
plotdrawn=widgetData.plotsDrawn(panelNr)
control = *plotinfo.USR_PTR1


;if keyword_set(OVERPLOT) then overplot=OVERPLOT else overplot=0
IF NOT keyword_set(low_en) THEN low_en = 13

if keyword_set(NOWIDGET) then BEGIN
   slice_widget={resolution:fix(control.pa_res),$
                 low_en:0, $
;                  average:fix(control.mus), $
;                  pitch:fix(control.pa_ctr_3dd), $
;                  label:fix(control.pa_lbl_3dd), $
;                  have_pitch:0, $
                  overplot:overplot, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}    
    goto, plotonly
ENDIF

; see if slice window needs to be closed --------------------------------------
if keyword_set(CLOSE) then begin
    if xregistered('USER_MPA_PA_Slice_lanl_mpa_pitch') then $
      if n_elements(slice_widget) gt 0 then begin
	if slice_widget.base ne 0 then $
          widget_control, slice_widget.base, /destroy
        slice_widget.base=0
      endif
    return
ENDIF 

; if slice window exists, bring it in front. ELSE, create it ------------------
if not xregistered('USER_MPA_PA_Slice_lanl_mpa_pitch') then begin
    
    slice_base=widget_base(column=1, group=widgetdata.main_base, $
                           title='PA_Slice_lanl_mpa_pitch')
    
;-- setup draw area -----------------------------------------------------------
    draw_base=WIDGET_BASE(Slice_base, COLUMN=1, /FRAME)
    dw_Slice  =WIDGET_DRAW(draw_base, XSIZE=600, YSIZE=500)

;-- set up label fields for user data -----------------------------------------
    user_label_base=WIDGET_BASE(slice_base, COLUMN=1, /FRAME)
    lbl_slice1=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
    lbl_slice2=WIDGET_LABEL(user_label_base,VALUE=strcenter('uservalues',80))
   
;-- set up some additional options for this slice -----------------------------
    Opt_base = WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    ef_res = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
                     TITLE = 'Pitch angle resolution', $
                     VALUE = control.pa_res, $
                     UVALUE='ef_res')

;    ef_low_en = CW_FIELD(Opt_base, /INTEGER, /RETURN_EVENTS, XSIZE = 4, $
;                     TITLE = 'Lowest energy channel:', $
;                     VALUE = low_en, $
;                     UVALUE='ef_low_en')
;

    
;-- set up buttons for the Actions --------------------------------------------
    Butt_Base=WIDGET_BASE(Slice_Base, ROW=1, /FRAME)
    pb_help=WIDGET_BUTTON(Butt_Base, Value=strcenter('Help', 12), $
                          UVALUE='pb_help')
    
    pb_print=WIDGET_BUTTON(Butt_Base, Value=strcenter('Print', 12), $
                           UVALUE='pb_print')
    pb_output=WIDGET_BUTTON(Butt_Base, Value=strcenter('Output', 12), $
                           UVALUE='pb_output')    

    pb_done=WIDGET_BUTTON(Butt_Base, Value=strcenter('Close', 12), $
                          UVALUE='pb_done')

    slice_widget={rsl_lines:0, $ 
                 resolution:fix(control.pa_res), $
                 low_en:0, $
                  base:slice_base, $
                  dw_slice:dw_slice, $
		  lbl_slice1:lbl_slice1, $
                  lbl_slice2:lbl_slice2, $
                  pb_output:pb_output, $
                  slice_call:'', $
	          panelNr:panelNr, $
                  lastPanelDrawn:-1}
    
    widget_control, slice_base, /realize
    xmanager, 'USER_MPA_PA_Slice_lanl_mpa_pitch', slice_base, /JUST_REG
    
 ENDIF                          ; end construct slice window
    
 IF keyword_set(NOPLOT) THEN return
 
 WIDGET_CONTROL,/HOURGLASS

;--- now, we need to get the data --------------------------------------------
;    but we will only read the data, if the panel the user clicked is not the
;    panel that he last clicked into (in this case, the data is already read).
 IF slice_widget.lastPanelDrawn NE panelNr THEN begin
    papco_get_slice_data, panelNr
    READDATA=1
 endif else READDATA=0  
  
;--- prepare the drawing widget for plotting ----------------------------------
 if !D.name eq 'X' then begin
    widget_control, slice_widget.dw_slice, get_value=drawindex
    wset, drawindex
 endif  
 
; set the panel window for the slice to full window
 !p.position=[0.1,0.1,0.75,0.95]  
 IF keyword_set(PANEL) THEN panel = panel ELSE panel = [0, 1, 1]
 panelset, panel
 
plotonly:
 
;--- store the current coordinate system --------------------------------------
 old_x = !x  &  old_y = !y  &  old_p = !p
 
; NOW do plot into draw window!!!
; ---------------------------------------------
 
 

; restrict data to actual time range requested - this makes for faster zooming
; mpa_pitch data is in TAI. For plotting, always start at zero to get max
; resolution (plot works in float only, large values get rounded! 
  tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
;  tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})
  index=where(mpa_pitch_data.tai ge tai_1) & index = index(0) 
;AND (mpa_pitch_data.tai le tai_2) $
           ;   AND mpa_pitch_data.b_flag NE 4 ,c)
;  if c GT 2 then data=mpa_pitch_data(index) ELSE GOTO, no_plot
 
; select the index for the time chosen out of the time array
 
;make time of middle of averaged intervals:
 t = (xarr(*, 0)+xarr(*, 1))/2.0
 
 DiffTimes=ABS(t - seconds)     ; this sets timeIndex to the index with
 result=MIN(DiffTimes, timeIndex)  &  timeIndex = TimeIndex+index; time(timeIndex,0) closest to 'seconds'

;check for nearest time found. Needs to be within diff*2
 diff = 84

; should return the correct time.
 time_str=papco_cursor_time(seconds,panelNr)
 time_mjdt=papco_cursor_time(seconds,panelNr,/mjdt)
 if result gt diff*2 then BEGIN
    message, 'No data found at: '+time_str,/cont
    xl=!p.position(0)  &  xr=!p.position(2)
    yb=!p.position(1)  &  yt=!p.position(3)
    plots, [xl, xr, xr, xl, xl], [yb, yb, yt, yt, yb], COLOR=1, /NORMAL
    goto, no_plot
 ENDIF


; select the index for the energy chosen out of the time array

 en_ind = where(yValue GT reverse(yarr(*, 0)) AND yValue LT reverse(yarr(*, 1)))
 

 sc_id = control.sc_id
 species = control.species

;IF NOT keyword_set(pa_res) THEN pa_res = 5



  mpa_energy_info  = r_lanl_mpa_channels(sc_id,species)
  mpa_center_en = reverse((mpa_energy_info(*, 0)+mpa_energy_info(*, 1))/2)

  IF species EQ 0 THEN BEGIN  ; electrons
     species = 'electrons'
     mpa_pa = mpa_pitch_data(timeIndex).e_mag(en_ind, *, *)
     mpa_data = mpa_pitch_data(timeIndex).e_unbinned(en_ind, *, *)
  ENDIF ELSE BEGIN  ; ions
     species = 'ions'
     mpa_pa = mpa_pitch_data(timeIndex).p_mag(en_ind, *, *)
     mpa_data = mpa_pitch_data(timeIndex).p_unbinned(en_ind, *, *)
  ENDELSE 


  nodata = -1e31

  IF max(mpa_data) EQ !values.f_infinity THEN mpa_data(where(mpa_data EQ !values.f_infinity)) = nodata


  pa_rng = papco_pitch_bin(control.pa_res, pa_idx = pa_idx) 
  pa_bins = n_elements(pa_rng)/2
  pa_data = fltarr(pa_bins)

  mpa_data = reform(mpa_data,6*24)
  mpa_pa = reform(mpa_pa, 6*24)

;-----bin data in pitch angle------------------------------------------------------

     FOR j = 0, pa_bins-1 DO BEGIN 
        ind = where(mpa_pa(*) GT pa_rng(j, 0) AND mpa_pa(*) LE pa_rng(j, 1) $
                                                 AND mpa_pa NE nodata AND mpa_data NE nodata, in_box)
        IF in_box GT 0 THEN pa_data(j) = total(mpa_data(ind))/in_box ELSE pa_data(j) = nodata
     ENDFOR
     sxarr = pa_rng
     xlog = 0 &  ylog = 1
     
     lowen = strtrim(string(mpa_energy_info(40-en_ind, 0)), 2) &  lowen = lowen(0)
     highen = strtrim(string(mpa_energy_info(40-en_ind, 1)), 2) & highen = highen(0)

     utitle=sat_names(sc_id)+' '+species+'!C'+lowen+'-'+ $
               highen+' eV'
     ztitle = ''
     xtitle = 'Pitch Angle (deg)'
     ytitle = 'Energy flux'
     xticks = 5
     xtickv = [0, 45, 90,135,180]
     


;     zscl(panelNr,3) = plotdrawn.ZLINLOG


; use the old zscaling if available, else calculate
     IF control.pl_type LT 4 THEN BEGIN 
        yst=plotdrawn.ZSCL_MIN  
        yen=plotdrawn.ZSCL_MAX 
     ENDIF  ELSE BEGIN
;do automatic scaling for full dynamic range
        papco_autorange,pa_data,yst,yen, log=plotdrawn.ZLINLOG, exclude=0, nodata=nodata

     ENDELSE  





;yst=plotdrawn.YSCL_MIN  &  yen=plotdrawn.YSCL_MAX
;yst = syarr(0, 0) &  yen = syarr(n_elements(syarr(*, 0))-1, 1)
xst = sxarr(0, 0) &  xen = sxarr(n_elements(sxarr(*, 0))-1, 1)
papco_y_label, yst, yen, log=ylog
 
extra_plotPar={xrange:[xst,xen], yrange:[yst,yen],  $
               xlog:xlog,ylog:plotdrawn.ZLINLOG, $
               color:black,xstyle:1, ystyle:1, $
               xtickformat:'',xtitle:xtitle,xticklen:0.01,$
               xticks:xticks, xminor:0, xtickv:xtickv, yticks:0,$
               yminor:0, ytickv:0,ytickformat:'', ytitle:ytitle}

if keyword_set(NOWIDGET) then begin
    extra_plotPar.xtickformat='noticks'
    extra_plotPar.xtitle=''
    extra_plotPar=create_struct(extra_plotPar, {ytickformat:'noticks'})
endif ELSE erase 
  

; nodata specifies the "nodata" flag
; in your data.

userx=[-0.5,0.0,0.5] & usery=[0.0,1.0,0.0] & usersym,userx,usery,/fill
sxarr = (sxarr(*, 0)+sxarr(*, 1))/2.

PLOT, sxarr, pa_data, ytitle=uztitle, title = title, $
   psym=4,  symsize = 0.75, _extra = extra_plotPar

;papco_plot_colorspec, zmat, sxarr, syarr, nodata=nodata, _extra=extra_plotPar  

;papco_color_bar, _extra=extra_plotPar


no_plot:



if keyword_set(NOWIDGET) then begin
    if NOWIDGET eq 2 then begin
      panelset, PANEL  ;set to normal panel size!
      ;papco_color_bar, _extra=extra_plotPar
      right_side_label,panel,utitle,/rot90
    ENDIF
    ;restore the coordinate-system
    !x = old_x  &  !y = old_y  &  !p = old_p
    return
endif  
 

; plot y-axis label at left of plot. Use scalable routine!  
;left_side_label,panel,uytitle,/rot90

  
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
right_side_label,panel,utitle,/rot90
  
; these two info strings are user-definable output strings to label the slice 
; window. To calculate the time use the start time in mjdt and the seconds 
; passed in. If your plot uses the papco philosophy then the time axis is in 
; running seconds since the start of the start day.
   
info1='This string not used yet'
info2=string(time_str, yValue, format="('time: ',A20,'   Y-value: ',f9.2)")
   
; these calls put your info strings in the provided labels of the slice widget
WIDGET_CONTROL, slice_widget.lbl_slice1, SET_VALUE=info1
WIDGET_CONTROL, slice_widget.lbl_slice2, SET_VALUE=info2 
  
;--- restore the coordinate-system --------------------------------------------
!x = old_x  &  !y = old_y  &  !p = old_p 
  
;--- keep old panel number- used for detecting panel changed of mouse ---------
slice_widget.lastPanelDrawn = panelNr   

END 
;******************************************************************************
;* PROCEDURE:
;*      PRO USER_PA_Slice_lanl_mpa_pitch_Event, Event
;*
;* DESCRIPTION:
;*	This slice displays a PA slice through the data, when the user clicks
;*      in the drawing-widget. This procedure handles events for that window.
;*
;* INPUTS:
;*	Event	 	an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     written November 2003, Arne Aasnes
;******************************************************************************
PRO USER_MPA_PA_Slice_lanl_mpa_pitch_Event, Event

  COMMON USER_PA_Slice_lanl_mpa_pitch, slice_widget
  COMMON plot_composer, widgetdata
  COMMON slice_calls, slice_calls

  widget_control,event.id,get_uvalue=ev
  
  CASE ev OF 
     'ef_res': BEGIN 
        slice_widget.resolution = Event.value 
;        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
;        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
;        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
;        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue
   end 
;     'ef_low_en':BEGIN 
;        slice_widget.low_en = Event.value
;        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
;        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
;        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
;        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue
;     end 
     'spb_pitch': slice_widget.pitch = event.select
     'spb_label': slice_widget.label = event.select
     'pb_done' :  widget_control, slice_widget.base, /destroy
     'pb_help':   xdisplayfile, papco_helpfile('papco_slice.help'), $
                  title='papco online-help', group=event.top
     'pb_print':  begin         ;loop through slices for all panels if needed
        slice_widget.rsl_lines=0
        for i=0,n_elements(slice_calls)-1 do begin
           close_dev=[i,n_elements(slice_calls)-1]
           papco_printmenu_event,slice_calls(i), CLOSE_DEV=close_dev
        endfor  
     end
     'pb_output': begin
                                ;establish error handler
        if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
        if error_status ne 0 then begin ;call error handler
           papco_error_handler, $
          error_status, proc_call, '', [0,0,0], /NO_ERR_DRAW
           error_status=0 & catch, /CANCEL ;disable error catching
           return
        endif
        tmp_str=slice_widget.slice_call  &  tmp_str=str_sep(tmp_str,',')
        proc_call=tmp_str(0)  &  panelNr=fix(tmp_str(1))
        seconds=float(tmp_str(2))  &  yValue=float(tmp_str(3))
        CALL_PROCEDURE,proc_call,panelNr,seconds,yValue,/OUTPUT
        catch, /CANCEL          ;disable error catching 
     end  
  else : begin & end
  ENDCASE 
END  
