;******************************************************************************
;* file papcoadd_lanl_mpa
;*      papcoadd_lanl_mpa_line
;*      papcoadd_lanl_mpa_spec
;*      papcoadd_lanl_mpa_lt
;*      papcoadd_lanl_mpa_pitch
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for a mpa. 
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_lanl_mpa,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*      This procedure adds a button for this mpa to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*      package, so that they can be automatically called when needed.
;*      The button itself can contain access to a sub-menu for further
;*      mpas.
;*
;* INPUTS:       
;*      plottypes_base  the widget ID of the parent widget to contain the
;*                      plottype buttons.
;*
;* OUTPUTS: 
;*      pb_plottype     the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for sub-types.
;*
;* KEYWORDS:
;*       none
;* 
;* CALLING SEQUENCE:
;*      addplottype_lanl_mpa,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified  May 2003, Arne Aasnes at  LANL
;*           added option for pitch angle plots
;******************************************************************************
pro addplottype_lanl_mpa, plottypes_base, buttonGeometry, pdm_lanl_mpa

MenuDesc = [ $
        { CW_PDMENU_S, 3, strcenter('LANL-MPA',10) }, $ 
        { CW_PDMENU_S, 0, ' configure mpa     ' }, $
        { CW_PDMENU_S, 0, ' mpa line plot     ' }, $
        { CW_PDMENU_S, 0, ' mpa spectrum plot ' }, $
        { CW_PDMENU_S, 0, ' mpa LT - UT         ' }, $
        { CW_PDMENU_S, 2, ' mpa pitch angle plot ' }]
  
pdm_lanl_mpa = CW_PDMENU(plottypes_base, MenuDesc, UVALUE='pdm_lanl_mpa')

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_lanl_mpa
;*
;* DESCRIPTION:  
;*      Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      Using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled. The
;*      desired default values for the plotinfo tags need to be edited by
;*      the user. This plotinfo structure will be passed to the panel editor
;*      When it is first called for the mpa.
;*      "panelEditor" must be set to the name of the panel editor routine.
;*
;* INPUTS:       
;*      event           event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;* OUTPUTS: 
;*      none
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      addpanel_lanl_mpa
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;***************************************************************************
pro addpanel_lanl_mpa,event
  
common get_error, get_err_no, get_err_msg
  
; The case statement here needs an entry for every submenu type you
; added in  the first procedure in this file (addplottype_lanl_mpa).
; For each submenu entry, you will need to specify the panelEditor to use.  
  
; The FIRST time you use this module you will have to configure it,
; i.e. set the data paths etc. Once you have done this PAPCO will
; write a mpa.config file into your papco_user_XX/papco_modules_defaults
; directory. 
; So here we check if this mpa.config file exists, and if not,
; we diallow the execution of any other option apart from "configure
; mpa". In this way the user is forced to configure the module
; before using it the first time!
  
r_module_config, strlowcase('lanl_mpa'), config_data, msg
if (get_err_no ne 0) and (event.value ne 1) then begin
    message,'You need to configure the module first', /cont
    papco_module_configure,'lanl_mpa'
    return
endif  
 
CASE event.value of
    1: BEGIN
        papco_module_configure,'lanl_mpa'
    end
    2: BEGIN
        panelKind='lanl_mpa_line'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    3: BEGIN
        panelKind='lanl_mpa_spec'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
    end
    4: BEGIN
        panelKind='lanl_mpa_lt'
        panelEditor=panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults     
     END
     5: BEGIN 
        panelKind = 'lanl_mpa_pitch'
        panelEditor = panelkind+'_panel'
        defaults=papco_getPlotInfoStruct() ;returns a basic plotinfo structure
;        defaults.typeVector=[0,0,0,0]      ;set some tags to desired defaults
;        defaults.ioptions = [0, 0, 0]
        papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
     END
    else: begin
      message,'Nothing defined for this submenu entry', /cont
      return
    end  
ENDCASE
      
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO plottable_lanl_mpa, plottable
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate plottable with / without orbit.
;*      PAPCO provides "set time by orbit" feature for CRRES and POLAR
;*      so far. Included for compatibility with older CRRES modules
;*      which ONLY use orbit as input to their read/plot routines
;*      (plotting over orbit boundaries is not enabled for CRRES)
;* 
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "plottable_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_lanl_mpa.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      plottable        a boolean 0 if plottable only by orbit
;*                                 1 if plottable by time also.
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      plottable_lanl_mpa, plottable
;*
;* MODIFICATION HISTORY:       
;*      written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_lanl_mpa_line, plottable

plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
pro plottable_lanl_mpa_spec, plottable

plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END

;******************************************************************************
pro plottable_lanl_mpa_lt, plottable

plottable=1     ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END
;******************************************************************************
PRO plottable_lanl_mpa_pitch, plottable

plottable = 1   ; 1 -> plotable without orbit info, by time 
                ; 0 -> plotable only by orbit number not by time 
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO canget_lanl_mpa, canget
;*
;* DESCRIPTION:  
;*      Sets a variable to indicate that mpa is capable of
;*      returning data from it's plot routine into a standard PAPCO
;*      array structure. Used for the scratch pad utility module.
;*      
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "canget_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_lanl_mpa.
;*
;* INPUTS:       
;*      none
;*
;* OUTPUTS: 
;*      canget         a boolean 0 if this feature is not available
;*                                1 if this feature is available
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      canget_lanl_mpa, canget
;*
;* MODIFICATION HISTORY:       
;*      written November 1997, Reiner Friedel
;******************************************************************************
pro canget_lanl_mpa_line, canget

canget=  1      ; 1 -> plot routine can return just data to file
                ; 0 -> feature disabled
END

;******************************************************************************
pro canget_lanl_mpa_spec, canget

canget=  1      ; 1 -> plot routine can return just data to file
                ; 0 -> feature disabled
END

;******************************************************************************
pro canget_lanl_mpa_lt, canget

canget=  1      ; 1 -> plot routine can return just data to file
                ; 0 -> feature disabled
END
;******************************************************************************
pro canget_lanl_mpa_pitch, canget

canget=  1      ; 1 -> plot routine can return just data to file
                ; 0 -> feature disabled
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_lanl_mpa, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*      Constructs the string used to label plots in the panel window.
;*      The plotinfo structure passed in here is the panel's plotinfo
;*      as returned by the panel editor for this mpa. You can use
;*      the information in that structure to construct some informative
;*      label for the panel which then appears in the Panel Compose list.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "descriptor_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_lanl_mpa.
;*
;* INPUTS:       
;*      plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*      descriptStr     a string containing the label
;*
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      descriptor_lanl_mpa, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     August 1998 written by Haje Korth
;*     May 2003, modified by Arne Aasnes at LANL, added lanl_mpa_pitch 
;******************************************************************************
pro descriptor_lanl_mpa_line, plotInfo, descriptStr

COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents

descriptStr=descriptStr+'Line Plot, '+sat_names(PlotInfo.typeVector(0))

CASE PlotInfo.typeVector(1) OF
     0:descriptStr=descriptStr+'Av. El. Flux, Channel: '+ $
                      strtrim(string(plotInfo.typevector(2)), 2)+')'
     1:descriptStr=descriptStr+'Av. Ion Flux, Channel: '+ $
                      strtrim(string(plotInfo.typevector(2)), 2)+')'
     2:descriptStr=descriptStr+'Background)'
     3:descriptStr=descriptStr+'S/C Potential)'
     4:descriptStr=descriptStr+'X Geo)'
     5:descriptStr=descriptStr+'Y Geo)'
     6:descriptStr=descriptStr+'Z Geo)'
     7:descriptStr=descriptStr+'S/C Mag. Lat.)'
     8:descriptStr=descriptStr+'S/C Mag. LT)'
     9:descriptStr=descriptStr+'Density LP)'
     10:descriptStr=descriptStr+'Velocity LP)'
     11:descriptStr=descriptStr+'Density HP)'
     12:descriptStr=descriptStr+'T-Par. HP)'
     13:descriptStr=descriptStr+'T-Perp. HP)'
     14:descriptStr=descriptStr+'Theta HP)'
     15:descriptStr=descriptStr+'Phi HP)'
     16:descriptStr=descriptStr+'Density HE)'
     17:descriptStr=descriptStr+'T-Par. HE)'
     18:descriptStr=descriptStr+'T-Perp. HE)'
     19:descriptStr=descriptStr+'Theta HE)'
     20:descriptStr=descriptStr+'Phi HE)'
     21:descriptStr=descriptStr+'Pressure HP)'
ENDCASE

END

pro descriptor_lanl_mpa_spec, plotInfo, descriptStr

COMMON lanl_mpa  ;see lanl_mpa_set_commons for contents

descriptStr=descriptStr+'Spectrum Plot, '+sat_names(PlotInfo.typeVector(0))

CASE PlotInfo.typeVector(1) OF
     0:descriptStr=descriptStr+', Elec, '
     1:descriptStr=descriptStr+', Ions, '
ENDCASE

CASE PlotInfo.typeVector(2) OF
     0:descriptStr=descriptStr+'Energy Flux'
     1:descriptStr=descriptStr+'Diff. Flux'
     2:descriptStr=descriptStr+'Ph. Sp. Dens.'
ENDCASE

END

;******************************************************************************
pro descriptor_lanl_mpa_lt, plotInfo, descriptStr

descriptStr=descriptStr+'mpa LT plot'

END
;******************************************************************************
pro descriptor_lanl_mpa_pitch, plotInfo, descriptStr

COMMON lanl_mpa
  control = *plotinfo.USR_PTR1
  descriptStr=descriptStr+'Pitch Angle Plot '+sat_names(control.sc_id)
  CASE control.species OF
     0:descriptStr=descriptStr+', Elec, '
     1:descriptStr=descriptStr+', Ions, '
  ENDCASE

  CASE control.pl_type OF
     0:descriptStr=descriptStr+'All PA @ 1 E'
     1:descriptStr=descriptStr+'All E @ 1 PA '
     2:descriptStr=descriptStr+'All PA for E range'
     3:descriptStr=descriptStr+'All E for PA range'
     4:descriptStr=descriptStr+'Anisotropy spect.'
     5:descriptStr=descriptStr+'PA range @ E range'
     6:descriptStr=descriptStr+'Anisotropy line pl.'
     7:descriptStr = descriptStr+'MF Direction'
  ENDCASE
  
END
;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_lanl_mpa, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*      This procedure performs the actual data read call and the plot call
;*      for this plottype. The user need to pass to his plot routine all the
;*      parameters needed, typically these parameters are stored in the 
;*      currplotinfo structure or constructed from them. The currplotinfo
;*      structure tags normally used for passing information from the panel
;*      editor to the plot routine are:
;*          type_vector - a four element array for choosing the type of data
;*                        (eg. elec or ion, diff or integ., pitch angle, 
;*                             or sector, channel ....)
;*          switch      - a three element vector for specific plot appearance
;*                        (eg. smoothing/averaging, interpolating ...)
;*          filename    - a filename
;*      
;*      These tags have grown out of traditional usage and can be used for
;*      anything the user wants to use them for. There are plenty variables
;*      for most applications: Adding a new tag to the plotinfo structure
;*      is not user-definable but need to be done at the papco core level.
;* 
;*      The panelVector MUST be passed to your plotroutine as it is used
;*      to set the panel position.
;*      The _EXTRA=extra_PlotInfo should be passed if you intend to make use
;*      of papco functionality such as overplotting, linestyle choice and
;*      color bar choice.
;*
;* NB!  You will need one of these routines for each submenu entry you
;*      add. The name of the routine MUST consist of "draw_" plus
;*      the panel_kind you have set for your submenu entry in 
;*      addpanel_lanl_mpa.
;*
;* INPUTS:       
;*      panelVector    position of the plot (see papco_conventions.text, 6)
;*      currPlotInfo   structure containing all data for this plot.
;*                     (structure papco_PLOTINFO, see papco_variables.text) 
;*
;* OUTPUTS: 
;*      get_data_call  a string containing the data call. Used by other
;*                     routines that need to know how to get the data
;*                     in a panel, e.g. to draw spectra, distribution functions
;*      IsRead         a boolean which is set when the the data is read the
;*                     first time. Useful if a given data-read puts data
;*                     into commons which can be plotted in many ways.
;*                     When set, data does not have to be read again.
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 switches. Currently
;                 there are 5 papco slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = papco Zoom                 (2^0)
;                     2 = papco View Slice           (2^1)
;                     4 = papco Access level Zero    (2^2)
;                     8 = papco Write Paneldata      (2^3)
;                    16 = papco Hair Cursor          (2^4)
;                    32 = USER slice A or not used   (2^5)
;                    64 = USER slice B or not used   (2^6)
;                    "     "     "   "  "  "    "      "    
;                 32768 = USER slice Z or not used  (2^15)
;                        
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero
;*
;* KEYWORDS:
;*      OUTPUT        if set then plot routine should output data to file,
;*                    as well as to screen.
;*
;* 
;* CALLING SEQUENCE:
;*      draw_lanl_mpa, panelVector, currPlotInfo, orbitNo, $
;*                      get_data_call, slice_type, IsRead, NODATA=NODATA
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;*     modified Augist 1998, Haje Korth
;******************************************************************************
pro draw_lanl_mpa_line, panelVector, currPlotInfo, orbitNo, $
                        get_data_call, slice_type, IsRead, $
                        OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
COMMON MPA_MEMORY, bird_memory

bird_mem_def=SIZE(bird_memory)
bird_mem_def=bird_mem_def(1)
IF (bird_mem_def EQ 0) THEN bird_memory=-1
    
get_data_call="r_lanl_mpa"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
bird=currPlotInfo.typeVector(0)
dataType=currPlotInfo.typeVector(1)
channel=currPlotInfo.typeVector(2)
  
; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
  slice_assigned_value=papco_slice_value('USER Defined Slice A')
  
; set up which mouse functions to allow
slice_type=1+8+16

IF NOT IsRead OR (bird NE bird_memory) THEN BEGIN  ;check if data read
    r_lanl_mpa,currPlotInfo                ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
ENDIF
  
IF IsRead THEN BEGIN
    p_lanl_mpa_line, panelVector, bird, dataType, channel, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo 
ENDIF

; Note: The structure extra_PlotInfo passed to the plot routine contains the
;       following tags: 
;                  overPlot               - overplot switch
;                  plots_AtThisPosition   - no. of overplots here
;                  subtable               - index of papco colortable 
;                  PSYM                   - linestyle keyword for plot
;                  COLOR                  - linecolor keyword for plot
;               
;       They are set by papco and your plot routine can use them if you
;       want to support that functionality.

bird_memory=bird

END

;******************************************************************************
pro draw_lanl_mpa_spec, panelVector, currPlotInfo, orbitNo, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
COMMON MPA_MEMORY, bird_memory

bird_mem_def=SIZE(bird_memory)
bird_mem_def=bird_mem_def(1)
IF (bird_mem_def EQ 0) THEN bird_memory=-1
    
get_data_call="r_lanl_mpa"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
bird=currPlotInfo.typeVector(0)
dataType=currPlotInfo.typeVector(1)
option=currPlotInfo.typeVector(2)
scPot=currPlotInfo.typeVector(3)
  
; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
  
user_slice_1 = PAPCO_slice_value('USER MPA Energy Flux Slice')
user_slice_2 = PAPCO_slice_value('USER MPA Diff Flux Slice')
user_slice_3 = PAPCO_slice_value('USER MPA Phase Space Density Slice')
  
; set up which mouse functions to allow
slice_type=1+2+8+16+user_slice_1+user_slice_2+user_slice_3

IF NOT IsRead OR (bird NE bird_memory) THEN BEGIN      ; check if data read
    r_lanl_mpa,currPlotInfo                ; read data
    ; check the read for success.
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel)       THEN IsRead=1
ENDIF
  
IF IsRead THEN BEGIN
    p_lanl_mpa_spec,panelVector,bird, dataType, option, scPot, OUTPUT=OUTPUT, $
      _EXTRA=extra_PlotInfo 
ENDIF

bird_memory=bird

END


;******************************************************************************
pro draw_lanl_mpa_lt, panelVector, currPlotInfo, orbitNo, $
                      get_data_call, slice_type, IsRead, $
                      OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
COMMON MPA_MEMORY, bird_memory

  bird_mem_def=SIZE(bird_memory)
  bird_mem_def=bird_mem_def(1)
  IF (bird_mem_def EQ 0) THEN bird_memory=-1
    
  get_data_call="r_lanl_mpa"   ; data read call used by slices

; set parameter for plot routine. In this example, channel 1 -10 chooses the 
; channel number for a line plot, channel=0 chooses a spectrum color plot.
  type=currPlotInfo.typeVector
  birdbits=type(1)

; Read event file
  IF (currPlotInfo.filename NE '') THEN BEGIN
    eventPath=papco_getenv('MPA_EVENT_DATA')
    eventFile=eventPath+currPlotInfo.filename
    eventStr=''
    eventTable=STRARR(4)

    OPENR, unit, eventFile, /GET_LUN

    WHILE NOT EOF(unit) DO BEGIN
      READF,unit,eventStr
      event=STR_SEP(eventStr,' ')
      IF (N_ELEMENTS(event) EQ 3) THEN event=[event,'']
      eventTable=[[eventTable],[event]]
    ENDWHILE

    CLOSE, unit
    FREE_LUN, unit
    
    eventNum=SIZE(eventTable)
    eventNum=eventNum(2)-1
    eventTable=eventTable(*,1:eventNum)

    time={MJDT, MJD:0l, T:0.0}
    dummy={mpa_event_times,satid:'',dom:0,month:0,year:0, $
                           hour:0,minute:0,lt_mod:'',mjdt_time:time}
    mpaEventStruct=REPLICATE(dummy,eventNum)

    FOR i=0,eventNum-1 DO BEGIN
      mpaEventStruct(i).satid=eventTable(0,i)
      date=STR_SEP(eventTable(1,i),'/')
      dom=FIX(date(1))
      month=FIX(date(0))
      year=FIX(date(2))
      mpaEventStruct(i).month=month
      mpaEventStruct(i).dom=dom
      mpaEventStruct(i).year=year
      time=STR_SEP(eventTable(2,i),':')
      hour=FIX(time(0))
      minute=FIX(time(1))
      mpaEventStruct(i).hour=hour
      mpaEventStruct(i).minute=minute
      mpaEventStruct(i).mjdt_time=convert_date_to_t90(day=dom,month=month,year=year, $
                                                      hour=hour,minutes=minute,/MJDT)
      mpaEventStruct(i).lt_mod=eventTable(3,i)
    ENDFOR

    plotEvents=1
  ENDIF ELSE BEGIN
    mpaEventStruct=0
    plotEvents=0
  ENDELSE
  
; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
  
; set up which mouse functions to allow
  slice_type=1+8+16

  plotnum=1
  FOR ibit= 0,15 DO BEGIN
    bird=birdbits AND LONG(2)^ibit
    yesdraw=bird EQ  LONG(2)^ibit

    IF yesdraw THEN BEGIN

      CASE bird OF
        1: currPlotInfo.typeVector(0)=0
        2: currPlotInfo.typeVector(0)=1
        4: currPlotInfo.typeVector(0)=2
        8: currPlotInfo.typeVector(0)=3
       16: currPlotInfo.typeVector(0)=4
      ENDCASE
      bird=currPlotInfo.typeVector(0)

      IF NOT IsRead OR (bird NE bird_memory) THEN BEGIN      ; check if data has already been read
        r_lanl_mpa,currPlotInfo                ; read data
        ; check the read for success.
        IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                                  panelVector, orbitNr, $
                                  currPlotInfo.typeVector, $
                                  currPlotInfo.channel)       THEN IsRead=1 $
        ELSE IsRead=0
      ENDIF

      IF IsRead THEN BEGIN
         p_lanl_mpa_lt, panelVector, bird, plotnum, plotEvents, mpaEventStruct, $
                   OUTPUT=OUTPUT, _EXTRA=extra_PlotInfo 
      ENDIF

      bird_memory=bird
      plotnum=plotnum+1
    ENDIF
  ENDFOR

END

;******************************************************************************
pro draw_lanl_mpa_pitch, panelVector, currPlotInfo, orbitNo, $
                    get_data_call, slice_type, IsRead, $
                    OUTPUT=OUTPUT

COMMON extra, extra_PlotInfo  ; passes in papco-set keywords structure
COMMON MPA_PA_MEMORY, bird_pa_memory, mag_corr_memory

IF n_elements(bird_pa_memory) EQ 0 THEN bird_pa_memory=-1
IF n_elements(mag_corr_memory) EQ 0 THEN mag_corr_memory = -1

control = *currPlotInfo.USR_PTR1
bird = control.sc_id
mag_corr = control.mag_corr
    
get_data_call='r_lanl_mpa_pitch'   ; data read call used by slices
plot_call = 'p_lanl_mpa_pitch'
  
; as papco dynamically allocates slice values for the user defined
; slices, one needs to be able to find out this assignment! Use the
; following function to find out the slice values for any slices you
; might have added for your module, so that you know which values to enable!
user_slice_1 = PAPCO_slice_value('USER Energy PA Slice')
user_slice_2 = PAPCO_SLICE_value('USER PA Slice')
  
; set up which mouse functions to allow
slice_type=1+2+8+16+user_slice_1+user_slice_2

IF NOT IsRead OR (bird NE bird_pa_memory)OR (mag_corr NE mag_corr_memory) THEN BEGIN      ; check if data read
    call_procedure, get_data_call, currPlotInfo                ; read data

    ; check the read for success.This draws tha papco error box in the
    ; plot if needed. Inhibit if OUTPUT is set.    
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel, $
                              OUTPUT = OUTPUT)  THEN IsRead=1 ELSE IsRead = 0
ENDIF
 
IF IsRead THEN BEGIN
    call_procedure, plot_call,panelVector, currPlotInfo, OUTPUT=OUTPUT, $
              _EXTRA=extra_PlotInfo 
ENDIF

bird_pa_memory=bird
mag_corr_memory = mag_corr

END
